/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.inapp.tasks;

import android.app.Activity;
import android.app.AlertDialog;
import android.content.Context;
import android.content.DialogInterface;
import android.view.View;
import com.moengage.core.Logger;
import com.moengage.core.MoEUtils;
import com.moengage.core.executor.SDKTask;
import com.moengage.core.executor.TaskResult;
import com.moengage.inapp.InAppConstants;
import com.moengage.inapp.InAppController;
import com.moengage.inapp.InAppInjector;
import com.moengage.inapp.InAppUtils;
import com.moengage.inapp.ViewCreationMeta;
import com.moengage.inapp.model.TestCampaignResponse;
import com.moengage.inapp.model.ViewDimension;
import com.moengage.inapp.repository.InAppRepository;

/**
 * @author Umang Chamaria
 * Date: 2019-10-14
 */
public class ShowTestInAppTask extends SDKTask {

  private static final String TAG = InAppConstants.MODULE_TAG + "ShowTestInAppTask";

  private String campaignId;

  public ShowTestInAppTask(Context context, String campaignId) {
    super(context);
    this.campaignId = campaignId;
  }

  @Override public TaskResult execute() {
    try {
      Logger.v(TAG + " execute() : Executing ShowTestInAppTask.");
      if (!InAppController.getInstance().isModuleEnabled(context)) {
        Logger.v(TAG + " execute() : InApp Module is disabled. Cannot show in-app.");
        return taskResult;
      }
      if (MoEUtils.isEmptyString(campaignId)) {
        Logger.v(TAG + " execute() : Empty campaign id. Cannot show test in-app.");
        taskResult.setIsSuccess(false);
        return taskResult;
      }
      if (InAppController.getInstance().isTablet(context)) {
        Logger.v(TAG + " execute() : Cannot show in-app on tablet.");
        showErrorDialog(InAppConstants.ERROR_MESSAGE_UNSUPPORTED_TABLET);
        return taskResult;
      }
      InAppRepository repository = InAppInjector.getInstance().getInAppRepository(context);
      TestCampaignResponse campaignResponse = repository.fetchTestCampaignPayload(campaignId);
      if (campaignResponse == null) {
        showErrorDialog(InAppConstants.ERROR_MESSAGE_TEST_CAMPAIGN + "\n Draft-id: " + campaignId);
        return taskResult;
      }
      if (!campaignResponse.isSuccess) {
        String message = campaignResponse.errorMessage;
        if (MoEUtils.isEmptyString(message)) {
          message = InAppConstants.ERROR_MESSAGE_TEST_CAMPAIGN;
        }
        message += "\n Draft-id: " + campaignId;
        showErrorDialog(message);
        return taskResult;
      }
      if (campaignResponse.campaignPayload == null) {
        Logger.v(TAG + " execute() : Campaign payload empty.");
        return taskResult;
      }
      if (InAppConstants.IN_APP_TEMPLATE_TYPE_SELF_HANDLED.equals(
          campaignResponse.campaignPayload.templateType)) {
        InAppController.getInstance().onSelfHandledAvailable(campaignResponse.campaignPayload);
        taskResult.setIsSuccess(true);
        return taskResult;
      }
      ViewCreationMeta meta = new ViewCreationMeta(InAppUtils.getScreenDimension(context),
          InAppUtils.getStatusBarHeight(context));
      View inAppView = InAppController.getInstance().buildInApp(
          campaignResponse.campaignPayload, meta);
      if (inAppView == null) {
        Logger.v(TAG + " execute() : Cannot show in-app. View creation failed.");
        showErrorDialog(
            InAppConstants.ERROR_MESSAGE_COULD_NOT_CREATE_VIEW + "Draft-id: " + campaignId);
        return taskResult;
      }
      ViewDimension inAppDimensions = InAppUtils.getUnspecifiedViewDimension(inAppView);
      if (InAppUtils.isInAppExceedingScreen(inAppDimensions,
          InAppUtils.getScreenDimension(context))) {
        showErrorDialog(InAppConstants.ERROR_MESSAGE_INAPP_EXCEEDING_DEVICE_DIMENSION);
        return taskResult;
      }
      if (InAppController.getInstance().isLandscapeMode(context)) {
        Logger.v(TAG + " execute() : Cannot show in-app in landscape mode.");
        showErrorDialog(InAppConstants.ERROR_MESSAGE_UNSUPPORTED_LANDSCAPE_MODE);
        return taskResult;
      }
      InAppController.getInstance()
          .addInAppToViewHierarchy(InAppController.getInstance().getCurrentActivity(),
              inAppView, campaignResponse.campaignPayload);
      taskResult.setIsSuccess(true);

      Logger.v(TAG + " execute() : Completed executing ShowTestInAppTask.");
    } catch (Exception e) {
      Logger.e(TAG + " execute() : ", e);
    }
    return taskResult;
  }

  @Override public String getTaskTag() {
    return TAG_TEST_INAPP_TASK;
  }

  @Override public boolean isSynchronous() {
    return true;
  }

  private void showErrorDialog(String errorMessage) {
    final Activity activity = InAppController.getInstance().getCurrentActivity();
    if (activity == null) {
      Logger.v(TAG + " showErrorDialog() : Cannot show error dialog. Activity instance is null.");
      return;
    }
    final AlertDialog.Builder alertDialog = new AlertDialog.Builder(activity);
    alertDialog.setMessage(errorMessage)
        .setTitle("Test in-app error")
        .setPositiveButton("OK", new DialogInterface.OnClickListener() {
          @Override public void onClick(DialogInterface dialog, int which) {
            dialog.dismiss();
          }
        });
    activity.runOnUiThread(new Runnable() {
      @Override public void run() {
        AlertDialog dialog = alertDialog.create();
        dialog.show();
      }
    });
  }
}
