/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.inapp.repository;

import android.content.ContentValues;
import android.database.Cursor;
import androidx.annotation.Nullable;
import com.moe.pushlibrary.providers.MoEDataContract;
import com.moe.pushlibrary.providers.MoEDataContract.InAppStatsEntity;
import com.moe.pushlibrary.providers.MoEDataContract.InAppV3Entity;
import com.moengage.core.Logger;
import com.moengage.core.MoEUtils;
import com.moengage.inapp.InAppConstants;
import com.moengage.inapp.model.StatModel;
import com.moengage.inapp.model.meta.CampaignMeta;
import com.moengage.inapp.model.meta.CampaignState;
import com.moengage.inapp.model.meta.InAppCampaign;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import org.json.JSONException;
import org.json.JSONObject;

/**
 * @author Umang Chamaria
 * Date: 2019-07-25
 */
class MarshallingHelper {

  private static final String TAG = InAppConstants.MODULE_TAG + "MarshallingHelper";

  ContentValues campaignToContentValues(InAppCampaign inAppCampaign) {
    ContentValues contentValue = new ContentValues();
    if (inAppCampaign._id != -1) {
      contentValue.put(MoEDataContract.InAppV3Entity._ID, inAppCampaign._id);
    }
    contentValue.put(InAppV3Entity.CAMPAIGN_ID, inAppCampaign.campaignMeta.campaignId);
    contentValue.put(InAppV3Entity.CAMPAIGN_TYPE, inAppCampaign.campaignType);
    contentValue.put(InAppV3Entity.CAMPAIGN_STATUS, inAppCampaign.status);
    contentValue.put(InAppV3Entity.CAMPAIGN_STATE,
        CampaignState.toJson(inAppCampaign.campaignState).toString());
    contentValue.put(InAppV3Entity.PRIORITY, inAppCampaign.campaignMeta.deliveryControl.priority);
    contentValue.put(InAppV3Entity.LAST_UPDATED_TIME, inAppCampaign.campaignMeta.lastUpdatedTime);
    contentValue.put(InAppV3Entity.TEMPLATE_TYPE, inAppCampaign.campaignMeta.templateType);
    contentValue.put(InAppV3Entity.DELETION_TIME, inAppCampaign.deletionTime);
    contentValue.put(InAppV3Entity.LAST_RECEIVED_TIME, inAppCampaign.lastReceivedTime);
    contentValue.put(InAppV3Entity.CAMPAIGN_META, CampaignMeta.toJson(inAppCampaign.campaignMeta).toString());
    return contentValue;
  }

  ContentValues campaignStateToContentValues(CampaignState campaignState) {
    ContentValues contentValue = new ContentValues();
    contentValue.put(InAppV3Entity.CAMPAIGN_STATE, CampaignState.toJson(campaignState).toString());
    return contentValue;
  }

  @Nullable ArrayList<InAppCampaign> campaignListFromCursor(Cursor cursor) {
    if (cursor == null || !cursor.moveToFirst()) return null;
    ArrayList<InAppCampaign> inAppCampaigns = new ArrayList<>();
    do {
      try {
        inAppCampaigns.add(inAppCampaignFromCursor(cursor));
      } catch (Exception e) {
        Logger.e(TAG + " campaignListFromCursor() : ", e);
      }
    } while (cursor.moveToNext());
    return inAppCampaigns;
  }

  InAppCampaign inAppCampaignFromCursor(Cursor cursor) throws JSONException {
    return new InAppCampaign(
        cursor.getLong(MoEDataContract.InAppV3Entity.COLUMN_INDEX_ID),
        cursor.getString(MoEDataContract.InAppV3Entity.COLUMN_INDEX_CAMPAIGN_TYPE),
        cursor.getString(InAppV3Entity.COLUMN_INDEX_CAMPAIGN_STATUS),
        cursor.getLong(InAppV3Entity.COLUMN_INDEX_DELETION_TIME),
        cursor.getLong(InAppV3Entity.COLUMN_INDEX_LAST_RECEIVED_TIME),
        cursorToCampaignMeta(cursor),
        cursorToCampaignState(cursor)
    );
  }


  private CampaignMeta cursorToCampaignMeta(Cursor cursor) throws JSONException {
    String campaignMeta = cursor.getString(InAppV3Entity.COLUMN_INDEX_CAMPAIGN_META);
    if (MoEUtils.isEmptyString(campaignMeta)){
      String campaignId = cursor.getString(InAppV3Entity.COLUMN_INDEX_CAMPAIGN_ID);
      return new CampaignMeta(campaignId, null, 0, 0, null, null, null, null);
    }
    return CampaignMeta.fromJson(new JSONObject(campaignMeta));
  }


  private CampaignState cursorToCampaignState(Cursor cursor) throws JSONException {
    String campaignState = cursor.getString(InAppV3Entity.COLUMN_INDEX_CAMPAIGN_STATE);
    return CampaignState.fromJson(new JSONObject(campaignState));
  }

  @Nullable HashSet<String> campaignIdsFromCursor(Cursor cursor) {
    if (cursor == null || !cursor.moveToFirst()) return null;
    HashSet<String> campaignIds = new HashSet<String>();
    do {
      campaignIds.add(cursor.getString(0));
    } while (cursor.moveToNext());
    return campaignIds;
  }

  ContentValues statToContentValues(StatModel stat){
    ContentValues values = new ContentValues();
    if (stat._id != -1) {
      values.put(InAppStatsEntity._ID, stat._id);
    }
    values.put(InAppStatsEntity.TIMESTAMP, stat.timestamp);
    values.put(InAppStatsEntity.REQUEST_ID, stat.requestId);
    values.put(InAppStatsEntity.PAYLOAD, stat.statsJson.toString());
    return values;
  }

  StatModel statFromCursor(Cursor cursor) throws JSONException {
    return new StatModel(
      cursor.getLong(InAppStatsEntity.COLUMN_INDEX_ID),
      cursor.getLong(InAppStatsEntity.COLUMN_INDEX_TIMESTAMP),
      cursor.getString(InAppStatsEntity.COLUMN_INDEX_REQUEST_ID),
        new JSONObject(cursor.getString(InAppStatsEntity.COLUMN_INDEX_PAYLOAD))
    );
  }

  @Nullable Map<String, InAppCampaign> campaignMapFromCursor(Cursor cursor) {
    if (cursor == null || !cursor.moveToFirst()) return null;
    Map<String, InAppCampaign> inAppCampaigns = new HashMap<>();
    do {
      try {
        InAppCampaign campaign = inAppCampaignFromCursor(cursor);
        if (campaign != null){
          inAppCampaigns.put(campaign.campaignMeta.campaignId, campaign);
        }
      } catch (Exception e) {
        Logger.e(TAG + " campaignListFromCursor() : ", e);
      }
    } while (cursor.moveToNext());
    return inAppCampaigns;
  }
}
