/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.inapp.repository;

import android.content.Context;
import androidx.annotation.Nullable;
import com.moengage.core.ConfigurationProvider;
import com.moengage.core.MoEUtils;
import com.moengage.core.SdkConfig;
import com.moengage.core.model.BaseRequest;
import com.moengage.inapp.model.InAppGlobalState;
import com.moengage.inapp.model.StatModel;
import com.moengage.inapp.model.meta.CampaignState;
import com.moengage.inapp.model.meta.InAppCampaign;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.json.JSONException;
import org.json.JSONObject;

/**
 * @author Umang Chamaria
 * Date: 2019-05-02
 */
public class LocalRepository {

  private ConfigurationProvider provider;
  private InAppDAO dao;

  public LocalRepository(Context context){
    provider = ConfigurationProvider.getInstance(context);
    dao = new InAppDAO(context);
  }

  void saveApiSyncInterval(long syncInterval){
    dao.saveApiSyncInterval(syncInterval);
  }

  public long getApiSyncInterval() {
    return provider.getInAppApiSyncDelay();
  }

  void saveLastApiSyncTime(long nextSyncTime) {
    dao.saveLastSyncTime(nextSyncTime);
  }

  void saveGlobalDelay(long globalDelay) {
    dao.saveGlobalDelay(globalDelay);
  }

  void addOrUpdateInApp(List<InAppCampaign> campaignMetaList) {
    dao.addOrUpdateCampaign(campaignMetaList);
  }

  public InAppGlobalState getGlobalState() {
    return new InAppGlobalState(provider.getInAppGlobalDelay(), provider.getLastInAppShownTime(),
        MoEUtils.currentSeconds());
  }

  public void deleteExpiredCampaigns() {
    dao.deleteExpiredCampaigns();
    dao.deleteExpiredImages();
  }

  void clearData() {
    dao.clearData();
  }

  @Nullable public List<String> getBlockedActivityList() {
    return SdkConfig.getConfig().inAppOptOutList;
  }

  @Nullable List<InAppCampaign> generalCampaigns() {
    return dao.getGeneralCampaigns();
  }

  @Nullable Set<String> primaryTriggerEvents() {
    return dao.getPrimaryTriggerEvents();
  }

  @Nullable List<InAppCampaign> selfHandledCampaign() {
    return dao.selfHandledCampaigns();
  }

  public long getLastSyncTime(){
    return provider.getLastInAppSyncTime();
  }

  public BaseRequest baseRequest() throws JSONException {
    return dao.baseRequest();
  }

  public List<InAppCampaign> getCampaignsForEvent(String eventName) {
    return dao.getCampaignsForEvent(eventName);
  }

  public int updateCampaignState(CampaignState state, String campaignId){
    return dao.updateStateForCampaign(state, campaignId);
  }

  public void updateLastShowTime(long time){
    dao.updateLastInAppShowTime(time);
  }

  @Nullable public List<InAppCampaign> getAllActiveCampaigns(){
    return dao.getActiveCampaigns();
  }

  public InAppCampaign getCampaignById(String campaignId){
    return dao.getCampaignFromId(campaignId);
  }

  @Nullable public Map<String, InAppCampaign> embeddedCampaign(){
    return dao.getEmbeddedCampaigns();
  }

  @Nullable public List<StatModel> getStats(int batchSize){
    return dao.getStats(batchSize);
  }

  @Nullable public void deleteStatById(StatModel stat){
    dao.deleteStatById(stat);
  }

  public void writeStats(JSONObject stats){
    dao.writeStats(new StatModel(MoEUtils.currentSeconds(), MoEUtils.getRequestId(), stats));
  }

}