package com.moengage.inapp.repository;

import android.content.Context;
import androidx.annotation.Nullable;
import androidx.annotation.RestrictTo;
import androidx.annotation.RestrictTo.Scope;
import androidx.annotation.WorkerThread;
import com.moengage.core.Logger;
import com.moengage.core.MoEUtils;
import com.moengage.core.SdkConfig;
import com.moengage.inapp.InAppConstants;
import com.moengage.inapp.StatsLogger;
import com.moengage.inapp.model.CampaignPayload;
import com.moengage.inapp.model.CampaignRequest;
import com.moengage.inapp.model.InAppCampaignResponse;
import com.moengage.inapp.model.InAppMetaRequest;
import com.moengage.inapp.model.InAppMetaResponse;
import com.moengage.inapp.model.TestCampaignResponse;
import com.moengage.inapp.repository.remote.RemoteRepository;

/**
 * @author Umang Chamaria
 */
public class InAppRepository {

  private static final String TAG = InAppConstants.MODULE_TAG + "InAppRepository";

  public final LocalRepository localRepository;
  public final RemoteRepository remoteRepository;
  public final InAppCache cache;

  public InAppRepository(LocalRepository localRepository, RemoteRepository remoteRepository,
      InAppCache inAppCache) {
    this.localRepository = localRepository;
    this.remoteRepository = remoteRepository;
    this.cache = inAppCache;
  }

  @WorkerThread public boolean fetchInAppCampaignMeta() {
    try {
      Logger.v(TAG + " fetchCampaignMeta() : Fetching in-app campaign meta");
      InAppMetaRequest request =
          new InAppMetaRequest(localRepository.baseRequest());
      InAppMetaResponse response = remoteRepository.fetchCampaignMeta(request);

      Logger.v(TAG + " fetchInAppCampaignMeta() : Sync Success: " + response.isSyncSuccess);
      Logger.v(TAG + " fetchInAppCampaignMeta() : Sync Interval: " + response.syncInterval);
      Logger.v(TAG + " fetchInAppCampaignMeta() : Global Delay: " + response.globalDelay);

      long currentTime = MoEUtils.currentSeconds();

      if (!response.isSyncSuccess) return false;

      localRepository.saveLastApiSyncTime(currentTime);

      localRepository.addOrUpdateInApp(response.campaignMetaList);

      if (response.syncInterval > 0) {
        localRepository.saveApiSyncInterval(response.syncInterval);
      }

      if (response.globalDelay >= 0) {
        localRepository.saveGlobalDelay(response.globalDelay);
      }
      return true;
    } catch (Exception e) {
      Logger.e(TAG + " fetchCampaignMeta() : Exception ", e);
    }
    return false;
  }

  @Nullable @WorkerThread public CampaignPayload fetchCampaignPayload(CampaignRequest request,
      boolean isPersistent) {
    try {
      Logger.v(TAG + " fetchCampaignPayload() : Will try to fetch campaign payload.");
      InAppCampaignResponse response = remoteRepository.fetchCampaignPayload(request);
      if (!response.isSuccess) {
        if (response.hasParsingException) {
          StatsLogger.getInstance().updateStatForCampaign(request.campaignId,
              MoEUtils.currentISOTime(), StatsLogger.DELIVERY_STAGE_MANDATORY_PARAM_MISSING);
        }
        if (response.responseCode != 409 && response.responseCode != 200) {
          StatsLogger.getInstance().updateStatForCampaign(request.campaignId,
              MoEUtils.currentISOTime(), StatsLogger.DELIVERY_STAGE_API_FAILURE);
        }

        return null;
      }
      if (response.campaignPayload != null &&
          response.campaignPayload.primaryWidget == -1 &&
          !isPersistent) {
        Logger.e(TAG + " fetchCampaignPayload() : Primary widget missing for a campaign, which "
            + "isn't persistent. Cannot show campaign. ");
        return null;
      }
      return response.campaignPayload;
    } catch (Exception e) {
      Logger.e(TAG + " fetchCampaignPayload() : Exception ", e);
    }
    return null;
  }

  public boolean isLifeCycleCallbackOptedOut() {
    return SdkConfig.getConfig().isLifecycleInAppOptedOut;
  }

  @RestrictTo(Scope.LIBRARY)
  public void onLogout(Context context) {
    StatsLogger.getInstance().writeStatsToStorage(context);
    uploadStats();
    localRepository.clearData();
    updateCache();
  }

  public void updateCache() {
    cache.updateCache(localRepository);
  }

  @WorkerThread @Nullable public TestCampaignResponse fetchTestCampaignPayload(String campaignId) {
    try {
      return remoteRepository.fetchTestCampaign(
          new CampaignRequest(localRepository.baseRequest(),
              campaignId));
    } catch (Exception e) {
      Logger.e(TAG + " fetchTestCampaignPayload() : Exception ", e);
    }
    return null;
  }

  @Nullable @WorkerThread public CampaignPayload fetchSelfHandledPayload(CampaignRequest request) {
    InAppCampaignResponse response = remoteRepository.fetchSelfHandledCampaign(request);
    if (response.isSuccess) return response.campaignPayload;

    if (response.hasParsingException) {
      StatsLogger.getInstance().updateStatForCampaign(request.campaignId,
          MoEUtils.currentISOTime(), StatsLogger.DELIVERY_STAGE_MANDATORY_PARAM_MISSING);
    }
    if (response.responseCode != 409 && response.responseCode != 200) {
      StatsLogger.getInstance().updateStatForCampaign(request.campaignId,
          MoEUtils.currentISOTime(), StatsLogger.DELIVERY_STAGE_API_FAILURE);
    }
    return null;
  }

  private final Object syncObj = new Object();

  public void uploadStats() {
/*    try {
      synchronized (syncObj){
        for (; ; ) {
          boolean isSuccess = true;
          List<StatModel> statList = localRepository.getStats(InAppConstants.STATISTICS_BATCH_SIZE);
          if (statList == null || statList.isEmpty()) {
            Logger.v(TAG + " uploadStats() : No pending batches.");
            return;
          }
          for (StatModel model : statList) {
            StatsUploadResponse response = remoteRepository.uploadStats(
                new StatsUploadRequest(localRepository.baseRequest(), model));
            if (!response.isSuccess) {
              isSuccess = false;
              break;
            }
            localRepository.deleteStatById(model);
          }
          if (!isSuccess) break;
        }
      }
    } catch (Exception e) {
      Logger.e(TAG + " uploadStats() : ", e);
    }*/
  }
}
