/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.inapp.repository;

import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import androidx.annotation.Nullable;
import com.moe.pushlibrary.utils.MoEHelperUtils;
import com.moengage.core.Logger;
import com.moengage.core.MoEFileManager;
import com.moengage.core.MoEUtils;
import java.io.File;
import java.io.InputStream;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Set;

/**
 * @author Umang Chamaria
 */
public class InAppImageManager {

  private MoEFileManager fileManager;

  public InAppImageManager(Context context) {
    fileManager = new MoEFileManager(context);
  }

  private static final String TAG = "InAppImageManager";

  @Nullable public Bitmap getImageFromUrl(Context context, String url, String campaignId) {
    try {
      if (isRemoteResource(url)) {
        return getRemoteImage(url, campaignId);
      } else {
        return getBundledImageIfPresent(context, url);
      }
    } catch (Exception e) {
      Logger.e(TAG + " getImageFromUrl() : ", e);
    }
    return null;
  }

  public void deleteImagesForCampaignIds(Set<String> campaignIdList) {
    if (campaignIdList == null) return;
    for (String campaignId : campaignIdList) {
      fileManager.deleteFolder(campaignId);
    }
  }

  private Bitmap getRemoteImage(String imageURI, String campaignId)
      throws NoSuchAlgorithmException {
    String fileName = getMd5FromString(imageURI);
    if (fileManager.fileExistsInDirectory(campaignId, fileName)) {
      return BitmapFactory.decodeFile(
          fileManager.getPathForFile(campaignId, fileName));
    } else {
      Bitmap bitmap = MoEHelperUtils.downloadImageBitmap(imageURI);
      if (bitmap == null) return null;
      fileManager.saveImageFile(campaignId, fileName, bitmap);
      return bitmap;
    }
  }

  private Bitmap getBundledImageIfPresent(Context context, String imageName) {
    int res = context.getResources().getIdentifier(imageName, "drawable", context.getPackageName());
    if (res == 0) {
      return null;
    } else {
      return BitmapFactory.decodeResource(context.getResources(), res);
    }
  }

  private boolean isRemoteResource(String imageUrl) {
    return imageUrl.startsWith("https://") || imageUrl.startsWith("http://");
  }

  private String getMd5FromString(String imageUrl) throws NoSuchAlgorithmException {
    MessageDigest messageDigest = MessageDigest.getInstance("MD5");
    messageDigest.update(imageUrl.getBytes());
    return MoEUtils.bytesToHex(messageDigest.digest());
  }

  @Nullable public File getGifFromUrl(String url, String campaignId){
    try {
      String fileName = getMd5FromString(url) + ".gif";
      if (fileManager.fileExistsInDirectory(campaignId, fileName)){
        return fileManager.getFileByName(campaignId, fileName);
      }
      InputStream inputStream = new java.net.URL(url).openStream();
      return fileManager.saveGif(campaignId, fileName, inputStream);
    } catch (Exception e) {
      Logger.e( TAG + " getGifFromUrl() : ", e);
    }
    return null;
  }
}
