/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.inapp.repository;

import android.content.ContentProviderOperation;
import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.net.Uri;
import androidx.annotation.Nullable;
import androidx.annotation.WorkerThread;
import com.moe.pushlibrary.providers.MoEDataContract;
import com.moe.pushlibrary.providers.MoEDataContract.InAppStatsEntity;
import com.moe.pushlibrary.providers.MoEDataContract.InAppV3Entity;
import com.moengage.core.ConfigurationProvider;
import com.moengage.core.Logger;
import com.moengage.core.MoEUtils;
import com.moengage.core.model.BaseRequest;
import com.moengage.core.RestUtils;
import com.moengage.inapp.InAppConstants;
import com.moengage.inapp.model.StatModel;
import com.moengage.inapp.model.meta.CampaignState;
import com.moengage.inapp.model.meta.InAppCampaign;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import org.json.JSONException;

/**
 * @author Umang Chamaria
 */
final class InAppDAO {

  private Context context;
  private static final String TAG = "InAppDAO";
  private MarshallingHelper marshallingHelper;
  private Uri campaignContentUri;
  private Uri statsContentUri;

  InAppDAO(Context context) {
    this.context = context;
    marshallingHelper = new MarshallingHelper();
    campaignContentUri = MoEDataContract.InAppV3Entity.getContentUri(context);
    statsContentUri = MoEDataContract.InAppStatsEntity.getContentUri(context);
  }

  void addOrUpdateCampaign(List<InAppCampaign> campaignList) {
    try {
      Map<String, InAppCampaign> storedCampaigns = getStoredCampaigns();
      ArrayList<ContentProviderOperation> addOperation = new ArrayList<>();
      ArrayList<ContentProviderOperation> updateOperation = new ArrayList<>();
      if (storedCampaigns == null) {
        for (InAppCampaign campaign : campaignList) {
          addOperation.add(ContentProviderOperation.newInsert(campaignContentUri)
              .withValues(marshallingHelper.campaignToContentValues(campaign))
              .build());
        }
      } else {
        for (InAppCampaign campaign : campaignList) {
          InAppCampaign inAppCampaign = storedCampaigns.get(campaign.campaignMeta.campaignId);
          if (inAppCampaign != null) {
            campaign._id = inAppCampaign._id;
            campaign.campaignState = inAppCampaign.campaignState;
            updateOperation.add(ContentProviderOperation.newUpdate(campaignContentUri.buildUpon()
                .appendPath(String.valueOf(campaign._id))
                .build()).withValues(marshallingHelper.campaignToContentValues(campaign)).build());
            storedCampaigns.remove(inAppCampaign.campaignMeta.campaignId);
          } else {
            addOperation.add(ContentProviderOperation.newInsert(campaignContentUri)
                .withValues(marshallingHelper.campaignToContentValues(campaign))
                .build());
          }
        }
        for (Entry<String, InAppCampaign> entry : storedCampaigns.entrySet()) {
          InAppCampaign campaign = entry.getValue();
          campaign.status = InAppConstants.IN_APP_CAMPAIGN_STATUS_IN_ACTIVE;
          updateOperation.add(ContentProviderOperation.newUpdate(campaignContentUri.buildUpon()
              .appendPath(String.valueOf(campaign._id))
              .build()).withValues(marshallingHelper.campaignToContentValues(campaign)).build());
        }
      }
      if (!updateOperation.isEmpty()) {
        context.getContentResolver()
            .applyBatch(MoEDataContract.getAuthority(context), updateOperation);
      }
      if (!addOperation.isEmpty()) {
        context.getContentResolver()
            .applyBatch(MoEDataContract.getAuthority(context), addOperation);
      }
    } catch (Exception e) {
      Logger.e(TAG + " addOrUpdateCampaign() : ", e);
    }
  }

  /**
   * Returns of all the stored campaigns.
   *
   * @return all stored campaigns.
   */
  @Nullable Map<String, InAppCampaign> getStoredCampaigns() {
    Cursor cursor = null;
    try {
      Map<String, InAppCampaign> campaignsMap = new HashMap<>();
      cursor = context.getContentResolver()
          .query(campaignContentUri, InAppV3Entity.PROJECTION, null, null, null);
      if (cursor == null || !cursor.moveToFirst()) return null;
      do {
        try {
          InAppCampaign campaign = marshallingHelper.inAppCampaignFromCursor(cursor);
          if (campaign != null) {
            campaignsMap.put(campaign.campaignMeta.campaignId, campaign);
          }
        } catch (Exception e) {
          Logger.e(TAG + " campaignListFromCursor() : ", e);
        }
      } while (cursor.moveToNext());
      return campaignsMap;
    } catch (Exception e) {
      Logger.e(TAG + " getStoredCampaigns() : ", e);
    } finally {
      closeCursor(cursor);
    }
    return null;
  }

  /**
   * Returns all active campaigns of TYPE {@link InAppConstants#IN_APP_CAMPAIGN_TYPE_SMART}  if
   * any, else null.
   *
   * @return {@link List} of {@link InAppCampaign}
   */
  @Nullable @WorkerThread List<InAppCampaign> getTriggerCampaigns() {
    Cursor cursor = null;
    try {
      cursor = context.getContentResolver()
          .query(campaignContentUri, InAppV3Entity.PROJECTION,
              MoEDataContract.InAppV3Entity.CAMPAIGN_STATUS
                  + " = ? "
                  + " AND "
                  + MoEDataContract.InAppV3Entity.CAMPAIGN_TYPE
                  + " = ? ", new String[] {
                  InAppConstants.IN_APP_CAMPAIGN_STATUS_ACTIVE,
                  InAppConstants.IN_APP_CAMPAIGN_TYPE_SMART
              }, InAppV3Entity.DEFAULT_SORT_ORDER);
      return marshallingHelper.campaignListFromCursor(cursor);
    } catch (Exception e) {
      Logger.e(TAG + " getTriggerCampaigns() : ", e);
    } finally {
      closeCursor(cursor);
    }
    return null;
  }

  /**
   * Returns all active embedded campaigns if any, else null.
   *
   * @return {@link List} of {@link InAppCampaign}
   */
  @Nullable Map<String, InAppCampaign> getEmbeddedCampaigns() {
    Cursor cursor = null;
    try {
      cursor = context.getContentResolver()
          .query(campaignContentUri, InAppV3Entity.PROJECTION,
              MoEDataContract.InAppV3Entity.CAMPAIGN_STATUS
                  + " = ? "
                  + " AND "
                  + MoEDataContract.InAppV3Entity.CAMPAIGN_TYPE
                  + " = ? "
                  + " AND "
                  + MoEDataContract.InAppV3Entity.TEMPLATE_TYPE
                  + " = ? ", new String[] {
                  InAppConstants.IN_APP_CAMPAIGN_STATUS_ACTIVE,
                  InAppConstants.IN_APP_CAMPAIGN_TYPE_GENERAL,
                  InAppConstants.IN_APP_TEMPLATE_TYPE_EMBEDDED
              }, InAppV3Entity.DEFAULT_SORT_ORDER);
      return marshallingHelper.campaignMapFromCursor(cursor);
    } catch (Exception e) {
      Logger.e(TAG + " getEmbeddedCampaigns() : ", e);
    } finally {
      closeCursor(cursor);
    }
    return null;
  }

  /**
   * Returns all active self-handled campaigns if any, else null.
   *
   * @return {@link List} of {@link InAppCampaign}
   */
  @Nullable List<InAppCampaign> selfHandledCampaigns() {
    Cursor cursor = null;
    try {
      cursor = context.getContentResolver()
          .query(campaignContentUri, InAppV3Entity.PROJECTION,
              MoEDataContract.InAppV3Entity.CAMPAIGN_STATUS
                  + " = ? "
                  + " AND "
                  + MoEDataContract.InAppV3Entity.CAMPAIGN_TYPE
                  + " = ? "
                  + " AND "
                  + MoEDataContract.InAppV3Entity.TEMPLATE_TYPE
                  + " = ? ", new String[] {
                  InAppConstants.IN_APP_CAMPAIGN_STATUS_ACTIVE,
                  InAppConstants.IN_APP_CAMPAIGN_TYPE_GENERAL,
                  InAppConstants.IN_APP_TEMPLATE_TYPE_SELF_HANDLED
              }, InAppV3Entity.DEFAULT_SORT_ORDER);
      return marshallingHelper.campaignListFromCursor(cursor);
    } catch (Exception e) {
      Logger.e(TAG + " selfHandledCampaigns() : ", e);
    } finally {
      closeCursor(cursor);
    }
    return null;
  }

  /**
   * Returns all active general campaigns with template type POP_UP and FULL_SCREEN if any, else
   * null.
   *
   * @return {@link List} of {@link InAppCampaign}
   */
  @Nullable List<InAppCampaign> getGeneralCampaigns() {
    Cursor cursor = null;
    try {
      cursor = context.getContentResolver()
          .query(campaignContentUri, InAppV3Entity.PROJECTION,
              MoEDataContract.InAppV3Entity.CAMPAIGN_STATUS
                  + " = ? "
                  + " AND "
                  + MoEDataContract.InAppV3Entity.CAMPAIGN_TYPE
                  + " = ? "
                  + " AND "
                  + MoEDataContract.InAppV3Entity.TEMPLATE_TYPE
                  + " IN ( ?, ? ) ", new String[] {
                  InAppConstants.IN_APP_CAMPAIGN_STATUS_ACTIVE,
                  InAppConstants.IN_APP_CAMPAIGN_TYPE_GENERAL,
                  InAppConstants.IN_APP_TEMPLATE_TYPE_POP_UP,
                  InAppConstants.IN_APP_TEMPLATE_TYPE_FULL_SCREEN
              }, InAppV3Entity.DEFAULT_SORT_ORDER);
      return marshallingHelper.campaignListFromCursor(cursor);
    } catch (Exception e) {
      Logger.e(TAG + " selfHandledCampaigns() : ", e);
    } finally {
      closeCursor(cursor);
    }
    return null;
  }

  /**
   * Gets campaign for the given id else null.
   *
   * @param campaignId id for which campaign should be returned.
   * @return instance of {@link InAppCampaign}
   */
  @Nullable InAppCampaign getCampaignFromId(String campaignId) {
    Cursor cursor = null;
    try {
      cursor = context.getContentResolver()
          .query(campaignContentUri, InAppV3Entity.PROJECTION,
              MoEDataContract.InAppV3Entity.CAMPAIGN_ID + " = ? ", new String[] { campaignId },
              null);
      if (cursor != null && cursor.moveToFirst()) {
        return marshallingHelper.inAppCampaignFromCursor(cursor);
      }
    } catch (Exception e) {
      Logger.e(TAG + " getCampaignFromId() : ", e);
    } finally {
      closeCursor(cursor);
    }
    return null;
  }

  int updateStateForCampaign(CampaignState campaignState, String campaignId) {
    int updateCount = -1;
    try {
      Logger.v(TAG
          + " updateStateForCampaign() : Updating campaign state for campaign id: "
          + campaignId);
      updateCount = context.getContentResolver()
          .update(campaignContentUri,
              marshallingHelper.campaignStateToContentValues(campaignState),
              InAppV3Entity.CAMPAIGN_ID + " = ? ",
              new String[] { campaignId });
      if (updateCount < 0) {
        Logger.v(TAG
            + " updateStateForCampaign() : update failed might not exist. Campaign Id: "
            + campaignId);
      } else {
        Logger.v(
            TAG + " updateStateForCampaign() : Campaign state updated. Campaign Id: " + campaignId);
      }
    } catch (Exception e) {
      Logger.e(TAG + " updateStateForCampaign() : ", e);
    }
    return updateCount;
  }

  void saveApiSyncInterval(long syncDelay) {
    ConfigurationProvider.getInstance(context).saveInAppApiSyncDelay(syncDelay);
  }

  void updateLastInAppShowTime(long showTime) {
    ConfigurationProvider.getInstance(context).setLastInAppShownTime(showTime);
  }

  void saveLastSyncTime(long nextSyncTime) {
    ConfigurationProvider.getInstance(context).saveLastInAppSyncTime(nextSyncTime);
  }

  void saveGlobalDelay(long globalDelay) {
    ConfigurationProvider.getInstance(context).saveInAppGlobalDelay(globalDelay);
  }

  void deleteExpiredCampaigns() {
    try {
      int rows = context.getContentResolver()
          .delete(campaignContentUri, InAppV3Entity.DELETION_TIME
              + " < ? ", new String[] {
              String.valueOf(MoEUtils.currentSeconds())
          });
      Logger.v(
          TAG + "deleteExpiredCampaigns(): Number of device triggers records deleted: " + rows);
    } catch (Exception e) {
      Logger.e(TAG + " deleteExpiredCampaigns() : ", e);
    }
  }

  /**
   * All campaign ids eligible for deletion.
   *
   * @return {@link Set} of {@link String}
   */
  @Nullable Set<String> campaignsEligibleForDeletion() {
    Cursor cursor = null;
    try {
      MarshallingHelper helper = new MarshallingHelper();
      cursor = context.getContentResolver()
          .query(campaignContentUri, new String[] { InAppV3Entity.CAMPAIGN_ID },
              InAppV3Entity.DELETION_TIME
                  + " < ? ", new String[] {
                  String.valueOf(MoEUtils.currentSeconds())
              }, null);
      return helper.campaignIdsFromCursor(cursor);
    } catch (Exception e) {
      Logger.e(TAG + " campaignsEligibleForDeletion() : ", e);
    } finally {
      closeCursor(cursor);
    }
    return null;
  }

  private void closeCursor(Cursor cursor) {
    if (cursor != null) {
      cursor.close();
    }
  }

  /**
   * Clear all cached campaigns, images, sync time etc.
   */
  void clearData() {
    new InAppImageManager(context).deleteImagesForCampaignIds(getAllCampaignIds());
    context.getContentResolver().delete(MoEDataContract.InAppV3Entity.getContentUri(context),
        null, null);
    saveLastSyncTime(0);
    updateLastInAppShowTime(0);
    deleteAllStats();
  }

  void deleteExpiredImages() {
    new InAppImageManager(context).deleteImagesForCampaignIds(campaignsEligibleForDeletion());
  }

  BaseRequest baseRequest() throws JSONException {
    return RestUtils.getBaseRequest(context);
  }

  /**
   * Set of event names for triggered campaigns.
   *
   * @return {@link Set} of {@link String}
   */
  Set<String> getPrimaryTriggerEvents() {
    HashSet<String> triggerEventSet = new HashSet<>();
    try {
      List<InAppCampaign> triggerCampaigns = getTriggerCampaigns();
      if (triggerCampaigns != null) {
        for (InAppCampaign campaign : triggerCampaigns) {
          triggerEventSet.add(campaign.campaignMeta.trigger.primaryCondition.eventName);
        }
      }
    } catch (Exception e) {
      Logger.e(TAG + " getPrimaryTriggerEvents() : Exception ", e);
    }
    return triggerEventSet;
  }

  /**
   * Get active campaigns for given event.
   *
   * @param eventName event name for which campaign should be fetched.
   * @return list of {@link InAppCampaign}
   */
  List<InAppCampaign> getCampaignsForEvent(String eventName) {
    ArrayList<InAppCampaign> campaigns = new ArrayList<>();
    try {
      List<InAppCampaign> triggerCampaigns = getTriggerCampaigns();
      if (triggerCampaigns != null) {
        for (InAppCampaign campaign : triggerCampaigns) {
          if (eventName.equals(campaign.campaignMeta.trigger.primaryCondition.eventName)) {
            campaigns.add(campaign);
          }
        }
      }
    } catch (Exception e) {
      Logger.e(TAG + " getCampaignsForEvent() : ", e);
    }
    return campaigns;
  }

  /**
   * Fetches all active campaigns irrespective of type and alignment.
   *
   * @return {@link List} of {@link InAppCampaign}
   */
  List<InAppCampaign> getActiveCampaigns() {
    Cursor cursor = null;
    try {
      cursor = context.getContentResolver()
          .query(campaignContentUri, InAppV3Entity.PROJECTION,
              MoEDataContract.InAppV3Entity.CAMPAIGN_STATUS
                  + " = ? "
              , new String[] {
                  InAppConstants.IN_APP_CAMPAIGN_STATUS_ACTIVE,
              }, InAppV3Entity.DEFAULT_SORT_ORDER);
      return marshallingHelper.campaignListFromCursor(cursor);
    } catch (Exception e) {
      Logger.e(TAG + " selfHandledCampaigns() : ", e);
    } finally {
      closeCursor(cursor);
    }
    return null;
  }

  /**
   * Get a set of all campaign ids.
   *
   * @return {@link Set} of {@link String}
   */
  @Nullable Set<String> getAllCampaignIds() {
    Cursor cursor = null;
    try {
      MarshallingHelper helper = new MarshallingHelper();
      cursor = context.getContentResolver()
          .query(campaignContentUri, new String[] { InAppV3Entity.CAMPAIGN_ID }, null, null, null);
      return helper.campaignIdsFromCursor(cursor);
    } catch (Exception e) {
      Logger.e(TAG + " campaignsEligibleForDeletion() : ", e);
    } finally {
      closeCursor(cursor);
    }
    return null;
  }

  void insertCampaign(InAppCampaign campaign) {
    context.getContentResolver().insert(campaignContentUri,
        marshallingHelper.campaignToContentValues(campaign));
  }

  void deleteCampaignById(String campaignId) {
    context.getContentResolver()
        .delete(campaignContentUri, InAppV3Entity.CAMPAIGN_ID
            + " = ? ", new String[] {
            campaignId
        });
  }

  void writeStats(StatModel stat){
    try {
      if (stat == null){
        Logger.v( TAG + " writeStats() : Stats model empty. Cannot write.");
        return;
      }
      ContentValues values = marshallingHelper.statToContentValues(stat);
      Uri uri = context.getContentResolver().insert(statsContentUri, values);
      if (uri != null){
        Logger.v(TAG + " writeStats() : New stat added. Uri: " + uri.toString());
      }else {
        Logger.v(TAG + " writeStats() : Unable to write stat.");
      }
    } catch (Exception e) {
      Logger.e( TAG + " writeStats() : ", e);
    }
  }

  void deleteAllStats(){
    try {
      context.getContentResolver().delete(statsContentUri, null, null);
    }catch (Exception e){
      Logger.e( TAG + " deleteAllStats() : ", e);
    }
  }

  @Nullable List<StatModel> getStats(int batchSize){
    Uri contentUri =
        statsContentUri.buildUpon().appendQueryParameter(MoEDataContract.QUERY_PARAMETER_LIMIT,
            String.valueOf(batchSize)).build();
    Cursor cursor = null;
    List<StatModel> statsList = null;
    try {
      cursor = context.getContentResolver().query(contentUri, InAppStatsEntity.PROJECTION, null,
          null, null);
      if (cursor == null || !cursor.moveToFirst() || cursor.getCount() == 0){
        Logger.v(TAG + " getStats() : No entry found.");
        closeCursor(cursor);
        return null;
      }
      statsList = new ArrayList<>(cursor.getCount());
      do {
        try {
          StatModel statModel = marshallingHelper.statFromCursor(cursor);
          if (statModel != null){
            statsList.add(statModel);
          }
        }catch (Exception e){
          Logger.e( TAG + " getStats() : ", e);
        }
      }while (cursor.moveToNext());
    }catch (Exception e){
      Logger.e( TAG + " getStats() : ", e);
    }finally {
      closeCursor(cursor);
    }
    return statsList;
  }

  void deleteStatById(StatModel stat) {
    try {

      ArrayList<ContentProviderOperation> operations = new ArrayList<>();
      ContentProviderOperation operation;
      operation = ContentProviderOperation.newDelete(statsContentUri)
          .withSelection(InAppStatsEntity._ID + " = ?", new String[] { String.valueOf(stat._id) })
          .build();
      operations.add(operation);
      context.getContentResolver().applyBatch(MoEDataContract.getAuthority(context), operations);
    } catch (Exception e) {
      Logger.e(TAG + " deleteStatById() : ", e);
    }
  }
}
