package com.moengage.inapp.model.meta;

import androidx.annotation.Nullable;
import com.moengage.core.Logger;
import com.moengage.core.MoEUtils;
import org.json.JSONException;
import org.json.JSONObject;

/**
 * InAppCampaign meta data.
 *
 * @author Umang Chamaria
 */
public class CampaignMeta {

  private static final String TAG = "CampaignMeta";
  /**
   * Unique identifier for the campaign.
   */
  public final String campaignId;
  /**
   * Name of the campaign.
   */
  public final String campaignName;
  /**
   * Expiry time of the campaign.
   */
  public final long expiryTime;
  /**
   * Last time the campaign was updated.
   */
  public final long lastUpdatedTime;
  /**
   * Rules on displaying or dismissing the campaign.
   */
  public final DisplayControl displayControl;
  /**
   * Type of template. Embedded, Full Screen, Pop up or Self Handled.
   */
  public final String templateType;
  /**
   * Control the delivery parameters of the campaign.
   */
  public final DeliveryControl deliveryControl;
  /**
   * Condition on which the in-app should be triggered.
   */
  @Nullable public final Trigger trigger;

  public CampaignMeta(String campaignId, String campaignName, long expiryTime, long lastUpdatedTime,
      DisplayControl displayControl, String templateType,
      DeliveryControl deliveryControl,
      @Nullable Trigger trigger) {
    this.campaignId = campaignId;
    this.campaignName = campaignName;
    this.expiryTime = expiryTime;
    this.lastUpdatedTime = lastUpdatedTime;
    this.displayControl = displayControl;
    this.templateType = templateType;
    this.deliveryControl = deliveryControl;
    this.trigger = trigger;
  }

  @Override public String toString() {
    try {
      JSONObject jsonObject = toJson(this);
      if (jsonObject != null) {
        return jsonObject.toString(4);
      }
    } catch (JSONException e) {

    }
    return super.toString();
  }

  public static CampaignMeta fromJson(JSONObject metaJson) throws JSONException {
    return new CampaignMeta(
        metaJson.getString(CAMPAIGN_ID),
        metaJson.getString(CAMPAIGN_NAME),
        MoEUtils.secondsFromIsoString(metaJson.getString(EXPIRY_TIME)),
        MoEUtils.secondsFromIsoString(metaJson.getString(LAST_UPDATED_TIME)),
        DisplayControl.fromJson(metaJson.optJSONObject(DISPLAY_CONTROL)),
        metaJson.getString(TEMPLATE_TYPE),
        DeliveryControl.fromJson(metaJson.getJSONObject(DELIVERY_CONTROL)),
        Trigger.fromJson(metaJson.optJSONObject(TRIGGER))
    );
  }

  public static JSONObject toJson(CampaignMeta meta) {
    try {
      JSONObject metaJson = new JSONObject();
      metaJson.put(CAMPAIGN_ID, meta.campaignId)
          .put(CAMPAIGN_NAME, meta.campaignName)
          .put(EXPIRY_TIME, MoEUtils.isoStringFromSeconds(meta.expiryTime))
          .put(LAST_UPDATED_TIME, MoEUtils.isoStringFromSeconds(meta.lastUpdatedTime))
          .put(DISPLAY_CONTROL, DisplayControl.toJson(meta.displayControl))
          .put(TEMPLATE_TYPE, meta.templateType)
          .put(DELIVERY_CONTROL, DeliveryControl.toJson(meta.deliveryControl))
          .put(TRIGGER, Trigger.toJson(meta.trigger));
      return metaJson;
    } catch (Exception e) {
      Logger.e(TAG + " toJson() : ", e);
    }
    return null;
  }

  @Override public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;

    CampaignMeta meta = (CampaignMeta) o;

    if (expiryTime != meta.expiryTime) return false;
    if (lastUpdatedTime != meta.lastUpdatedTime) return false;
    if (!campaignId.equals(meta.campaignId)) return false;
    if (!campaignName.equals(meta.campaignName)) return false;
    if (!displayControl.equals(meta.displayControl)) return false;
    if (!templateType.equals(meta.templateType)) return false;
    if (!deliveryControl.equals(meta.deliveryControl)) return false;
    return trigger != null ? trigger.equals(meta.trigger) : meta.trigger == null;
  }

  private static final String CAMPAIGN_ID = "campaign_id";
  private static final String CAMPAIGN_NAME = "campaign_name";
  private static final String EXPIRY_TIME = "expiry_time";
  private static final String LAST_UPDATED_TIME = "updated_time";
  private static final String DISPLAY_CONTROL = "display";
  private static final String TEMPLATE_TYPE = "template_type";
  private static final String DELIVERY_CONTROL = "delivery";
  private static final String TRIGGER = "trigger";
}
