/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.inapp;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.Color;
import android.graphics.Typeface;
import android.graphics.drawable.Drawable;
import android.graphics.drawable.GradientDrawable;
import android.os.Handler;
import android.os.Looper;
import android.util.TypedValue;
import android.view.Gravity;
import android.view.KeyEvent;
import android.view.View;
import android.view.View.OnClickListener;
import android.view.ViewGroup;
import android.view.animation.Animation;
import android.view.animation.AnimationUtils;
import android.widget.Button;
import android.widget.ImageView;
import android.widget.ImageView.ScaleType;
import android.widget.LinearLayout;
import android.widget.RelativeLayout;
import android.widget.RelativeLayout.LayoutParams;
import android.widget.TextView;
import androidx.annotation.ColorInt;
import androidx.annotation.Nullable;
import androidx.annotation.WorkerThread;
import androidx.core.content.res.ResourcesCompat;
import com.bumptech.glide.Glide;
import com.moengage.core.Logger;
import com.moengage.core.MoEUtils;
import com.moengage.core.SdkConfig;
import com.moengage.inapp.exceptions.CouldNotCreateViewException;
import com.moengage.inapp.exceptions.ImageNotFoundException;
import com.moengage.inapp.model.Border;
import com.moengage.inapp.model.CampaignPayload;
import com.moengage.inapp.model.InAppComponent;
import com.moengage.inapp.model.InAppContainer;
import com.moengage.inapp.model.InAppWidget;
import com.moengage.inapp.model.Margin;
import com.moengage.inapp.model.Padding;
import com.moengage.inapp.model.Spacing;
import com.moengage.inapp.model.ViewDimension;
import com.moengage.inapp.model.Widget;
import com.moengage.inapp.model.actions.Action;
import com.moengage.inapp.model.enums.Orientation;
import com.moengage.inapp.model.enums.TemplateAlignment;
import com.moengage.inapp.model.enums.ViewType;
import com.moengage.inapp.model.enums.WidgetType;
import com.moengage.inapp.model.style.ButtonStyle;
import com.moengage.inapp.model.style.CloseStyle;
import com.moengage.inapp.model.style.ContainerStyle;
import com.moengage.inapp.model.style.ImageStyle;
import com.moengage.inapp.model.style.InAppStyle;
import com.moengage.inapp.model.style.RatingStyle;
import com.moengage.inapp.model.style.TextStyle;
import com.moengage.inapp.repository.InAppImageManager;
import com.moengage.widgets.MoERatingBar;
import java.io.File;
import java.util.List;

/**
 * @author Umang Chamaria
 */
public class ViewEngine {
  private static final String TAG = InAppConstants.MODULE_TAG + "ViewEngine";

  private CampaignPayload campaignPayload;
  private Context context;
  private InAppImageManager imageManager;
  private ViewDimension parentViewDimensions;
  private View popUpView;
  private int statusBarHeight;
  private float densityScale;
  private int popUpId;
  private Activity activity;
  private View inAppView;

  public ViewEngine(Activity activity,
      CampaignPayload campaignPayload, ViewCreationMeta viewCreationMeta) {
    this.activity = activity;
    this.context = activity.getApplicationContext();
    this.campaignPayload = campaignPayload;
    imageManager = new InAppImageManager(activity.getApplicationContext());
    this.parentViewDimensions = viewCreationMeta.deviceDimensions;
    this.statusBarHeight = viewCreationMeta.statusBarHeight;
    densityScale = activity.getResources().getDisplayMetrics().density;
  }

  @SuppressWarnings("ConstantConditions")
  @SuppressLint("WrongThread") @WorkerThread @Nullable View createInApp() {
    try {
      Logger.v(TAG
          + " createInApp() : Will try to create in-app view for campaign-id: "
          + campaignPayload.campaignId);

      Logger.v(TAG + " createInApp() : Device Dimensions: " + parentViewDimensions +
          "Status Bar height: " + statusBarHeight);

      inAppView = createPrimaryContainer(campaignPayload.primaryContainer);
      if (inAppView == null) return null;
      handleBackPress(inAppView);
      Logger.v(TAG + " createInApp() : InApp creation complete, returning created view.");
      ContainerStyle style = (ContainerStyle) campaignPayload.primaryContainer.style;
      if (style.animation != null && style.animation.entry != -1) {
        Animation animation = AnimationUtils.loadAnimation(context, style.animation.entry);
        animation.setFillAfter(true);
        inAppView.setAnimation(animation);
      }
      inAppView.setClickable(true);
      return inAppView;
    } catch (Exception e) {
      Logger.e(TAG + " createInApp() : ", e);
      if (e instanceof UnsupportedOperationException) {
        StatsLogger.getInstance().updateStatForCampaign(campaignPayload.campaignId,
            MoEUtils.currentISOTime(), StatsLogger.IMPRESSION_STAGE_GIF_LIBRARY_NOT_PRESENT);
      } else if (e instanceof ImageNotFoundException) {
        StatsLogger.getInstance().updateStatForCampaign(campaignPayload.campaignId,
            MoEUtils.currentISOTime(), StatsLogger.IMPRESSION_STAGE_IMAGE_DOWNLOAD_FAILURE);
      }
    }
    return null;
  }

  private View createPrimaryContainer(InAppContainer container)
      throws CouldNotCreateViewException, ImageNotFoundException, IllegalStateException {
    Logger.v(TAG + " createPrimaryContainer() : ");
    RelativeLayout containerLayout = new RelativeLayout(context);
    containerLayout.setId(CONTAINER_BASE_ID + container.id);
    View widgetView = null;
    Widget widget = getWidgetFromList(container.widgets, WidgetType.CONTAINER);
    if (widget == null) {
      throw new IllegalStateException("Unexpected Widget type");
    }
    widgetView = createPopUp((InAppContainer) widget.inAppWidget);
    if (widgetView == null) {
      throw new CouldNotCreateViewException("One of the container/widget "
          + "creation wasn't successful cannot create view further");
    }
    popUpView = widgetView;
    containerLayout.addView(widgetView);
    widget = getWidgetFromList(container.widgets, WidgetType.WIDGET);
    if (widget == null) {
      throw new IllegalStateException("Unexpected Widget type");
    }
    InAppWidget closeWidget = (InAppWidget) widget.inAppWidget;
    if (closeWidget.viewType != ViewType.CLOSE_BUTTON) {
      throw new IllegalStateException("Unexpected Widget type. Expected widget type is close "
          + "button.");
    }

    ViewDimension campaignDimensions = getViewDimensionsFromPercentage(container.style);
    Logger.v(
        TAG + " createPrimaryContainer() : Campaign Dimension: " + campaignDimensions);
    ViewDimension unspecifiedDimension = getUnspecifiedViewDimension(containerLayout);
    Logger.v(
        TAG + " createPrimaryContainer() : Computed Dimension: " + unspecifiedDimension);
    campaignDimensions.height = Math.max(campaignDimensions.height, unspecifiedDimension.height);

    if (closeWidget.component.style.display) {
      widgetView = createCloseButton(closeWidget, campaignDimensions);
      alignCloseButton(widgetView, (CloseStyle) closeWidget.component.style);
      containerLayout.addView(widgetView);
    }

    RelativeLayout.LayoutParams layoutParams =
        new RelativeLayout.LayoutParams(campaignDimensions.width, campaignDimensions.height);
    Spacing spacingMargin = transformMargin(container.style.margin);
    layoutParams.setMargins(spacingMargin.left, spacingMargin.top, spacingMargin.right,
        spacingMargin.bottom);
    containerLayout.setLayoutParams(layoutParams);
    Spacing spacingPadding = transformPadding(container.style.padding);
    containerLayout.setPadding(spacingPadding.left, spacingPadding.top, spacingPadding.right,
        spacingPadding.bottom);

    styleContainer(containerLayout, (ContainerStyle) container.style,
        campaignDimensions);

    return containerLayout;
  }

  private Widget getWidgetFromList(List<Widget> widgetList, WidgetType widgetType) {
    for (Widget widget : widgetList) {
      if (widget.type == widgetType) return widget;
    }
    return null;
  }

  private void alignCloseButton(View widgetView, CloseStyle style)
      throws CouldNotCreateViewException {
    if (style.position == null) {
      throw new CouldNotCreateViewException("Cannot create in-app position "
          + "of close button is missing Campaign-id:"
          + campaignPayload.campaignId);
    }
    RelativeLayout.LayoutParams layoutParams =
        (RelativeLayout.LayoutParams) widgetView.getLayoutParams();
    switch (style.position) {
      case LEFT:
        if (campaignPayload.templateType.equals(
            InAppConstants.IN_APP_TEMPLATE_TYPE_POP_UP)) {
          layoutParams.addRule(RelativeLayout.ALIGN_TOP, popUpView.getId());
          layoutParams.addRule(RelativeLayout.ALIGN_LEFT, popUpView.getId());
          layoutParams.leftMargin += transformViewDimension(style.margin.left,
              parentViewDimensions.width) - (CLOSE_BUTTON_MARGIN * densityScale);
        } else if (InAppConstants.IN_APP_TEMPLATE_TYPE_EMBEDDED.equals(campaignPayload.templateType)){
          layoutParams.addRule(RelativeLayout.ALIGN_TOP, popUpView.getId());
          layoutParams.addRule(RelativeLayout.ALIGN_LEFT, popUpView.getId());
        } else {
          layoutParams.addRule(RelativeLayout.ALIGN_PARENT_LEFT);
          if (SdkConfig.getConfig().isNavBarOptedOut) layoutParams.topMargin = statusBarHeight;
        }
        break;
      case RIGHT:
        if (campaignPayload.templateType.equals(
            InAppConstants.IN_APP_TEMPLATE_TYPE_POP_UP)) {
          layoutParams.rightMargin += transformViewDimension(style.margin.right,
              parentViewDimensions.width) - (CLOSE_BUTTON_MARGIN * densityScale);
          layoutParams.addRule(RelativeLayout.ALIGN_TOP, popUpView.getId());
          layoutParams.addRule(RelativeLayout.ALIGN_RIGHT, popUpView.getId());
        } else if (InAppConstants.IN_APP_TEMPLATE_TYPE_EMBEDDED.equals(
            campaignPayload.templateType)) {
          layoutParams.addRule(RelativeLayout.ALIGN_TOP, popUpView.getId());
          layoutParams.addRule(RelativeLayout.ALIGN_RIGHT, popUpView.getId());
        } else {
          layoutParams.addRule(RelativeLayout.ALIGN_PARENT_RIGHT);
          if (SdkConfig.getConfig().isNavBarOptedOut) layoutParams.topMargin = statusBarHeight;
        }
        break;
    }
    if (campaignPayload.templateType.equals(
        InAppConstants.IN_APP_TEMPLATE_TYPE_POP_UP)) {
      layoutParams.topMargin = layoutParams.topMargin - (int) (CLOSE_BUTTON_MARGIN * densityScale);
    }
    widgetView.setLayoutParams(layoutParams);
  }

  private View createCloseButton(InAppWidget widget, ViewDimension primaryContainerDimension) {
    Logger.v(TAG + " createCloseButton() : Will create close button. " + widget);
    Bitmap imageBitmap = imageManager.getImageFromUrl(context, widget.component.content,
        campaignPayload.campaignId);
    if (imageBitmap == null) {
      int res =
          context.getResources().getIdentifier("moe_close", "drawable", context.getPackageName());
      imageBitmap = BitmapFactory.decodeResource(context.getResources(), res);
    }
    ImageView imageView = new ImageView(context);
    int dimension = (int) (CLOSE_BUTTON_SIZE * densityScale);
    ViewDimension imageDimension = new ViewDimension(dimension,
        Math.min(dimension, primaryContainerDimension.height));
    int imageSize =
        campaignPayload.templateType.equals(InAppConstants.IN_APP_TEMPLATE_TYPE_EMBEDDED) ? (int) (
            CLOSE_BUTTON_IMAGE_SIZE_NUDGE
                * densityScale) : (int) (CLOSE_BUTTON_IMAGE_SIZE * densityScale);
    imageView.setImageBitmap(getScaledBitmap(imageBitmap, new ViewDimension(imageSize, imageSize)));
    RelativeLayout.LayoutParams layoutParams =
        new RelativeLayout.LayoutParams(imageDimension.width,
            imageDimension.height);

    Spacing padding = null;
    if (campaignPayload.templateType.equals(InAppConstants.IN_APP_TEMPLATE_TYPE_EMBEDDED)){
      int fixedPadding = (int) (14 * densityScale);
      padding = new Spacing(fixedPadding, 0, 0, fixedPadding);
    }else {
      int fixedPadding = (int) (CLOSE_BUTTON_PADDING * densityScale);
      padding = new Spacing(fixedPadding, fixedPadding, fixedPadding, fixedPadding);
    }
    imageView.setPadding(padding.left, padding.top, padding.right, padding.bottom);
    imageView.setLayoutParams(layoutParams);
    imageView.setClickable(true);
    addAction(imageView, widget.actions);
    return imageView;
  }

  @SuppressLint("ResourceType") private View createPopUp(InAppContainer container)
      throws CouldNotCreateViewException, ImageNotFoundException {
    RelativeLayout popUpLayout = new RelativeLayout(context);
    popUpId = container.id;
    View popUpView = createContainer(container);
    if (popUpView == null) {
      throw new CouldNotCreateViewException(
          "One of the container/widget creation wasn't successful cannot create view further");
    }
    RelativeLayout.LayoutParams layoutParams =
        new RelativeLayout.LayoutParams(ViewGroup.LayoutParams.WRAP_CONTENT,
            ViewGroup.LayoutParams.WRAP_CONTENT);
    addMarginToLayout(layoutParams, container.style);
    popUpLayout.setLayoutParams(layoutParams);
    //popUpView.he
    ViewDimension popUpDimensions = new ViewDimension(
        getViewDimensionsFromPercentage(container.style).width,
        getUnspecifiedViewDimension(popUpView).height);
    Logger.v(TAG + " createPopUp() : Pop up view Dimensions: " + popUpDimensions);
    styleContainer(popUpLayout, (ContainerStyle) container.style, popUpDimensions);
    popUpLayout.addView(popUpView);
    alignContainer(popUpLayout, campaignPayload.alignment);
    popUpLayout.setId(12345);
    return popUpLayout;
  }

  private void alignContainer(View view, TemplateAlignment alignment) {
    RelativeLayout.LayoutParams layoutParams = (RelativeLayout.LayoutParams) view.getLayoutParams();
    layoutParams.addRule(RelativeLayout.CENTER_VERTICAL);
    layoutParams.addRule(RelativeLayout.CENTER_HORIZONTAL);
    view.setLayoutParams(layoutParams);
  }

  private View createContainer(InAppContainer container)
      throws CouldNotCreateViewException, ImageNotFoundException {
    LinearLayout containerLayout = new LinearLayout(context);
    switch (container.orientation) {
      case VERTICAL:
        containerLayout.setOrientation(LinearLayout.VERTICAL);
        break;
      case HORIZONTAL:
        containerLayout.setOrientation(LinearLayout.HORIZONTAL);
        containerLayout.setGravity(Gravity.CENTER_HORIZONTAL);
        break;
    }
    View widgetView = null;
    for (Widget widget : container.widgets) {
      switch (widget.type) {
        case WIDGET:
          InAppWidget inAppWidget = (InAppWidget) widget.inAppWidget;
          if (!inAppWidget.component.style.display) {
            Logger.v(TAG + " createContainer() : Display type of widget is false. Will not create"
                + " widget. " + inAppWidget);
            continue;
          }
          widgetView = createWidget(inAppWidget, container.orientation);
          break;
        case CONTAINER:
          InAppContainer inAppContainer = (InAppContainer) widget.inAppWidget;
          if (!inAppContainer.style.display) {
            Logger.v(TAG + " createContainer() : Display type of container is false. Will not "
                + "create container. " + inAppContainer);
            continue;
          }
          widgetView = createContainer(inAppContainer);
          break;
      }
      if (widgetView == null) {
        throw new CouldNotCreateViewException("One of the container/widget "
            + "creation wasn't successful cannot create view further");
      }
      containerLayout.addView(widgetView);
    }
    Logger.v(TAG + " createContainer() : " + container.style);
    setViewDimensionsPopUp(containerLayout, container.style);
    if (popUpId != container.id) {
      RelativeLayout.LayoutParams layoutParams =
          (RelativeLayout.LayoutParams) containerLayout.getLayoutParams();
      addMarginToLayout(layoutParams, container.style);
      containerLayout.setLayoutParams(layoutParams);
      Spacing spacing = transformPadding(container.style.padding);
      containerLayout.setPadding(spacing.left, spacing.top, spacing.right, spacing.bottom);
      styleContainer(containerLayout, (ContainerStyle) container.style);
    }
    containerLayout.setId(CONTAINER_BASE_ID + container.id);
    return containerLayout;
  }

  private void setViewDimensionsPopUp(View view, InAppStyle style) {

    ViewDimension campaignDimension = getViewDimensionsFromPercentage(style);
    Logger.v(
        TAG + " setViewDimensionsPopUp() : Campaign Dimension " + campaignDimension);
    ViewDimension computedDimension = getUnspecifiedViewDimension(view);
    Logger.v(
        TAG + " setViewDimensionsPopUp() : Computed dimension: " + computedDimension);

    campaignDimension.height = Math.max(campaignDimension.height, computedDimension.height);

    RelativeLayout.LayoutParams layoutParams =
        new RelativeLayout.LayoutParams(campaignDimension.width,
            campaignDimension.height);
    view.setLayoutParams(layoutParams);
  }

  private View createWidget(InAppWidget widget, Orientation parentOrientation)
      throws ImageNotFoundException, CouldNotCreateViewException {
    Logger.v(TAG + " createWidget() : Creating widget: " + widget);
    View view = null;
    switch (widget.viewType) {
      case TEXT:
        view = createTextView(widget, parentOrientation);
        break;
      case IMAGE:
        view = createImageView(widget, parentOrientation);
        break;
      case BUTTON:
        view = createButton(widget, parentOrientation);
        break;
      case RATING:
        view = createRatingBar(widget, parentOrientation);
        break;
    }
    if (view == null) {
      throw new CouldNotCreateViewException(
          "View type not recognised. Type " + widget.viewType);
    }
    view.setId(WIDGET_BASE_ID + widget.id);
    view.setClickable(true);
    addAction(view, widget.actions);
    return view;
  }

  private TextView createTextView(InAppWidget widget, Orientation parentOrientation) {
    Logger.v(TAG + " createTextView() : Will create text widget: " + widget);
    TextView textView = new TextView(context);
    setTextContent(textView, widget.component);
    TextStyle textStyle = (TextStyle) widget.component.style;

    textView.setTextSize(textStyle.font.size);

    if (textStyle.font.color != null) {
      textView.setTextColor(getColor(textStyle.font.color));
    }
    int resourceId = context.getResources().getIdentifier(textStyle.font.name, "font", context
        .getPackageName());
    if (resourceId > 0) {
      Typeface typeface = ResourcesCompat.getFont(context, resourceId);
      textView.setTypeface(typeface);
    }

    ViewDimension campaignDimension = getViewDimensionsFromPercentage(widget.component.style);
    Logger.v(TAG + " createTextView() : Campaign Dimension: " + campaignDimension);
    campaignDimension.height = -2;
    Spacing paddingSpacing = transformPadding(textStyle.padding);
    Logger.v(TAG + " createTextView() : Padding: " + paddingSpacing);
    textView.setPadding(paddingSpacing.left, paddingSpacing.top, paddingSpacing.right,
        paddingSpacing.bottom);
    Logger.v(TAG + " createTextView() : Final Dimensions: " + campaignDimension);
    LinearLayout.LayoutParams layoutParams =
        new LinearLayout.LayoutParams(campaignDimension.width,
            campaignDimension.height);
    setLayoutGravity(layoutParams, parentOrientation);
    Spacing marginSpacing = transformMargin(textStyle.margin);
    layoutParams.setMargins(marginSpacing.left, marginSpacing.top, marginSpacing.right,
        marginSpacing.bottom);
    textView.setLayoutParams(layoutParams);
    GradientDrawable drawable = new GradientDrawable();
    if (textStyle.background != null && textStyle.background.color != null) {
      drawable.setColor(getColor(textStyle.background.color));
    }
    if (textStyle.border != null) {
      getBorder(textStyle.border, drawable);
    }
    applyBackgroundToView(textView, drawable);
    textView.setGravity(Gravity.CENTER);

    return textView;
  }

  @SuppressLint("CheckResult") private LinearLayout createImageView(InAppWidget widget,
      Orientation parentOrientation)
      throws ImageNotFoundException {
    Logger.v(TAG + " createImageView() : Will create this widget: " + widget);
    ImageStyle imageStyle = (ImageStyle) widget.component.style;
    if (MoEUtils.isGif(widget.component.content) && !InAppUtils.hasGifSupport()) {
      Logger.w(TAG + " createImageView() : Image is of gif type, gif dependency not add");
      throw new UnsupportedOperationException("library not support gif not added.");
    }
    final ImageView imageView = new ImageView(context);
    if (MoEUtils.isGif(widget.component.content)) {
      final File gifFile = imageManager.getGifFromUrl(widget.component.content,
          campaignPayload.campaignId);
      if (gifFile == null || !gifFile.exists()) {
        throw new ImageNotFoundException("Gif Download failure");
      }
      Logger.v(TAG + " createImageView() : Real dimensions: " + new ViewDimension(
          (int) imageStyle.realWidth, (int) imageStyle.realHeight));
      ViewDimension campaignDimension = getViewDimensionsFromPercentage(imageStyle);
      Logger.v(TAG + " createImageView() : Campaign Dimension: " + campaignDimension);
      campaignDimension.height =
          (int) ((imageStyle.realHeight * campaignDimension.width) / imageStyle.realWidth);
      Logger.v(TAG + " createImageView() : Final Dimensions: " + campaignDimension);

      LinearLayout.LayoutParams layoutParams =
          new LinearLayout.LayoutParams(campaignDimension.width, campaignDimension.height);
      imageView.setLayoutParams(layoutParams);
      imageView.setScaleType(ScaleType.FIT_XY);
      Handler mainThread = new Handler(Looper.getMainLooper());
      mainThread.post(new Runnable() {
        @Override public void run() {
          Glide.with(context).asGif().load(gifFile).fitCenter().into(imageView);
        }
      });
    } else {
      Bitmap imageBitmap = imageManager.getImageFromUrl(context, widget.component.content,
          campaignPayload.campaignId);
      if (imageBitmap == null) throw new ImageNotFoundException("Image Download failure");

      ViewDimension campaignDimension = getViewDimensionsFromPercentage(widget.component.style);
      Logger.v(TAG + " createImageView() : Campaign Dimensions: " + campaignDimension);
      ViewDimension imageDimension = new ViewDimension(imageBitmap.getWidth(),
          imageBitmap.getHeight());
      Logger.v(TAG + " createImageView() : Image dimensions: " + imageDimension);
      campaignDimension.height =
          (imageDimension.height * campaignDimension.width) / imageDimension.width;
      Logger.v(TAG + " createImageView() : Final dimensions: " + campaignDimension);
      LinearLayout.LayoutParams layoutParams =
          new LinearLayout.LayoutParams(campaignDimension.width, campaignDimension.height);
      imageView.setLayoutParams(layoutParams);
      imageView.setImageBitmap(getScaledBitmap(imageBitmap, campaignDimension));
    }

    Spacing paddingSpacing = transformPadding(imageStyle.padding);
    imageView.setPadding(paddingSpacing.left, paddingSpacing.top, paddingSpacing.right,
        paddingSpacing.bottom);

    LinearLayout linearLayout = new LinearLayout(context);
    LinearLayout.LayoutParams linearLayoutParams = new LinearLayout.LayoutParams(
        imageView.getLayoutParams().width, imageView.getLayoutParams().height);
    Spacing marginSpacing = transformMargin(imageStyle.margin);
    linearLayoutParams.setMargins(marginSpacing.left, marginSpacing.top, marginSpacing.right,
        marginSpacing.bottom);
    linearLayoutParams.leftMargin = marginSpacing.left;
    linearLayoutParams.rightMargin = marginSpacing.right;
    linearLayoutParams.topMargin = marginSpacing.top;
    linearLayoutParams.bottomMargin = marginSpacing.bottom;
    setLayoutGravity(linearLayoutParams, parentOrientation);
    linearLayout.setLayoutParams(linearLayoutParams);
    int borderWidth = 0;
    if (imageStyle.border != null) {
      borderWidth = transformToPx(imageStyle.border.width);
    }

    linearLayout.setPadding(borderWidth, borderWidth, borderWidth, borderWidth);
    if (imageStyle.border != null) {
      applyBackgroundToView(linearLayout, getBorder(imageStyle.border));
    }
    linearLayout.addView(imageView);
    return linearLayout;
  }

  private Button createButton(InAppWidget widget, Orientation parentOrientation) {
    Logger.v(TAG + " createButton() : Will create button widget " + widget);
    Button button = new Button(context);
    setTextContent(button, widget.component);
    ButtonStyle style = (ButtonStyle) widget.component.style;
    Logger.v(TAG + " createButton() : Style: " + style);
    button.setTextSize(style.font.size);

    if (style.font.color != null) {
      button.setTextColor(getColor(style.font.color));
    }
    int resourceId = context.getResources().getIdentifier(style.font.name, "font", context
        .getPackageName());
    if (resourceId > 0) {
      Typeface typeface = ResourcesCompat.getFont(context, resourceId);
      button.setTypeface(typeface);
    }
    ViewDimension campaignDimension = getViewDimensionsFromPercentage(widget.component.style);
    Logger.v(TAG + " createButton() : Campaign Dimension: " + campaignDimension);
    Spacing spacing = transformPadding(style.padding);
    Logger.v(TAG + " createButton() : Padding: " + spacing);
    button.setPadding(spacing.left, spacing.top, spacing.right, spacing.bottom);
    ViewDimension unSpecifiedDimension = getUnspecifiedViewDimension(button);
    Logger.v(TAG + " createButton() : Calculated Dimensions: " + unSpecifiedDimension);
    int minimumHeight = transformToPx(style.minHeight);
    Logger.v(TAG + " createButton() : Minimum height for widget: " + minimumHeight);
    if (minimumHeight > unSpecifiedDimension.height){
      campaignDimension.height = minimumHeight;
    }
    Logger.v(TAG + " createButton() : Final Dimensions: " + campaignDimension);
    LinearLayout.LayoutParams layoutParams =
        new LinearLayout.LayoutParams(campaignDimension.width,
            campaignDimension.height);

    setLayoutGravity(layoutParams, parentOrientation);
    Spacing marginSpacing = transformMargin(style.margin);
    layoutParams.setMargins(marginSpacing.left, marginSpacing.top, marginSpacing.right,
        marginSpacing.bottom);
    button.setLayoutParams(layoutParams);

    GradientDrawable drawable = new GradientDrawable();

    if (style.background != null && style.background.color != null) {
      drawable.setColor(getColor(style.background.color));
    }
    if (style.border != null) {
      getBorder(style.border, drawable);
    }
    applyBackgroundToView(button, drawable);
    button.setGravity(Gravity.CENTER);

    return button;
  }

  private MoERatingBar createRatingBar(InAppWidget widget, Orientation parentOrientation) {
    Logger.v(TAG + " createRatingBar() : Will create rating widget: " + widget);
    MoERatingBar ratingBar = new MoERatingBar(context);
    ratingBar.setIsIndicator(false);

    RatingStyle style = (RatingStyle) widget.component.style;
    ratingBar.setNumStars(style.numberOfStars);
    if (style.isHalfStepAllowed) {
      ratingBar.setStepSize(0.5f);
    } else {
      ratingBar.setStepSize(1.0f);
    }
    ratingBar.setColor(getColor(style.color));
    ViewDimension campaignDimension =
        new ViewDimension(getViewDimensionsFromPercentage(style).width,
            (int) (style.realHeight * densityScale));
    Logger.v(TAG + " createRatingBar() : Campaign dimensions: " + campaignDimension);
    LinearLayout.LayoutParams layoutParams =
        new LinearLayout.LayoutParams(campaignDimension.width, campaignDimension.height);
    setLayoutGravity(layoutParams, parentOrientation);
    Spacing marginSpacing = transformMargin(style.margin);
    layoutParams.setMargins(marginSpacing.left, marginSpacing.top, marginSpacing.right,
        marginSpacing.bottom);
    ratingBar.setLayoutParams(layoutParams);
    GradientDrawable drawable = new GradientDrawable();

    if (style.border != null) {
      getBorder(style.border, drawable);
    }
    applyBackgroundToView(ratingBar, drawable);
    return ratingBar;
  }

  private void applyBackgroundToView(View view, Drawable drawable) {
    view.setBackground(drawable);
  }

  private GradientDrawable getBorder(Border border) {
    GradientDrawable drawable = new GradientDrawable();
    return getBorder(border, drawable);
  }

  private GradientDrawable getBorder(Border border, GradientDrawable drawable) {
    if (border.radius != 0.0) {
      drawable.setCornerRadius((float) border.radius * densityScale);
    }
    if (border.color != null && border.width != 0.0) {
      drawable.setStroke((int) (border.width * densityScale), getColor(border.color));
    }
    return drawable;
  }

  private void addAction(final View view, final List<Action> actions) {
    if (actions == null) {
      Logger.v(TAG + " addAction() : View does not have any actionType.");
      return;
    }
    Logger.v(TAG + " addAction() : Will try to execute actionType: " + actions);
    view.setOnClickListener(new OnClickListener() {
      @Override public void onClick(View v) {
        markClickedIfRequired(v.getId());
        ActionManager actionManager = new ActionManager();
        for (Action action : actions) {
          Logger.v(TAG + " onClick() : Will execute actionType: " + action);
          actionManager.onActionPerformed(activity, inAppView, action, campaignPayload);
        }
      }
    });
  }

  private void markClickedIfRequired(int id) {
    if (campaignPayload.primaryWidget + WIDGET_BASE_ID == id) {
      InAppController.getInstance().logPrimaryWidgetClicked(context, campaignPayload.campaignId);
    }
  }

  private Bitmap getScaledBitmap(Bitmap imageBitmap, ViewDimension bitmapDimension) {
    return Bitmap.createScaledBitmap(imageBitmap, bitmapDimension.width,
        bitmapDimension.height, true);
  }

  private void setTextContent(TextView view, InAppComponent component) {
    view.setText(component.content);
    view.setAllCaps(false);
  }

  private void addMarginToLayout(LayoutParams layoutParams, InAppStyle style) {
    Margin margin = style.margin;
    layoutParams.leftMargin = margin.left == 0 ? 0
        : transformViewDimension(margin.left, parentViewDimensions.width);
    layoutParams.rightMargin = margin.right == 0 ? 0
        : transformViewDimension(margin.right, parentViewDimensions.width);
    layoutParams.topMargin = margin.top == 0 ? 0
        : transformViewDimension(margin.top, parentViewDimensions.height);
    layoutParams.bottomMargin = margin.bottom == 0 ? 0
        : transformViewDimension(margin.bottom, parentViewDimensions.height);
  }

  private ViewDimension getUnspecifiedViewDimension(View view) {
    view.measure(View.MeasureSpec.UNSPECIFIED, View.MeasureSpec.UNSPECIFIED);
    return new ViewDimension(view.getMeasuredWidth(), view.getMeasuredHeight());
  }

  private ViewDimension getViewDimensionsFromPercentage(InAppStyle style) {
    return new ViewDimension(transformViewDimension(style.width, parentViewDimensions.width),
        style.height == -2 ? -2
            : transformViewDimension(style.height, parentViewDimensions.height));
  }

  private int transformViewDimension(double dimension, int containerReference) {
    return (int) ((dimension * containerReference) / 100);
  }

  private void styleContainer(LinearLayout containerLayout, ContainerStyle style) {

    if (style.background != null && style.background.color != null) {
      containerLayout.setBackgroundColor(getColor(style.background.color));
    }
    //background without image
    if (style.border != null) {
      GradientDrawable backgroundDrawable = getBorder(style.border);
      if (style.background != null && style.background.color != null) {
        backgroundDrawable.setColor(getColor(style.background.color));
      }
      applyBackgroundToView(containerLayout, backgroundDrawable);
    }
  }

  private void styleContainer(RelativeLayout containerLayout, ContainerStyle style,
      ViewDimension containerDimensions) throws ImageNotFoundException {
    if (style.background == null) return;
    int borderWidth = 0;
    if (style.border != null) {
      borderWidth = (int) ((int) style.border.width * densityScale);
    }
    if (borderWidth != 0) {
      Spacing paddingSpacing = new Spacing(containerLayout.getPaddingLeft(),
          containerLayout.getPaddingRight(), containerLayout.getPaddingTop(),
          containerLayout.getPaddingBottom());
      containerLayout.setPadding(paddingSpacing.left + borderWidth,
          paddingSpacing.top + borderWidth, paddingSpacing.right + borderWidth,
          paddingSpacing.bottom + borderWidth);
    }
    if (style.background.content != null) {
      //background with image
      final ImageView imageView = new ImageView(context);
      RelativeLayout.LayoutParams layoutParams =
          new RelativeLayout.LayoutParams(containerDimensions.width, containerDimensions.height);
      imageView.setLayoutParams(layoutParams);

      //imageView.setPadding(borderWidth / 2, borderWidth / 2, borderWidth / 2, borderWidth / 2);
      if (MoEUtils.isGif(style.background.content)) {
        final File gifFile = imageManager.getGifFromUrl(style.background.content,
            campaignPayload.campaignId);
        if (gifFile == null || !gifFile.exists()) {
          throw new ImageNotFoundException("Gif Download failure");
        }
        InAppController.getInstance().mainThreadHandler.post(new Runnable() {
          @Override public void run() {
            Glide.with(context).asGif().load(gifFile).centerCrop().into(imageView);
          }
        });
      } else {
        Bitmap imageBitmap = imageManager.getImageFromUrl(context, style.background.content,
            campaignPayload.campaignId);
        if (imageBitmap == null) throw new ImageNotFoundException("Image Download failure");
        imageView.setImageBitmap(imageBitmap);
        imageView.setScaleType(ScaleType.FIT_XY);
      }
      containerLayout.addView(imageView);
    }
    GradientDrawable drawable = new GradientDrawable();

    if (style.background.color != null) {
      drawable.setColor(getColor(style.background.color));
    }
    if (style.border != null) {
      getBorder(style.border, drawable);
    }

    applyBackgroundToView(containerLayout, drawable);
  }

  private void handleBackPress(View inAppView) {
    inAppView.setFocusableInTouchMode(true);
    inAppView.requestFocus();
    inAppView.setOnKeyListener(new View.OnKeyListener() {
      @Override public boolean onKey(View v, int keyCode, KeyEvent event) {
        try {
          if (event.getAction() == KeyEvent.ACTION_DOWN && keyCode == KeyEvent.KEYCODE_BACK) {
            Logger.v(TAG + " handleBackPress() : on back button pressed");
            if (campaignPayload.isCancellable) {
              ContainerStyle containerStyle =
                  (ContainerStyle) campaignPayload.primaryContainer.style;
              com.moengage.inapp.model.Animation animation = containerStyle.animation;
              if (animation != null && animation.exit != -1) {
                Animation viewAnimation = AnimationUtils.loadAnimation(context, animation.exit);
                viewAnimation.setFillAfter(true);
                v.setAnimation(viewAnimation);
              }
              ((ViewGroup) v.getParent()).removeView(v);
              InAppController.getInstance().handleDismiss(campaignPayload);
              return true;
            } else {
              // this is done to ensure in-app is not shown when user comes back to the screen
              // again.
              ((ViewGroup) v.getParent()).removeView(v);
              return false;
            }
          }
        } catch (Exception e) {
          Logger.e(TAG + " onKey() : ", e);
        }
        return false;
      }
    });
  }

  @ColorInt
  private int getColor(com.moengage.inapp.model.Color color) {
    return Color.argb((int) (color.alpha * 255.0f + 0.5f), color.red, color.green, color.blue);
  }

  private Spacing transformPadding(Padding padding) {
    Spacing spacing = new Spacing(
        padding.left == 0 ? 0
            : transformViewDimension(padding.left, parentViewDimensions.width),
        padding.right == 0 ? 0
            : transformViewDimension(padding.right, parentViewDimensions.width),
        padding.top == 0 ? 0
            : transformViewDimension(padding.top, parentViewDimensions.height),
        padding.bottom == 0 ? 0
            : transformViewDimension(padding.bottom, parentViewDimensions.height)
    );
    Logger.v(TAG + " transformPadding() : Padding: " + spacing);
    return spacing;
  }

  private Spacing transformMargin(Margin margin) {
    Spacing spacing = new Spacing(
        margin.left == 0 ? 0
            : transformViewDimension(margin.left, parentViewDimensions.width),
        margin.right == 0 ? 0
            : transformViewDimension(margin.right, parentViewDimensions.width),
        margin.top == 0 ? 0
            : transformViewDimension(margin.top, parentViewDimensions.height),
        margin.bottom == 0 ? 0
            : transformViewDimension(margin.bottom, parentViewDimensions.height)
    );
    Logger.v(TAG + " transformMargin() : Margin: " + spacing);
    return spacing;
  }

  private int transformToPx(double dp) {
    return (int) TypedValue.applyDimension(TypedValue.COMPLEX_UNIT_DIP, (float) dp,
        activity.getResources().getDisplayMetrics());
  }

  private void setLayoutGravity(LinearLayout.LayoutParams layoutParams,
      Orientation parentOrientation) {
    if (Orientation.VERTICAL == parentOrientation) {
      layoutParams.gravity = Gravity.CENTER_HORIZONTAL;
    }
  }

  static final int CONTAINER_BASE_ID = 20000;
  static final int WIDGET_BASE_ID = 30000;
  private static final int CLOSE_BUTTON_SIZE = 42;
  private static final int CLOSE_BUTTON_PADDING = 6;
  private static final int CLOSE_BUTTON_MARGIN = CLOSE_BUTTON_SIZE / 2;
  private static final int CLOSE_BUTTON_IMAGE_SIZE = 24;
  private static final int CLOSE_BUTTON_IMAGE_SIZE_NUDGE = 16;
}
