/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.inapp;

import android.content.Context;
import androidx.annotation.NonNull;
import com.moe.pushlibrary.MoEHelper;
import com.moengage.core.Logger;
import com.moengage.core.MoEConstants;
import com.moengage.core.MoEUtils;
import com.moengage.core.Properties;
import com.moengage.core.SdkConfig;
import com.moengage.core.executor.TaskManager;
import com.moengage.inapp.listeners.InAppMessageListener;
import com.moengage.inapp.model.MoEInAppCampaign;
import com.moengage.inapp.model.enums.StateUpdateType;
import com.moengage.inapp.tasks.UpdateCampaignStateTask;

/**
 * Helper class for showing in-app.
 *
 * @author Umang Chamaria
 * Date: 02/04/19
 * @since 9.4.00
 */
public class MoEInAppHelper {

  private static final String TAG = InAppConstants.MODULE_TAG + "MoEInAppHelper";

  private MoEInAppHelper() {
    inAppMessageListener = new InAppMessageListener();
  }

  private InAppMessageListener inAppMessageListener;

  /**
   * Register a callback to be get notified whenever in-app is shown, clicked, closed.
   *
   * @param listener instance of {@link InAppMessageListener}
   * @since 10.0.00
   */
  public void registerListener(InAppMessageListener listener) {
    inAppMessageListener = listener;
  }

  /**
   * Returns instance of {@link InAppMessageListener} listener is set by
   * {@link MoEInAppHelper#registerListener(InAppMessageListener)} else null.
   *
   * @return instance of {@link InAppMessageListener}.
   * @since 10.0.00
   */
  public InAppMessageListener getListener() {
    return inAppMessageListener;
  }

  private static MoEInAppHelper instance = null;

  /**
   * Returns a singleton instance of {@link MoEInAppHelper}
   *
   * @return instance of {@link MoEInAppHelper}
   */
  public static MoEInAppHelper getInstance() {
    if (instance == null) {
      synchronized (MoEInAppHelper.class) {
        if (instance == null) instance = new MoEInAppHelper();
      }
    }
    return instance;
  }

  /**
   * Try to show a In-App Message
   *
   * @param context instance of {@link Context}
   * @since 9.4.00
   */
  public void showInApp(Context context) {
    Logger.v(TAG + " showInApp() : Inside showInApp() will try to show in-app if possible.");
    if (!InAppController.getInstance().isModuleEnabled(context)){
      Logger.v(TAG + " showInApp() : InApp Module is disabled. Cannot show in-app.");
      return;
    }
    if (!SdkConfig.getConfig().isLifecycleInAppOptedOut) {
      Logger.w(TAG
          + " showInApp() : Hasn't opted out of lifecycle campaign. Cannot show "
          + "campaigns using this method. Opt out of default display in the MoEngage.Builder "
          + "to use this API");
      return;
    }
    if (!InAppController.getInstance().isInAppSynced()) {
      Logger.v(TAG
          + " showInApp() : Either campaign sync is pending or campaign cannot be shown because "
          + "of global delay.");
      InAppController.getInstance().setShowInAppPending(true);
      return;
    }
    Logger.v(TAG + " showInApp() : Will try to show in-app");
    InAppController.getInstance().tryToShowInApp(context);
  }

  /**
   * Try to return a self handled in-app to the callback listener. Ensure self handled in-app
   * listener is set before you call this.<br>
   *
   * @param context instance of {@link Context}
   * @since 9.4.00
   */
  public void getSelfHandledInApp(Context context) {
    Logger.v(TAG + " getSelfHandledInApp() : Inside getSelfHandledInApp() will try to return "
        + "in-app if possible.");
    if (!InAppController.getInstance().isModuleEnabled(context)) {
      Logger.v(TAG + " getSelfHandledInApp() : InApp Module is disabled. Cannot show in-app.");
      return;
    }
    if (!InAppController.getInstance().isInAppSynced()) {
      Logger.v(TAG
          + " getSelfHandledInApp() : Either campaign sync is pending or campaign cannot be shown"
          + " because of global delay.");
      InAppController.getInstance().setSelfHandledInAppPending(true);
      return;
    }
    Logger.v(TAG + " getSelfHandledInApp() : Will try to return self handled in-app.");
    InAppController.getInstance().tryToShowSelfHandledInApp(context);
  }

  /**
   * Mark self-handled campaign as shown.
   * <b>Note:</b> If this API is called for any other campaign type request will be ignored.
   *
   * @param context instance of {@link Context}
   * @param campaign instance of {@link MoEInAppCampaign} passed in the callback.
   * @since 10.0.00
   */
  public void selfHandledShown(Context context, @NonNull MoEInAppCampaign campaign) {
    if (campaign == null || MoEUtils.isEmptyString(campaign.campaignId)) {
      Logger.w(TAG + " selfHandledCampaignShown() : Campaign object or Id is null. Ignoring "
          + "request.");
      return;
    }
    InAppController.getInstance().trackInAppShown(context, campaign.campaignId, campaign.campaignName);
    TaskManager.getInstance().addTaskToQueue(new UpdateCampaignStateTask(context,
        campaign.campaignId, StateUpdateType.SHOWN, true));
  }

  /**
   * Mark self-handled campaign as clicked.
   * <b>Note:</b> If this API is called for any other campaign type request will be ignored.
   *
   * @param context instance of {@link Context}
   * @param campaign instance of {@link MoEInAppCampaign} passed in the callback.
   * @since 10.0.00
   */
  public void selfHandledClicked(Context context, @NonNull MoEInAppCampaign campaign) {
    selfHandledClicked(context, campaign, -1);
  }

  /**
   * Mark self-handled campaign as clicked.
   * <b>Note:</b> If this API is called for any other campaign type request will be ignored.
   *
   * @param context instance of {@link Context}
   * @param campaign instance of {@link MoEInAppCampaign} passed in the callback.
   * @param widgetId id of the widget which was clicked.
   * @since 10.0.00
   */
  public void selfHandledClicked(Context context, @NonNull MoEInAppCampaign campaign,
      int widgetId) {
    Logger.v(TAG + " selfHandledClicked() : Will log self handled click. " + campaign);
    if (campaign == null || campaign.selfHandledCampaign == null || MoEUtils.isEmptyString(
        campaign.campaignId)) {
      Logger.w(TAG
          + " selfHandledClicked() : Ignoring request, Reason - one of the following - \n 1. "
          + "Campaign object is null \n 2. Campaign is not of type self handled \n 3. Campaign is"
          + " an empty string.");
      return;
    }
    Properties properties = new Properties();
    properties.addAttribute(MoEConstants.MOE_CAMPAIGN_ID, campaign.campaignId)
        .addAttribute(MoEConstants.MOE_CAMPAIGN_NAME, campaign.campaignName);
    if (widgetId > 0) {
      properties.addAttribute(MoEConstants.ATTRIBUTE_WIDGET_ID, widgetId);
    }
    MoEHelper.getInstance(context).trackEvent(MoEConstants.EVENT_IN_APP_CLICKED, properties);
  }

  /**
   * Mark self-handled campaign as primary clicked.
   * <b>Note:</b> If this API is called for any other campaign type request will be ignored.
   *
   * @param context instance of {@link Context}
   * @param campaign instance of {@link MoEInAppCampaign} passed in the callback.
   * @since 10.0.00
   */
  public void selfHandledPrimaryClicked(Context context, @NonNull MoEInAppCampaign campaign) {
    if (campaign == null || MoEUtils.isEmptyString(campaign.campaignId)) {
      Logger.w(TAG + " selfHandledPrimaryClicked() : Ignoring request, Reason - one of the "
          + "following - \n 1. Campaign object is null \n 2. Campaign is not of type self handled"
          + " \n 3. Campaign is an empty string.");
      return;
    }
    TaskManager.getInstance().addTaskToQueue(new UpdateCampaignStateTask(context,
        campaign.campaignId, StateUpdateType.CLICKED, true));
  }

  /**
   * Notify the SDK that self-handled campaign has been dismissed.
   * <b>Note:</b> If this API is called for any other campaign type request will be ignored.
   *
   * @param context instance of {@link Context}
   * @param campaign instance of {@link MoEInAppCampaign} passed in the callback.
   * @since 10.0.00
   */
  public void selfHandledDismissed(Context context, MoEInAppCampaign campaign) {
    Logger.v(TAG + " selfHandledDismissed() : Will log self-handled dismissed. " + campaign);
    if (campaign == null || campaign.selfHandledCampaign == null ||
        MoEUtils.isEmptyString(campaign.campaignId)) {
      Logger.w(TAG + " selfHandledDismissed() : Campaign object or Id is null. Ignoring request, "
          + "Reason - one of the following - \n 1. Campaign object is null \n 2. Campaign is not "
          + "of type self handled \n 3. Campaign is an empty string.");
      return;
    }
    Properties properties = new Properties();
    properties.addAttribute(MoEConstants.MOE_CAMPAIGN_ID, campaign.campaignId)
        .addAttribute(MoEConstants.MOE_CAMPAIGN_NAME, campaign.campaignName);
    MoEHelper.getInstance(context).trackEvent(MoEConstants.EVENT_IN_APP_DISMISSED, properties);
  }
}
