/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.inapp;

import android.content.Context;
import android.util.DisplayMetrics;
import android.view.View;
import com.moe.pushlibrary.MoEHelper;
import com.moengage.core.Logger;
import com.moengage.core.MoEUtils;
import com.moengage.inapp.model.InAppGlobalState;
import com.moengage.inapp.model.ViewDimension;
import java.util.ArrayList;
import java.util.List;

/**
 * @author Umang Chamaria
 * Date: 2019-09-01
 */
public class InAppUtils {

  private static final String TAG = InAppConstants.MODULE_TAG + "InAppUtils";

  private InAppUtils() {

  }

  public static int getStatusBarHeight(Context context) {
    int marginTop = 0;
    int resource = context.getResources().getIdentifier("status_bar_height", "dimen", "android");
    if (resource > 0) {
      marginTop = context.getResources().getDimensionPixelSize(resource);
    }
    return marginTop;
  }

  public static ViewDimension getScreenDimension(Context context) {
    DisplayMetrics displayMetrics = context.getResources().getDisplayMetrics();
    return new ViewDimension(displayMetrics.widthPixels, displayMetrics.heightPixels);
  }

  public static void logCurrentInAppState(Context context) {

    Logger.v(
        TAG + " logCurrentInAppState() : Current Activity Name: " + InAppController.getInstance()
            .getCurrentActivityName());
    List<String> contextList = MoEHelper.getInstance(context).getAppContext();
    if (contextList != null) {
      Logger.v(TAG + " logCurrentInAppState() : Current context: " + contextList);
    } else {
      Logger.v(TAG + " logCurrentInAppState() : No context set.");
    }
    InAppGlobalState globalState =
        InAppInjector.getInstance().getInAppRepository(context).localRepository.getGlobalState();
    Logger.v(TAG + " logCurrentInAppState() : Global Delay: " + globalState.globalDelay
        + "\n Last campaign shown at: " + MoEUtils.isoStringFromSeconds(globalState.lastShowTime)
        + "\n Current time: " + MoEUtils.isoStringFromSeconds(globalState.currentDeviceTime)
    );
  }

  public static boolean hasGifSupport() {
    try {
      // Check for a subset of classes used from the Glide library.
      ClassLoader staticClassLoader = InAppUtils.class.getClassLoader();
      boolean isGlideOnPath = true;
      for (String classPath : USED_GLIDE_CLASSES) {
        if (Class.forName(classPath, false, staticClassLoader) == null) {
          isGlideOnPath = false;
          break;
        }
      }
      return isGlideOnPath;
    } catch (Exception e) {
      Logger.v(TAG + " hasGifSupport() Glide library not found");
    } catch (NoClassDefFoundError ncd) {
      Logger.v(TAG + " hasGifSupport() Glide library not found");
    } catch (Throwable t) {
      Logger.v(TAG + " hasGifSupport() Glide library not found");
    }
    return false;
  }

  public static ViewDimension getUnspecifiedViewDimension(View view) {
    view.measure(View.MeasureSpec.UNSPECIFIED, View.MeasureSpec.UNSPECIFIED);
    return new ViewDimension(view.getMeasuredWidth(), view.getMeasuredHeight());
  }

  public static boolean isInAppExceedingScreen(ViewDimension inAppDimensions,
      ViewDimension screenDimensions) {
    return screenDimensions.height < inAppDimensions.height;
  }

  private static final String[] USED_GLIDE_CLASSES = {
      "com.bumptech.glide.load.resource.gif.GifDrawable",
      "com.bumptech.glide.Glide"
  };

  private static final ArrayList<String> OEMS_DONT_EXCLUDE_STATUS_BAR = new ArrayList<>();

  static {
    OEMS_DONT_EXCLUDE_STATUS_BAR.add("HMD Global");
    OEMS_DONT_EXCLUDE_STATUS_BAR.add("OnePlus");
  }
}
