/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.inapp;

import androidx.annotation.Nullable;
import com.moengage.core.Logger;
import com.moengage.core.MoEUtils;
import com.moengage.core.SdkConfig;
import com.moengage.inapp.model.InAppGlobalState;
import com.moengage.inapp.model.enums.EvaluationStatusCode;
import com.moengage.inapp.model.meta.CampaignMeta;
import com.moengage.inapp.model.meta.CampaignState;
import com.moengage.inapp.model.meta.InAppCampaign;
import java.util.List;
import java.util.Set;

/**
 * @author Umang Chamaria
 */
public class InAppEvaluator {

  private static final String TAG = InAppConstants.MODULE_TAG + "InAppEvaluator";

  public boolean canShowInAppOnActivity(String activityName,
      @Nullable List<String> blockedActivityList) {
    if (blockedActivityList != null && blockedActivityList.contains(activityName)) {
      Logger.i(TAG
          + " canShowInAppOnActivity() : reason: in-app blocked on screen. Screen Name: "
          + activityName);
      return false;
    }
    return true;
  }

  public EvaluationStatusCode isCampaignEligibleForDisplay(InAppCampaign inAppCampaign, List<String> contextList,
      String currentActivityName, InAppGlobalState globalState) {
    CampaignMeta meta = inAppCampaign.campaignMeta;
    CampaignState campaignState = inAppCampaign.campaignState;
    Logger.v(TAG
        + " isCampaignEligibleForDisplay() : Evaluating campaign: "
        + meta.campaignId
        + "\n Campaign meta: "
        + meta
        + "\n Campaign state: "
        + campaignState);

    if (!canShowInAppOnActivity(currentActivityName, SdkConfig.getConfig().inAppOptOutList)) {
      Logger.i(TAG
          + " isCampaignEligibleForDisplay() : Cannot show campaign: "
          + meta.campaignId
          + "reason: in-app blocked on screen.");
      return EvaluationStatusCode.BLOCKED_ON_SCREEN;
    }
    Logger.v(TAG + " isCampaignEligibleForDisplay() : InApp Blocked on screen check passed.");
    if ((globalState.lastShowTime + globalState.globalDelay
        > globalState.currentDeviceTime)
        && !meta.deliveryControl.frequencyCapping.ignoreGlobalDelay) {
      Logger.i(TAG
          + " isCampaignEligibleForDisplay() : Cannot show campaign: "
          + meta.campaignId
          + "reason: global delay failure");
      return EvaluationStatusCode.GLOBAL_DELAY;
    }
    Logger.v(TAG + " isCampaignEligibleForDisplay() : Global minimum delay check passed.");
    if (meta.expiryTime < globalState.currentDeviceTime) {
      Logger.i(TAG
          + " isCampaignEligibleForDisplay() : Cannot show campaign: "
          + meta.campaignId
          + "reason: campaign expired");
      return EvaluationStatusCode.EXPIRY;
    }
    Logger.v(TAG + " isCampaignEligibleForDisplay() : Campaign expiry check passed.");
    if (meta.displayControl.rules.screenName != null
        && !meta.displayControl.rules.screenName.equals(currentActivityName)) {
      Logger.i(TAG
          + " isCampaignEligibleForDisplay() : Cannot show campaign: "
          + meta.campaignId
          + "reason: cannot show in-app on this screen");
      return EvaluationStatusCode.INVALID_SCREEN;
    }
    Logger.v(TAG + " isCampaignEligibleForDisplay(): Show only in screen check has passed ");
    if (meta.displayControl.rules.context != null && !meta.displayControl.rules.context.isEmpty()) {
      if (contextList == null) return EvaluationStatusCode.INVALID_CONTEXT;
      boolean contextExists = false;
      Set<String> contextToShowIn = meta.displayControl.rules.context;
      for (String context : contextList) {
        if (contextToShowIn.contains(context)) {
          contextExists = true;
        }
      }
      if (!contextExists) {
        Logger.i(TAG
            + " isCampaignEligibleForDisplay() : Cannot show campaign: "
            + meta.campaignId
            + "reason: current contextList not as");
        return EvaluationStatusCode.INVALID_CONTEXT;
      }
    }
    Logger.v(TAG + " isCampaignEligibleForDisplay(): Context check has passed.");
    if (!meta.deliveryControl.persistent && campaignState.isClicked) {
      Logger.i(TAG
          + " isCampaignEligibleForDisplay() : Cannot show campaign: "
          + meta.campaignId
          + "reason: already clicked and campaign is not persistent");
      return EvaluationStatusCode.PERSISTENT;
    }
    Logger.v(TAG + " isCampaignEligibleForDisplay(): Persistent check passed.");
    if (meta.deliveryControl.frequencyCapping.maxCount > 0
        && campaignState.showCount >= meta.deliveryControl.frequencyCapping.maxCount) {
      Logger.i(TAG
          + " isCampaignEligibleForDisplay() : Cannot show campaign: "
          + meta.campaignId
          + "reason: already shown max times");
      return EvaluationStatusCode.MAX_COUNT;
    }
    Logger.v(TAG + " isCampaignEligibleForDisplay(): Max count check passed.");
    if (campaignState.lastShowTime + meta
        .deliveryControl.frequencyCapping.minimumDelay > globalState.currentDeviceTime) {
      Logger.i(TAG
          + " isCampaignEligibleForDisplay() : Cannot show campaign: "
          + meta.campaignId
          + "reason: minimum delay between same campaign");
      return EvaluationStatusCode.CAMPAIGN_DELAY;
    }
    Logger.v(
        TAG + " isCampaignEligibleForDisplay(): Minimum delay between same campaign check passed.");
    return EvaluationStatusCode.SUCCESS;
  }

  @Nullable public InAppCampaign getEligibleCampaignFromList(List<InAppCampaign> campaignList,
      InAppGlobalState globalState, List<String> appContext) {
    StatsLogger.getInstance().logCampaignAttempted(campaignList);
    InAppCampaign suitableCampaign = null;
    int index = 0;
    for (;index<campaignList.size();index++){
      InAppCampaign campaign = campaignList.get(index);
      EvaluationStatusCode statusCode = isCampaignEligibleForDisplay(campaign, appContext,
          InAppController.getInstance().getCurrentActivityName(), globalState);
      if ( statusCode == EvaluationStatusCode.SUCCESS) {
        suitableCampaign = campaign;
        break;
      }else {
        StatsLogger.getInstance().logPriorityStageFailure(campaign, statusCode);
      }
    }
    if (suitableCampaign != null){
      index++;
      String timestamp = MoEUtils.currentISOTime();
      for (; index < campaignList.size(); index++) {
        InAppCampaign campaign = campaignList.get(index);
        StatsLogger.getInstance().updateStatForCampaign(campaign.campaignMeta.campaignId,
            timestamp, StatsLogger.PRIORITY_STAGE_HIGHER_PRIORITY_CAMPAIGN_AVAILABLE);
      }
    }
    return suitableCampaign;
  }

  public boolean isServerSyncRequired(long lastSyncTime, long currentTime,
      long syncInterval, boolean isInAppSynced) {
    return !isInAppSynced || lastSyncTime + syncInterval < currentTime;
  }
}
