/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.inapp;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.content.Context;
import android.os.Bundle;
import android.os.Handler;
import android.os.Looper;
import android.view.View;
import android.view.ViewGroup;
import android.view.animation.Animation;
import android.view.animation.AnimationUtils;
import android.widget.FrameLayout;
import androidx.annotation.Nullable;
import androidx.annotation.RestrictTo;
import androidx.annotation.RestrictTo.Scope;
import com.moe.pushlibrary.MoEHelper;
import com.moe.pushlibrary.models.Event;
import com.moengage.core.ConfigurationProvider;
import com.moengage.core.Logger;
import com.moengage.core.MoEConstants;
import com.moengage.core.MoEUtils;
import com.moengage.core.Properties;
import com.moengage.core.RemoteConfig;
import com.moengage.core.SdkConfig;
import com.moengage.core.executor.TaskManager;
import com.moengage.inapp.listeners.InAppMessageListener;
import com.moengage.inapp.model.CampaignPayload;
import com.moengage.inapp.model.MoEInAppCampaign;
import com.moengage.inapp.model.SelfHandledCampaign;
import com.moengage.inapp.model.enums.EvaluationStatusCode;
import com.moengage.inapp.model.enums.StateUpdateType;
import com.moengage.inapp.model.meta.InAppCampaign;
import com.moengage.inapp.model.style.ContainerStyle;
import com.moengage.inapp.repository.remote.FetchMetaTask;
import com.moengage.inapp.tasks.ShowInAppTask;
import com.moengage.inapp.tasks.ShowSelfHandledInAppTask;
import com.moengage.inapp.tasks.ShowTestInAppTask;
import com.moengage.inapp.tasks.ShowTriggerInAppTask;
import com.moengage.inapp.tasks.UpdateCampaignStateTask;
import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Observer;
import java.util.Set;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import org.json.JSONObject;

/**
 * @author Umang Chamaria
 */
@RestrictTo(Scope.LIBRARY)
public class InAppController {

  private static final String TAG = InAppConstants.MODULE_TAG + "InAppController";

  private static InAppController instance = null;
  private WeakReference<Activity> currentActivity = null;
  private AtomicBoolean inAppManagerState;
  private boolean isInAppSynced = false;
  private boolean isShowInAppPending = false;
  private boolean isSelfHandledInAppPending = false;
  private boolean isInAppShowing = false;

  public Handler mainThreadHandler;
  private List<Event> pendingTriggerEvents;
  private SyncCompleteObservable syncObservable;
  private ScheduledExecutorService scheduledExecutorService;

  private InAppController() {
    inAppManagerState = new AtomicBoolean();
    mainThreadHandler = new Handler(Looper.getMainLooper());
    pendingTriggerEvents = new ArrayList<>();
    syncObservable = new SyncCompleteObservable();
    visibleOrProcessingNudge = new HashSet<>();

  }

  public static InAppController getInstance() {
    if (instance == null) {
      synchronized (InAppController.class) {
        if (instance == null) instance = new InAppController();
      }
    }
    return instance;
  }

  void registerActivity(Activity activity) {
    updateCurrentActivityContext(activity);
    inAppManagerState.set(true);
  }

  void unRegisterActivity(Activity activity) {
    try {
      if (currentActivity != null && currentActivity.getClass()
          .getName()
          .equals(activity.getClass().getName())) {
        updateCurrentActivityContext(null);
        inAppManagerState.set(false);
      }
    } catch (Exception e) {
      Logger.e(TAG + " unRegisterActivity() : ", e);
      inAppManagerState.set(false);
    }
  }

  private void updateCurrentActivityContext(Activity activity) {
    currentActivity = activity == null ? null : new WeakReference<>(activity);
  }

  @Nullable public String getCurrentActivityName() {
    if (currentActivity == null) return null;
    Activity activity = currentActivity.get();
    if (activity == null) return null;
    return activity.getClass().getName();
  }

  void tryToShowSelfHandledInApp(Context context) {
    TaskManager.getInstance().startTask(new ShowSelfHandledInAppTask(context));
  }

  public boolean canShowInAppForConfig(Context context, List<InAppCampaign> campaignList) {
    if (isTablet(context)) {
      Logger.v(TAG + "canShowInAppForConfig() : Cannot show in-app on tablet will return.");
      return false;
    }
    if (isLandscapeMode(context)) {
      Logger.v(TAG + "canShowInAppForConfig() : Cannot show in-app in landscape mode will return");
      StatsLogger.getInstance().logDeviceOrientationNotSupported(campaignList);
      return false;
    }
    return true;
  }

  void tryToShowInApp(Context context) {
    TaskManager.getInstance().startTask(new ShowInAppTask(context));
  }

  public boolean isTablet(Context context) {
    return context.getResources().getBoolean(R.bool.moeIsTablet);
  }

  public boolean isLandscapeMode(Context context) {
    return context.getResources().getBoolean(R.bool.moeIsLand);
  }

  public void buildAndShowInApp(Context context, InAppCampaign suitableInApp,
      CampaignPayload campaignPayload) {
    ViewCreationMeta viewCreationMeta = new ViewCreationMeta(InAppUtils.getScreenDimension(context),
        InAppUtils.getStatusBarHeight(context));

    View view = buildInApp(campaignPayload, viewCreationMeta);
    if (view == null) {
      Logger.v(TAG + " buildAndShowInApp() : Could not create view for in-app "
          + "campaign " + suitableInApp.campaignMeta.campaignId);
      return;
    }

    if (canShowInApp(context, suitableInApp, view)) {
      showInApp(view, viewCreationMeta, campaignPayload);
    }
  }

  @Nullable public View buildInApp(CampaignPayload campaignPayload,
      ViewCreationMeta viewCreationMeta) {
    Activity activity = currentActivity.get();
    if (activity == null) {
      Logger.v(TAG + " buildInApp() : Cannot build in-app without activity. Aborting in-app "
          + "creation");
      return null;
    }
    return new ViewEngine(activity, campaignPayload, viewCreationMeta).createInApp();
  }

  private void showInApp(View view, ViewCreationMeta viewCreationMeta,
      CampaignPayload campaignPayload) {
    Logger.v(TAG
        + " showInApp() : Will try to show in-app. Campaign id: "
        + campaignPayload.campaignId);
    Activity activity = currentActivity.get();
    if (activity == null) {
      Logger.v(TAG + " showInApp() : Cannot show campaign activity reference is null");
      return;
    }
    addInAppToViewHierarchy(activity, view, campaignPayload);
  }

  public void addInAppToViewHierarchy(final Activity currentActivity, final View inAppView,
      final CampaignPayload campaignPayload) {
    hideStatusBarIfRequired(currentActivity);
    mainThreadHandler.post(new Runnable() {
      @Override public void run() {
        final FrameLayout root = getWindowRoot(currentActivity);
        root.addView(inAppView);
        isInAppShowing = true;
        autoDismissInAppIfRequired(root, campaignPayload, inAppView, currentActivity);
        onInAppShown(currentActivity.getApplicationContext(), campaignPayload);
      }
    });
  }

  private void autoDismissInAppIfRequired(final FrameLayout root,
      final CampaignPayload campaignPayload,
      final View inAppView, final Activity currentActivity) {
    if (campaignPayload.dismissInterval > 0) {
      mainThreadHandler.postDelayed(new Runnable() {
        @Override public void run() {
          if (root.indexOfChild(inAppView) == -1) {
            Logger.v(TAG
                + " autoDismissInAppIfRequired() : in-app was closed before being auto dismissed");
          } else {
            removeViewFromHierarchy(campaignPayload, currentActivity, inAppView);
            onAutoDismiss(currentActivity.getApplicationContext(), campaignPayload);
          }
        }
      }, campaignPayload.dismissInterval * 1000);
    }
  }

  FrameLayout getWindowRoot(Activity activity) {
    return (FrameLayout) activity.getWindow()
        .getDecorView()
        .findViewById(android.R.id.content)
        .getRootView();
  }

  @SuppressLint("ResourceType") void removeViewFromHierarchy(CampaignPayload campaignPayload,
      Context context, View inAppView) {
    try {
      ContainerStyle style = (ContainerStyle) campaignPayload.primaryContainer.style;
      if (style.animation != null &&
          style.animation.exit != -1) {
        Animation animation = AnimationUtils.loadAnimation(context, style.animation.exit);
        inAppView.setAnimation(animation);
      }
      ((ViewGroup) inAppView.getParent()).removeView(inAppView);
    } catch (Exception e) {
      Logger.e(TAG + " removeViewFromHierarchy() : ", e);
    }
  }

  private boolean canShowInApp(Context context, InAppCampaign inAppCampaign, View inAppView) {
    if (isInAppShowing) {
      Logger.i(TAG + " canShowInApp() : InApp is already being shown. Cannot show another in-app.");
      StatsLogger.getInstance().updateStatForCampaign(inAppCampaign.campaignMeta.campaignId,
          MoEUtils.currentISOTime(), StatsLogger.IMPRESSION_STAGE_ANOTHER_CAMPAIGN_VISIBLE);
      return false;
    }
    if (isLandscapeMode(context)) {
      Logger.i(TAG + " canShowInApp() : Cannot show in-app for config.");
      StatsLogger.getInstance().updateStatForCampaign(inAppCampaign.campaignMeta.campaignId,
          MoEUtils.currentISOTime(), StatsLogger.IMPRESSION_STAGE_ORIENTATION_UNSUPPORTED);
      return false;
    }
    InAppEvaluator evaluator = new InAppEvaluator();
    EvaluationStatusCode statusCode = evaluator.isCampaignEligibleForDisplay(inAppCampaign,
        MoEHelper.getInstance(context).getAppContext(),
        getCurrentActivityName(),
        InAppInjector.getInstance().getInAppRepository(context).localRepository.getGlobalState());
    if (statusCode != EvaluationStatusCode.SUCCESS) {
      Logger.i(TAG + " canShowInApp() : Cannot show in-app, conditions don't satisfy.");
      StatsLogger.getInstance().logImpressionStageFailure(inAppCampaign, statusCode);
      return false;
    }
    if (InAppUtils.isInAppExceedingScreen(InAppUtils.getUnspecifiedViewDimension(inAppView),
        InAppUtils.getScreenDimension(context))){
      Logger.i( TAG + " canShowInApp() : Cannot show in-app, view dimensions exceed device "
          + "dimensions.");
      StatsLogger.getInstance().updateStatForCampaign(inAppCampaign.campaignMeta.campaignId,
          MoEUtils.currentISOTime(), StatsLogger.IMPRESSION_STAGE_HEIGHT_EXCEEDS_DEVICE);
      return false;
    }
    return true;
  }

  public boolean isInAppSynced() {
    return isInAppSynced;
  }

  void setInAppSynced(boolean inAppSynced) {
    isInAppSynced = inAppSynced;
  }

  private boolean isShowInAppPending() {
    return isShowInAppPending;
  }

  void setShowInAppPending(boolean showInAppPending) {
    isShowInAppPending = showInAppPending;
  }

  private boolean isSelfHandledInAppPending() {
    return isSelfHandledInAppPending;
  }

  void setSelfHandledInAppPending(boolean selfHandledInAppPending) {
    isSelfHandledInAppPending = selfHandledInAppPending;
  }

  @RestrictTo(Scope.LIBRARY)
  public void tryToShowTriggerInAppIfPossible(Context context, Event event) {
    if (!isInAppSynced) {
      Logger.v(TAG + " tryToShowTriggerInAppIfPossible() : In-App has not synced. Will show "
          + "try to show trigger in-app after sync, queueing event.");
      pendingTriggerEvents.add(event);
      return;
    }
    Set<String> triggerEvents =
        InAppInjector.getInstance().getInAppRepository(context).cache.triggerEvents;
    if (triggerEvents != null && triggerEvents.contains(event.eventName)) {
      TaskManager.getInstance().startTask(new ShowTriggerInAppTask(context, event));
    }
  }

  void syncInApps(Context context) {
    try {
      if (isTablet(context)) {
        Logger.i(
            "$tag syncInAppIfRequired() : Cannot show in-apps on tablet. No point making a sync "
                + "request.");
        return;
      }
      TaskManager.getInstance().addTaskToQueue(new FetchMetaTask(context));
    } catch (Exception e) {
      Logger.i(TAG + " syncInAppIfRequired() : ", e);
    }
  }

  void handleDismiss(CampaignPayload campaignPayload) {
    isInAppShowing = false;
    removeProcessingNudge(campaignPayload.campaignId);
    notifyInAppClosedListeners(campaignPayload);
    Activity activity = getCurrentActivity();
    if (activity != null){
      showStatusBarIfRequired(activity);
    }
  }

  private void notifyInAppShownListeners(CampaignPayload campaign) {
    final MoEInAppCampaign inAppCampaign = new MoEInAppCampaign(campaign.campaignId,
        campaign.campaignName);
    mainThreadHandler.post(new Runnable() {
      @Override public void run() {
        InAppMessageListener listener = MoEInAppHelper.getInstance().getListener();
        listener.onShown(inAppCampaign);
      }
    });
  }

  private void notifyInAppClosedListeners(CampaignPayload campaignPayload) {
    final MoEInAppCampaign inAppCampaign = new MoEInAppCampaign(campaignPayload.campaignId,
        campaignPayload.campaignName);
    mainThreadHandler.post(new Runnable() {
      @Override public void run() {
        InAppMessageListener listener = MoEInAppHelper.getInstance().getListener();
        listener.onClosed(inAppCampaign);
      }
    });
  }

  public void onSyncSuccess(Context context) {
    Logger.v(TAG + " onSyncSuccess() : Sync successful will try to process pending showInApp if "
        + "required.");
    setInAppSynced(true);
    syncObservable.onSyncSuccess();
    if (InAppInjector.getInstance().getInAppRepository(context).isLifeCycleCallbackOptedOut()) {
      Logger.v(TAG + " onSyncSuccess() : Lifecycle callbacks is opted out, will check if explicit "
          + "calls are pending.");
      if (isShowInAppPending()) {
        MoEInAppHelper.getInstance().showInApp(context);
        setShowInAppPending(false);
      }
      if (isSelfHandledInAppPending()) {
        MoEInAppHelper.getInstance().getSelfHandledInApp(context);
        setSelfHandledInAppPending(false);
      }
    } else {
      Logger.v(TAG + " onSyncSuccess() : Lifecycle in-apps are pending, will try to show.");
      tryToShowInApp(context);
      tryToShowSelfHandledInApp(context);
    }
  }

  @RestrictTo(Scope.LIBRARY)
  public void onInAppShown(Context context, CampaignPayload inAppCampaign) {
    trackInAppShown(context, inAppCampaign.campaignId, inAppCampaign.campaignName);
    notifyInAppShownListeners(inAppCampaign);

    TaskManager.getInstance().addTaskToQueue(new UpdateCampaignStateTask(context,
        StateUpdateType.SHOWN, inAppCampaign.campaignId));
  }

  void logPrimaryWidgetClicked(Context context, String campaignId) {
    TaskManager.getInstance()
        .addTaskToQueue(new UpdateCampaignStateTask(context, StateUpdateType.CLICKED,
            campaignId));
  }

  private void onAutoDismiss(Context context, CampaignPayload campaignPayload) {
    handleDismiss(campaignPayload);
    Properties properties = new Properties();
    properties.addAttribute(MoEConstants.MOE_CAMPAIGN_NAME, campaignPayload.campaignName)
        .addAttribute(MoEConstants.MOE_CAMPAIGN_ID, campaignPayload.campaignId)
        .setNonInteractive();
    MoEHelper.getInstance(context)
        .trackEvent(MoEConstants.EVENT_IN_APP_AUTO_DISMISS, properties);
  }

  @RestrictTo(Scope.LIBRARY)
  public void clearPendingEvents() {
    Logger.v(TAG + " clearPendingEvents() : Will clear pending events.");
    pendingTriggerEvents.clear();
  }

  public List<Event> getPendingTriggerEvents() {
    return pendingTriggerEvents;
  }

  @Nullable public Activity getCurrentActivity() {
    if (currentActivity == null) return null;
    return currentActivity.get();
  }

  public void registerSyncObserver(Observer observer) {
    syncObservable.addObserver(observer);
  }

  public void unregisterSyncObserver(Observer observer) {
    syncObservable.deleteObserver(observer);
  }

  void showInAppFromPush(final Context context, Bundle pushPayload) {
    try {
      Logger.v(TAG
          + " showInAppFromPush() : Will try to show inapp from push. Metadata: \n"
          + pushPayload);
      String campaignId;
      boolean isTest;
      long timeDelay = 5;
      if (pushPayload.containsKey(MoEConstants.PUSH_EXTRA_INAPP_META)){
        JSONObject inAppMetaJson = new JSONObject(pushPayload.getString(
            MoEConstants.PUSH_EXTRA_INAPP_META));
        campaignId = inAppMetaJson.getString(InAppConstants.PUSH_ATTR_CAMPAIGN_ID);
        isTest = inAppMetaJson.optBoolean(InAppConstants.PUSH_ATTR_IS_TEST_CAMPAIGN, false);
        timeDelay = inAppMetaJson.optLong(InAppConstants.PUSH_ATTR_TRIGGER_DELAY, 5);
      }else if (pushPayload.containsKey(MoEConstants.PUSH_EXTRA_INAPP_LEGACY_META)){
        campaignId = pushPayload.getString(MoEConstants.PUSH_EXTRA_INAPP_LEGACY_META);
        isTest = true;
      }else {
        Logger.v(TAG + " showInAppFromPush() : InApp meta data missing cannot show campaign.");
        return;
      }
      if (MoEUtils.isEmptyString(campaignId)) {
        Logger.v(TAG + " showInAppFromPush() : Cannot show in-app. Campaign id missing.");
        return;
      }
      if (scheduledExecutorService == null || scheduledExecutorService.isShutdown()) {
        scheduledExecutorService = Executors.newScheduledThreadPool(1);
      }

      final boolean finalIsTest = isTest;
      final String finalCampaignId = campaignId;
      scheduledExecutorService.schedule(new Runnable() {
        @Override public void run() {
          if (finalIsTest) {
            TaskManager.getInstance().startTask(new ShowTestInAppTask(context, finalCampaignId));
          }
        }
      }, timeDelay, TimeUnit.SECONDS);
    } catch (Exception e) {
      Logger.e(TAG + " showInAppFromPush() : ", e);
    }
  }

  void onAppClose(Context context) {
    clearPendingEvents();
    if (scheduledExecutorService != null) {
      scheduledExecutorService.shutdown();
    }
  }

  private void hideStatusBarIfRequired(final Activity activity) {
    if (SdkConfig.getConfig().isNavBarOptedOut) return;
    activity.runOnUiThread(new Runnable() {
      @Override public void run() {
        View decorView = activity.getWindow().getDecorView();
        decorView.setSystemUiVisibility(View.SYSTEM_UI_FLAG_FULLSCREEN);
      }
    });
  }

  private void showStatusBarIfRequired(final Activity activity) {
    if (SdkConfig.getConfig().isNavBarOptedOut) return;
    if (activity != null) {
      View decorView = activity.getWindow().getDecorView();
      decorView.setSystemUiVisibility(View.SYSTEM_UI_FLAG_VISIBLE);
    }
  }

  public void onSelfHandledAvailable(final CampaignPayload campaignPayload){
    mainThreadHandler.post(new Runnable() {
      @Override public void run() {
        MoEInAppHelper.getInstance().getListener().onSelfHandledAvailable(
            new MoEInAppCampaign(campaignPayload.campaignId,
                campaignPayload.campaignName,
                new SelfHandledCampaign(campaignPayload.customPayload,
                    campaignPayload.dismissInterval,
                    campaignPayload.isCancellable)));
      }
    });
  }

  /**
   * Checks whether in-app module should be enabled or not. Takes into account GDPR flag, account
   * status and in-app status from config api.
   * @param context instance of {@link Context}
   * @return true is module is enabled, else false.
   */
  public boolean isModuleEnabled(Context context){
    RemoteConfig config = RemoteConfig.getConfig();
    return !ConfigurationProvider.getInstance(context).isInAppOptedOut()
        && config.isInAppEnabled
        && config.isAppEnabled;
  }

  private final Object lock = new Object();
  private Set<String> visibleOrProcessingNudge;

  public InAppCampaign getEligibleNudgeView(Context context,
      Map<String, InAppCampaign> campaignMap){
    synchronized (lock) {
      Logger.v(TAG + " getEligibleNudgeView() : Campaigns Being show or processed: " + visibleOrProcessingNudge);
      if (campaignMap == null) return null;
      InAppEvaluator evaluator = new InAppEvaluator();
      for (String campaignId : visibleOrProcessingNudge) {
        campaignMap.remove(campaignId);
      }
      if (campaignMap.isEmpty()) return null;
      InAppCampaign campaign =
          evaluator.getEligibleCampaignFromList(new ArrayList<>(campaignMap.values()),
          InAppInjector.getInstance().getInAppRepository(context).localRepository.getGlobalState(),
          MoEHelper.getInstance(context).getAppContext());
      if (campaign == null) return null;
      addProcessingNudge(campaign.campaignMeta.campaignId);
      return campaign;
    }
  }

  public void addProcessingNudge(String campaignId) {
    visibleOrProcessingNudge.add(campaignId);
  }

  public void removeProcessingNudge(String campaignId){
    Logger.v(TAG + " removeProcessingNudge() : Removing campaign from processing list, id: "
        + "" + campaignId);
    visibleOrProcessingNudge.remove(campaignId);
  }

  void trackInAppShown(Context context, String campaignId, String campaignName){
    Properties properties = new Properties();
    properties.addAttribute(MoEConstants.MOE_CAMPAIGN_ID, campaignId)
        .addAttribute(MoEConstants.MOE_CAMPAIGN_NAME, campaignName)
        .setNonInteractive();
    MoEHelper.getInstance(context).trackEvent(MoEConstants.EVENT_IN_APP_SHOWN, properties);
  }
}
