/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.inapp;

import android.app.Activity;
import android.content.Context;
import android.content.Intent;
import android.net.Uri;
import android.os.Bundle;
import android.view.View;
import com.moe.pushlibrary.MoEHelper;
import com.moe.pushlibrary.activities.MoEActivity;
import com.moe.pushlibrary.utils.MoEHelperConstants;
import com.moe.pushlibrary.utils.MoEHelperUtils;
import com.moengage.core.ActionManagerBase;
import com.moengage.core.Logger;
import com.moengage.core.MoEConstants;
import com.moengage.core.MoEUtils;
import com.moengage.core.Properties;
import com.moengage.evaluator.ConditionEvaluator;
import com.moengage.inapp.listeners.InAppMessageListener;
import com.moengage.inapp.model.CampaignPayload;
import com.moengage.inapp.model.MoEInAppCampaign;
import com.moengage.inapp.model.actions.Action;
import com.moengage.inapp.model.actions.CallAction;
import com.moengage.inapp.model.actions.Condition;
import com.moengage.inapp.model.actions.ConditionAction;
import com.moengage.inapp.model.actions.CopyAction;
import com.moengage.inapp.model.actions.CustomAction;
import com.moengage.inapp.model.actions.NavigationAction;
import com.moengage.inapp.model.actions.ShareAction;
import com.moengage.inapp.model.actions.SmsAction;
import com.moengage.inapp.model.actions.TrackAction;
import com.moengage.inapp.model.actions.UserInputAction;
import com.moengage.inapp.model.enums.ActionType;
import com.moengage.widgets.MoERatingBar;
import java.util.Map;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import static com.moengage.inapp.model.enums.NavigationType.RICH_LANDING;

/**
 * @author Umang Chamaria
 */
public class ActionManager extends ActionManagerBase {
  private static final String TAG = InAppConstants.MODULE_TAG + "ActionManager";

  void onActionPerformed(Activity context, View inAppView, Action action,
      CampaignPayload campaignPayload) {
    try {
      switch (action.actionType) {
        case SMS:
          smsAction(context, action, campaignPayload.campaignId);
          break;
        case CALL:
          callAction(context, action, campaignPayload.campaignId);
          break;
        case SHARE:
          shareAction(context, action, campaignPayload.campaignId);
          break;
        case NAVIGATE:
          navigateAction(context, action, campaignPayload);
          break;
        case DISMISS:
          dismissAction(context, action, inAppView, campaignPayload);
          break;
        case TRACK_DATA:
          trackAction(context, action, campaignPayload.campaignId);
          break;
        case COPY_TEXT:
          copyAction(context, action, campaignPayload.campaignId);
          break;
        case CONDITION_ACTION:
          conditionAction(context, inAppView, action, campaignPayload);
          break;
        case CUSTOM_ACTION:
          customAction(action, campaignPayload);
          break;
        case USER_INPUT:
          userInputAction(context, inAppView, action, campaignPayload);
          break;
          default:
            Logger.v( TAG + " onActionPerformed() : Not a valid action.");
      }
    } catch (Exception e) {
      Logger.e(TAG + " onActionPerformed() : ", e);
    }
  }

  private void navigateAction(Activity context, Action action, CampaignPayload campaignPayload)
      throws ClassNotFoundException{
    Logger.v(TAG + " navigateAction() : Will try to navigate user");
    if (!(action instanceof NavigationAction)) {
      Logger.v(TAG
          + " navigateAction() : Not a valid navigation action. Campaign-id: "
          + campaignPayload.campaignId);
      return;
    }
    NavigationAction navigationAction = (NavigationAction) action;
    Logger.v(TAG + " navigateAction() : Navigation Action: " + navigationAction.toString());
    if (MoEUtils.isEmptyString(navigationAction.navigationUrl)) {
      Logger.v(TAG
          + " navigateAction() : Navigation url empty, aborting action. Campaign id: "
          + campaignPayload.campaignId);
      return;
    }
    InAppMessageListener listener = MoEInAppHelper.getInstance().getListener();
    if (navigationAction.navigationType != RICH_LANDING) {
      MoEInAppCampaign moEInAppCampaign =
          new MoEInAppCampaign(campaignPayload.campaignId, campaignPayload.campaignName,
              navigationAction);
      if (listener.onNavigation(moEInAppCampaign)) {
        Logger.v(TAG
            + " navigateAction() : Navigation handled by the app, SDK need not take any "
            + "action");
        return;
      }
    }
    Intent redirectIntent = null;
    switch (navigationAction.navigationType) {
      case SCREEN:
        redirectIntent = new Intent(context, Class.forName(navigationAction.navigationUrl));
        Bundle extras = new Bundle();
        if (navigationAction.keyValuePairs != null) {
          for (Map.Entry<String, Object> entry : navigationAction.keyValuePairs.entrySet()) {
            extras.putString(entry.getKey(), String.valueOf(entry.getValue()));
          }
          if (!extras.isEmpty()) {
            redirectIntent.putExtras(extras);
          }
        }
        break;
      case RICH_LANDING:
        redirectIntent = new Intent(context, MoEActivity.class);
        redirectIntent.putExtra(MoEHelperConstants.GCM_EXTRA_WEB_URL,
            buildUriFromString(navigationAction).toString());
        redirectIntent.putExtra(MoEConstants.EXTRA_IS_EMBEDDED_WEB_VIEW, true);
        break;
      case DEEP_LINKING:
        redirectIntent = new Intent(Intent.ACTION_VIEW, buildUriFromString(navigationAction));
        break;
    }
    if (redirectIntent != null) {
      context.startActivity(redirectIntent);
    }
  }

  private Uri buildUriFromString(NavigationAction action){
    Uri uri = Uri.parse(action.navigationUrl);
    if (action.keyValuePairs == null) return uri;
    Uri.Builder builder = uri.buildUpon();
    for (Map.Entry<String, Object> entry : action.keyValuePairs.entrySet()) {
      builder.appendQueryParameter(entry.getKey(), String.valueOf(entry.getValue()));
    }
    return builder.build();
  }

  private void trackAction(Context context, Action action, String campaignId) {
    Logger.v(TAG + " trackAction() : Will try to track data.");
    if (!(action instanceof TrackAction)) {
      Logger.v(TAG + " trackAction() : Not a valid track action. Campaign-id: " + campaignId);
      return;
    }
    TrackAction trackAction = (TrackAction) action;

    switch (trackAction.trackType) {
      case EVENT:
        trackEvent(context, trackAction, campaignId);
        break;
      case USER_ATTRIBUTE:
        trackUserAttribute(context, trackAction, campaignId);
        break;
    }
  }

  private void trackUserAttribute(Context context, TrackAction action, String campaignId) {
    Logger.v(TAG + " trackUserAttribute() : Will try to track user attribute");
    if (MoEUtils.isEmptyString(action.name.trim())) {
      Logger.v(TAG + " trackUserAttribute() : Cannot track user attribute with empty name. "
          + "Campaign id: " + campaignId);
      return;
    }
    MoEHelper.getInstance(context).setUserAttribute(action.name.trim(), action.value);
  }

  private void trackEvent(Context context, TrackAction action, String campaignId) {
    Logger.v(TAG + " trackEvent() : Will try to track event");
    if (MoEUtils.isEmptyString(action.name.trim())) {
      Logger.v(TAG
          + " trackEvent() : Event name is empty cannot track event. Campaign id: "
          + campaignId);
      return;
    }
    Properties properties = new Properties();
    if (action.attributes != null) {
      for (Map.Entry<String, Object> entry : action.attributes.entrySet()) {
        properties.addAttribute(entry.getKey(), entry.getValue());
      }
    }
    MoEHelper.getInstance(context).trackEvent(action.name.trim(), properties);
  }

  private void customAction(Action action, final CampaignPayload campaignPayload) {
    Logger.v(TAG + " customAction() : will try to trigger a custom action callback");
    if (!(action instanceof CustomAction)) {
      Logger.v(TAG
          + " customAction() : Not a valid custom action. Camapaign-id: "
          + campaignPayload.campaignId);
      return;
    }
    final CustomAction customAction = (CustomAction) action;
    final InAppMessageListener listener = MoEInAppHelper.getInstance().getListener();
    InAppController.getInstance().mainThreadHandler.post(new Runnable() {
      @Override public void run() {
        listener.onCustomAction(new MoEInAppCampaign(campaignPayload.campaignId,
            campaignPayload.campaignName, customAction));
      }
    });
  }

  private void copyAction(Context context, Action action, String campaignId) {
    Logger.v(TAG + " copyAction() : will try to copy text on clipboard");
    if (!(action instanceof CopyAction)) {
      Logger.v(TAG + " copyAction() : Not a valid copy action. Campaign-id: " + campaignId);
      return;
    }
    CopyAction copyAction = (CopyAction) action;
    Logger.v(TAG + " copyAction() : Copy Action: " + copyAction);
    if (MoEUtils.isEmptyString(copyAction.textToCopy)) {
      Logger.v(TAG
          + " copyAction() : Text to copy is empty, aborting action. Campaign id: "
          + campaignId);
    }
    MoEHelperUtils.copyTextToClipboardAndShowToast(context, copyAction.textToCopy,
        copyAction.message);
  }

  private void shareAction(Context context, Action action, String campaignId) {
    Logger.v(TAG + " shareAction() : will try to share text");
    if (!(action instanceof ShareAction)) {
      Logger.v(TAG + " shareAction() : Not a valid share action. Campaign-id: " + campaignId);
      return;
    }
    ShareAction shareAction = (ShareAction) action;
    Logger.v(TAG + " shareAction() : Share Action: " + shareAction);
    if (MoEUtils.isEmptyString(shareAction.shareText)) {
      Logger.v(TAG + " shareAction() : Text to share is empty, aborting action. Campaign "
          + "id: " + campaignId);
    }
    triggerShareIntent(context, shareAction.shareText);
  }

  private void callAction(Context context, Action action, String campaignId) {
    Logger.v(TAG + " callAction() : will try to trigger call intent");
    if (!(action instanceof CallAction)) {
      Logger.v(TAG + " callAction() : Not a valid call action. Campaign-id: " + campaignId);
      return;
    }
    CallAction callAction = (CallAction) action;
    Logger.v(TAG + " callAction() : Call Action: " + callAction);
    if (MoEUtils.isEmptyString(callAction.phoneNumber) || !isPhoneNumberValid(
        callAction.phoneNumber)) {
      Logger.v(TAG
          + " callAction() : Phone number is empty, aborting action. Campaign-id: " + campaignId);
      return;
    }
    triggerCallIntent(context, callAction.phoneNumber);
  }

  private void smsAction(Context context, Action action, String campaignId) {
    Logger.v(TAG + " smsAction() : will try to trigger sms intent");
    if (!(action instanceof SmsAction)) {
      Logger.v(TAG + " smsAction() : Not a valid sms action returning. Campaign-id: " + campaignId);
      return;
    }
    SmsAction smsAction = (SmsAction) action;
    Logger.v(TAG + " smsAction() : Sms Action: " + smsAction);
    if (MoEUtils.isEmptyString(smsAction.phoneNumber) || MoEUtils.isEmptyString(
        smsAction.message)) {
      Logger.v(TAG + " smsAction() : Either number or body is null Campaign id: " + campaignId);
      return;
    }
    Uri uri = Uri.parse("smsto:" + smsAction.phoneNumber);
    Intent intent = new Intent(Intent.ACTION_SENDTO, uri);
    intent.putExtra("sms_body", smsAction.message);
    context.startActivity(intent);
  }

  private void dismissAction(Activity activity, Action action, View inAppView,
      CampaignPayload campaignPayload) {
    Logger.v(TAG + " dismissAction() : Will try to dismiss in-app");
    InAppController.getInstance().removeViewFromHierarchy(campaignPayload,
        activity.getApplicationContext(), inAppView);
    InAppController.getInstance().handleDismiss(campaignPayload);
  }

  private void userInputAction(Activity context, View inAppView, Action action,
      CampaignPayload campaignPayload) {
    Logger.v(TAG + " userInputAction() : Will try to capture user input.");
    if (!(action instanceof UserInputAction)) {
      Logger.v(TAG + " userInputAction() : Not a valid User-Input action returning. "
          + "Campaign-id:" + campaignPayload);
      return;
    }
    UserInputAction inputAction = (UserInputAction) action;
    Logger.v(TAG + " userInputAction() : User Input Action: " + inputAction);

    switch (inputAction.userInputType){
      case RATING:
        View view = inAppView.findViewById(ViewEngine.WIDGET_BASE_ID + inputAction.widgetId);
        if (view == null) {
          Logger.v( TAG + " userInputAction() : Did not find widget with the given id.");
          return;
        }
        if (!(view instanceof MoERatingBar)) {
          Logger.v( TAG + " userInputAction() : Given view is not a rating widget.");
          return;
        }
        MoERatingBar ratingView = (MoERatingBar) view;
        float ratingInput = ratingView.getRating();
        for (Action actionItem: inputAction.actionList){
          if (actionItem.actionType == ActionType.TRACK_DATA){
            TrackAction trackAction = (TrackAction) actionItem;
            switch (trackAction.trackType){
              case EVENT:
                trackAction.attributes.put(InAppConstants.IN_APP_RATING_ATTRIBUTE, ratingInput);
                trackEvent(context, trackAction, campaignPayload.campaignId);
                break;
              case USER_ATTRIBUTE:
                MoEHelper.getInstance(context).setUserAttribute(trackAction.name.trim(), ratingInput);
                break;
            }
          } else {
            onActionPerformed(context, inAppView, actionItem, campaignPayload);
          }
        }
        break;
    }
  }

  private void conditionAction(Activity context, View inAppView, Action action,
      CampaignPayload campaignPayload) {
    try {
      Logger.v(TAG + " conditionAction() : Will try to perform actionType based on condition.");
      if (!(action instanceof ConditionAction)) {
        Logger.v(TAG + " conditionAction() : Not a valid Condition actionType. Will return. "
            + "Campaign-id: " + campaignPayload.campaignId);
        return;
      }
      ConditionAction conditionAction = (ConditionAction) action;
      Logger.v(TAG + " conditionAction() : Condition Action: " + conditionAction);

      View view = inAppView.findViewById(ViewEngine.WIDGET_BASE_ID + conditionAction.widgetId);
      if (view == null) {
        Logger.v(TAG + " userInputAction() : Did not find widget with the given id.");
        return;
      }
      if (!(view instanceof MoERatingBar)) {
        Logger.v(TAG + " userInputAction() : Given view is not a rating widget.");
        return;
      }
      MoERatingBar ratingView = (MoERatingBar) view;
      float ratingInput = ratingView.getRating();
      JSONObject userInput = new JSONObject();
      userInput.put(InAppConstants.IN_APP_RATING_ATTRIBUTE, ratingInput);
      for (Condition condition: conditionAction.conditionList){
        ConditionEvaluator evaluator =
            new ConditionEvaluator(transformConditionAttributeForPackage(condition.conditionAttribute),
            userInput);
        if (evaluator.evaluate()){
          for (Action actionItem: condition.actionList){
            onActionPerformed(context, inAppView, actionItem, campaignPayload);
          }
        }
      }
    } catch (Exception e) {
      Logger.e( TAG + " conditionAction() : ", e);
    }
  }

  private JSONObject transformConditionAttributeForPackage(JSONObject conditionAttribute)
      throws JSONException {
    JSONObject attributes = new JSONObject();
    attributes.put("filter_operator", "and");
    JSONArray filters = new JSONArray();
    filters.put(conditionAttribute);
    attributes.put("filters", filters);
    return attributes;
  }
}