/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.moengage.push;

import android.content.Context;
import androidx.annotation.Nullable;
import androidx.annotation.RestrictTo;
import androidx.annotation.RestrictTo.Scope;
import com.moengage.core.Logger;
import com.moengage.core.MoEConstants;
import com.moengage.core.MoEUtils;

/**
 * This is a Manager class which has callbacks for handling GCM payload handling and token refresh
 *
 * @author MoEngage (abhishek@moengage.com)
 * @since 5.3
 */
public class PushManager {

  private static final String TAG = MoEConstants.MODULE_TAG + "PushManager";

  private PushBaseHandler pushBaseHandler;

  private PushManager() {
    //check for PushHandler on class path
    loadPushHandler();
  }

  private void loadPushHandler() {
    try {
      Class baseHandler = Class.forName("com.moengage.pushbase.PushBaseHandlerImpl");
      pushBaseHandler = (PushBaseHandler) baseHandler.newInstance();
      Class handler = Class.forName("com.moengage.firebase.PushHandlerImpl");
      pushHandler = (PushHandler) handler.newInstance();
      Logger.v("PushManager:loadPushHandler FCM Enabled");
    } catch (Exception e) {
      Logger.i(TAG + " loadPushHandler() : Did not find push module.");
    }
  }

  private static PushManager instance = null;

  /**
   * Returns a singleton instance of {@link PushManager}
   * @return instance of {@link PushManager}
   */
  public static PushManager getInstance() {
    if (instance == null) {
      synchronized (PushManager.class){
       if (instance == null) instance = new PushManager();
      }
    }
    return instance;
  }

  /**
   * Use MoEPushHelper.getInstance().setMessageListener()
   */
  @Deprecated public void setMessageListener(Object messageListener) {
    if (pushBaseHandler != null) {
      pushBaseHandler.setPushMessageListener(messageListener);
    }
  }

  /**
   * @deprecated Use MoEFireBaseHelper.getInstance().passPushToken()
   */
  @Deprecated
  public void refreshToken(Context context, String token) {
    if (context == null || MoEUtils.isEmptyString(token) || pushHandler == null) return;
    pushHandler.passPushToken(context, token);
  }

  @RestrictTo(Scope.LIBRARY)
  public void registerForPush(Context context) {
    if (pushHandler != null) {
      pushHandler.registerForPushToken(context);
    }
  }

  @Nullable
  public PushHandler getPushHandler() {
    return pushHandler;
  }

  private PushHandler pushHandler;

  /**
   * @deprecated This interface is deprecated because all the callbacks related to Firebase events
   *             will be handled by com.moengage.firebase.listener.FirebaseEventListener.
   */
  @Deprecated
  public interface OnTokenReceivedListener {
    /**
     * Called whenever push token is generated.
     *
     * @param token Push token.
     */
    void onTokenReceived(String token);
  }

  /**
   * @deprecated MoEFireBaseHelper.getInstance().setEventListener()
   */
  @Deprecated
  public void setTokenObserver(OnTokenReceivedListener tokenObserver) {
    tokenListener = tokenObserver;
  }

  private OnTokenReceivedListener tokenListener;

  public void notifyTokenChange(String token) {
    try {
      if (tokenListener != null) {
        tokenListener.onTokenReceived(token);
      }
    } catch (Exception e) {
      Logger.e(TAG + " notifyTokenChange() : ", e);
    }
  }

  @RestrictTo(Scope.LIBRARY)
  public void onAppOpen(Context context) {
    if (context != null && pushBaseHandler != null) {
      pushBaseHandler.onAppOpen(context);
    }
  }
}
