/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.moengage.core.utils;

import com.moengage.core.Logger;
import java.lang.reflect.InvocationTargetException;

/**
 * A Utility class which helps to invoke methods using Java Reflection API in order to avoid
 * @author MoEngage (abhishek@moenegage.com)
 * @since 5.0
 * @version 5.0
 */
public final class ReflectionUtils {

  private ReflectionUtils() {
    throw new IllegalStateException("This class cannot be instantiated");
  }

  public static Object invokeStatic(Class<?> classObject, String methodName, Class<?>[] types,
      Object[] args) {
    return invocationHelper(null, classObject, null, methodName, null, types, args);
  }

  public static Object invokeStatic(String className, String methodName, Class<?>[] types,
      Object[] args) {
    return invocationHelper(null, null, className, methodName, null, types, args);
  }

  public static Object getAttribute(String className, String attrName){
    if( null == attrName || attrName.isEmpty()){
      return null;
    }
    return invocationHelper(null, null, className, attrName, null, null, null);
  }

  public static Object invokeInstance(Object target, String methodName, Class<?>[] types,
      Object[] args) {
    return invocationHelper(target, null, null, methodName, null, types, args);
  }
  private static Object invocationHelper(Object target, Class<?> classObject, String className,
      String methodName, String attributeName, Class<?>[] argTypes, Object[] args) {
    try {
      Class cls;
      if (classObject != null) {
        cls = classObject;
      } else {
        if (target != null) {
          cls = target.getClass();
        } else {
          cls = Class.forName(className);
        }
      }
      if( attributeName != null){
        return cls.getField(attributeName);
      }else if(null != methodName ){
        return cls.getMethod(methodName, argTypes).invoke(target, args);
      }else{
        return cls;
      }
    } catch (NoSuchMethodException e) {
      Logger.e("ReflectionUtils:invocationHelper", e);
    } catch (IllegalAccessException e) {
      Logger.e("ReflectionUtils:invocationHelper", e);
    } catch (InvocationTargetException e) {
      Logger.e("ReflectionUtils:invocationHelper", e);
    } catch (ClassNotFoundException e) {
      Logger.e("ReflectionUtils:invocationHelper", e);
    } catch (Exception e){
      Logger.e("ReflectionUtils:invocationHelper", e);
    }
    return null;
  }

  public static boolean isClassPresentInPath(String className){
    try{
     Class.forName(className);
      return true;
    }catch(Exception e){
      Logger.e("ReflectionUtils:isClassPresentInPath" + e.getMessage());
    }
    return false;
  }
}
