/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.core.rest;

import android.net.Uri;
import androidx.annotation.NonNull;
import com.moengage.core.MoEUtils;
import com.moengage.core.rest.exceptions.InvalidRequestException;
import com.moengage.core.rest.exceptions.UTF8EncodingException;
import java.security.InvalidKeyException;
import java.util.HashMap;
import java.util.Map;
import org.json.JSONObject;

/**
 * @author Umang Chamaria
 */
public class RequestBuilder {

  private RequestType requestType;
  private Map<String, String> headersMap;
  private JSONObject jsonBody;
  private String contentType;
  private Uri uri;
  private int connectionTimeout;
  private String encryptionKey;
  private boolean shouldEncrypt;
  private boolean shouldLogRequest = true;

  /**
   * Supported Request types.
   */
  public enum RequestType {GET, POST, PUT, DELETE}

  /**
   * Builder to build a request to make an API call
   *
   * @param requestType One of the supported HTTP request types, one of {@link RequestType}
   */
  public RequestBuilder(Uri uri, RequestType requestType) {
    this.uri = uri;
    this.requestType = requestType;
    headersMap = new HashMap<>();
    contentType = RestConstants.DEFAULT_CONTENT_TYPE;
    connectionTimeout = 10;
  }

  /**
   * Key value pair to be added as a header.
   *
   * @param headerKey key name for the header.
   * @param headerValue value for the header.
   * @return instance of {@link RequestBuilder}
   */
  public RequestBuilder addHeader(String headerKey, String headerValue) {
    headersMap.put(headerKey, headerValue);
    return this;
  }

  /**
   * Set of headers to be added to the headers.
   *
   * @param headersMap headers to be added to the request.
   * @return instance of {@link RequestBuilder}
   */
  public RequestBuilder addHeaders(Map<String, String> headersMap) {
    this.headersMap.putAll(headersMap);
    return this;
  }

  /**
   * Add body to the request.
   *
   * @param jsonBody instance of {@link JSONObject}
   * @return instance of {@link RequestBuilder}
   */
  public RequestBuilder addBody(JSONObject jsonBody) {
    this.jsonBody = jsonBody;
    return this;
  }

  /**
   * Set the content type of the request. Defaults to <i>application/json</i>
   *
   * @param contentType content type of the request.
   * @return instance of {@link RequestBuilder}
   */
  public RequestBuilder setContentType(String contentType) {
    this.contentType = contentType;
    return this;
  }

  public RequestBuilder setConnectionTimeOut(int connectionTimeOut) {
    this.connectionTimeout = connectionTimeOut;
    return this;
  }

  public RequestBuilder enableEncryption(@NonNull String encryptionKey) {
    this.encryptionKey = encryptionKey;
    this.shouldEncrypt = true;
    return this;
  }

  public RequestBuilder disableRequestLogging() {
    shouldLogRequest = false;
    return this;
  }

  public Request build() throws UTF8EncodingException, InvalidRequestException,
      InvalidKeyException {
    if (requestType == RequestType.GET && jsonBody != null) {
      throw new InvalidRequestException("GET request cannot have a body.");
    }

    if (shouldEncrypt && MoEUtils.isEmptyString(encryptionKey)) {
      throw new InvalidKeyException("Encryption key cannot be null.");
    }
    return new Request(uri, requestType, headersMap,
        jsonBody, contentType, connectionTimeout, encryptionKey, shouldEncrypt, shouldLogRequest);
  }
}