/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.core.model;

import androidx.annotation.Nullable;
import com.moengage.core.Logger;
import com.moengage.core.MoEConstants;
import com.moengage.core.MoEUtils;
import com.moengage.core.utils.JsonBuilder;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

/**
 * @author Umang Chamaria
 * Date: 2019-05-21
 */
public class UserSession {

  private static final String TAG = MoEConstants.MODULE_TAG + "UserSession";

  public final String sessionId;
  public final String startTime;
  @Nullable public TrafficSource trafficSource;
  public long lastInteractionTime;

  public UserSession(String sessionId, String startTime,
      @Nullable TrafficSource trafficSource, long lastInteractionTime) {
    this.sessionId = sessionId;
    this.startTime = startTime;
    this.trafficSource = trafficSource;
    this.lastInteractionTime = lastInteractionTime;
  }

  @Nullable
  public static JSONObject toJson(UserSession userSession) {
    try {
      JsonBuilder builder = new JsonBuilder();
      builder.putString(SESSION_ID, userSession.sessionId)
          .putString(START_TIME, userSession.startTime)
          .putLong(LAST_INTERACTION_TIME, userSession.lastInteractionTime);
      JSONArray sourceArray = new JSONArray();
      JSONObject sourceJson = TrafficSource.toJson(userSession.trafficSource);
      if (MoEUtils.hasKeys(sourceJson)) {
        sourceArray.put(sourceJson);
      }
      if (sourceArray.length() > 0) {
        builder.putJsonArray(SOURCE_ARRAY, sourceArray);
      }

      return builder.build();
    } catch (Exception e) {
      Logger.e(TAG + " toJson() : Exception: ", e);
    }
    return null;
  }

  @Nullable
  public static UserSession fromJsonString(String jsonString) {
    try {
      if (MoEUtils.isNullOrEmpty(jsonString)) return null;
      JSONObject sessionJson = new JSONObject(jsonString);
      return new UserSession(
          sessionJson.getString(SESSION_ID),
          sessionJson.getString(START_TIME),
          getTrafficSource(sessionJson),
          sessionJson.getLong(LAST_INTERACTION_TIME)
      );
    } catch (Exception e) {
      Logger.e(TAG + " fromJsonString() : Exception: ", e);
    }
    return null;
  }

  private static TrafficSource getTrafficSource(JSONObject sessionJson) throws JSONException {
    if (sessionJson.has(SOURCE_ARRAY)) {
      return TrafficSource.fromJson(sessionJson.getJSONArray(SOURCE_ARRAY).getJSONObject(0));
    }
    return null;
  }

  @Override public String toString() {
    return "{" +
        "sessionId : '" + sessionId + '\'' +
        ", startTime : '" + startTime + '\'' +
        ", trafficSource : " + trafficSource +
        ", lastInteractionTime : " + lastInteractionTime +
        '}';
  }

  private static final String SESSION_ID = "session_id";
  private static final String START_TIME = "start_time";
  public static final String LAST_INTERACTION_TIME = "last_interaction_time";
  public static final String SOURCE_ARRAY = "source_array";
  public static final String INITIATED_IN_BACKGROUND = "background_initiated";
}
