/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.core.model;

import androidx.annotation.Nullable;
import com.moengage.core.Logger;
import com.moengage.core.MoEUtils;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map.Entry;
import java.util.Set;
import org.json.JSONObject;

/**
 * @author Umang Chamaria
 * Date: 2019-05-14
 */
public class TrafficSource {

  private static final String TAG = "TrafficSource";

  public TrafficSource() {
    extras = new HashMap<>();
  }

  @Nullable public String source;
  @Nullable public String medium;
  @Nullable public String campaignName;
  @Nullable public String campaignId;
  @Nullable public String sourceUrl;
  @Nullable public String content;
  @Nullable public String term;
  public HashMap<String, String> extras;

  public TrafficSource(@Nullable String source, @Nullable String medium,
      @Nullable String campaignName, @Nullable String campaignId,
      @Nullable String content, @Nullable String term) {
    this(source, medium, campaignName, campaignId, content, term, null);
  }

  public TrafficSource(@Nullable String source, @Nullable String medium,
      @Nullable String campaignName, @Nullable String campaignId,
      @Nullable String content, @Nullable String term, @Nullable String sourceUrl) {
    this(source, medium, campaignName, campaignId, content, term, sourceUrl,
        new HashMap<String, String>());
  }

  public TrafficSource(@Nullable String source, @Nullable String medium,
      @Nullable String campaignName, @Nullable String campaignId,
      @Nullable String content, @Nullable String term, @Nullable String sourceUrl,
      HashMap<String, String> extras) {
    this.source = source;
    this.medium = medium;
    this.campaignName = campaignName;
    this.campaignId = campaignId;
    this.content = content;
    this.term = term;
    this.sourceUrl = sourceUrl;
    this.extras = extras;
  }

  @Nullable
  public static JSONObject toJson(TrafficSource source) {
    try {
      if (source == null) return null;
      JSONObject jsonObject = new JSONObject();
      if (source.source != null) {
        jsonObject.put(SOURCE, source.source);
      }
      if (source.medium != null) {
        jsonObject.put(MEDIUM, source.medium);
      }
      if (source.campaignName != null) {
        jsonObject.put(CAMPAIGN_NAME, source.campaignName);
      }
      if (source.campaignId != null) {
        jsonObject.put(CAMPAIGN_ID, source.campaignId);
      }
      if (source.sourceUrl != null) {
        jsonObject.put(SOURCE_URL, source.sourceUrl);
      }
      if (source.content != null) {
        jsonObject.put(CONTENT, source.content);
      }
      if (source.term != null) {
        jsonObject.put(TERM, source.term);
      }
      JSONObject extrasJson = new JSONObject();
      Set<Entry<String, String>> entrySet = source.extras.entrySet();
      for (Entry<String, String> entry : entrySet) {
        extrasJson.put(entry.getKey(), entry.getValue());
      }
      if (extrasJson.length() > 0) {
        jsonObject.put(EXTRAS, extrasJson);
      }
      return jsonObject;
    } catch (Exception e) {
      Logger.e(TAG + " toJson() : Exception ", e);
    }
    return null;
  }

  public static boolean isEmpty(TrafficSource source) {
    if (source == null) return true;
    return MoEUtils.isEmptyString(source.source)
        && MoEUtils.isEmptyString(source.medium)
        && MoEUtils.isEmptyString(source.campaignName)
        && MoEUtils.isEmptyString(source.campaignId)
        && MoEUtils.isEmptyString(source.content)
        && MoEUtils.isEmptyString(source.term)
        && source.extras.isEmpty();
  }

  @Nullable public static TrafficSource fromJson(JSONObject sourceJson) {
    try {
      HashMap<String, String> extras = new HashMap<>();
      if (sourceJson.has(EXTRAS)) {
        JSONObject extrasJson = sourceJson.getJSONObject(EXTRAS);
        Iterator<String> iterator = extrasJson.keys();
        while (iterator.hasNext()) {
          String key = iterator.next();
          String value = extrasJson.getString(key);
          extras.put(key, value);
        }
      }
      return new TrafficSource(
          sourceJson.optString(SOURCE, null),
          sourceJson.optString(MEDIUM, null),
          sourceJson.optString(CAMPAIGN_NAME, null),
          sourceJson.optString(CAMPAIGN_ID, null),
          sourceJson.optString(CONTENT, null),
          sourceJson.optString(TERM, null),
          sourceJson.optString(SOURCE_URL, null),
          extras
      );
    } catch (Exception e) {
      Logger.e(TAG + " fromJson() : Exception: ", e);
    }
    return null;
  }

  @Override public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;

    TrafficSource source1 = (TrafficSource) o;

    if (source != null ? !source.equals(source1.source) : source1.source != null) return false;
    if (medium != null ? !medium.equals(source1.medium) : source1.medium != null) return false;
    if (campaignName != null ? !campaignName.equals(source1.campaignName)
        : source1.campaignName != null) {
      return false;
    }
    if (campaignId != null ? !campaignId.equals(source1.campaignId) : source1.campaignId != null) {
      return false;
    }
/*    if (sourceUrl != null ? !sourceUrl.equals(source1.sourceUrl) : source1.sourceUrl != null) {
      return false;
    }*/
    if (content != null ? !content.equals(source1.content) : source1.content != null) return false;
    if (term != null ? !term.equals(source1.term) : source1.term != null) return false;
    return extras.equals(source1.extras);
  }

  @Override public String toString() {
    return "{" +
        "source : '" + source + '\'' +
        ", medium : '" + medium + '\'' +
        ", campaignName : '" + campaignName + '\'' +
        ", campaignId : '" + campaignId + '\'' +
        ", sourceUrl : '" + sourceUrl + '\'' +
        ", content : '" + content + '\'' +
        ", term : '" + term + '\'' +
        ", extras : " + extras.toString() +
        '}';
  }

  private static final String SOURCE = "source";
  private static final String MEDIUM = "medium";
  private static final String CAMPAIGN_NAME = "campaign_name";
  private static final String CAMPAIGN_ID = "campaign_id";
  private static final String SOURCE_URL = "source_url";
  private static final String CONTENT = "content";
  private static final String TERM = "term";
  private static final String EXTRAS = "extras";
}
