/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.core.internal.security;

import android.util.Base64;
import androidx.annotation.NonNull;
import androidx.annotation.RestrictTo;
import androidx.annotation.RestrictTo.Scope;
import com.moengage.core.Logger;
import com.moengage.core.MoEConstants;
import com.moengage.core.MoEUtils;
import java.security.SecureRandom;
import javax.crypto.Cipher;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;

/**
 * @author Arshiya Khanum
 * Date: 2020/07/22
 */
@RestrictTo(Scope.LIBRARY)
public class AESHandler {

  private static final String TAG = MoEConstants.MODULE_TAG + "AESHandler";
  private static final int BLOCK_SIZE = 16;
  private static final String CHARSET_NAME = "UTF-8";

  public static String encrypt(@NonNull String key, String plainText) throws Exception {
    if (MoEUtils.isEmptyString(key) || MoEUtils.isEmptyString(plainText)) {
      Logger.v(TAG + " encrypt() : key/text is null.");
      return null;
    }
    Cipher cipher = Cipher.getInstance("AES/CBC/PKCS7Padding");
    SecretKeySpec secretKeySpec = new SecretKeySpec(key.getBytes(), "AES");
    IvParameterSpec ivParameterSpec = generateIvParameterSpec();

    cipher.init(Cipher.ENCRYPT_MODE, secretKeySpec, ivParameterSpec);

    return Base64.encodeToString(addIV(cipher.doFinal(plainText.getBytes(CHARSET_NAME)),
        ivParameterSpec.getIV()),
        Base64.DEFAULT);
  }

  public static String decrypt(@NonNull String key, String encryptedText) throws Exception {
    if (MoEUtils.isEmptyString(key) || MoEUtils.isEmptyString(encryptedText)) {
      Logger.v(TAG + " decrypt() : key/text is null.");
      return null;
    }

    byte[] encryptedBytes = Base64.decode(encryptedText.getBytes(CHARSET_NAME), Base64.DEFAULT);

    Cipher cipher = Cipher.getInstance("AES/CBC/PKCS7Padding");
    SecretKeySpec secretKeySpec = new SecretKeySpec(key.getBytes(), "AES");
    IvParameterSpec ivParameterSpec = getIvParameterSpec(encryptedBytes);

    cipher.init(Cipher.DECRYPT_MODE, secretKeySpec, ivParameterSpec);

    return new String(cipher.doFinal(removeIV(encryptedBytes)));
  }

  private static IvParameterSpec generateIvParameterSpec() {
    byte[] iv = new byte[BLOCK_SIZE];
    SecureRandom secureRandom = new SecureRandom();
    secureRandom.nextBytes(iv);

    return new IvParameterSpec(iv);
  }

  private static IvParameterSpec getIvParameterSpec(byte[] encryptedBytes) {
    byte[] iv = new byte[BLOCK_SIZE];
    System.arraycopy(encryptedBytes, 0, iv, 0, BLOCK_SIZE);

    return new IvParameterSpec(iv);
  }

  private static byte[] addIV(byte[] plainBytes, byte[] ivBytes) {
    int ivLen = ivBytes.length;
    int textLen = plainBytes.length;
    byte[] plaintTextWithIv = new byte[ivBytes.length + plainBytes.length];
    System.arraycopy(ivBytes, 0, plaintTextWithIv, 0, ivLen);
    System.arraycopy(plainBytes, 0, plaintTextWithIv, ivLen, textLen);

    return plaintTextWithIv;
  }

  private static byte[] removeIV(byte[] encryptedBytes) {
    int textLen = encryptedBytes.length;
    byte[] encryptedBytesNoIv = new byte[textLen - BLOCK_SIZE];
    System.arraycopy(encryptedBytes, BLOCK_SIZE, encryptedBytesNoIv, 0,
        textLen - BLOCK_SIZE);

    return encryptedBytesNoIv;
  }
}
