/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.core.internal.reports;

import android.content.Context;
import android.os.Build;
import androidx.annotation.Nullable;
import com.moengage.core.APIManager;
import com.moengage.core.ConfigurationProvider;
import com.moengage.core.Logger;
import com.moengage.core.MoEConstants;
import com.moengage.core.MoEDAO;
import com.moengage.core.MoEUtils;
import com.moengage.core.RestUtils;
import com.moengage.core.SdkConfig;
import com.moengage.core.internal.AdvertisingIdClient.AdInfo;
import com.moengage.core.internal.storage.StorageProvider;
import com.moengage.core.model.BatchData;
import com.moengage.core.model.DevicePreferences;
import com.moengage.core.model.PushTokens;
import com.moengage.core.rest.Response;
import com.moengage.core.utils.JsonBuilder;
import java.util.ArrayList;
import java.util.TimeZone;
import org.json.JSONException;
import org.json.JSONObject;

/**
 * Help class to sync Data batches to MoEngage Server.
 *
 * @author Umang Chamaria
 * Date: 2020/09/28
 */
class DataSyncHelper {

  private static final String TAG = MoEConstants.MODULE_TAG + "DataSyncHelper";

  private final Object lock = new Object();

  void syncData(Context context, String appId, int attemptNumber) {
    synchronized (lock) {
      boolean result = false;
      ArrayList<BatchData> batchedData = null;

      String endpoint =
          MoEConstants.API_ENDPOINT_REPORT_ADD + "/" + appId;
      if (shouldSendDataToTestServer(context)) {
        endpoint = MoEConstants.API_ENDPOINT_INTEGRATION_VERIFICATION_REPORT_ADD;
      }

      ReportsBatchHelper batchHelper = new ReportsBatchHelper();
      for (; ; ) {
        batchedData = MoEDAO.getInstance(context).getBatchedData(DataManager.BATCH_SIZE);
        if (batchedData == null || batchedData.isEmpty()) {
          Logger.v(TAG + " syncData() : Nothing found to send.");
          return;
        }
        for (BatchData data : batchedData) {
          BatchData batch = batchHelper.updateBatchIfRequired(context, data);
          try {
            String requestId = getRequestID(data.batchDataJson);
            JSONObject requestBody =
                cleanBody(data.batchDataJson)
                    .put(MoEConstants.REQUEST_ATTR_QUERY_PARAMS, getQueryParams(context));
            Response response = APIManager.reportAdd(appId, endpoint, requestId,
                requestBody);
            if (response != null && (response.responseCode == 200
                || response.responseCode == MoEConstants.ENCRYPTION_FAIL)) {
              result = true;
            }
          } catch (Exception e) {
            Logger.e(TAG + " syncData() : ", e);
            result = false;
          }
          if (result) {
            Logger.v(TAG + " syncData() : Batch sent successfully deleting batch");
            MoEDAO.getInstance(context).deleteBatch(batch);
          } else {
            Logger.v(TAG + " syncData() : Will try to schedule retry.");
            scheduleRetryIfRequired(context, attemptNumber);
            break;
          }
        }
        if (!result) break;
        batchedData.clear();
      }
    }
  }

  private void scheduleRetryIfRequired(Context context, int attemptNumber) {
    Logger.v(TAG + " scheduleRetryIfRequired() : Attempt Number: " + attemptNumber);
    switch (attemptNumber) {
      case DataManager.RETRY_ONE:
        DataManager.getInstance().scheduleImmediateRetry(context,
            DataManager.INTERVAL_RETRY_ONE, DataManager.RETRY_TWO);
        break;
      case DataManager.RETRY_TWO:
        DataManager.getInstance().scheduleImmediateRetry(context,
            DataManager.INTERVAL_RETRY_TWO, DataManager.NO_RETRY);
        break;
      default:
        Logger.v(TAG + " scheduleRetryIfRequired() : Will not retry.");
    }
  }

  /**
   * Checks whether device is registered as a test device or not
   *
   * @return true if registered as test device, else false
   */
  private boolean shouldSendDataToTestServer(Context context) {
    ConfigurationProvider provider = ConfigurationProvider.getInstance(context);
    return provider.isDeviceRegisteredForVerification()
        && ((provider.getVerificationRegistrationTime() + MoEUtils.minutesToMillis(
        DataManager.MAX_TEST_DEVICE_TIME)) > MoEUtils.currentMillis());
  }

  @Nullable private String getRequestID(JSONObject batchData) {
    try {
      return batchData.getString(MoEConstants.REQUEST_HEADER_REQUEST_ID);
    } catch (JSONException e) {
      Logger.e("SendInteractionDataTask: getRequestID(): Exception ", e);
    }
    return null;
  }

  private JSONObject cleanBody(JSONObject batchData) {
    batchData.remove(MoEConstants.REQUEST_HEADER_REQUEST_ID);
    return batchData;
  }

  private JSONObject getQueryParams(Context context) throws JSONException {
    ConfigurationProvider provider = ConfigurationProvider.getInstance(context);
    JsonBuilder builder = RestUtils.getDefaultParams(context);
    builder.putString(MoEConstants.GENERIC_PARAM_V2_KEY_TIMEZONE, TimeZone.getDefault().getID());

    DevicePreferences devicePreferences = MoEDAO.getInstance(context).getDevicePreferences();

    if (!devicePreferences.isPushOptedOut) {
      PushTokens pushTokens =
          StorageProvider.INSTANCE.getRepository(context, SdkConfig.getConfig()).getPushTokens();
      if (!MoEUtils.isEmptyString(pushTokens.fcmToken)) {
        builder.putString(MoEConstants.GENERIC_PARAM_V2_KEY_FCM_PUSH_ID, pushTokens.fcmToken);
      }
      if (!MoEUtils.isEmptyString(pushTokens.oemToken)) {
        builder.putString(MoEConstants.GENERIC_PARAM_V2_KEY_OEM_TOKEN, pushTokens.oemToken);
      }
    }

    if (!devicePreferences.isDataTrackingOptedOut) {
      String androidId = MoEUtils.getAndroidID(context);
      if (!MoEUtils.isEmptyString(androidId)) {
        builder.putString(MoEConstants.GENERIC_PARAM_V2_KEY_ANDROID_ID, androidId);
      }

      if (!SdkConfig.getConfig().trackingOptOut.isGaidOptedOut()) {
        String gaid = provider.getStoredGAID();
        if (MoEUtils.isEmptyString(gaid)) {
          AdInfo adInfo = MoEUtils.getAdvertisementInfo(context);
          if (adInfo != null) {
            gaid = adInfo.getId();
            provider.storeGAID(gaid);
          }
        }
        if (!MoEUtils.isEmptyString(gaid)) {
          builder.putString(MoEConstants.GENERIC_PARAM_V2_KEY_GAID, gaid);
        }
      }
      builder.putString(MoEConstants.GENERIC_PARAM_V2_KEY_OS_VERSION,
          String.valueOf(Build.VERSION.SDK_INT))
          .putString(MoEConstants.GENERIC_PARAM_V2_KEY_MODEL, Build.MODEL)
          .putString(MoEConstants.GENERIC_PARAM_V2_KEY_APP_VERSION_NAME,
              provider.getAppVersionName());

      String nwType = MoEUtils.getNetworkType(context);
      if (!MoEUtils.isEmptyString(nwType)) {
        builder.putString(MoEConstants.GENERIC_PARAM_KEY_NW_TYPE, nwType);
      }
    }

    return builder.build();
  }
}
