/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.core.internal.reports;

import android.annotation.TargetApi;
import android.app.AlarmManager;
import android.app.PendingIntent;
import android.app.job.JobInfo;
import android.app.job.JobInfo.Builder;
import android.app.job.JobScheduler;
import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.os.Build.VERSION;
import android.os.Build.VERSION_CODES;
import android.os.Looper;
import android.os.PersistableBundle;
import androidx.annotation.NonNull;
import androidx.annotation.RestrictTo;
import androidx.annotation.RestrictTo.Scope;
import androidx.annotation.WorkerThread;
import com.moengage.core.DataSyncJob;
import com.moengage.core.Logger;
import com.moengage.core.MoEAlarmReceiver;
import com.moengage.core.MoEConstants;
import com.moengage.core.MoEUtils;
import com.moengage.core.MoEngage;
import com.moengage.core.RemoteConfig;
import com.moengage.core.SdkConfig;
import com.moengage.core.internal.analytics.AnalyticsHelper;
import com.moengage.core.executor.TaskManager;
import com.moengage.core.model.MoEJobParameters;

/**
 * @author Arshiya Khanum
 * Date: 2020/08/21
 * @since 10.4.00
 */
@RestrictTo(Scope.LIBRARY)
public class DataManager {

  private static final String TAG = MoEConstants.MODULE_TAG + "DataManager";

  private static DataManager instance = null;

  private final ReportsBatchHelper batchHelper;

  private final DataSyncHelper dataSyncHelper;

  private DataManager() {
    batchHelper = new ReportsBatchHelper();
    dataSyncHelper = new DataSyncHelper();
  }

  public static DataManager getInstance() {
    if (instance == null) {
      synchronized (DataManager.class) {
        if (instance == null) instance = new DataManager();
      }
    }
    return instance;
  }

  @SuppressWarnings("ConstantConditions")
  public void scheduleDataSending(@NonNull Context context) {
    if (context == null) {
      Logger.v(TAG + "scheduleDataSending() : context is null");
      return;
    }
    try {
      if (VERSION.SDK_INT < VERSION_CODES.LOLLIPOP) {
        scheduleDataSendingAlarm(context, APP_CLOSE_DATA_SYNC_ALARM_ID, APP_CLOSE_SYNC_INTERVAL);
      } else {
        scheduleDataSendingJob(context, APP_CLOSE_DATA_SYNC_JOB_ID, APP_CLOSE_SYNC_INTERVAL,
            RETRY_ONE);
      }
      scheduleBackgroundSyncIfRequired(context);
    } catch (Exception e) {
      Logger.e(TAG + "scheduleDataSending() :  Exception: ", e);
    }
  }

  void scheduleImmediateRetry(Context context, int seconds, int attemptNumber) {
    Logger.v(
        TAG + " scheduleImmediateRetry() : Scheduling immediate retry, delay: " + seconds);
    if (VERSION.SDK_INT < VERSION_CODES.LOLLIPOP) {
      scheduleDataSendingAlarm(context, IMMEDIATE_RETRY_DATA_SYNC_ALARM_ID, seconds);
    } else {
      scheduleDataSendingJob(context, IMMEDIATE_RETRY_DATA_SYNC_JOB_ID, seconds, attemptNumber);
    }
  }

  public void backgroundSync(Context context, int attemptNumber, MoEJobParameters jobParameters) {
    Logger.v(TAG + " backgroundSync() : ");
    queueBatchingDataTask(context, jobParameters, attemptNumber);
    scheduleBackgroundSyncIfRequired(context);
  }

  private void scheduleBackgroundSyncIfRequired(Context context) {
    if (!SdkConfig.getConfig().isBackgroundSyncEnabled) return;
    long syncInterval =
        RemoteConfig.getConfig().dataSyncRetryInterval / 1000;
    Logger.v(TAG + " scheduleBackgroundSyncIfRequired() : Will schedule background sync.");
    if (VERSION.SDK_INT < VERSION_CODES.LOLLIPOP) {
      scheduleDataSendingAlarm(context, BACKGROUND_DATA_SYNC_ALARM_ID, syncInterval);
    } else {
      scheduleDataSendingJob(context, BACKGROUND_DATA_SYNC_JOB_ID, syncInterval, RETRY_ONE);
    }
    Logger.v(TAG + " scheduleBackgroundSyncIfRequired() : Background sync scheduled.");
  }

  private void scheduleDataSendingAlarm(@NonNull Context context, int alarmId, long syncInterval) {
    Logger.v(TAG + " scheduleDataSending() alarmId: " + alarmId + " interval: " + syncInterval);
    Intent alarmIntent = new Intent(context, MoEAlarmReceiver.class);
    PendingIntent pendingIntent =
        PendingIntent.getBroadcast(context, alarmId, alarmIntent,
            PendingIntent.FLAG_UPDATE_CURRENT);
    AlarmManager alarmManager = (AlarmManager) context.getSystemService(Context.ALARM_SERVICE);
    if (alarmManager != null) {
      alarmManager.set(AlarmManager.RTC_WAKEUP,
          MoEUtils.secondsToMillis(MoEUtils.currentSeconds() + syncInterval), pendingIntent);
    }
  }

  @TargetApi(VERSION_CODES.LOLLIPOP)
  private void scheduleDataSendingJob(@NonNull Context context, int jobId, long syncInterval,
      int attemptNumber) {
    Logger.v(TAG + " scheduleDataSendingJob() JobId: " + jobId + " interval: " + syncInterval);
    ComponentName serviceComponent = new ComponentName(context, DataSyncJob.class);
    JobInfo.Builder builder = new Builder(jobId, serviceComponent);
    builder.setRequiredNetworkType(JobInfo.NETWORK_TYPE_ANY)
        .setOverrideDeadline(MoEUtils.secondsToMillis(syncInterval * EXPONENTIAL_COUNTER_LATENCY_RETRY))
        .setMinimumLatency(MoEUtils.secondsToMillis(syncInterval));
    PersistableBundle bundle = new PersistableBundle();
    bundle.putInt(ATTEMPT_NUMBER, attemptNumber);
    builder.setExtras(bundle);
    JobScheduler jobScheduler =
        (JobScheduler) context.getSystemService(Context.JOB_SCHEDULER_SERVICE);
    if (jobScheduler != null) {
      jobScheduler.schedule(builder.build());
    }
  }

  @RestrictTo(Scope.LIBRARY) public void batchAndSyncData(Context context) {
    Logger.v(TAG + " batchAndSyncData() : Will batch and sync data");
    int attemptNumber = NO_RETRY;
    if (!MoEngage.isAppForeground()) {
      attemptNumber = RETRY_ONE;
    }
    queueBatchingDataTask(context, null, attemptNumber);
  }

  private void queueBatchingDataTask(Context context, MoEJobParameters jobParameters,
      int attemptNumber) {
    Logger.v(TAG + " queueBatchingDataTask() : Will queue batching data task.");
    TaskManager.getInstance()
        .addTaskToQueue(new BatchDataTask(context, true, jobParameters, attemptNumber));
  }

  /**
   * Creates and saves batches of data-points.<br/>
   * <b>Note:</b> This API does not sync the batched data. Only creates the batch and persists it.
   *
   * @param context instance of {@link Context}
   */
  @RestrictTo(Scope.LIBRARY) @WorkerThread public void batchData(Context context) {
    if (Looper.getMainLooper() == Looper.myLooper()) {
      Logger.v(TAG + " batchData() : Cannot process this request on MAIN THREAD.");
      return;
    }
    Logger.v(TAG + " batchData() : Will batch data points.");
    batchHelper.createAndSaveBatches(context, AnalyticsHelper.getInstance(context).getSession());
  }

  /**
   * Syncs created data batches to MoEngage Server.<br/>
   * This is a blocking call should not be called on the MAIN-THREAD
   *
   * @param context instance of {@link Context}
   * @param appId APP-ID of the MoEngage account.
   */
  @RestrictTo(Scope.LIBRARY) @WorkerThread public void sendData(Context context, String appId,
      int attemptNumber) {
    if (Looper.getMainLooper() == Looper.myLooper()) {
      Logger.v(TAG + " sendData() : Cannot process this request on MAIN THREAD.");
      return;
    }
    if (MoEUtils.isEmptyString(appId)) {
      Logger.w(TAG + " sendData() : App-id is empty cannot send data");
      return;
    }
    Logger.v(TAG + " sendData() : Will send data to server.");
    dataSyncHelper.syncData(context, appId, attemptNumber);
  }

  public static String ACTION_DATA_SENDING = "MOE_ACTION_DATA_SENDING";

  private static final int APP_CLOSE_DATA_SYNC_JOB_ID = 90001;
  private static final int APP_CLOSE_DATA_SYNC_ALARM_ID = 90002;
  public static final int BACKGROUND_DATA_SYNC_JOB_ID = 90003;
  public static final int BACKGROUND_DATA_SYNC_ALARM_ID = 90004;
  public static final int IMMEDIATE_RETRY_DATA_SYNC_JOB_ID = 90005;
  public static final int IMMEDIATE_RETRY_DATA_SYNC_ALARM_ID = 90006;

  public static final int BATCH_SIZE = 100;

  public static int MAX_TEST_DEVICE_TIME = 60;
  public static int INTERVAL_RETRY_ONE = 60;
  public static int INTERVAL_RETRY_TWO = 180;
  public static int APP_CLOSE_SYNC_INTERVAL = 3;

  public static String ATTEMPT_NUMBER = "attempt_number";

  public static final int NO_RETRY = -1;
  public static final int RETRY_ONE = 1;
  public static final int RETRY_TWO = 2;

  private static final int EXPONENTIAL_COUNTER_LATENCY_RETRY = 2;
}
