/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.core.internal.logger;

import com.moengage.core.LogLevel;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.net.UnknownHostException;
import java.util.HashMap;

/**
 * @author Umang Chamaria
 * Date: 11/05/20
 */
public class LogUtils {

  /**
   * @return Stack trace in form of String
   */
  public static String getStackTraceString(Throwable tr) {
    if (tr == null) {
      return "";
    }

    // This is to reduce the amount of log spew that apps do in the non-error
    // condition of the network being unavailable.
    Throwable t = tr;
    while (t != null) {
      if (t instanceof UnknownHostException) {
        return "";
      }
      t = t.getCause();
    }

    StringWriter sw = new StringWriter();
    PrintWriter pw = new PrintWriter(sw);
    tr.printStackTrace(pw);
    pw.flush();
    return sw.toString();
  }

  public static boolean isValidLevel(int level) {
    return level >= LogLevel.NO_LOG && level <= LogLevel.VERBOSE;
  }

  public static HashMap<String, Integer> LOG_TYPE_TO_LEVEL_MAPPING = new HashMap<>();

  public static HashMap<Integer, String> LOG_LEVEL_TO_LOG_TYPE_MAPPING = new HashMap<>();

  public static final String LOG_LEVEL_NO_LOG = "no_log";
  private static final String LOG_LEVEL_ERROR = "error";
  private static final String LOG_LEVEL_WARN = "warn";
  private static final String LOG_LEVEL_INFO = "info";
  private static final String LOG_LEVEL_DEBUG = "debug";
  private static final String LOG_LEVEL_VERBOSE = "verbose";

  static {
    LOG_TYPE_TO_LEVEL_MAPPING.put(LOG_LEVEL_NO_LOG, LogLevel.NO_LOG);
    LOG_TYPE_TO_LEVEL_MAPPING.put(LOG_LEVEL_ERROR, LogLevel.ERROR);
    LOG_TYPE_TO_LEVEL_MAPPING.put(LOG_LEVEL_WARN, LogLevel.WARN);
    LOG_TYPE_TO_LEVEL_MAPPING.put(LOG_LEVEL_INFO, LogLevel.INFO);
    LOG_TYPE_TO_LEVEL_MAPPING.put(LOG_LEVEL_DEBUG, LogLevel.DEBUG);
    LOG_TYPE_TO_LEVEL_MAPPING.put(LOG_LEVEL_VERBOSE, LogLevel.VERBOSE);

    LOG_LEVEL_TO_LOG_TYPE_MAPPING.put(LogLevel.NO_LOG, LOG_LEVEL_NO_LOG);
    LOG_LEVEL_TO_LOG_TYPE_MAPPING.put(LogLevel.ERROR, LOG_LEVEL_ERROR);
    LOG_LEVEL_TO_LOG_TYPE_MAPPING.put(LogLevel.WARN, LOG_LEVEL_WARN);
    LOG_LEVEL_TO_LOG_TYPE_MAPPING.put(LogLevel.INFO, LOG_LEVEL_INFO);
    LOG_LEVEL_TO_LOG_TYPE_MAPPING.put(LogLevel.DEBUG, LOG_LEVEL_DEBUG);
    LOG_LEVEL_TO_LOG_TYPE_MAPPING.put(LogLevel.VERBOSE, LOG_LEVEL_VERBOSE);
  }
}
