/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.core.internal.inapp;

import android.app.Activity;
import android.content.Context;
import android.os.Bundle;
import androidx.annotation.Nullable;
import com.moe.pushlibrary.models.Event;
import com.moengage.core.ConfigurationProvider;
import com.moengage.core.Logger;
import com.moengage.core.MoEConstants;
import com.moengage.core.RemoteConfig;
import com.moengage.core.model.InAppV2Meta;
import com.moengage.core.model.InAppV3Meta;

/**
 * @author Umang Chamaria
 */
public class InAppManager {

  private static final String TAG = MoEConstants.MODULE_TAG + "InAppManager";

  private InAppHandler inAppHandler;

  private static InAppManager instance;

  private InAppManager() {
    loadInAppHandler();
  }

  private void loadInAppHandler() {
    try {
      Class handler = Class.forName("com.moengage.inapp.InAppHandlerImpl");
      inAppHandler = (InAppHandler) handler.newInstance();
      Logger.v("InAppManager:loadInAppHandler InApp Module present");
    } catch (Exception e) {
      Logger.v(TAG + " loadInAppHandler : InApp Module not present ");
    }
  }

  public static InAppManager getInstance() {
    if (instance == null) {
      synchronized (InAppManager.class) {
        if (instance == null) instance = new InAppManager();
      }
    }
    return instance;
  }

  public void showInAppIfRequired(Context context) {
    if (isInAppModuleActive(context)) {
      inAppHandler.showInAppIfRequired(context);
    }
  }

  public void registerInAppManager(Activity activity) {
    if (isInAppModuleActive(activity.getApplicationContext())) {
      inAppHandler.registerInAppManager(activity);
    }
  }

  public void unregisterInAppManager(Activity activity) {
    if (isInAppModuleActive(activity.getApplicationContext())) {
      inAppHandler.unregisterInAppManager(activity);
    }
  }

  public void showTriggerInAppIfPossible(Context context, Event action) {
    if (isInAppModuleActive(context)) {
      inAppHandler.showTriggerInAppIfPossible(context, action);
    }
  }

  public void onLogout(Context context) {
    if (inAppHandler != null) {
      inAppHandler.onLogout(context);
    }
  }

  public void onAppClose(Context context){
    if (inAppHandler != null){
      inAppHandler.onAppClose(context);
    }
  }

  private boolean isInAppModuleActive(Context context) {
    RemoteConfig config = RemoteConfig.getConfig();
      return inAppHandler != null
          && !ConfigurationProvider.getInstance(context).isInAppOptedOut()
          && config.isInAppEnabled
          && config.isAppEnabled;
  }

  public void syncInAppsIfRequired(Context context) {
    if (isInAppModuleActive(context)) {
      inAppHandler.syncInAppIfRequired(context);
    }
  }

  public void showInAppFromPush(Context context, Bundle pushPayload){
    if (isInAppModuleActive(context)){
      inAppHandler.showInAppFromPush(context, pushPayload);
    }
  }

  @Nullable
  public InAppV3Meta generateMetaForV2Campaign(InAppV2Meta inAppV2Meta){
    if (inAppHandler != null){
      return inAppHandler.generateMetaForV2Campaign(inAppV2Meta);
    }
    return null;
  }

}