/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.core.internal.events;

import android.content.Context;
import com.moe.pushlibrary.models.Event;
import com.moengage.core.ConfigurationProvider;
import com.moengage.core.Logger;
import com.moengage.core.MoEConstants;
import com.moengage.core.MoEDTManager;
import com.moengage.core.MoEDispatcher;
import com.moengage.core.RemoteConfig;
import com.moengage.core.internal.analytics.AnalyticsHelper;
import com.moengage.core.executor.SDKTask;
import com.moengage.core.executor.TaskResult;
import com.moengage.core.internal.inapp.InAppManager;

/**
 *
 * Tracks an event.
 * @author Umang Chamaria
 */
class TrackEventTask extends SDKTask {
  private static final String TAG = MoEConstants.MODULE_TAG + "TrackEventTask";

  private Event event;

  TrackEventTask(Context context, Event event) {
    super(context);
    this.event = event;
  }

  @Override public TaskResult execute() {
    try {
      Logger.v("TrackEventTask : executing task");
      if (event.eventName == null){
        Logger.w( TAG + " execute() : Event name is null cannot track it.");
        return null;
      }
      MoEEventManager eventManager = MoEEventManager.getInstance(context);
      ConfigurationProvider configProvider = ConfigurationProvider.getInstance(context);
      // check for gdpr opt out and event blacklisting
      if (!eventManager.eventHandler.shouldTrackEvent(configProvider.isDataTrackingOptedOut(),
          RemoteConfig.getConfig().gdprWhitelistEventList,
          RemoteConfig.getConfig().blacklistedEvents,
          event.eventName)){
        Logger.i( TAG + " execute() :  Either data tracking is opted out and this is not a GDPR "
            + "whitelist event cannot track or event is blacklisted Event Name: " + event.eventName);
        return taskResult;
      }
      // check and show in-app smart trigger if required
      InAppManager.getInstance().showTriggerInAppIfPossible(context, event);

      // update session if required
      AnalyticsHelper.getInstance(context).onEventTracked(event, context);
      //check and show device triggers if possible
      MoEDTManager.getInstance().showTriggerIfPossible(context, event.eventName, event.attributes);
      // write event to storage
      eventManager.writeDataPointToStorage(event);
      // sync data points if required
      eventManager.flushIfRequired(event);
      // increment cached event counter
      MoEEventManager.getInstance(context).incrementEventCounter();
      Logger.v(TAG + " execute() : Cached event count: " + MoEEventManager.getInstance
          (context).getEventCounter());
      // update session info

      // immediate sync for unique id
      if(eventManager.getEventCounter() == RemoteConfig.getConfig().eventBatchCount){
        Logger.d("Unique Id set, So will try to send data");
        MoEDispatcher.getInstance(context).sendInteractionData();
      }
      Logger.v("TrackEventTask : completed execution");
    } catch (Exception e) {
      Logger.e( TAG + " execute() : Exception: ", e);
    }
    return null;
  }

  @Override public String getTaskTag() {
    return SDKTask.TAG_TRACK_EVENT;
  }

  @Override public boolean isSynchronous() {
    return false;
  }
}
