/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.core.internal.analytics;

import android.app.Activity;
import android.content.Intent;
import android.net.Uri;
import android.os.Bundle;
import androidx.annotation.Nullable;
import com.moengage.core.model.TrafficSource;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Set;

/**
 * @author Umang Chamaria
 * Date: 2019-05-22
 */
// TODO: 2019-05-22 change class name
public class SourceProcessor {

  private static final String TAG = "SourceProcessor";

  private static List<String> sourceKeys;
  private static List<String> mediumKeys;
  private static List<String> campaignNameKeys;
  private static List<String> campaignIdKeys;
  private static List<String> contentKeys;
  private static List<String> termKeys;

  private static final String SOURCE_PARAM_SOURCE = "source";
  private static final String SOURCE_PARAM_UTM_SOURCE = "utm_source";

  private static final String CAMPAIGN_PARAM_UTM_CAMPAIGN = "utm_campaign";
  private static final String CAMPAIGN_PARAM_CAMPAIGN = "campaign";

  private static final String MEDIUM_PARAM_UTM_MEDIUM = "utm_medium";
  private static final String MEDIUM_PARAM_MEDIUM = "medium";

  private static final String CAMPAIGN_ID_PARAM_UTM_CAMPAIGN_ID = "utm_id";
  private static final String CAMPAIGN_ID_PARAM_CAMPAIGN_ID = "id";

  private static final String CONTENT_PARAM_UTM_CONTENT = "utm_content";
  private static final String CONTENT_PARAM_CONTENT = "content";

  private static final String TERM_PARAM_UTM_TERM = "utm_term";
  private static final String TERM_PARAM_TERM = "term";

  private static final int SOURCE_PROPERTIES_LIST_SIZE = 2;

  static {
    sourceKeys = new ArrayList<>(SOURCE_PROPERTIES_LIST_SIZE);
    sourceKeys.add(SOURCE_PARAM_UTM_SOURCE);
    sourceKeys.add(SOURCE_PARAM_SOURCE);

    campaignNameKeys = new ArrayList<>(SOURCE_PROPERTIES_LIST_SIZE);
    campaignNameKeys.add(CAMPAIGN_PARAM_UTM_CAMPAIGN);
    campaignNameKeys.add(CAMPAIGN_PARAM_CAMPAIGN);

    mediumKeys = new ArrayList<>(SOURCE_PROPERTIES_LIST_SIZE);
    mediumKeys.add(MEDIUM_PARAM_UTM_MEDIUM);
    mediumKeys.add(MEDIUM_PARAM_MEDIUM);

    campaignIdKeys = new ArrayList<>(SOURCE_PROPERTIES_LIST_SIZE);
    campaignIdKeys.add(CAMPAIGN_ID_PARAM_UTM_CAMPAIGN_ID);
    campaignIdKeys.add(CAMPAIGN_ID_PARAM_CAMPAIGN_ID);

    contentKeys = new ArrayList<>(SOURCE_PROPERTIES_LIST_SIZE);
    contentKeys.add(CONTENT_PARAM_UTM_CONTENT);
    contentKeys.add(CONTENT_PARAM_CONTENT);

    termKeys = new ArrayList<>(SOURCE_PROPERTIES_LIST_SIZE);
    termKeys.add(TERM_PARAM_UTM_TERM);
    termKeys.add(TERM_PARAM_TERM);
  }

  public SourceProcessor() {
  }

  TrafficSource getTrafficSourceFromActivity(Activity activity,
      Set<String> extraSourceIdentifiers) {
    Intent intent = activity.getIntent();
    if (intent == null) return null;
    TrafficSource source = null;
    Uri intentUri = intent.getData();
    Bundle extras = intent.getExtras();
    if (intentUri != null) {
      // get source from uri
      TrafficSource uriSource = getTrafficSourceFromUrl(intentUri, extraSourceIdentifiers);
      if (!TrafficSource.isEmpty(uriSource)) {
        source = uriSource;
      }
    }
    if (source == null && extras != null){
      TrafficSource sourceIntentExtra = getTrafficSourceFromExtras(extras, extraSourceIdentifiers);
      if (!TrafficSource.isEmpty(sourceIntentExtra)){
        source = sourceIntentExtra;
      }
    }
    if (source == null){
      source = new TrafficSource();
    }
    return source;
  }

  public TrafficSource getTrafficSourceFromUrl(Uri intentUri, Set<String> extraSourceIdentifiers) {
    Set<String> queryParamKeys = intentUri.getQueryParameterNames();
    return new TrafficSource(
        paramValueForParamName(intentUri, sourceKeys, queryParamKeys),
        paramValueForParamName(intentUri, mediumKeys, queryParamKeys),
        paramValueForParamName(intentUri, campaignNameKeys, queryParamKeys),
        paramValueForParamName(intentUri, campaignIdKeys, queryParamKeys),
        paramValueForParamName(intentUri, contentKeys, queryParamKeys),
        paramValueForParamName(intentUri, termKeys, queryParamKeys),
        intentUri.toString(),
        getExtraIdentifiersIfPresent(intentUri, queryParamKeys, extraSourceIdentifiers)
    );
  }

  public TrafficSource getTrafficSourceFromExtras(Bundle extras,
      Set<String> extraSourceIdentifiers) {
    Set<String> bundleKeySet = extras.keySet();
    return new TrafficSource(
        paramValueForParamName(extras, bundleKeySet, sourceKeys),
        paramValueForParamName(extras, bundleKeySet, mediumKeys),
        paramValueForParamName(extras, bundleKeySet, campaignNameKeys),
        paramValueForParamName(extras, bundleKeySet, campaignIdKeys),
        paramValueForParamName(extras, bundleKeySet, contentKeys),
        paramValueForParamName(extras, bundleKeySet, termKeys),
        null,
        getExtraIdentifiersIfPresent(extras, bundleKeySet, extraSourceIdentifiers)
    );
  }

  @Nullable private String paramValueForParamName(Uri uri, List<String> keys,
      Set<String> queryParamKeys) {
    for (String key : keys) {
      if (queryParamKeys.contains(key)) {
        return uri.getQueryParameter(key);
      }
    }
    return null;
  }

  private String paramValueForParamName(Bundle extras, Set<String> extraKeySet, List<String> keys) {
    for (String key : keys) {
      if (extraKeySet.contains(key)) {
        return extras.getString(key);
      }
    }
    return null;
  }

  private HashMap<String, String> getExtraIdentifiersIfPresent(Uri uri,
      Set<String> queryParamsKeys, Set<String> extraSourceIdentifiers) {
    HashMap<String, String> extraIdentifierMap = new HashMap<>();
    if (queryParamsKeys == null || extraSourceIdentifiers == null) return extraIdentifierMap;
    for (String identifier : extraSourceIdentifiers) {
      if (queryParamsKeys.contains(identifier)) {
        extraIdentifierMap.put(identifier, uri.getQueryParameter(identifier));
      }
    }
    return extraIdentifierMap;
  }

  private HashMap<String, String> getExtraIdentifiersIfPresent(Bundle extras,
      Set<String> bundleKeySet, Set<String> extraSourceIdentifiers) {
    HashMap<String, String> extraIdentifierMap = new HashMap<>();
    if (bundleKeySet == null || extraSourceIdentifiers == null) return extraIdentifierMap;
    for (String identifier : extraSourceIdentifiers) {
      if (bundleKeySet.contains(identifier)) {
        extraIdentifierMap.put(identifier, extras.getString(identifier));
      }
    }
    return extraIdentifierMap;
  }

}
