/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.core.internal.analytics;

import android.app.Activity;
import android.content.Context;
import androidx.annotation.Nullable;
import androidx.annotation.RestrictTo;
import androidx.annotation.RestrictTo.Scope;
import androidx.annotation.WorkerThread;
import com.moe.pushlibrary.models.Event;
import com.moengage.core.Logger;
import com.moengage.core.MoEConstants;
import com.moengage.core.MoECoreEvaluator;
import com.moengage.core.MoEDAO;
import com.moengage.core.MoEUtils;
import com.moengage.core.MoEngage;
import com.moengage.core.RemoteConfig;
import com.moengage.core.model.TrafficSource;
import com.moengage.core.model.UserSession;
import com.moengage.core.internal.reports.DataManager;
import java.util.UUID;

/**
 * @author Umang Chamaria
 * Date: 2019-05-21
 */
@RestrictTo(Scope.LIBRARY_GROUP)
public class AnalyticsHelper {

  private static final String TAG = MoEConstants.MODULE_TAG + "AnalyticsHelper";

  private static AnalyticsHelper instance = null;

  private UserSession session;

  private MoECoreEvaluator evaluator;

  private boolean hasProcessedAppOpen;

  private final Object lock = new Object();

  private AnalyticsHelper(Context context) {
    session = MoEDAO.getInstance(context).getLastSavedSession();
    evaluator = new MoECoreEvaluator();
  }

  public static AnalyticsHelper getInstance(Context context) {
    if (instance == null) {
      synchronized (AnalyticsHelper.class) {
        if (instance == null) instance = new AnalyticsHelper(context);
      }
    }
    return instance;
  }

  public void onEventTracked(Event event, Context context) {
    try {
      Logger.v(TAG
          + " onEventTracked() : Will update last interaction time if required. Event: "
          + event.details);
      // check if event is non-interactive
      if (!event.isInteractiveEvent) {
        Logger.v(
            TAG + " onEventTracked() : No operation required. Tracked event is non-interactive");
        return;
      }
      // user attribute tracked no action required.
      if (MoEConstants.EVENT_ACTION_USER_ATTRIBUTE.equals(event.eventName)) {
        Logger.v(TAG + " updateSession() : Need not update session info since user attribute is "
            + "tracked.");
        return;
      }
      // event tracked before sdk initialisation
      if (!hasProcessedAppOpen && evaluator.hasSessionExpired(
          session == null ? 0 : session.lastInteractionTime,
          RemoteConfig.getConfig().sessionInActiveTime, MoEUtils.currentMillis())) {
        Logger.v(TAG + " onEventTracked() : Source not processed yet. Will create new session");
        batchPreviousDataAndCreateNewSession(context, null);
        return;
      }
      // app in foreground no action required.
      if (MoEngage.isAppForeground()) {
        Logger.v(TAG + " updateSession() : App is in foreground no action required.");
        return;
      }
      // no sessions created till now
      if (session == null) {
        Logger.v(TAG + " onEventTracked() : No previous session. Will create a new session");
        batchPreviousDataAndCreateNewSession(context, null);
        return;
      }
      // existing session expired create a new session.
      if (evaluator.hasSessionExpired(session.lastInteractionTime,
          RemoteConfig.getConfig().sessionInActiveTime,
          MoEUtils.currentMillis())) {
        Logger.v(TAG + " onEventTracked() : Session has expired.");
        batchPreviousDataAndCreateNewSession(context, null);
        return;
      }
      //update last interaction
      updateLastInteractionTime(MoEUtils.currentMillis());
    } catch (Exception e) {
      Logger.e(TAG + " onEventTracked() : Exception: ", e);
    }
  }

  void updateLastInteractionTime(long time) {
    if (session != null) {
      session.lastInteractionTime = time;
    }
  }

  // update session if required
  @WorkerThread public void onActivityStart(Activity activity) {
    Logger.v(TAG + " onActivityStart() : Will try to process traffic information.");
    if (session != null) {
      Logger.v(TAG + " onActivityStart() : Existing Session " + session.toString());
    }
    if (hasProcessedAppOpen) {
      Logger.v(TAG + " onActivityStart() : App Open already processed. Ignoring");
      return;
    }
    updateUserSessionIfRequired(activity);
    hasProcessedAppOpen = true;
  }

  public void onNotificationClicked(Context context, TrafficSource source) {
    try {
      Logger.v(TAG + " onNotificationClicked() : Source " + source);
      updateSessionIfRequired(context, source);
    } catch (Exception e) {
      Logger.e(TAG + " onNotificationClicked() : ", e);
    }
  }

  private void updateUserSessionIfRequired(Activity activity) {
    try {
      Logger.v(TAG + " updateUserSessionIfRequired() : ");
      Context context = activity.getApplicationContext();
      SourceProcessor sourceProcessor = new SourceProcessor();
      TrafficSource computedSource = sourceProcessor.getTrafficSourceFromActivity(activity,
          RemoteConfig.getConfig().additionalSourceIdentifiers);
      Logger.v(TAG + " updateUserSessionIfRequired() : Computed Source " + computedSource);
      updateSessionIfRequired(context, computedSource);
    } catch (Exception e) {
      Logger.e(TAG + " onAppOpen() : Exception: ", e);
    }
  }

  private void updateSessionIfRequired(Context context, TrafficSource currentSource) {
    synchronized (lock) {
      Logger.v(TAG + " updateSessionIfRequired() : New source: " + currentSource);
      if (session == null) {
        Logger.v(TAG
            + " updateSessionIfRequired() : No saved session for user will create a new session.");
        batchPreviousDataAndCreateNewSession(context, currentSource);
        return;
      }
      Logger.v(TAG + " updateSessionIfRequired() : Current Session: " + session);
      if (evaluator.canUpdateSourceInCurrentSession(session,
          MoEUtils.currentMillis())) {
        Logger.v(TAG + " updateSessionIfRequired() : Updating Traffic source.");
        session.trafficSource = currentSource;
        Logger.v(TAG + " updateSessionIfRequired() : Updated session: " + session);
        return;
      }
      Logger.v(TAG + " updateSessionIfRequired() : Cannot update existing session, will "
          + "create a new session if required.");
      // previous session expired
      boolean hasSessionExpired =
          evaluator.hasSessionExpired(session.lastInteractionTime,
              RemoteConfig.getConfig().sessionInActiveTime,
              MoEUtils.currentMillis());
      if (hasSessionExpired) {
        Logger.v(TAG
            + " updateSessionIfRequired() : Previous session has expired. Will create a new "
            + "session");
        batchPreviousDataAndCreateNewSession(context, currentSource);
        return;
      }
      // source changed
      TrafficSource savedSource = session.trafficSource;
      boolean hasSourceChanged = evaluator.hasSourceChanged(savedSource, currentSource);
      if (hasSourceChanged) {
        Logger.v(TAG + " updateSessionIfRequired() : Source changed. will create a new session");
        batchPreviousDataAndCreateNewSession(context, currentSource);
      }
    }
  }

  @WorkerThread private void batchPreviousDataAndCreateNewSession(Context context,
      TrafficSource currentSource) {
    synchronized (lock) {
      Logger.v(TAG + " batchPreviousDataAndCreateNewSession() : Will try to batch data and create"
          + " new session");
      DataManager.getInstance().batchData(context);
      createAndPersistNewSession(context, currentSource);
    }
  }

  private UserSession createAndPersistNewSession(Context context, TrafficSource currentSource) {
    session = createNewSession(currentSource);
    Logger.v(TAG + " createAndPersistNewSession() : New session: " + session.toString());
    persistUserSession(context, session);
    return session;
  }

  private UserSession createNewSession(@Nullable TrafficSource currentSource) {
    long currentTime = MoEUtils.currentMillis();
    return new UserSession(
        UUID.randomUUID().toString(),
        MoEUtils.getTimeInISO(currentTime),
        currentSource,
        currentTime
    );
  }

  private void persistUserSession(Context context, UserSession userSession) {
    MoEDAO.getInstance(context).saveUserSession(userSession);
  }

  public void onAppClose(Context context) {
    Logger.v(TAG + " onAppClose() : ");
    hasProcessedAppOpen = false;
    updateLastInteractionTime(MoEUtils.currentMillis());
    persistUserSession(context, session);
  }

  @WorkerThread public void onLogout(Context context) {
    createAndPersistNewSession(context, null);
  }

  @Nullable public UserSession getSession() {
    return session;
  }

  public static final int MAX_INTERVAL_FOR_SOURCE_UPDATE = 3;
}
