/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.core.internal;

import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.content.ServiceConnection;
import android.content.pm.PackageManager;
import android.os.IBinder;
import android.os.IInterface;
import android.os.Looper;
import android.os.Parcel;
import android.os.RemoteException;
import java.io.IOException;
import java.util.concurrent.LinkedBlockingQueue;

public class AdvertisingIdClient {

  public static final class AdInfo {
    private final String advertisingId;
    private final int limitAdTrackingEnabled;

    public AdInfo(String advertisingId, int limitAdTrackingEnabled) {
      this.advertisingId = advertisingId;
      this.limitAdTrackingEnabled = limitAdTrackingEnabled;
    }

    public String getId() {
      return this.advertisingId;
    }

    public int isLimitAdTrackingEnabled() {
      return this.limitAdTrackingEnabled;
    }
  }

  public static AdInfo getAdvertisingIdInfo(Context context) throws Exception {
    if(Looper.myLooper() == Looper.getMainLooper()) throw new IllegalStateException("Cannot be called from the main thread");

    PackageManager pm = context.getPackageManager(); pm.getPackageInfo("com.android.vending", 0);

    AdvertisingConnection connection = new AdvertisingConnection();
    Intent intent = new Intent("com.google.android.gms.ads.identifier.service.START");
    intent.setPackage("com.google.android.gms");
    if(context.bindService(intent, connection, Context.BIND_AUTO_CREATE)) {
      try {
        AdvertisingInterface adInterface = new AdvertisingInterface(connection.getBinder());
        return new AdInfo(adInterface.getId(), adInterface.isLimitAdTrackingEnabled(true));
      } finally {
        context.unbindService(connection);
      }
    }
    throw new IOException("Google Play connection failed");
  }

  private static final class AdvertisingConnection implements ServiceConnection {
    boolean retrieved = false;
    private final LinkedBlockingQueue<IBinder> queue = new LinkedBlockingQueue<IBinder>(1);

    public void onServiceConnected(ComponentName name, IBinder service) {
      try { this.queue.put(service); }
      catch (InterruptedException localInterruptedException){}
      catch (Exception exception){}
    }

    public void onServiceDisconnected(ComponentName name){}

    public IBinder getBinder() throws InterruptedException {
      if (this.retrieved) throw new IllegalStateException();
      this.retrieved = true;
      return (IBinder)this.queue.take();
    }
  }

  private static final class AdvertisingInterface implements IInterface {
    private IBinder binder;

    public AdvertisingInterface(IBinder pBinder) {
      binder = pBinder;
    }

    public IBinder asBinder() {
      return binder;
    }

    public String getId() throws RemoteException {
      Parcel data = Parcel.obtain();
      Parcel reply = Parcel.obtain();
      String id;
      try {
        data.writeInterfaceToken("com.google.android.gms.ads.identifier.internal.IAdvertisingIdService");
        binder.transact(1, data, reply, 0);
        reply.readException();
        id = reply.readString();
      } finally {
        reply.recycle();
        data.recycle();
      }
      return id;
    }

    public int isLimitAdTrackingEnabled(boolean paramBoolean) throws RemoteException {
      Parcel data = Parcel.obtain();
      Parcel reply = Parcel.obtain();
      int limitAdTracking;
      try {
        data.writeInterfaceToken("com.google.android.gms.ads.identifier.internal.IAdvertisingIdService");
        data.writeInt(paramBoolean ? 1 : 0);
        binder.transact(2, data, reply, 0);
        reply.readException();
        limitAdTracking = reply.readInt();
      } finally {
        reply.recycle();
        data.recycle();
      }
      return limitAdTracking;
    }
  }
}
