/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.core;

import androidx.annotation.Nullable;
import com.moengage.core.config.CardConfig;
import com.moengage.core.config.LogConfig;
import com.moengage.core.config.PushConfig;
import com.moengage.core.config.TrackingOptOutConfig;
import java.util.List;

/**
 * @author Umang Chamaria
 * Date: 17/10/19
 */
public class SdkConfig {
  /**
   * MoEngage App-id
   */
  public String appId;
  /**
   * List of activities on which in-app should not be shown.
   */
  @Nullable public List<String> inAppOptOutList;
  /**
   * List activities that should not be tracked. Activities in this list will be tracked if they
   * are landing screen when the app comes to foreground.
   */
  @Nullable public List<String> activityTrackingOptOutList;
  /**
   * If true sdk will not hide/show navigation bar below Android M.
   */
  public boolean isNavBarOptedOut;
  /**
   * If true GAID will not be tracked by the sdk.
   */
  public boolean isGaidTrackingOptedOut;
  /**
   * If true sdk will not track user location.
   */
  public boolean isLocationTrackingOptedOut;
  /**
   * If true sdk will not geo-fences.
   */
  public boolean isGeofenceTrackingOptedOut;
  /**
   * Region to which data should be sent.
   */
  public DataCenter dataCenter = DataCenter.DATA_CENTER_1;
  /**
   * Time interval after which data is synced when app is in foreground. Unit - Seconds
   */
  public long flushInterval = -1;
  /**
   * If true sdk will periodically sync data when the app is in foreground.
   */
  public boolean isPeriodicFlushEnabled = true;
  /**
   * If true sdk will try to perform location related action i.e. - track location, set
   * geo-fences etc.
   */
  public boolean isLocationServiceEnabled;
  /**
   * If true sdk will try to sync data periodically even if the app is in background.
   */
  public boolean isBackgroundSyncEnabled = true;
  /**
   * If true sdk will try to sync device trigger campaigns in background.
   */
  public boolean isRealTimeTriggerBackgroundSyncEnabled = true;
  /**
   * If true app is treated as segment integration.
   */
  public boolean isSegmentIntegration;
  /**
   * If true in-app will not be shown via activity lifecycle callbacks.
   */
  public boolean isLifecycleInAppOptedOut;
  /**
   * If true, location will be fetched when geofence syncs in background.
   */
  public boolean isBackgroundLocationFetchEnabled;
  /**
   * If true, geofence campaigns will be fetched in the background.
   */
  public boolean isGeofenceBackgroundSyncEnabled = true;
  /**
   * Card configuration
   */
  public CardConfig cardConfig;
  /**
   * Push Configuration
   */
  public PushConfig pushConfig;
  /**
   * Log Configuration
   */
  public LogConfig logConfig;
  /**
   * Tracking out-opt configuration
   */
  public TrackingOptOutConfig trackingOptOut;

  /**
   * If true, rest calls will be encrypted
   */
  public boolean isEncryptionEnabled;

  public SdkConfig(String appId) {
    this.appId = appId;
    cardConfig = new CardConfig(-1, -1, MoEConstants.CARD_DEFAULT_DATE_FORMAT);
    pushConfig = new PushConfig();
    logConfig = new LogConfig();
    trackingOptOut = new TrackingOptOutConfig(false, false, false, false);
  }

  SdkConfig() {
    this(null);
  }

  @Override public String toString() {
    return "{\n"
        +
        "\"appId\": \""
        + appId
        + "\" ,\n"
        +
        " \"inAppOptOutList\": "
        + inAppOptOutList
        + ",\n"
        +
        " \"activityTrackingOptOutList\": "
        + activityTrackingOptOutList
        + ",\n"
        +
        " \"isNavBarOptedOut\": "
        + isNavBarOptedOut
        + ",\n"
        +
        " \"isLocationTrackingOptedOut\": "
        + isLocationTrackingOptedOut
        + ",\n"
        +
        " \"isGeofenceTrackingOptedOut\": "
        + isGeofenceTrackingOptedOut
        + ",\n"
        +
        " \"dataRegion\": "
        + dataCenter
        + ",\n"
        +
        " \"flushInterval\": "
        + flushInterval
        + ",\n"
        +
        " \"isPeriodicFlushEnabled\": "
        + isPeriodicFlushEnabled
        + ",\n"
        +
        " \"isLocationServiceEnabled\": "
        + isLocationServiceEnabled
        + ",\n"
        +
        " \"isBackgroundSyncEnabled\": "
        + isBackgroundSyncEnabled
        + ",\n"
        +
        " \"isRealTimeTriggerBackgroundSyncEnabled\": "
        + isRealTimeTriggerBackgroundSyncEnabled
        + ",\n"
        +
        " \"isSegmentIntegration\": "
        + isSegmentIntegration
        + ",\n"
        +
        " \"isLifecycleInAppOptedOut\": "
        + isLifecycleInAppOptedOut
        + ",\n"
        +
        " \"isBackgroundLocationFetchEnabled\": "
        + isBackgroundLocationFetchEnabled
        + ",\n"
        +
        " \"isGeofenceBackgroundSyncEnabled\": "
        + isGeofenceBackgroundSyncEnabled
        + ",\n"
        + " \"cardConfig\": " + cardConfig + ",\n"
        + " \"pushConfig\": " + pushConfig + ",\n"
        + " \"isEncryptionEnabled\": " + isEncryptionEnabled + ",\n"
        + " \"logConfig\": " + logConfig + ",\n"
        + " \"trackingOptOut\": " + trackingOptOut + "\n"
        + '}';
  }

  private static SdkConfig sdkConfig;

  public static SdkConfig getConfig() {
    if (sdkConfig == null) {
      synchronized (SdkConfig.class) {
        if (sdkConfig == null) sdkConfig = new SdkConfig();
      }
    }
    return sdkConfig;
  }

  static void setSdkConfig(SdkConfig config) {
    sdkConfig = config;
  }
}
