/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.core;

import androidx.annotation.NonNull;
import com.moengage.core.internal.logger.LogUtils;
import com.moengage.core.utils.ApiUtility;
import java.util.Set;
import org.json.JSONException;
import org.json.JSONObject;

/**
 * @author Umang Chamaria
 * Date: 2020/10/01
 */
public class RemoteConfigParser {

  private static final String TAG = MoEConstants.MODULE_TAG + "RemoteConfigParser";

  public static RemoteConfig fromJson(JSONObject configurationJson) {
    try {
      RemoteConfig configuration = new RemoteConfig();
      if (configurationJson.has(RESPONSE_ATTR_PUSH_AMP_SYNC_DELAY)) {
        long syncDelay = configurationJson.getLong(RESPONSE_ATTR_PUSH_AMP_SYNC_DELAY);
        if (syncDelay > 0) {
          configuration.pushAmpSyncDelay = syncDelay * 1000;
        }
      }
      if (configurationJson.has(RESPONSE_ATTR_BLACKLIST_EVENT)) {
        configuration.blacklistedEvents = ApiUtility.jsonArrayToStringSet(
            configurationJson.getJSONArray(RESPONSE_ATTR_BLACKLIST_EVENT));
      }
      if (configurationJson.has(RESPONSE_ATTR_APP_STATE)) {
        configuration.isAppEnabled = getStateFromResponse(configurationJson,
            RESPONSE_ATTR_APP_STATE,
            RemoteConfigDefault.ACCOUNT_STATUS);
      }
      if (configurationJson.has(RESPONSE_ATTR_IN_APP_STATE)) {
        configuration.isInAppEnabled =
            getStateFromResponse(configurationJson, RESPONSE_ATTR_IN_APP_STATE,
                RemoteConfigDefault.IN_APP_STATUS);
      }
      if (configurationJson.has(RESPONSE_ATTR_GEO_STATE)) {
        configuration.isGeofenceEnabled =
            getStateFromResponse(configurationJson, RESPONSE_ATTR_GEO_STATE,
                RemoteConfigDefault.GEO_FENCE_STATUS);
      }
      if (configurationJson.has(RESPONSE_ATTR_PUSH_AMP_STATE)) {
        configuration.isPushAmpEnabled =
            getStateFromResponse(configurationJson, RESPONSE_ATTR_PUSH_AMP_STATE,
                RemoteConfigDefault.PUSH_AMP_STATUS);
      }
      if (configurationJson.has(RESPONSE_ATTR_EVENT_SYNC_COUNT)) {
        configuration.eventBatchCount = configurationJson.getInt(RESPONSE_ATTR_EVENT_SYNC_COUNT);
      }
      if (configurationJson.has(RESPONSE_ATTR_DATA_SYNC_RETRY_INTERVAL)) {
        configuration.dataSyncRetryInterval =
            configurationJson.getLong(RESPONSE_ATTR_DATA_SYNC_RETRY_INTERVAL) * 1000;
      }
      if (configurationJson.has(RESPONSE_ATTR_FLUSH_EVENTS)) {
        Set<String> events = ApiUtility.jsonArrayToStringSet(
            configurationJson.getJSONArray(RESPONSE_ATTR_FLUSH_EVENTS));
        if (events != null) {
          configuration.flushEvents.addAll(events);
        }
      }
      if (configurationJson.has(RESPONSE_ATTR_PERIODIC_FLUSH_STATE)) {
        configuration.isPeriodicFlushEnabled =
            getStateFromResponse(configurationJson, RESPONSE_ATTR_PERIODIC_FLUSH_STATE,
                RemoteConfigDefault.PERIODIC_FLUSH_STATE);
      }
      if (configurationJson.has(RESPONSE_ATTR_PERIODIC_FLUSH_TIME)) {
        configuration.periodicFlushTime =
            configurationJson.getLong(RESPONSE_ATTR_PERIODIC_FLUSH_TIME);
      }
      if (configurationJson.has(RESPONSE_ATTR_CAMPAIGN_ID_EXPIRY)) {
        configuration.pushAmpCampaignExpiryTime =
            configurationJson.getLong(RESPONSE_ATTR_CAMPAIGN_ID_EXPIRY);
      }
      if (configurationJson.has(RESPONSE_ATTR_REAL_TIME_TRIGGER_STATE)) {
        configuration.isRealTimeTriggerEnabled =
            getStateFromResponse(configurationJson, RESPONSE_ATTR_REAL_TIME_TRIGGER_STATE,
                RemoteConfigDefault.REAL_TIME_TRIGGER_STATUS);
      }
      if (configurationJson.has(RESPONSE_ATTR_REAL_TIME_TRIGGER_SYNC_TIME)) {
        configuration.realTimeTriggerBackgroundSyncInterval =
            configurationJson.getLong(RESPONSE_ATTR_REAL_TIME_TRIGGER_SYNC_TIME) * 1000;
      }
      if (configurationJson.has(RESPONSE_ATTR_GDPR_WHITELIST_EVENTS)) {
        Set<String> whitelistedEvents = ApiUtility.jsonArrayToStringSet(
            configurationJson.getJSONArray(RESPONSE_ATTR_GDPR_WHITELIST_EVENTS));
        if (whitelistedEvents != null) {
          configuration.gdprWhitelistEventList.addAll(whitelistedEvents);
        }
      }

      if (configurationJson.has(RESPONSE_ATTR_USER_ATTRIBUTE_CACHING_TIME)) {
        configuration.userAttributeCachingTime =
            configurationJson.getLong(RESPONSE_ATTR_USER_ATTRIBUTE_CACHING_TIME) * 1000;
      }
      if (configurationJson.has(RESPONSE_ATTR_BLOCKED_UNIQUE_ID_REGEX)) {
        configuration.blockedUniqueIdRegex = ApiUtility.jsonArrayToStringList(
            configurationJson.getJSONArray(RESPONSE_ATTR_BLOCKED_UNIQUE_ID_REGEX));
      }
      if (configurationJson.has(RESPONSE_ATTR_SESSION_INACTIVE_TIME)) {
        configuration.sessionInActiveTime = configurationJson.getLong(
            RESPONSE_ATTR_SESSION_INACTIVE_TIME) * 1000;
      }
      if (configurationJson.has(RESPONSE_ATTR_TRAFFIC_SOURCE_EXTRA_IDENTIFIERS)) {
        Set<String> identifiers = ApiUtility.jsonArrayToStringSet(
            configurationJson.getJSONArray(RESPONSE_ATTR_TRAFFIC_SOURCE_EXTRA_IDENTIFIERS));
        if (identifiers != null) {
          configuration.additionalSourceIdentifiers = identifiers;
        }
      }
      if (configurationJson.has(RESPONSE_ATTR_MI_PUSH_STATUS)) {
        configuration.isPushAmpPlusEnabled = getStateFromResponse(configurationJson,
            RESPONSE_ATTR_MI_PUSH_STATUS, RemoteConfigDefault.MI_PUSH_APP_STATUS);
      }
      if (configurationJson.has(RESPONSE_ATTR_DATA_ENCRYPTION_KEY)) {
        String key = configurationJson.getString(RESPONSE_ATTR_DATA_ENCRYPTION_KEY);
        if (MoEUtils.isEmptyString(key)) {
          key = RemoteConfigDefault.DEFAULT_DATA_ENCRYPTION_KEY;
        }
        configuration.encryptionKey = key;
      }
      if (configurationJson.has(RESPONSE_ATTR_REMOTE_LOGGING_STATUS)) {
        configuration.isRemoteLoggingEnabled = getStateFromResponse(configurationJson,
            RESPONSE_ATTR_REMOTE_LOGGING_STATUS, RemoteConfigDefault.DEFAULT_REMOTE_LOGGING_STATE);
      }

      String logLevel =
          configurationJson.optString(RESPONSE_ATTR_REMOTE_LOGGING_LEVEL,
              LogUtils.LOG_LEVEL_NO_LOG);
      if (MoEUtils.isEmptyString(logLevel)) {
        logLevel = LogUtils.LOG_LEVEL_NO_LOG;
      }
      configuration.logLevel = LogUtils.LOG_TYPE_TO_LEVEL_MAPPING.get(logLevel);

      return configuration;
    } catch (Exception e) {
      Logger.e(TAG + " parseConfigApiResponse() : Exception ", e);
    }
    return null;
  }

  private static boolean getStateFromResponse(@NonNull JSONObject responseJSON,
      @NonNull String responseAttr, boolean defaultValue) {
    try {
      String state = responseJSON.getString(responseAttr);
      switch (state) {
        case "blocked":
          return false;
        case "allowed":
          return true;
      }
    } catch (JSONException e) {
      Logger.e(" getStateFromResponse ", e);
    }
    return defaultValue;
  }

  private static final String RESPONSE_ATTR_PUSH_AMP_SYNC_DELAY = "m_s_t";

  private static final String RESPONSE_ATTR_BLACKLIST_EVENT = "b_e";

  private static final String RESPONSE_ATTR_APP_STATE = "a_s";

  private static final String RESPONSE_ATTR_IN_APP_STATE = "i_s";

  private static final String RESPONSE_ATTR_GEO_STATE = "g_s";

  private static final String RESPONSE_ATTR_PUSH_AMP_STATE = "in_s";

  private static final String RESPONSE_ATTR_EVENT_SYNC_COUNT = "e_b_c";

  private static final String RESPONSE_ATTR_DATA_SYNC_RETRY_INTERVAL = "d_s_r_i";

  private static final String RESPONSE_ATTR_FLUSH_EVENTS = "f_e";

  private static final String RESPONSE_ATTR_PERIODIC_FLUSH_TIME = "p_f_t";

  private static final String RESPONSE_ATTR_PERIODIC_FLUSH_STATE = "p_f_s";

  private static final String RESPONSE_ATTR_CAMPAIGN_ID_EXPIRY = "cid_ex";

  private static final String RESPONSE_ATTR_REAL_TIME_TRIGGER_STATE = "d_t";

  private static final String RESPONSE_ATTR_REAL_TIME_TRIGGER_SYNC_TIME = "dt_s_t";

  private static final String RESPONSE_ATTR_GDPR_WHITELIST_EVENTS = "d_t_w_e";

  private static final String RESPONSE_ATTR_USER_ATTRIBUTE_CACHING_TIME = "u_a_c_t";

  private static final String RESPONSE_ATTR_BLOCKED_UNIQUE_ID_REGEX = "b_uid_r";

  private static final String RESPONSE_ATTR_SESSION_INACTIVE_TIME = "s_i_d";

  private static final String RESPONSE_ATTR_TRAFFIC_SOURCE_EXTRA_IDENTIFIERS = "src_ext";

  private static final String RESPONSE_ATTR_MI_PUSH_STATUS = "mi_p_s";

  private static final String RESPONSE_ATTR_DATA_ENCRYPTION_KEY = "d_e_k";

  private static final String RESPONSE_ATTR_REMOTE_LOGGING_STATUS = "s_log";

  private static final String RESPONSE_ATTR_REMOTE_LOGGING_LEVEL = "log_level";
}
