/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.core;

import androidx.annotation.Nullable;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

/**
 * @author Umang Chamaria
 */
public class RemoteConfig {

  /**
   * Time interval after which push-amp api is synced when the app is in background.
   */
  public long pushAmpSyncDelay;
  /**
   * Events which shouldn't be sent to server even if tracked by the client.
   */
  @Nullable public Set<String> blacklistedEvents;
  /**
   * If true account is enabled.
   */
  public boolean isAppEnabled;
  /**
   * If true in-app is enabled for the account.
   */
  public boolean isInAppEnabled;
  /**
   * If true geofence is enabled for the account.
   */
  public boolean isGeofenceEnabled;
  /**
   * If true push-amp is enabled for the account.
   */
  public boolean isPushAmpEnabled;
  /**
   * Event count threshold for syncing data.
   */
  public int eventBatchCount;
  /**
   * Time interval after which sdk tries to sync data when the app is in background.
   */
  public long dataSyncRetryInterval;
  /**
   * Set of events which should be synced immediately.
   */
  public Set<String> flushEvents;
  /**
   * If true sdk will try to sync data periodically when the app is in foreground.
   */
  public boolean isPeriodicFlushEnabled;
  /**
   * Time interval after which sdk attempts to sync data.
   */
  public long periodicFlushTime;
  /**
   * Time duration after which campaign-ids cached for push-amp should be removed.
   */
  public long pushAmpCampaignExpiryTime;
  /**
   * If true real-time trigger is enabled for the account.
   */
  public boolean isRealTimeTriggerEnabled;
  /**
   * Time duration after which real time trigger is synced after app close.
   */
  public long realTimeTriggerBackgroundSyncInterval;
  /**
   * Events to be tracked even if user has opted out of data tracking.
   */
  public Set<String> gdprWhitelistEventList;
  /**
   * Time duration before which same user attribute value shouldn't be sent to server.
   */
  public long userAttributeCachingTime;
  /**
   * USER_ATTRIBUTE_UNIQUE_ID matching this regex should not be tracked.
   */
  @Nullable public List<String> blockedUniqueIdRegex;
  /**
   * Time duration after which session is marked as in-active.
   */
  public long sessionInActiveTime;
  /**
   * Additional keys to identify traffic source.
   */
  public Set<String> additionalSourceIdentifiers;
  /**
   * If true push-amp+ is enabled for the account.
   */
  public boolean isPushAmpPlusEnabled;
  /**
   * Secret key
   */
  String encryptionKey;
  /**
   * Level of logging for remote logging.
   */
  public @LogLevel int logLevel;
  /**
   * If true, remote logging is enabled else false.
   */
  public boolean isRemoteLoggingEnabled;

  public RemoteConfig() {
    pushAmpSyncDelay = RemoteConfigDefault.PUSH_AMP_SYNC_INTERVAL;
    isAppEnabled = RemoteConfigDefault.ACCOUNT_STATUS;
    isInAppEnabled = RemoteConfigDefault.IN_APP_STATUS;
    isGeofenceEnabled = RemoteConfigDefault.GEO_FENCE_STATUS;
    isPushAmpEnabled = RemoteConfigDefault.PUSH_AMP_STATUS;
    eventBatchCount = RemoteConfigDefault.EVENT_BATCH_COUNT;
    dataSyncRetryInterval = RemoteConfigDefault.DATA_SYNC_RETRY_INTERVAL;
    isPeriodicFlushEnabled = RemoteConfigDefault.PERIODIC_FLUSH_STATE;
    periodicFlushTime = RemoteConfigDefault.PERIODIC_FLUSH_TIME;
    pushAmpCampaignExpiryTime = RemoteConfigDefault.PUSH_AMP_CAMPAIGN_EXPIRY_TIME;
    isRealTimeTriggerEnabled = RemoteConfigDefault.REAL_TIME_TRIGGER_STATUS;
    realTimeTriggerBackgroundSyncInterval = RemoteConfigDefault.REAL_TIME_TRIGGER_SYNC_INTERVAL;
    userAttributeCachingTime = RemoteConfigDefault.USER_ATTRIBUTE_CACHING_TIME;
    sessionInActiveTime = RemoteConfigDefault.DEFAULT_SESSION_INACTIVE_TIME;
    additionalSourceIdentifiers = new HashSet<>();
    flushEvents = new HashSet<>();
    flushEvents.addAll(RemoteConfigDefault.DEFAULT_FLUSH_EVENTS);
    gdprWhitelistEventList = new HashSet<>();
    gdprWhitelistEventList.addAll(RemoteConfigDefault.DEFAULT_GDPR_WHITELIST_EVENTS);
    isPushAmpPlusEnabled = RemoteConfigDefault.MI_PUSH_APP_STATUS;
    encryptionKey = RemoteConfigDefault.DEFAULT_DATA_ENCRYPTION_KEY;
    isRemoteLoggingEnabled = RemoteConfigDefault.DEFAULT_REMOTE_LOGGING_STATE;
    logLevel = RemoteConfigDefault.DEFAULT_REMOTE_LOGGING_LEVEL;
  }

  @Override public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;

    RemoteConfig that = (RemoteConfig) o;
    if (pushAmpSyncDelay != that.pushAmpSyncDelay) return false;
    if (isAppEnabled != that.isAppEnabled) return false;
    if (isInAppEnabled != that.isInAppEnabled) return false;
    if (isGeofenceEnabled != that.isGeofenceEnabled) return false;
    if (isPushAmpEnabled != that.isPushAmpEnabled) return false;
    if (eventBatchCount != that.eventBatchCount) return false;
    if (dataSyncRetryInterval != that.dataSyncRetryInterval) return false;
    if (isPeriodicFlushEnabled != that.isPeriodicFlushEnabled) return false;
    if (periodicFlushTime != that.periodicFlushTime) return false;
    if (pushAmpCampaignExpiryTime != that.pushAmpCampaignExpiryTime) return false;
    if (isRealTimeTriggerEnabled != that.isRealTimeTriggerEnabled) return false;
    if (realTimeTriggerBackgroundSyncInterval != that.realTimeTriggerBackgroundSyncInterval) return false;
    if (userAttributeCachingTime != that.userAttributeCachingTime) return false;
    if (sessionInActiveTime != that.sessionInActiveTime) return false;

    if (blacklistedEvents != null ? !blacklistedEvents.equals(that.blacklistedEvents)
        : that.blacklistedEvents != null) {
      return false;
    }
    if (flushEvents != null ? !flushEvents.equals(that.flushEvents)
        : that.flushEvents != null) {
      return false;
    }
    if (gdprWhitelistEventList != null ? !gdprWhitelistEventList.equals(that.gdprWhitelistEventList)
        : that.gdprWhitelistEventList != null) {
      return false;
    }
    return blockedUniqueIdRegex != null ? blockedUniqueIdRegex.equals(that.blockedUniqueIdRegex)
        : that.blockedUniqueIdRegex == null;
  }

  private static RemoteConfig remoteConfig;

  public static RemoteConfig getConfig() {
    if (remoteConfig == null) {
      synchronized (RemoteConfig.class) {
        if (remoteConfig == null) remoteConfig = new RemoteConfig();
      }
    }
    return remoteConfig;
  }

  public static void setRemoteConfig(RemoteConfig config) {
    remoteConfig = config;
  }

  @Override public String toString() {
    return "RemoteConfig{" +
        "pushAmpSyncDelay=" + pushAmpSyncDelay +
        ", blacklistedEvents=" + blacklistedEvents +
        ", isAppEnabled=" + isAppEnabled +
        ", isInAppEnabled=" + isInAppEnabled +
        ", isGeofenceEnabled=" + isGeofenceEnabled +
        ", isPushAmpEnabled=" + isPushAmpEnabled +
        ", eventBatchCount=" + eventBatchCount +
        ", dataSyncRetryInterval=" + dataSyncRetryInterval +
        ", flushEvents=" + flushEvents +
        ", isPeriodicFlushEnabled=" + isPeriodicFlushEnabled +
        ", periodicFlushTime=" + periodicFlushTime +
        ", pushAmpCampaignExpiryTime=" + pushAmpCampaignExpiryTime +
        ", isRealTimeTriggerEnabled=" + isRealTimeTriggerEnabled +
        ", realTimeTriggerBackgroundSyncInterval=" + realTimeTriggerBackgroundSyncInterval +
        ", gdprWhitelistEventList=" + gdprWhitelistEventList +
        ", userAttributeCachingTime=" + userAttributeCachingTime +
        ", blockedUniqueIdRegex=" + blockedUniqueIdRegex +
        ", sessionInActiveTime=" + sessionInActiveTime +
        ", additionalSourceIdentifiers=" + additionalSourceIdentifiers +
        ", isPushAmpPlusEnabled=" + isPushAmpPlusEnabled +
        ", encryptionKey='" + encryptionKey + '\'' +
        ", logLevel=" + logLevel +
        ", isRemoteLoggingEnabled=" + isRemoteLoggingEnabled +
        '}';
  }
}
