/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use
 * of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that
 * all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or
 * promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written
 * permission. Any such redistribution must retain the above copyright notice, this list of
 * conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR
 *  IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 *  CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.core;

import android.location.Location;
import androidx.annotation.NonNull;
import androidx.annotation.RestrictTo;
import androidx.annotation.RestrictTo.Scope;
import com.moe.pushlibrary.PayloadBuilder;
import com.moe.pushlibrary.models.GeoLocation;
import java.util.Date;
import org.json.JSONArray;
import org.json.JSONObject;

/**
 * A helper class for building attributes
 *
 * @author Umang Chamaria
 * @since 10.0.00
 * Date: 12/02/20
 */
public class Properties {
  private static final String TAG = MoEConstants.MODULE_TAG + "Properties";
  private PayloadBuilder payloadBuilder;

  public Properties() {
    payloadBuilder = new PayloadBuilder();
  }

  /**
   * Add an attribute with value of type Object.
   * <br/> <b>Note:</b> This API only accepts the below data-types:<br/>
   *
   * String<br/>
   * Integer<br/>
   * Long<br/>
   * Double<br/>
   * Float<br/>
   * Boolean<br/>
   * Date<br/>
   * GeoLocation<br/>
   * JSONArray<br/>
   * JSONObject<br/>
   *
   * If any other data-type is passed payload would be rejected.
   *
   * @param attributeName Attribute name
   * @param attributeValue Attribute value
   * @return Instance of {@link Properties}
   * @since 10.0.00
   */
  @SuppressWarnings("ConstantConditions")
  public Properties addAttribute(@NonNull String attributeName, Object attributeValue) {
    if (isAcceptedDataType(attributeValue)) {
      addAttributeInternal(attributeName, attributeValue);
    }
    return this;
  }

  /**
   * Add an attribute with value of type date. The input string should in ISO 8601 format.<br/>
   * Accepted Format - [yyyy-MM-dd|yyyyMMdd][T(hh:mm[:ss[.sss]]|hhmm[ss[.sss]])]?[Z|[+-]hh:mm]]
   *
   * @param attributeName Attribute name
   * @param attributeValue Attribute value
   * @return Instance of {@link Properties}
   * @since 10.0.00
   */
  public Properties addDateIso(@NonNull String attributeName, String attributeValue) {
    payloadBuilder.putAttrISO8601Date(attributeName, attributeValue);
    return this;
  }

  /**
   * Add an attribute with value of type date. The input value should be in Epoch(milliseconds)
   *
   * @param attributeName Attribute name
   * @param attributeValue Attribute value
   * @return Instance of {@link Properties}
   * @since 10.0.00
   */
  public Properties addDateEpoch(@NonNull String attributeName, long attributeValue) {
    payloadBuilder.putAttrDateEpoch(attributeName, attributeValue);
    return this;
  }

  /**
   * Marks an event as non-interactive.
   *
   * @return Instance of {@link Properties}
   * @since 10.0.00
   */
  public Properties setNonInteractive() {
    payloadBuilder.setNonInteractive();
    return this;
  }

  @RestrictTo(Scope.LIBRARY) public PayloadBuilder getPayload() {
    return payloadBuilder;
  }

  private void addAttributeInternal(String attributeName, Object attributeValue) {
    try {
      if (MoEUtils.isEmptyString(attributeName)) return;
      if (attributeValue instanceof GeoLocation) {
        payloadBuilder.putAttrLocation(attributeName, (GeoLocation) attributeValue);
      } else if (attributeValue instanceof Date) {
        payloadBuilder.putAttrDate(attributeName, (Date) attributeValue);
      } else if (attributeValue instanceof Location) {
        payloadBuilder.putAttrLocation(attributeName, (Location) attributeValue);
      } else {
        payloadBuilder.putAttrObject(attributeName, attributeValue);
      }
    } catch (Exception e) {
      Logger.e(TAG + " addAttributeInternal() : ", e);
    }
  }

  private boolean isAcceptedDataType(Object attributeValue) {
    return attributeValue instanceof String
        || attributeValue instanceof Integer
        || attributeValue instanceof Long
        || attributeValue instanceof Double
        || attributeValue instanceof Float
        || attributeValue instanceof Boolean
        || attributeValue instanceof Date
        || attributeValue instanceof GeoLocation
        || attributeValue instanceof JSONArray
        || attributeValue instanceof JSONObject
        || attributeValue instanceof Location;
  }
}
