/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.core;

import android.app.Activity;
import android.content.Context;
import android.content.Intent;
import android.database.Cursor;
import android.os.Bundle;
import androidx.annotation.Nullable;
import androidx.annotation.RestrictTo;
import androidx.annotation.RestrictTo.Scope;
import androidx.annotation.WorkerThread;
import com.moe.pushlibrary.MoEHelper;
import com.moe.pushlibrary.models.Event;
import com.moe.pushlibrary.utils.MoEHelperConstants;
import com.moengage.core.internal.AppCloseTask;
import com.moengage.core.internal.MoEWorkerTask;
import com.moengage.core.internal.analytics.AnalyticsHelper;
import com.moengage.core.cards.CardManager;
import com.moengage.core.executor.ITask;
import com.moengage.core.executor.OnTaskCompleteListener;
import com.moengage.core.executor.SDKTask;
import com.moengage.core.executor.TaskManager;
import com.moengage.core.executor.TaskProcessor;
import com.moengage.core.executor.TaskResult;
import com.moengage.core.internal.inapp.InAppManager;
import com.moengage.core.internal.storage.StorageProvider;
import com.moengage.core.listeners.AppBackgroundListener;
import com.moengage.core.listeners.OnAppBackgroundListener;
import com.moengage.core.listeners.OnLogoutCompleteListener;
import com.moengage.core.internal.location.GeoManager;
import com.moengage.core.mipush.MiPushManager;
import com.moengage.core.internal.pushamp.PushAmpManager;
import com.moengage.core.internal.remoteconfig.ConfigApiNetworkTask;
import com.moengage.core.internal.reports.DataManager;
import com.moengage.push.PushManager;
import com.moengage.push.hms.puskit.PushKitManager;
import java.util.Locale;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import org.json.JSONObject;

/**
 * @author Umang Chamaria
 */
@RestrictTo(Scope.LIBRARY_GROUP)
public class MoEDispatcher implements OnTaskCompleteListener {

  private static final String TAG = MoEConstants.MODULE_TAG + "MoEDispatcher";

  private Context context;

  private static MoEDispatcher instance;

  private boolean shouldTrackUniqueId = false;

  private JSONObject uniqueIdAttribute = null;

  private ScheduledExecutorService scheduler;

  private DeviceAddManager deviceAddManager = null;

  /**
   * MoEDispatcher constructor. There should be only one instance of
   * MoEDispatcher per task instance of an APP
   *
   * @param context Application Context
   */
  private MoEDispatcher(Context context) {
    if (context != null) {
      this.context = context;
      TaskProcessor.getInstance().setOnTaskCompleteListener(this);
    } else {
      Logger.v(TAG + " context is null");
    }
  }

  public static MoEDispatcher getInstance(Context context) {
    if (instance == null) {
      synchronized (MoEDispatcher.class) {
        if (instance == null) instance = new MoEDispatcher(context);
      }
    }
    return instance;
  }

  /**
   * Lifecycle callback of an activity lifecycle
   *
   * @param activity activity reference {@link Activity}
   * @param intent intent calling that activity
   */
  public void onStart(Activity activity, Intent intent) {
    try {
      if (!RemoteConfig.getConfig().isAppEnabled) return;
      if (null == activity) {
        Logger.v(TAG + " onStart activity instance is null");
        return;
      }
      if (null == intent) {
        intent = activity.getIntent();
      }
      context = activity.getApplicationContext();
      Logger.v(TAG + " onStart ----");
      MoEUtils.dumpIntentExtras(TAG, intent.getExtras());
      TaskManager.getInstance().addTaskToQueue(new ActivityStartTask(activity));
      InAppManager.getInstance().showInAppIfRequired(context);

      MoEUtils.updateTestDeviceState(this.context);
    } catch (Exception e) {
      Logger.e(TAG + " onStart() : ", e);
    }
  }

  /**
   * Logs impression for notification click
   *
   * @param gtime notification clicked time
   */
  public void trackNotificationClicked(long gtime) {
    if (!RemoteConfig.getConfig().isAppEnabled) return;
    TaskManager.getInstance().addTaskToQueue(new NotificationClickedTask(context, gtime));
  }

  /**
   * Saves a user attribute
   *
   * @param userJson user attribute in json format
   */
  public void setUserAttribute(JSONObject userJson) {
    MoEHelper.getInstance(context).getAttributeManager().setUserAttribute(userJson);
  }

  public void onResume(Activity activity, boolean isRestoring) {
    if (!RemoteConfig.getConfig().isAppEnabled) return;
    if (!isRestoring) {
      showDialogAfterPushClick(activity);
    }
  }

  /**
   * Lifecycle callback of activity lifecycle
   *
   * @param activity activity reference {@link Activity}
   */
  public void onStop(Activity activity) {
    if (!RemoteConfig.getConfig().isAppEnabled) return;
    if (null == activity) return;
    int activityCounter = MoEHelper.getActivityCounter();
    Logger.d(TAG + " activity counter " + activityCounter);
    String activityName = activity.getClass().getName();
    if (!MoEUtils.isEmptyString(activityName)) {
      Logger.v(TAG + " " + activityName + " stopped");
    }
    if (activityCounter == 0 && SdkConfig.getConfig().isSegmentIntegration) {
      TaskManager.getInstance().addTaskToQueue(new AppCloseTask(context));
    }
  }

  /**
   * Logs out the user. Clears the data and tries to re-register for push in case push
   * registration is handled by MoEngage.
   */
  @RestrictTo(Scope.LIBRARY)
  @WorkerThread public void handleLogout(boolean isForcedLogout) {
    try {
      Logger.v(TAG + " handleLogout() : Started logout process");
      if (!RemoteConfig.getConfig().isAppEnabled) return;
      CardManager.getInstance().onLogout(context);
      trackLogoutEvent(isForcedLogout);
      DataManager.getInstance().batchData(context);
      DataManager.getInstance()
          .sendData(context, SdkConfig.getConfig().appId, DataManager.NO_RETRY);
      InAppManager.getInstance().onLogout(context);
      MoEDAO.getInstance(context).clearDataOnLogout();
      new MoEFileManager(context).clearMoEngageFiles();
      AnalyticsHelper.getInstance(context).onLogout(context);
      PushManager.getInstance().registerForPush(context);
      getDeviceAddManager().registerDevice(context);
      notifyLogoutCompleteListener();
      Logger.v(TAG + " handleLogout() : Logout process complete.");
    } catch (Exception e) {
      Logger.e(TAG + " handleLogout() : ", e);
    }
  }

  @WorkerThread private void trackLogoutEvent(boolean isForcedLogout) {
    try {
      Properties eventAttributes = new Properties();
      if (isForcedLogout) eventAttributes.addAttribute("type", "forced");
      eventAttributes.setNonInteractive();
      Event event =
          new Event(MoEConstants.LOGOUT_EVENT, eventAttributes.getPayload().build());
      MoEDAO.getInstance(context).addEvent(event);
    } catch (Exception e) {
      Logger.e(TAG + " trackLogoutEvent(): ", e);
    }
  }

  /**
   * Sends all the tracked events to the server.
   */
  @RestrictTo(Scope.LIBRARY_GROUP)
  public void sendInteractionData() {
    DataManager.getInstance().batchAndSyncData(context);
  }

  /**
   * Updates the local DB that the specified message has been clicked<br>
   * <b>Note : Don't call on UI Thread</b>.
   *
   * @param id The id associated with the inbox message that was clicked
   */
  @WorkerThread public void setInboxMessageClicked(long id) {
    MoEDAO.getInstance(context).setMessageClicked(id);
  }

  /**
   * Handles the app update event, registers,and tries to get new GCM registration ID<br>
   * <b>Note : Don't call on UI Thread</b>
   */
  @WorkerThread public void handleAppUpdateEvent() {
    //Logging an update event
    try {
      if (!RemoteConfig.getConfig().isAppEnabled) return;
      int prevVersion = ConfigurationProvider.getInstance(context).getStoredAppVersion();
      Properties eventObj = new Properties();
      eventObj.addAttribute(MoEHelperConstants.FROM_VERSION, prevVersion);
      eventObj.addAttribute(MoEHelperConstants.TO_VERSION,
          ConfigurationProvider.getInstance(context).getAppVersion());
      //update event
      Logger.i("Adding an update event");
      MoEHelper.getInstance(context).trackEvent(MoEHelperConstants.EVENT_APP_UPDATE, eventObj);
      if (!MoEngage.isAppForeground()) {
        sendInteractionData();
      }
    } catch (Exception e) {
      Logger.e("Adding update event", e);
    }
  }

  /**
   * Get a all inbox messages<br>
   * <b>Note : Don't call on UI Thread</b>
   *
   * @return A populated cursor with the inbox messages or null
   */
  @Nullable @WorkerThread public Cursor getAllMessages() {
    return MoEDAO.getInstance(context).getMessages(context);
  }

  /**
   * Returns the number of unread messages. Unread messages are counted based on clicks<br>
   * <b>Note : Don't call on UI Thread</b>
   *
   * @return The Count of unread messages
   */
  @WorkerThread public int getUnreadMessageCount() {
    return MoEDAO.getInstance(context).getUnreadMessageCount();
  }

  /**
   * Show a dialog after push was clicked
   *
   * @param activity The instance of the activity where the dialog needs to be created
   */
  private void showDialogAfterPushClick(Activity activity) {
    if (null == activity) return;
    try {
      Intent intent = activity.getIntent();
      if (intent != null) {
        Bundle extras = intent.getExtras();
        if (extras != null && extras.containsKey(MoEHelperConstants.GCM_EXTRA_SHOW_DIALOG)) {
          //remove the extra
          intent.removeExtra(MoEHelperConstants.GCM_EXTRA_SHOW_DIALOG);
          if (extras.containsKey(MoEHelperConstants.GCM_EXTRA_COUPON_CODE)) {
            MoEUtils.showCouponDialog(extras.getString(MoEConstants.PUSH_NOTIFICATION_MESSAGE),
                extras.getString(MoEHelperConstants.GCM_EXTRA_COUPON_CODE), activity);
            intent.removeExtra(MoEConstants.PUSH_NOTIFICATION_MESSAGE);
            intent.removeExtra(MoEHelperConstants.GCM_EXTRA_COUPON_CODE);
          } else {
            MoEUtils.showNormalDialogWithOk(extras.getString(MoEConstants.PUSH_NOTIFICATION_MESSAGE),
                activity);
            intent.removeExtra(MoEConstants.PUSH_NOTIFICATION_MESSAGE);
          }
        }
      }
    } catch (Exception e) {
      Logger.e(TAG + " showDialogAfterPushClick : ", e);
    }
  }

  /**
   * Use {@link TaskManager#addTaskToQueueBeginning(ITask)}
   */
  @Deprecated public void addTaskToQueueBeginning(ITask task) {
    TaskManager.getInstance().addTaskToQueueBeginning(task);
  }

  /**
   * Use {@link TaskManager#startTask(ITask)}
   */
  @Deprecated public void startTask(ITask task) {
    TaskManager.getInstance().startTask(task);
  }

  @Override public void onTaskComplete(String tag, TaskResult taskResult) {
    // removing task from running list
    Logger.v("Task completed : " + tag);
    switch (tag) {
      case SDKTask.TAG_TRACK_ATTRIBUTE:
        if (!taskResult.isSuccess()) {
          shouldTrackUniqueId = true;
          uniqueIdAttribute = (JSONObject) taskResult.getPayload();
        }
        break;
      case SDKTask.TAG_DEVICE_ADD:
        deviceAddManager.processTaskResult(context, taskResult);
        break;
      case SDKTask.TAG_LOGOUT_TASK:
        if (shouldTrackUniqueId) trackChangedUniqueId();
    }
  }

  @RestrictTo(Scope.LIBRARY_GROUP)
  public void syncConfigIfRequired() {
    if (StorageProvider.INSTANCE.getRepository(context, SdkConfig.getConfig()).getConfigSyncTime()
        + MoEConstants.CONFIG_API_SYNC_DELAY < MoEUtils.currentMillis()) {
      TaskManager.getInstance().startTask(new ConfigApiNetworkTask(context));
    }
    TaskManager.getInstance().startTask(new ConfigApiNetworkTask(context));
  }

  public void trackDeviceLocale() {
    try {
      if (!RemoteConfig.getConfig().isAppEnabled) return;
      trackDeviceAndUserAttribute("LOCALE_COUNTRY", Locale.getDefault().getCountry());
      ;
      trackDeviceAndUserAttribute("LOCALE_COUNTRY_DISPLAY",
          Locale.getDefault().getDisplayCountry());
      trackDeviceAndUserAttribute("LOCALE_LANGUAGE", Locale.getDefault().getLanguage());
      trackDeviceAndUserAttribute("LOCALE_LANGUAGE_DISPLAY",
          Locale.getDefault().getDisplayLanguage());
      trackDeviceAndUserAttribute("LOCALE_DISPLAY", Locale.getDefault().getDisplayName());
      trackDeviceAndUserAttribute("LOCALE_COUNTRY_ ISO3", Locale.getDefault().getISO3Country());
      trackDeviceAndUserAttribute("LOCALE_LANGUAGE_ISO3", Locale.getDefault().getISO3Language());
    } catch (Exception e) {
      Logger.e(TAG + " trackDeviceLocale", e);
    }
  }

  private void trackDeviceAndUserAttribute(String attrName, String attrValue) {
    try {
      JSONObject attribute = new JSONObject();
      attribute.put(attrName, attrValue);
      setUserAttribute(attribute);
      //setDeviceAttribute(attribute);
    } catch (Exception e) {
      Logger.e(TAG + " trackDeviceAndUserAttribute() ", e);
    }
  }

  public void logoutUser(final boolean isForcedLogout) {
    try {
      Bundle extras = new Bundle();
      extras.putBoolean(MoEConstants.SERVICE_LOGOUT_TYPE, isForcedLogout);
      TaskManager.getInstance()
          .addTaskToQueue(new MoEWorkerTask(context, MoEConstants.SERVICE_TYPE_LOGOUT, extras));
    } catch (Exception e) {
      Logger.e(TAG + " logoutUser() ", e);
    }
  }

  private void trackChangedUniqueId() {
    if (uniqueIdAttribute != null) {
      setUserAttribute(uniqueIdAttribute);
      uniqueIdAttribute = null;
      shouldTrackUniqueId = false;
    }
  }

  public void setAlias(JSONObject aliasJSON) {
    if (!RemoteConfig.getConfig().isAppEnabled) return;
    TaskManager.getInstance().addTaskToQueue(new SetAliasTask(context, aliasJSON));
  }

  public void setDeviceAttribute(JSONObject deviceAttribute) {
    TaskManager.getInstance().addTaskToQueue(new SetDeviceAttributeTask(context, deviceAttribute));
  }

  private void schedulePeriodicFlushIfRequired() {
    try {
      Logger.v(TAG + " schedulePeriodicFlushIfRequired() : Will try to schedule periodic flush if"
          + " enabled.");
      if (RemoteConfig.getConfig().isPeriodicFlushEnabled
          && SdkConfig.getConfig().isPeriodicFlushEnabled) {
        Runnable syncRunnable = new Runnable() {
          @Override public void run() {
            Logger.v(
                TAG + " schedulePeriodicFlushIfRequired() inside runnable, will sync "
                + "now");
            sendInteractionData();
          }
        };
        long timeDelay = RemoteConfig.getConfig().periodicFlushTime;
        if (SdkConfig.getConfig().flushInterval > timeDelay) {
          timeDelay = SdkConfig.getConfig().flushInterval;
        }
        Logger.v(TAG + " schedulePeriodicFlushIfRequired() scheduling periodic sync");
        scheduler = Executors.newScheduledThreadPool(1);
        scheduler.scheduleWithFixedDelay(syncRunnable, timeDelay, timeDelay, TimeUnit.SECONDS);
      }
    } catch (Exception e) {
      Logger.e(TAG + " schedulePeriodicFlushIfRequired() ", e);
    }
  }

  private void shutDownPeriodicFlush() {
    try {
      if (RemoteConfig.getConfig().isPeriodicFlushEnabled
          && SdkConfig.getConfig().isPeriodicFlushEnabled
          && scheduler != null) {
        Logger.v(TAG + " shutDownPeriodicFlush() shutting down periodic flush");
        scheduler.shutdownNow();
      }
    } catch (Exception e) {
      Logger.e(TAG + " shutDownPeriodicFlush() ", e);
    }
  }

  public void onAppOpen() {
    if (!RemoteConfig.getConfig().isAppEnabled) return;
    try {
      Logger.v(TAG + " onAppOpen(): Application coming to foreground.");
      TaskManager.getInstance().addTaskToQueue(new AppOpenTask(context));
      syncConfigIfRequired();
      updateFeatureConfigForOptOutIfRequired();
      schedulePeriodicFlushIfRequired();
      // enable logs if required.
      if (ConfigurationProvider.getInstance(context).isDebugLogEnabled()) {
        SdkConfig.getConfig().logConfig.isEnabledForSignedBuild = true;
        SdkConfig.getConfig().logConfig.level = LogLevel.VERBOSE;
      }
      updatePushService();
    } catch (Exception e) {
      Logger.e(TAG + " onAppOpen() ", e);
    }
  }

  void updateFeatureConfigForOptOutIfRequired() {
    ConfigurationProvider provider = ConfigurationProvider.getInstance(context);
    if (provider.isDataTrackingOptedOut()) {
      // gaid optOut
      SdkConfig.getConfig().trackingOptOut.setGaidOptedOut(true);
      //androidId optOut
      SdkConfig.getConfig().trackingOptOut.setAndroidIdOptedOut(true);
      //location optOut
      SdkConfig.getConfig().isLocationTrackingOptedOut = true;
      //geo-fence optOut
      SdkConfig.getConfig().isGeofenceTrackingOptedOut = true;
      //device attributes optOut
      SdkConfig.getConfig().trackingOptOut.setDeviceAttributeOptedOut(true);
      //setting location services state
      SdkConfig.getConfig().isLocationServiceEnabled = false;
    }
    if (provider.isPushNotificationOptedOut()) {
      Logger.v(TAG + " updateFeatureConfigForOptOutIfRequired() : Push opted out clearing Push "
          + "Token.");
      provider.clearPushToken();
    }
  }

  void notifyLogoutCompleteListener() {
    Logger.v(TAG + " notifyLogoutCompleteListener() : Notifying listeners");
    for (OnLogoutCompleteListener listener : MoECallbacks.getInstance().getLogoutListenerList()) {
      try {
        listener.logoutComplete();
      } catch (Exception e) {
        Logger.e(TAG + " notifyLogoutCompleteListener() : ", e);
      }
    }
  }

  public void onAppClose() {
    if (!RemoteConfig.getConfig().isAppEnabled) return;
    Logger.v(TAG + " onAppClose(): Application going to background.");

    // schedule data sending
    DataManager.getInstance().scheduleDataSending(context);
    // notify when app goes to background
    notifyOnAppBackground();
    // retry device add if token and app id is available
    getDeviceAddManager().retryDeviceRegistrationIfRequired(context);
    // shutdown periodic flush
    shutDownPeriodicFlush();
    // track exit intent
    trackAppExit();
    // update last active time
    AnalyticsHelper.getInstance(context).onAppClose(context);
    InAppManager.getInstance().onAppClose(context);
    //schedule device triggers
    MoEDTManager.getInstance().scheduleBackgroundSync(context);
    // save sent screens
    ConfigurationProvider.getInstance(context)
        .saveSentScreenNames(ConfigurationCache.getInstance().getSentScreenNames());
    //schedule message pull
    PushAmpManager.getInstance().scheduleServerSync(context);
    GeoManager.getInstance().scheduleBackgroundSync(context);
  }

  private void notifyOnAppBackground() {
    for (OnAppBackgroundListener listener : MoECallbacks.getInstance()
        .getBackgroundListenerList()) {
      try {
        listener.goingToBackground();
      } catch (Exception e) {
        Logger.e(TAG + " notifyOnAppBackground() : ", e);
      }
    }
    for (AppBackgroundListener listener : MoECallbacks.getInstance().getAppBackgroundListeners()) {
      try {
        listener.onAppBackground(context);
      } catch (Exception e) {
        Logger.e(TAG + " notifyOnAppBackground() : ", e);
      }
    }
  }

  private void trackAppExit() {
    MoEHelper.getInstance(context).trackEvent(MoEConstants.MOE_APP_EXIT_EVENT, new Properties());
  }

  public DeviceAddManager getDeviceAddManager() {
    if (deviceAddManager == null) {
      deviceAddManager = new DeviceAddManager();
    }
    return deviceAddManager;
  }

  public void showInAppFromPush(Bundle bundle) {
    try {
      InAppManager.getInstance().showInAppFromPush(context, bundle);
    } catch (Exception e) {
      Logger.e(TAG + " showInAppFromPush() : ", e);
    }
  }

  private void updatePushService() {
    if (!MiPushManager.getInstance().hasMiPushModule() && PushKitManager.getInstance()
        .hasPushKitModule()) {
      ConfigurationProvider.getInstance(context).savePushService(MoEConstants.PUSH_SERVICE_FCM);
    }
  }
}
