/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.core;

import android.content.Context;
import androidx.annotation.Keep;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import org.json.JSONObject;

/**
 * @author Umang Chamaria
 */
public class MoEDTManager {

  private static final String TAG = MoEConstants.MODULE_TAG + "MoEDTManager";
  private DTHandler triggerHandler;

  private MoEDTManager(){
    loadHandler();
  }

  private static MoEDTManager instance;

  public static MoEDTManager getInstance(){
    if (instance == null){
      synchronized (MoEDTManager.class) {
       if (instance == null) instance = new MoEDTManager();
      }
    }
    return instance;
  }

  private void loadHandler(){
    try {
      Class handler = Class.forName("com.moengage.addon.trigger.DTHandlerImpl");
      triggerHandler = (DTHandler) handler.newInstance();
    }catch (Exception e){
      Logger.i( TAG + "loadHandler() : DT module not found");
    }
  }

  @Nullable
  DTHandler getTriggerHandler(Context context){
    if (RemoteConfig.getConfig().isRealTimeTriggerEnabled &&
        !ConfigurationProvider.getInstance(context).isPushNotificationOptedOut()) return
        triggerHandler;
    return null;
  }

  public void showTriggerIfPossible(Context context, String eventName, JSONObject eventAttributes){
    DTHandler dtHandler = getTriggerHandler(context);
    if (dtHandler != null){
      dtHandler.showTriggerCampaignIfPossible(context, eventName, eventAttributes);
    }
  }

  public void syncDeviceTriggersIfRequired(Context context){
    DTHandler dtHandler = getTriggerHandler(context);
    if (dtHandler != null){
      dtHandler.syncTriggersIfRequired(context);
    }
  }

  public void forceSyncDeviceTriggers(Context context){
    DTHandler dtHandler = getTriggerHandler(context);
    if (dtHandler != null){
      dtHandler.forceSyncDeviceTriggers(context);
    }
  }

  void scheduleBackgroundSync(Context context){
    DTHandler dtHandler = getTriggerHandler(context);
    if (dtHandler != null){
      dtHandler.scheduleBackgroundSync(context);
    }
  }

  @Keep
  public interface DTHandler {

    /**
     * Fetch the trigger list from server if lastSyncTime + minimumDelay < currentTime
     * @param context Application Context
     */
    void syncTriggersIfRequired(@NonNull Context context);

    /**
     * Checks whether a given event is a device trigger event or not and shows device trigger
     * campaign if required.
     *
     * @param context Application context
     * @param eventName event name for which device trigger might be present
     * @param eventAttributes event attributes of the event in question
     */
    void showTriggerCampaignIfPossible(@NonNull Context context, @NonNull String eventName, @NonNull
        JSONObject
        eventAttributes);

    /**
     * Fetch device triggers from server without any condition.
     *
     * @param context Application Context
     */
    void forceSyncDeviceTriggers(@NonNull Context context);

    /**
     * Scheduling background sync
     *
     * @param context Application context
     */
    void scheduleBackgroundSync(Context context);
  }
}
