/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.core;

import com.moe.pushlibrary.utils.MoEHelperUtils;
import com.moengage.core.internal.analytics.AnalyticsHelper;
import com.moengage.core.model.TrafficSource;
import com.moengage.core.model.UserSession;
import java.util.List;
import java.util.regex.Pattern;
import org.json.JSONObject;

/**
 * @author Umang Chamaria
 */
public class MoECoreEvaluator {

  private static final String TAG = MoEConstants.MODULE_TAG + "MoECoreEvaluator";

  public boolean isValidUniqueId(List<String> uniqueIdRegexList, String trackedUniqueId) {
    try {
      if (uniqueIdRegexList == null) return true;
      for (String regex : uniqueIdRegexList) {
        if (Pattern.matches(regex, trackedUniqueId)) {
          return false;
        }
      }
    } catch (Exception e) {
      Logger.e(TAG + " isValidUniqueId() : Exception: ", e);
    }
    return true;
  }

  public boolean shouldTrackScreenName(String screenName, List<String> optedOutScreenNames) {
    if (optedOutScreenNames == null || optedOutScreenNames.isEmpty()) return true;
    return !optedOutScreenNames.contains(screenName);
  }

  public boolean hasSessionExpired(long lastActiveTime, long sessionInActivityDuration,
      long currentTime) {
    return lastActiveTime + sessionInActivityDuration < currentTime;
  }

  public boolean hasSourceChanged(TrafficSource savedSource,
      TrafficSource currentSource) {
    if (TrafficSource.isEmpty(savedSource) && TrafficSource.isEmpty(currentSource)) return false;
    if (TrafficSource.isEmpty(savedSource) && !TrafficSource.isEmpty(currentSource)) return true;
    if (!TrafficSource.isEmpty(savedSource) && TrafficSource.isEmpty(currentSource)) return false;
    return !savedSource.equals(currentSource);
  }

  public boolean isInteractiveEvent(String dataPointString){
    try {
      JSONObject dataPointJson = new JSONObject(dataPointString);
      if (!dataPointJson.has(MoEHelperUtils.EVENT_NON_INTERACTIVE)) return true;
      int isInteractive = dataPointJson.getInt(MoEHelperUtils.EVENT_NON_INTERACTIVE);
      return isInteractive == 0;
    } catch (Exception e) {
      Logger.e(TAG + " isInteractiveEvent() : Exception: ", e);
    }
    return true;
  }

  /**
   * Checks whether the given campaign can be at the given time.
   *
   * @return false if campaign cannot be shown, else true
   */
  public boolean canShowCampaignNow(long startTime, long endTime, int currentHour,
      int currentMinutes) {
    if (endTime == -1
        || startTime == -1
        || (startTime == 0 && endTime == 0)
        || endTime == startTime) {
      return false;
    }

    // get dnd start hours and minutes
    int dndStartTimeHours = 0;
    int dndStartTimeMinutes = 0;
    if (startTime != 0) {
      dndStartTimeHours = (int) startTime / 100;
      dndStartTimeMinutes = (int) startTime % 100;
    }

    // get dnd stop hours and minutes
    int dndStopTimeHours = 0;
    int dndStopTimeMinutes = 0;
    if (endTime != 0) {
      dndStopTimeHours = (int) endTime / 100;
      dndStopTimeMinutes = (int) endTime % 100;
    }

    //dnd spanning over 2 different days
    if (dndStartTimeHours > dndStopTimeHours) {
      if (dndStartTimeHours < currentHour || dndStopTimeHours > currentHour) return true;
      if (dndStartTimeHours == currentHour) return currentMinutes >= dndStartTimeMinutes;
      if (dndStopTimeHours == currentHour) return currentMinutes <= dndStopTimeMinutes;
    }
    //dnd on the same day
    else if (dndStartTimeHours < dndStopTimeHours) {
      if (currentHour > dndStartTimeHours && currentHour < dndStopTimeHours) return true;
      if (dndStartTimeHours == currentHour) return currentMinutes >= dndStartTimeMinutes;
      if (dndStopTimeHours == currentHour) return currentMinutes <= dndStopTimeMinutes;
    }
    // start and end has same hours
    else if (dndStartTimeHours == dndStopTimeHours) {
      return currentHour == dndStartTimeHours
          && currentMinutes >= dndStartTimeMinutes
          && currentMinutes <= dndStopTimeMinutes;
    }

    return false;
  }

  public boolean canUpdateSourceInCurrentSession(UserSession userSession, long currentTime) {
    if (userSession == null) return false;
    if (!TrafficSource.isEmpty(userSession.trafficSource)) return false;
    long sessionStartTime = ISO8601Utils.parse(userSession.startTime).getTime();
    return (currentTime - sessionStartTime) / 1000 <= AnalyticsHelper.MAX_INTERVAL_FOR_SOURCE_UPDATE;
  }
}
