/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.core;

import android.content.Context;
import com.moengage.core.executor.TaskManager;
import com.moengage.core.executor.TaskResult;
import com.moengage.core.internal.storage.StorageProvider;
import com.moengage.core.model.TokenState;

/**
 * @author Umang Chamaria
 */
public class DeviceAddManager {

  private static final String TAG = MoEConstants.MODULE_TAG + "DeviceAddManager";

  private boolean isDeviceAddInProgress;
  private boolean isFcmRegistrationPending;
  private boolean isSecondaryTokenRegistrationPending;
  private boolean isGdprRequestPending;

  public void registerGdprOptOut(Context context) {
    if (isDeviceAddInProgress){
      Logger.v(TAG + " registerGdprOptOut() : Device Add is in progress, will send gdpr opt-out "
          + "after current request completes.");
      isGdprRequestPending = true;
    }else {
      Logger.v(TAG + " registerGdprOptOut() : Initiating request to send GDPR opt out.");
      initiateDeviceAdd(context);
    }
  }

  public void registerFcmToken(Context context) {
    if (isDeviceAddInProgress) {
      Logger.v(TAG
          + " registerFcmToken() : Device add in progress will send FCM token to server "
          + "after current request completes.");
      isFcmRegistrationPending = true;
    } else {
      Logger.v(TAG + " registerFcmToken() : Initiating request for sending FCM token to server.");
      initiateDeviceAdd(context);
    }
  }

  public void registerOemPushToken(Context context) {
    if (isDeviceAddInProgress) {
      Logger.v(TAG + " registerOemPushToken() : Device add is in progress will send " +
          "token to server after current request is completed");

      isSecondaryTokenRegistrationPending = true;
      return;
    }
    Logger.v(TAG + " registerOemPushToken() : Initiating request for sending oem token to "
        + "server.");
    initiateDeviceAdd(context);
  }

  void retryDeviceRegistrationIfRequired(Context context) {
    if (!StorageProvider.INSTANCE.getRepository(context, SdkConfig.getConfig())
        .isDeviceRegistered()) {
      Logger.v(TAG + " retryDeviceRegistrationIfRequired() : Device not registered yet. Will try "
          + "to register device.");
      initiateDeviceAdd(context);
    }
  }

  void registerDevice(Context context){
    if (isDeviceAddInProgress){
      Logger.v(TAG + " registerDevice() : Device add is already in progress, will not make "
          + "another call.");
      return;
    }
    initiateDeviceAdd(context);
  }

  private void initiateDeviceAdd(Context context) {
    Logger.v(TAG + " initiateDeviceAdd() : Will initiate device add call.");
    if (!RemoteConfig.getConfig().isAppEnabled){
      Logger.i( TAG + " initiateDeviceAdd() : App is disabled. Will not make device add call.");
      return;
    }
    if (!isDeviceAddInProgress) {
      Logger.v(TAG + " initiateDeviceAdd() : Initiating device add call");
      StorageProvider.INSTANCE.getRepository(context, SdkConfig.getConfig())
          .setDeviceRegistrationState(false);
      isDeviceAddInProgress = TaskManager.getInstance().startTask(new DeviceAddTask(context,
          MoEDAO.getInstance(context).getDevicePreferences()));
      Logger.v(TAG + " initiateDeviceAdd() : Device add call initiated: " + isDeviceAddInProgress);
    } else {
      Logger.v(TAG
          + " initiateDeviceAdd() : Device Add is already in progress cannot schedule "
          + "another one");
    }
  }

  void processTaskResult(Context context, TaskResult taskResult) {
    try {
      isDeviceAddInProgress = false;
      Logger.v(TAG + " processTaskResult() : Device Add completed, processing response.");
      if (taskResult == null) return;
      StorageProvider.INSTANCE.getRepository(context, SdkConfig.getConfig())
          .setDeviceRegistrationState(taskResult.isSuccess());
      if (!taskResult.isSuccess()) {
        Logger.v(TAG + " processTaskResult() : Device add failed recently");
        return;
      }
      TokenState tokenState = (TokenState) taskResult.getPayload();
      if (tokenState == null) return;

      if (isSecondaryTokenRegistrationPending && !tokenState.isSendingSecondaryToken) {
        isSecondaryTokenRegistrationPending = false;
        registerOemPushToken(context);
      }

      if (isFcmRegistrationPending && !tokenState.isSendingFcmToken) {
        isFcmRegistrationPending = false;
        registerFcmToken(context);
      }

      if (isGdprRequestPending) {
        isGdprRequestPending = false;
        registerGdprOptOut(context);
      }
    } catch (Exception e) {
      Logger.e( TAG + " processTaskResult() : Exception ", e);
    }
  }
}
