/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.moengage.core;

import android.content.Context;
import android.content.SharedPreferences;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.text.TextUtils;
import androidx.annotation.Nullable;
import com.moe.pushlibrary.models.GeoLocation;
import com.moe.pushlibrary.utils.MoEHelperConstants;
import com.moengage.core.internal.storage.preference.SharedPrefHelper;
import com.moengage.core.internal.storage.StorageProvider;
import com.moengage.core.internal.storage.preference.SharedPrefKeysKt;
import com.moengage.core.model.UserAttribute;
import java.util.HashMap;
import java.util.Set;
import java.util.UUID;

public final class ConfigurationProvider {

  //Should have the Generic params and the commonly used params
  //SHOULD NOT have the
  private HashMap<String, Object> configMap;

  private boolean initialized = false;
  private static final String PREF_NAME = "pref_moe";
  private static final String CURRENT_APP_VERSION = "APP_VERSION";
  /**
   * <b>Preference key</b>.The version associated with the app . This is the
   * Version Code specified in the application manifest
   */
  private static final String PREF_KEY_APP_VERSION = "appVersion";

  private static final int NOTIFICATION_ID = 17987;
  private static final String PREF_KEY_LAST_NOTIFICATION_ID = "PREF_LAST_NOTIFICATION_ID";

  private Context context;

  private SharedPreferences preferences;

  private static final String TAG = MoEConstants.MODULE_TAG + "ConfigurationProvider";

  //constructor intentionally hidden
  private ConfigurationProvider(Context context) {
    if (context == null) {
      Logger.v("ConfigurationProvider : context passed is null");
      return;
    }
    this.context = context;
    preferences =
        StorageProvider.INSTANCE.getSharedPreference(context, SdkConfig.getConfig()).getPreference();
    init();
  }

  //A mutex to be used by all methods which need synchronisation
  private final Object lock = new Object();

  private void init() {
    synchronized (lock) {
      //return if a call was waiting and now is no more required
      if (initialized) return;
      configMap = new HashMap<>();
      try {
        PackageInfo packageInfo =
            context.getPackageManager().getPackageInfo(context.getPackageName(), 0);
        configMap.put(CURRENT_APP_VERSION, packageInfo.versionCode);
      } catch (PackageManager.NameNotFoundException e) {
        Logger.e("Could not get package name: ", e);
      } catch (Exception e) {
        Logger.e("Could not get package name: ", e);
      }
      initialized = true;
    }
  }

  private static ConfigurationProvider instance;

  public static ConfigurationProvider getInstance(Context context) {
    synchronized (ConfigurationProvider.class) {
      if (null == instance) {
        instance = new ConfigurationProvider(context);
      }
    }
    return instance;
  }


  private SharedPreferences getSharedPrefs() {
    if (preferences == null) {
      preferences = StorageProvider.INSTANCE.getSharedPreference(context, SdkConfig.getConfig())
          .getPreference();
    }
    return preferences;
  }

  private final Object gcmTokenLock = new Object();

  public String getFcmToken() {
    synchronized (gcmTokenLock) {
      return getSharedPrefs().getString(FCM_PUSH_TOKEN, null);
    }
  }

  public void setFcmToken(String token) {
    synchronized (gcmTokenLock) {
      getSharedPrefs().edit().putString(FCM_PUSH_TOKEN, token).apply();
    }
  }

  private final Object tokenLock = new Object();

  public String getToken(String tokenName) {
    synchronized (tokenLock) {
      return getSharedPrefs().getString(tokenName, "");
    }
  }

  public void setToken(String tokenName, String token) {
    synchronized (tokenLock) {
      getSharedPrefs().edit().putString(tokenName, token).apply();
    }
  }

   public int getAppVersion() {
    return (Integer) configMap.get(CURRENT_APP_VERSION);
  }

  public int getStoredAppVersion() {
    return getSharedPrefs().getInt(PREF_KEY_APP_VERSION, 0);
  }

  public void storeAppVersion(int version) {
    SharedPreferences pref = getSharedPrefs();
    pref.edit().putInt(PREF_KEY_APP_VERSION, version).apply();
  }

  private final Object userLock = new Object();

  /**
   * Gets the Unique ID from the shared preference file. If it is not found
   * then generates a Unique ID and stores it before returning the newly
   * generated UniqueId
   *
   * @return The Unique ID associated with this app
   */
  public String getCurrentUserId() {
    synchronized (userLock) {
      String prefUniqueId = getSharedPrefs().getString(PREF_KEY_APP_UUID_MOE, null);
      UserAttribute dbUniqueIdObj = MoEDAO.getInstance(context).getUserAttributeByName
          (PREF_KEY_APP_UUID_MOE);
      String dbUniqueId = null;
      if (dbUniqueIdObj != null){
        dbUniqueId = dbUniqueIdObj.userAttributeValue;
      }
      //only if it is using API v2 generate and store the UUID
      if (TextUtils.isEmpty(prefUniqueId) && TextUtils.isEmpty(dbUniqueId)){
        //no unique id present
        Logger.v(TAG + " getCurrentUserId() no uniqueId present generating new "
            + "id");
        return generateAndSaveUniqueId();
      }else if (!TextUtils.isEmpty(dbUniqueId)){
        //unique id present in db
        Logger.v(TAG + " getCurrentUserId() unique id present in db");
        getSharedPrefs().edit().putString(PREF_KEY_APP_UUID_MOE, dbUniqueId).apply();
        return dbUniqueId;
      }else if (!TextUtils.isEmpty(prefUniqueId)){
        //unique id present in preference
        Logger.v(TAG + " getCurrentUserId() unique id present in preference");
        return prefUniqueId;
      }
      Logger.v(TAG + " getCurrentUserId() generating user id from fallback "
          + "condition something went wrong");
      // fallback if everything fails
      return generateAndSaveUniqueId();
    }
  }

  public int getNotificationId() {
    return getSharedPrefs().getInt(PREF_KEY_LAST_NOTIFICATION_ID, NOTIFICATION_ID);
  }

  public void updateNotificationId(int notificationId) {
    if (notificationId - NOTIFICATION_ID >= 100) {
      notificationId = NOTIFICATION_ID;
    }
    SharedPreferences sp = getSharedPrefs();
    sp.edit().putInt(PREF_KEY_LAST_NOTIFICATION_ID, notificationId).apply();
  }

  private static final String PREF_KEY_APP_UUID_MOE = "APP_UUID";

  /**
   * Generates a unique ID using {@link UUID#randomUUID()}, stores the same in
   * the {@link SharedPreferences} file and returns the same
   *
   * @return the generated UUID
   */
  private String generateAndSaveUniqueId() {
    String uid = generateUUID();
    UserAttribute uniqueId = new UserAttribute(PREF_KEY_APP_UUID_MOE, uid);
    MoEDAO.getInstance(context).addOrUpdateUserAttribute(uniqueId);
    getSharedPrefs().edit().putString(PREF_KEY_APP_UUID_MOE, uid).apply();
    return uid;
  }

  /**
   * Get a randomly generated UID
   *
   * @return A randomly generated Sting representation of {@link UUID}
   */
  private String generateUUID() {
    UUID generatedUID = UUID.randomUUID();
    return generatedUID.toString();
  }

  /**
   * <b>Preference key</b>.The GCM registration ID retrieved after registering
   * with GCM
   */
  public static final String FCM_PUSH_TOKEN = "registration_id";
  /**
   * <b>Preference key</b> and <b>User attribute</b> key which denotes whether
   * the user has enabled limited ad tracking or not.
   */
  static final String PREF_KEY_MOE_ISLAT = "PREF_KEY_MOE_ISLAT";
  /**
   * <b>Preference key</b> and <b>User attribute</b> key for the Google
   * Advertisement ID associated with the Google user.
   */
  private static final String PREF_KEY_MOE_GAID = "PREF_KEY_MOE_GAID";

  /**
   * Store the Google Advertisement ID
   *
   * @param gaid The Google Advertisement ID
   */
  public void storeGAID(String gaid) {
    SharedPreferences sp = getSharedPrefs();
    sp.edit().putString(PREF_KEY_MOE_GAID, gaid).apply();
  }

  /**
   * Get the stored Google AdvertisementID
   *
   * @return the stored google advertisement ID
   */
  public String getStoredGAID() {
    SharedPreferences sp = getSharedPrefs();
    return sp.getString(PREF_KEY_MOE_GAID, "");
  }

  /**
   * Store the isLimitedAdTracking flag
   *
   * @param isLAT the integer representation of the limited ad tracking flag
   */
  void storeISLAT(int isLAT) {
    SharedPreferences sp = getSharedPrefs();
    sp.edit().putInt(PREF_KEY_MOE_ISLAT, isLAT).apply();
  }

  /**
   * Get the stored isLimitedAdTracking flag
   *
   * @return the stored limited ad tracking flag
   */
  int getStoredISLAT() {
    SharedPreferences sp = getSharedPrefs();
    return sp.getInt(PREF_KEY_MOE_ISLAT, 2);
  }

  /**
   * Removes the user related configuration which is stored int he system
   */
  void removeUserConfigurationOnLogout() {
    preferences.edit()
        .remove(PREF_KEY_INAPP_LAST_SHOWN_TIME)
        .remove(PREF_KEY_APP_UUID_MOE)
        .remove(PREF_KEY_DEVICE_TRIGGERS_LAST_SHOW_TIME)
        .remove(PREF_KEY_DEVICE_TRIGGERS_LAST_SYNC_TIME)
        .remove(PREF_KEY_DEVICE_TRIGGERS_MINIMUM_DELAY)
        .remove(PREF_KEY_DEVICE_TRIGGERS_DND_END_TIME)
        .remove(PREF_KEY_DEVICE_TRIGGERS_DND_START_TIME)
        .remove(PREF_KEY_DEVICE_TRIGGERS_MINIMUM_DELAY)
        .remove(SharedPrefKeysKt.KEY_LAST_CONFIG_SYNC_TIME)
        .remove(PREF_KEY_USER_ATTRIBUTE_UNIQUE_ID)
        .remove(SharedPrefKeysKt.KEY_DEVICE_REGISTRATION_STATE)
        .remove(PREF_KEY_LAST_SYNC_TIME_IN_APP)
        .remove(PREF_KEY_USER_SESSION)
        .remove(PREF_KEY_SEGMENT_ANONYMOUS_ID)
        .apply();
  }

  public void saveLocation(GeoLocation location) {
    getSharedPrefs().edit()
        .putString(PREF_KEY_USER_LOCATION, location.latitude + "," + location.longitude)
        .apply();
  }

  public GeoLocation getSavedLocation() {
    try {
      String location = getSharedPrefs().getString(PREF_KEY_USER_LOCATION, null);
      if (null != location) {
        String[] latLong = location.split(",");
        return new GeoLocation(Double.parseDouble(latLong[0]), Double.parseDouble(latLong[1]));
      }
    } catch (Exception e) {
      Logger.e("ConfigurationProvider: getSavedLocation", e);
    }
    return null;
  }

  /**
   * <b>Preference key</b>. Stores when the last in app message was shown to
   * the user.
   */
  private static final String PREF_KEY_INAPP_LAST_SHOWN_TIME = "MOE_LAST_IN_APP_SHOWN_TIME";

  private static final String PREF_KEY_USER_LOCATION = "key_geoinfo";


  public String getAppVersionName() {
    if (configMap.get("app_version_name") == null) {
      setAppVersionName();
    }
    return (String) configMap.get("app_version_name");
  }

  private void setAppVersionName() {
    PackageInfo pInfo = null;
    try {
      pInfo = context.getPackageManager().getPackageInfo(context.getPackageName(), 0);
      configMap.put("app_version_name", pInfo.versionName);
    } catch (PackageManager.NameNotFoundException e) {
      Logger.e("ConfigurationProvider#setAppVersionName : Package Name not found", e);
    }
  }

  public void saveGeoIDList(String list) {
    SharedPreferences sp = getSharedPrefs();
    sp.edit().putString(MoEHelperConstants.PREF_KEY_GEO_FENCE_LIST, list).apply();
  }

  public String getGeoIDList() {
    SharedPreferences sp = getSharedPrefs();
    return sp.getString(MoEHelperConstants.PREF_KEY_GEO_FENCE_LIST, null);
  }

  public long getLastInAppShownTime() {
    SharedPreferences sp = getSharedPrefs();
    return sp.getLong(PREF_KEY_INAPP_LAST_SHOWN_TIME, 0);
  }

  public void setLastInAppShownTime(long currentTime) {
    SharedPreferences sp = getSharedPrefs();
    sp.edit().putLong(PREF_KEY_INAPP_LAST_SHOWN_TIME, currentTime).apply();
  }

  private static final String PREF_LAST_MESSAGE_FETCH_TIME = "last_message_sync";

  public void setLastPushAmpSyncTime(long time) {
    SharedPreferences sp = getSharedPrefs();
    if (sp != null) sp.edit().putLong(PREF_LAST_MESSAGE_FETCH_TIME, time).apply();
  }

  public long getLastPushAmpSyncTime() {
    SharedPreferences sp = getSharedPrefs();
    if (sp != null) {
      return sp.getLong(PREF_LAST_MESSAGE_FETCH_TIME, 0);
    }
    return 0;
  }

  private static final String PREF_VERIFICATION_REGISTRATION_TIME = "verfication_registration_time";

  public void setVerificationRegistrationTime(long time) {
    SharedPreferences sp = getSharedPrefs();
    if (sp != null) sp.edit().putLong(PREF_VERIFICATION_REGISTRATION_TIME, time).apply();
  }

  public long getVerificationRegistrationTime() {
    SharedPreferences sp = getSharedPrefs();
    if (sp != null) {
      return sp.getLong(PREF_VERIFICATION_REGISTRATION_TIME, 0);
    }
    return 0;
  }

  private static final String PREF_HAS_REGISTERED_FOR_VERIFICATION =
      "has_registered_for_verification";

  public void setVerificationRegistration(boolean hasVerificationRegistered) {
    SharedPreferences sp = getSharedPrefs();
    if (sp != null) {
      sp.edit().putBoolean(PREF_HAS_REGISTERED_FOR_VERIFICATION, hasVerificationRegistered).apply();
    }
  }

  public boolean isDeviceRegisteredForVerification() {
    SharedPreferences sp = getSharedPrefs();
    if (sp != null) {
      return sp.getBoolean(PREF_HAS_REGISTERED_FOR_VERIFICATION, false);
    }
    return false;
  }

  private static final String PREF_KEY_ACTIVITY_SENT_LIST = "sent_activity_list";

  void saveSentScreenNames(Set<String> sentScreenNames){
    SharedPreferences.Editor editor = getSharedPrefs().edit();
    editor.putStringSet(PREF_KEY_ACTIVITY_SENT_LIST, sentScreenNames).apply();
  }

  public Set<String> getSentScreenNames(){
    return getSharedPrefs().getStringSet(PREF_KEY_ACTIVITY_SENT_LIST, null);
  }

  private static final  String PREF_KEY_ENABLE_DEBUG_LOGS = "enable_logs";

  public void setDebugLogStatus(boolean status){
    SharedPreferences.Editor editor = getSharedPrefs().edit();
    editor.putBoolean(PREF_KEY_ENABLE_DEBUG_LOGS, status).apply();
  }

  public boolean isDebugLogEnabled(){
    SharedPreferences sp = getSharedPrefs();
    return sp.getBoolean(PREF_KEY_ENABLE_DEBUG_LOGS, false);
  }

  private static final String PREF_KEY_SEGMENT_ANONYMOUS_ID = "segment_anonymous_id";

  public void saveSegmentAnonymousId(String anonymousId) {
    getSharedPrefs().edit().putString(PREF_KEY_SEGMENT_ANONYMOUS_ID, anonymousId).apply();
  }

  @Nullable public String getSegmentAnonymousId() {
    return getSharedPrefs().getString(PREF_KEY_SEGMENT_ANONYMOUS_ID, null);
  }

  private static final String PREF_KEY_DEVICE_TRIGGERS_LAST_SHOW_TIME = "dt_last_show_time";

  public void saveLastDTShowTime(long time){
    getSharedPrefs().edit().putLong(PREF_KEY_DEVICE_TRIGGERS_LAST_SHOW_TIME, time).apply();
  }

  public long getDTLastShowTime(){
    return getSharedPrefs().getLong(PREF_KEY_DEVICE_TRIGGERS_LAST_SHOW_TIME, 0);
  }

  private static final String PREF_KEY_DEVICE_TRIGGERS_LAST_SYNC_TIME = "dt_last_sync_time";

  public void saveLastDTSyncTime(long time){
    getSharedPrefs().edit().putLong(PREF_KEY_DEVICE_TRIGGERS_LAST_SYNC_TIME, time).apply();
  }

  public long getDTLastSyncTime(){
    return getSharedPrefs().getLong(PREF_KEY_DEVICE_TRIGGERS_LAST_SYNC_TIME, 0);
  }

  private static final String PREF_KEY_DEVICE_TRIGGERS_MINIMUM_DELAY = "dt_minimum_delay";

  public void saveDTMinimumDelay(long time) {
    getSharedPrefs().edit().putLong(PREF_KEY_DEVICE_TRIGGERS_MINIMUM_DELAY, time).apply();
  }

  public long getDTMinimumDelay() {
    return getSharedPrefs().getLong(PREF_KEY_DEVICE_TRIGGERS_MINIMUM_DELAY, 0);
  }

  private static final String PREF_KEY_DEVICE_TRIGGERS_DND_START_TIME = "dt_dnd_start";

  public void saveDTDNDStartTime(long time){
    getSharedPrefs().edit().putLong(PREF_KEY_DEVICE_TRIGGERS_DND_START_TIME, time).apply();
  }

  public long getDTDNDStartTime(){
    return getSharedPrefs().getLong(PREF_KEY_DEVICE_TRIGGERS_DND_START_TIME, -1);
  }

  private static final String PREF_KEY_DEVICE_TRIGGERS_DND_END_TIME = "dt_dnd_end";

  public void saveDNDEndTime(long time){
    getSharedPrefs().edit().putLong(PREF_KEY_DEVICE_TRIGGERS_DND_END_TIME, time).apply();
  }

  public long getDNDEndTime(){
    return getSharedPrefs().getLong(PREF_KEY_DEVICE_TRIGGERS_DND_END_TIME, -1);
  }

  private static final String DATA_TRACKING_OPT_OUT = "data_tracking_opt_out";

  void optOutOfDataTracking(boolean optOut){
    getSharedPrefs().edit().putBoolean(DATA_TRACKING_OPT_OUT, optOut).apply();
  }

  public boolean isDataTrackingOptedOut(){
    return getSharedPrefs().getBoolean(DATA_TRACKING_OPT_OUT, false);
  }

  public static final String PUSH_NOTIFICATION_OPT_OUT = "push_notification_opt_out";

  /**
   * Sets the optOut state for push notification.
   *
   * @param optOut true/false
   */
  void optOutOfPushNotification(boolean optOut){
    getSharedPrefs().edit().putBoolean(PUSH_NOTIFICATION_OPT_OUT, optOut).apply();
  }

  /**
   * Check if push notification is opted out or not.
   * @return true if Push notification is opted out, else false.
   */
  public boolean isPushNotificationOptedOut(){
    return getSharedPrefs().getBoolean(PUSH_NOTIFICATION_OPT_OUT,
        ConfigDefault.IS_PUSH_NOTIFICATION_OPTED_OUT);
  }

  private static final String IN_APP_NOTIFICATION_OPT_OUT = "in_app_notification_opt_out";

  void optOutOfInAppNotification(boolean optOut){
    getSharedPrefs().edit().putBoolean(IN_APP_NOTIFICATION_OPT_OUT, optOut).apply();
  }

  public boolean isInAppOptedOut(){
    return getSharedPrefs().getBoolean(IN_APP_NOTIFICATION_OPT_OUT, false);
  }

  private static final String PREF_KEY_USER_ATTRIBUTE_UNIQUE_ID = "user_attribute_unique_id";

  public void saveUserAttributeUniqueId(String uniqueId) {
    // do not change to apply, use of commit() intentional.
    getSharedPrefs().edit().putString(PREF_KEY_USER_ATTRIBUTE_UNIQUE_ID, uniqueId).commit();
  }

  public String getUserAttributeUniqueId() {
    return getSharedPrefs().getString(PREF_KEY_USER_ATTRIBUTE_UNIQUE_ID, null);
  }

  private static final String PREF_KEY_LAST_GEO_FENCE_SYNC_TIME = "last_geo_sync_time";

  public void setLastGeoFenceSyncTime(long time){
    getSharedPrefs().edit().putLong(PREF_KEY_LAST_GEO_FENCE_SYNC_TIME, time).apply();
  }

  public long getLastGeoFenceSyncTime(){
    return getSharedPrefs().getLong(PREF_KEY_LAST_GEO_FENCE_SYNC_TIME, 0);
  }

  public static final String OEM_PUSH_TOKEN = "mi_push_token";

  void clearPushToken() {
    removeKey(FCM_PUSH_TOKEN);
    removeKey(OEM_PUSH_TOKEN);
  }

  private static String PREF_KEY_USER_SESSION = "user_session";

  public void saveUserSession(String sessionString) {
    getSharedPrefs().edit().putString(PREF_KEY_USER_SESSION, sessionString).apply();
  }

  String getUserSession(){
    return getSharedPrefs().getString(PREF_KEY_USER_SESSION, null);
  }

  private static final String PREF_KEY_LAST_SYNC_TIME_IN_APP = "inapp_last_sync_time";

  public void saveLastInAppSyncTime(long nextSyncTime) {
    getSharedPrefs().edit().putLong(PREF_KEY_LAST_SYNC_TIME_IN_APP, nextSyncTime).apply();
  }

  public long getLastInAppSyncTime() {
    return getSharedPrefs().getLong(PREF_KEY_LAST_SYNC_TIME_IN_APP, 0);
  }

  private static final String PREF_KEY_MINIMUM_SYNC_DELAY_IN_APP = "inapp_api_sync_delay";

  public void saveInAppApiSyncDelay(long minimumSyncDelay) {
    getSharedPrefs().edit().putLong(PREF_KEY_MINIMUM_SYNC_DELAY_IN_APP, minimumSyncDelay).apply();
  }

  public long getInAppApiSyncDelay() {
    return getSharedPrefs().getLong(PREF_KEY_MINIMUM_SYNC_DELAY_IN_APP, 900);
  }

  private static final String PREF_KEY_GLOBAL_DELAY_IN_APP = "in_app_global_delay";

  public void saveInAppGlobalDelay(long globalDelay) {
    getSharedPrefs().edit().putLong(PREF_KEY_GLOBAL_DELAY_IN_APP, globalDelay).apply();
  }

  public long getInAppGlobalDelay() {
    return getSharedPrefs().getLong(PREF_KEY_GLOBAL_DELAY_IN_APP, 900);
  }

  private static String PREF_KEY_REMOTE_CONFIGURATION = "remote_configuration";

  public void setRemoteConfiguration(String configurationString){
    getSharedPrefs().edit().putString(PREF_KEY_REMOTE_CONFIGURATION, configurationString).apply();
  }

  public String getRemoteConfiguration(){
    return getSharedPrefs().getString(PREF_KEY_REMOTE_CONFIGURATION, null);
  }

  /**
   * Set install registered state as true
   */
  public void setInstallRegistered() {
    getSharedPrefs().edit().putBoolean(MoEConstants.PREF_KEY_INSTALL_LOGGED, true).apply();
  }

  /**
   * Check whether install has been registered or not
   *
   * @return true if the install has been registered and false otherwise
   */
  public boolean isInstallRegistered() {
    return getSharedPrefs().getBoolean(MoEConstants.PREF_KEY_INSTALL_LOGGED, false);
  }

  public static final String PREF_KEY_PUSH_SERVICE = "push_service";

  public void savePushService(String service) {
    getSharedPrefs().edit().putString(PREF_KEY_PUSH_SERVICE, service).apply();
  }

  public String getPushService() {
    return getSharedPrefs().getString(PREF_KEY_PUSH_SERVICE, MoEConstants.PUSH_SERVICE_FCM);
  }

  /**
   * Use {@link SharedPrefHelper#putString(String, String)}
   */
  @Deprecated
  public void putString(String key, String value) {
    preferences.edit().putString(key, value).apply();
  }

  /**
   * Use {@link SharedPrefHelper#getString(String, String)}
   */
  @Deprecated
  public String getString(String key, String defaultValue) {
    return preferences.getString(key, defaultValue);
  }

  /**
   * Use {@link SharedPrefHelper#putInt(String, int)}
   */
  @Deprecated
  public void putInt(String key, int value) {
    preferences.edit().putInt(key, value).apply();
  }

  /**
   * Use {@link SharedPrefHelper#getInt(String, int)}
   */
  @Deprecated
  public int getInt(String key, int defaultValue) {
    return preferences.getInt(key, defaultValue);
  }

  /**
   * Use {@link SharedPrefHelper#putLong(String, long)}
   */
  @Deprecated
  public void putLong(String key, long value) {
    preferences.edit().putLong(key, value).apply();
  }

  /**
   * Use {@link SharedPrefHelper#getLong(String, long)}
   *
   */
  @Deprecated
  public long getLong(String key, long defaultValue) {
    return preferences.getLong(key, defaultValue);
  }

  /**
   * Use {@link SharedPrefHelper#putBoolean(String, boolean)}
   */
  @Deprecated
  public void putBoolean(String key, boolean value) {
    preferences.edit().putBoolean(key, value).apply();
  }

  /**
   * Use {@link SharedPrefHelper#getBoolean(String, boolean)}
   */
  @Deprecated
  public boolean getBoolean(String key, boolean defaultValue) {
    return preferences.getBoolean(key, defaultValue);
  }

  /**
   * Use {@link SharedPrefHelper#putStringSet(String, Set)}
   */
  @Deprecated
  public void putStringSet(String key, Set<String> stringSet) {
    preferences.edit().putStringSet(key, stringSet).apply();
  }

  /**
   * Use {@link SharedPrefHelper#getStringSet(String, Set)}
   */
  @Deprecated
  public Set<String> getStringSet(String key, Set<String> defaultValue) {
    return preferences.getStringSet(key, defaultValue);
  }

  /**
   * Use {@link SharedPrefHelper#removeKey(String)}
   */
  public void removeKey(String key) {
    preferences.edit().remove(key).apply();
  }
}
