/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.core;

import android.content.Context;
import com.moe.pushlibrary.MoEHelper;
import com.moengage.core.cards.CardManager;
import com.moengage.core.executor.SDKTask;
import com.moengage.core.executor.TaskResult;
import com.moengage.core.internal.AdvertisingIdClient;
import com.moengage.core.internal.inapp.InAppManager;
import com.moengage.core.internal.location.GeoManager;
import com.moengage.core.mipush.MiPushManager;
import com.moengage.core.internal.pushamp.PushAmpManager;
import com.moengage.push.PushManager;
import com.moengage.push.hms.puskit.PushKitManager;

/**
 * @author Umang Chamaria
 * Date: 2020/09/18
 */
class AppOpenTask extends SDKTask {

  private static final String TAG = MoEConstants.MODULE_TAG + "AppOpenTask";

  public AppOpenTask(Context context) {
    super(context);
  }

  @Override public TaskResult execute() {
    try {
      MoEHelper.getInstance(context).trackEvent(MoEConstants.EVENT_ACTION_ACTIVITY_START,
          new Properties());
      // sync and update Geofence
      GeoManager.getInstance().updateFenceAndLocation(context);
      // sync in-app
      InAppManager.getInstance().syncInAppsIfRequired(context);
      // sync push-amp
      PushAmpManager.getInstance().forceServerSync(context, true);
      // generate token if SDK is registering for push
      PushManager.getInstance().registerForPush(context);
      // create channel if required
      PushManager.getInstance().onAppOpen(context);
      // Sync device triggers
      MoEDTManager.getInstance().forceSyncDeviceTriggers(context);
      // initialise Mi Push
      MiPushManager.getInstance().initMiPush(context);
      // initialise HMS Push Kit
      PushKitManager.getInstance().onAppOpen(context);
      // Fetch cards
      CardManager.getInstance().onAppOpen(context);
      // delete stale data, mainly expired campaigns.
      MoEDAO.getInstance(context).removeExpiredData();
      // check and update GAID if required.
      checkGoogleAdvertisementIDAndUpdate();
    } catch (Exception e) {
      Logger.e(TAG + " execute() : ", e);
    }
    return taskResult;
  }

  @Override public String getTaskTag() {
    return TAG_APP_OPEN_TASK;
  }

  @Override public boolean isSynchronous() {
    return false;
  }

  private void checkGoogleAdvertisementIDAndUpdate() {
    ConfigurationProvider provider = ConfigurationProvider.getInstance(context);
    if (SdkConfig.getConfig().trackingOptOut.isGaidOptedOut()) {
      Logger.i(TAG + " : Opted out of GAID Collection");
      return;
    }

    String storedGAID = provider.getStoredGAID();
    int storedISLAT = provider.getStoredISLAT();
    AdvertisingIdClient.AdInfo adInfo = MoEUtils.getAdvertisementInfo(context);
    if (adInfo == null) return;
    //check if the google advertisement id has changed or not
    if (!MoEUtils.isEmptyString(adInfo.getId()) && (MoEUtils.isEmptyString(storedGAID)
        || (!adInfo.getId().equals(storedGAID)))) {
      MoEHelper.getInstance(context).setUserAttribute(MoEConstants.ATTR_MOE_GAID,
          adInfo.getId());
      provider.storeGAID(adInfo.getId());
    }
    //check if the adTracking opt out has changed or not
    if (adInfo.isLimitAdTrackingEnabled() != storedISLAT) {
      MoEHelper.getInstance(context).setUserAttribute(MoEConstants.ATTR_IS_LAT,
          Integer.toString(adInfo.isLimitAdTrackingEnabled()));
      provider.storeISLAT(adInfo.isLimitAdTrackingEnabled());
    }
  }
}
