/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.core;

import android.app.Activity;
import com.moengage.core.internal.analytics.AnalyticsHelper;
import com.moengage.core.executor.SDKTask;
import com.moengage.core.executor.TaskResult;
import java.util.Set;

/**
 * Logs activity start event
 * Service constant MSG_ACTIVITY_START
 *
 * @author Umang Chamaria
 */
class ActivityStartTask extends SDKTask {

  private static final String TAG = MoEConstants.MODULE_TAG + "ActivityStartTask";

  private Activity activity;

  ActivityStartTask(Activity activity) {
    super(activity.getApplicationContext());
    this.activity = activity;
  }

  @Override public TaskResult execute() {
    try {
      Logger.v(TAG + " started execution");
      String activityName = activity.getClass().getName();
      Logger.v("ActivityLifecycleStart : " + activityName + " started");
      // update session if required.
      AnalyticsHelper.getInstance(context).onActivityStart(activity);
      // track activity name if required.
      trackActivityIfRequired(activityName);
      taskResult.setIsSuccess(true);
      Logger.v(TAG + " completed execution");
    } catch (Exception e) {
      Logger.e(TAG + " execute() : Exception: ", e);
    }
    return taskResult;
  }

  @Override public String getTaskTag() {
    return TAG_ACTIVITY_START;
  }

  @Override public boolean isSynchronous() {
    return false;
  }


  private boolean isActivityTracked(String activityName) {
    try {
      Set<String> trackedScreen = ConfigurationCache.getInstance().getSentScreenNames();
      return trackedScreen != null && trackedScreen.contains(activityName);
    } catch (Exception e) {
      Logger.e(TAG + " isActivityTracked() : ", e);
    }
    return false;
  }

  private void trackActivityIfRequired(String activityName){
    if (isActivityTracked(activityName)) return;
    MoECoreEvaluator evaluator = new MoECoreEvaluator();
    if (evaluator.shouldTrackScreenName(activityName,
        SdkConfig.getConfig().activityTrackingOptOutList)) {
      MoEUtils.trackActivityStates(MoEConstants.EVENT_ACTION_ACTIVITY_START, activityName,
          context);
      ConfigurationCache.getInstance().addScreenToSentList(activityName);
    }
  }
}