/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.moe.pushlibrary.providers;

import android.content.ContentProvider;
import android.content.ContentProviderOperation;
import android.content.ContentProviderResult;
import android.content.ContentUris;
import android.content.ContentValues;
import android.content.Context;
import android.content.OperationApplicationException;
import android.content.UriMatcher;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteQueryBuilder;
import android.net.Uri;
import android.text.TextUtils;
import androidx.annotation.NonNull;
import com.moe.pushlibrary.providers.MoEDataContract.AttributeCacheEntity;
import com.moe.pushlibrary.providers.MoEDataContract.BatchDataEntity;
import com.moe.pushlibrary.providers.MoEDataContract.CampaignListEntity;
import com.moe.pushlibrary.providers.MoEDataContract.DTEntity;
import com.moe.pushlibrary.providers.MoEDataContract.InAppMessageEntity;
import com.moe.pushlibrary.providers.MoEDataContract.InAppStatsEntity;
import com.moe.pushlibrary.providers.MoEDataContract.InAppV3Entity;
import com.moe.pushlibrary.providers.MoEDataContract.MessageEntity;
import com.moe.pushlibrary.providers.MoEDataContract.UserAttributeEntity;
import com.moengage.core.Logger;
import com.moengage.core.MoEConstants;
import com.moengage.core.internal.storage.database.contract.DataPointContract;
import com.moengage.core.storage.CardsDataContract.CardsEntity;
import java.util.ArrayList;
import java.util.HashMap;


/**
 * This is a content provider for accessing the MoEngage data.
 *
 * @author MoEngage (abhishek@moengage.com)
 * @since v5
 */
public class MoEProvider extends ContentProvider {

  private static final String TAG = "MoEProvider";

  /**
   * Denotes that the URI matched Messages
   */
  private static final int MESSAGES = 1;
  /**
   * Denotes that the URI matched a` single PromotionalMessage entry
   */
  private static final int MESSAGE_ID = 2;
  /**
   * Denotes that the URI matched Events
   */
  private static final int DATAPOINTS = 3;
  /**
   * Denotes that the URI matched a single Event entry
   */
  private static final int DATAPOINT_ID = 4;
  /**
   * Denotes that the URI matched InApps
   */
  private static final int INAPPS = 5;
  /**
   * Denotes that the URI matched a single InApp entry
   */
  private static final int INAPP_ID = 6;
  /**
   * Denotes that URI matched user attributes
   */
  private static final int USER_ATTRIBUTES = 9;
  /**
   * Denotes that URI matched a  user attribute
   */
  private static final int USER_ATTRIBUTES_ID = 10;
  /**
   * Denotes that URI matched campaign ids
   */
  private static final int CAMPAIGN_IDS = 11;
  /**
   * Denotes that URI matched a  campaign id
   */
  private static final int CAMPAIGN_ID = 12;
  /**
   * Denotes the uri matched to batches of data
   */
  private static final int DATA_BATCHES = 13;
  /**
   * Denotes the uri matched to a batch of data
   */
  private static final int DATA_BATCH = 14;

  private static final int DEVICE_TRIGGERS = 15;

  private static final int DEVICE_TRIGGER = 16;

  private static final int ATTRIBUTE_CACHE = 17;

  private static final int ATTRIBUTE_CACHE_ID = 18;

  private static final int INAPPV3_CAMPAIGNS = 19;

  private static final int INAPPV3_CAMPAIGN = 20;

  private static final int INAPP_STATS = 21;

  private static final int INAPP_STAT = 22;

  private static final int CARD_CAMPAIGNS = 23;

  private static final int CARD_CAMPAIGN = 24;

  /**
   * Instance of {@link UriMatcher} used for getting the type of the URI
   */
  private static UriMatcher uriMatcher;

  private static HashMap<String, String> messageProjectionMap;
  private static HashMap<String, String> eventProjectionMap;
  private static HashMap<String, String> inAppProjectionMap;
  private static HashMap<String, String> userAttributeProjectionMap;
  private static HashMap<String, String> campaignListProjectionMap;
  private static HashMap<String, String> batchDataProjectionMap;
  private static HashMap<String, String> dTProjectionMap;
  private static HashMap<String, String> inAppV3ProjectionMap;
  private static HashMap<String, String> attributeCacheMap;
  private static HashMap<String, String> inAppStatsMap;
  private static HashMap<String, String> cardCampaignMap;

  static {

    messageProjectionMap = new HashMap<>();
    messageProjectionMap.put(MessageEntity._ID, MessageEntity._ID);
    messageProjectionMap.put(MessageEntity.GTIME, MessageEntity.GTIME);
    messageProjectionMap.put(MessageEntity.MSG_DETAILS, MessageEntity.MSG_DETAILS);
    messageProjectionMap.put(MessageEntity.MSG_CLICKED, MessageEntity.MSG_CLICKED);
    messageProjectionMap.put(MessageEntity.MSG_TTL, MessageEntity.MSG_TTL);
    messageProjectionMap.put(MessageEntity.MSG_TAG, MessageEntity.MSG_TAG);
    messageProjectionMap.put(MessageEntity.CAMPAIGN_ID, MessageEntity.CAMPAIGN_ID);

    eventProjectionMap = new HashMap<>();
    eventProjectionMap.put(DataPointContract.DataPointEntity._ID, DataPointContract.DataPointEntity._ID);
    eventProjectionMap.put(DataPointContract.DataPointEntity.GTIME, DataPointContract.DataPointEntity.GTIME);
    eventProjectionMap.put(DataPointContract.DataPointEntity.DETAILS, DataPointContract.DataPointEntity.DETAILS);

    inAppProjectionMap = new HashMap<>();
    inAppProjectionMap.put(InAppMessageEntity._ID, InAppMessageEntity._ID);
    inAppProjectionMap.put(InAppMessageEntity.GTIME, InAppMessageEntity.GTIME);
    inAppProjectionMap.put(InAppMessageEntity.MSG_CAMPAIGN_ID, InAppMessageEntity.MSG_CAMPAIGN_ID);
    inAppProjectionMap.put(InAppMessageEntity.MSG_ALIGN_TYPE, InAppMessageEntity.MSG_ALIGN_TYPE);
    inAppProjectionMap.put(InAppMessageEntity.MSG_INAPP_TYPE, InAppMessageEntity.MSG_INAPP_TYPE);
    inAppProjectionMap.put(InAppMessageEntity.MSG_TTL, InAppMessageEntity.MSG_TTL);
    inAppProjectionMap.put(InAppMessageEntity.MSG_MIN_DELAY, InAppMessageEntity.MSG_MIN_DELAY);
    inAppProjectionMap.put(InAppMessageEntity.MSG_MAX_TIMES, InAppMessageEntity.MSG_MAX_TIMES);
    inAppProjectionMap.put(InAppMessageEntity.MSG_SHOWN_COUNT, InAppMessageEntity.MSG_SHOWN_COUNT);
    inAppProjectionMap.put(InAppMessageEntity.MSG_PERSISTENT, InAppMessageEntity.MSG_PERSISTENT);
    inAppProjectionMap.put(InAppMessageEntity.MSG_PRIORITY, InAppMessageEntity.MSG_PRIORITY);
    inAppProjectionMap.put(InAppMessageEntity.MSG_CONTEXT, InAppMessageEntity.MSG_CONTEXT);
    inAppProjectionMap.put(InAppMessageEntity.MSG_LAST_SHOWN, InAppMessageEntity.MSG_LAST_SHOWN);
    inAppProjectionMap.put(InAppMessageEntity.MSG_IS_CLICKED, InAppMessageEntity.MSG_IS_CLICKED);
    inAppProjectionMap.put(InAppMessageEntity.MSG_HAS_ERRORS, InAppMessageEntity.MSG_HAS_ERRORS);
    inAppProjectionMap.put(InAppMessageEntity.MSG_AUTODISMISS_TIME,
        InAppMessageEntity.MSG_AUTODISMISS_TIME);
    inAppProjectionMap.put(InAppMessageEntity.MSG_CANCELABLE, InAppMessageEntity.MSG_CANCELABLE);
    inAppProjectionMap.put(InAppMessageEntity.MSG_CONTENT, InAppMessageEntity.MSG_CONTENT);
    inAppProjectionMap.put(InAppMessageEntity.MSG_SHOW_ONLY_IN,
        InAppMessageEntity.MSG_SHOW_ONLY_IN);
    inAppProjectionMap.put(InAppMessageEntity.MSG_STATUS, InAppMessageEntity.MSG_STATUS);
    inAppProjectionMap.put(InAppMessageEntity.MSG_CONTAINER_STYLE,
        InAppMessageEntity.MSG_CONTAINER_STYLE);

    userAttributeProjectionMap = new HashMap<>();
    userAttributeProjectionMap.put(UserAttributeEntity._ID, UserAttributeEntity._ID);
    userAttributeProjectionMap.put(UserAttributeEntity.GTIME, UserAttributeEntity.GTIME);
    userAttributeProjectionMap.put(UserAttributeEntity.ATTRIBUTE_NAME, UserAttributeEntity
        .ATTRIBUTE_NAME);
    userAttributeProjectionMap.put(UserAttributeEntity.ATTRIBUTE_VALUE, UserAttributeEntity
        .ATTRIBUTE_VALUE);

    campaignListProjectionMap = new HashMap<>();
    campaignListProjectionMap.put(CampaignListEntity._ID, CampaignListEntity._ID);
    campaignListProjectionMap.put(CampaignListEntity.CAMPAIGN_ID, CampaignListEntity.CAMPAIGN_ID);
    campaignListProjectionMap.put(CampaignListEntity.CAMPAIGN_ID_TTL,
        CampaignListEntity.CAMPAIGN_ID_TTL);

    batchDataProjectionMap = new HashMap<>();
    batchDataProjectionMap.put(BatchDataEntity._ID, BatchDataEntity._ID);
    batchDataProjectionMap.put(BatchDataEntity.BATCHED_DATA, BatchDataEntity.BATCHED_DATA);

    dTProjectionMap = new HashMap<>();
    dTProjectionMap.put(DTEntity._ID, DTEntity._ID);
    dTProjectionMap.put(DTEntity.CAMPAIGN_ID, DTEntity.CAMPAIGN_ID);
    dTProjectionMap.put(DTEntity.TRIGGER_EVENT_NAME, DTEntity.TRIGGER_EVENT_NAME);
    dTProjectionMap.put(DTEntity.PAYLOAD, DTEntity.PAYLOAD);
    dTProjectionMap.put(DTEntity.CAMPAIGN_PAYLOAD, DTEntity.CAMPAIGN_PAYLOAD);
    dTProjectionMap.put(DTEntity.CAMPAIGN_TYPE, DTEntity.CAMPAIGN_TYPE);
    dTProjectionMap.put(DTEntity.MAX_COUNT, DTEntity.MAX_COUNT);
    dTProjectionMap.put(DTEntity.MINIMUM_DELAY, DTEntity.MINIMUM_DELAY);
    dTProjectionMap.put(DTEntity.SHOULD_SHOW_OFFLINE, DTEntity.SHOULD_SHOW_OFFLINE);
    dTProjectionMap.put(DTEntity.MAX_SYNC_DELAY_TIME, DTEntity.MAX_SYNC_DELAY_TIME);
    dTProjectionMap.put(DTEntity.EXPIRY_TIME, DTEntity.EXPIRY_TIME);
    dTProjectionMap.put(DTEntity.PRIORITY, DTEntity.PRIORITY);
    dTProjectionMap.put(DTEntity.LAST_SHOW_TIME, DTEntity.LAST_SHOW_TIME);
    dTProjectionMap.put(DTEntity.SHOW_COUNT, DTEntity.SHOW_COUNT);
    dTProjectionMap.put(DTEntity.LAST_UPDATED_TIME, DTEntity.LAST_UPDATED_TIME);
    dTProjectionMap.put(DTEntity.STATUS, DTEntity.STATUS);
    dTProjectionMap.put(DTEntity.SHOULD_IGNORE_DND, DTEntity.SHOULD_IGNORE_DND);
    dTProjectionMap.put(DTEntity.DELAY_BEFORE_SHOWING_NOTIFICATION, DTEntity
        .DELAY_BEFORE_SHOWING_NOTIFICATION);

    inAppV3ProjectionMap = new HashMap<>();
    inAppV3ProjectionMap.put(InAppV3Entity._ID, InAppV3Entity._ID);
    inAppV3ProjectionMap.put(InAppV3Entity.CAMPAIGN_ID, InAppV3Entity.CAMPAIGN_ID);
    inAppV3ProjectionMap.put(InAppV3Entity.CAMPAIGN_TYPE, InAppV3Entity.CAMPAIGN_TYPE);
    inAppV3ProjectionMap.put(InAppV3Entity.CAMPAIGN_STATUS, InAppV3Entity.CAMPAIGN_STATUS);
    inAppV3ProjectionMap.put(InAppV3Entity.CAMPAIGN_STATE, InAppV3Entity.CAMPAIGN_STATE);
    inAppV3ProjectionMap.put(InAppV3Entity.PRIORITY, InAppV3Entity.PRIORITY);
    inAppV3ProjectionMap.put(InAppV3Entity.LAST_UPDATED_TIME, InAppV3Entity.LAST_UPDATED_TIME);
    inAppV3ProjectionMap.put(InAppV3Entity.TEMPLATE_TYPE, InAppV3Entity.TEMPLATE_TYPE);
    inAppV3ProjectionMap.put(InAppV3Entity.DELETION_TIME, InAppV3Entity.DELETION_TIME);
    inAppV3ProjectionMap.put(InAppV3Entity.LAST_RECEIVED_TIME, InAppV3Entity.LAST_RECEIVED_TIME);
    inAppV3ProjectionMap.put(InAppV3Entity.CAMPAIGN_META, InAppV3Entity.CAMPAIGN_META);

    attributeCacheMap = new HashMap<>();
    attributeCacheMap.put(AttributeCacheEntity._ID, AttributeCacheEntity._ID);
    attributeCacheMap.put(AttributeCacheEntity.ATTRIBUTE_NAME,
        AttributeCacheEntity.ATTRIBUTE_NAME);
    attributeCacheMap.put(AttributeCacheEntity.ATTRIBUTE_VALUE,
        AttributeCacheEntity.ATTRIBUTE_VALUE);
    attributeCacheMap.put(AttributeCacheEntity.LAST_TRACKED_TIME,
        AttributeCacheEntity.LAST_TRACKED_TIME);
    attributeCacheMap.put(AttributeCacheEntity.DATA_TYPE, AttributeCacheEntity.DATA_TYPE);

    inAppStatsMap = new HashMap<>();
    inAppStatsMap.put(InAppStatsEntity._ID, InAppStatsEntity._ID);
    inAppStatsMap.put(InAppStatsEntity.PAYLOAD, InAppStatsEntity.PAYLOAD);
    inAppStatsMap.put(InAppStatsEntity.REQUEST_ID, InAppStatsEntity.REQUEST_ID);
    inAppStatsMap.put(InAppStatsEntity.TIMESTAMP, InAppStatsEntity.TIMESTAMP);

    cardCampaignMap = new HashMap<>();
    cardCampaignMap.put(CardsEntity._ID, CardsEntity._ID);
    cardCampaignMap.put(CardsEntity.CARD_ID, CardsEntity.CARD_ID);
    cardCampaignMap.put(CardsEntity.CATEGORY, CardsEntity.CATEGORY);
    cardCampaignMap.put(CardsEntity.CAMPAIGN_STATE, CardsEntity.CAMPAIGN_STATE);
    cardCampaignMap.put(CardsEntity.VISIBILITY_STATUS, CardsEntity.VISIBILITY_STATUS);
    cardCampaignMap.put(CardsEntity.LAST_UPDATED_TIME, CardsEntity.LAST_UPDATED_TIME);
    cardCampaignMap.put(CardsEntity.CAMPAIGN_PAYLOAD, CardsEntity.CAMPAIGN_PAYLOAD);
    cardCampaignMap.put(CardsEntity.IS_PINNED, CardsEntity.IS_PINNED);
    cardCampaignMap.put(CardsEntity.DELETION_TIME, CardsEntity.DELETION_TIME);
    cardCampaignMap.put(CardsEntity.IS_NEW_CARD, CardsEntity.IS_NEW_CARD);
    cardCampaignMap.put(CardsEntity.IS_DELETED, CardsEntity.IS_DELETED);
    cardCampaignMap.put(CardsEntity.PRIORITY, CardsEntity.PRIORITY);

  }

  /**
   * The database helper which is used to do all database related operations
   */
  private MoEDatabaseHelper databaseHelper = null;

  /*
   * (non-Javadoc)
   *
   * @see android.content.ContentProvider#onCreate()
   */
  @Override public boolean onCreate() {
    databaseHelper = new MoEDatabaseHelper(getContext());
    initializeUriMatcher();
    Logger.d("MoEProvider: Provider created");
    return true;
  }

  private void initializeUriMatcher() {
    uriMatcher = new UriMatcher(UriMatcher.NO_MATCH);
    Context context = getContext();
    uriMatcher.addURI(MoEDataContract.getAuthority(context), "messages", MESSAGES);
    uriMatcher.addURI(MoEDataContract.getAuthority(context), "messages/#", MESSAGE_ID);
    uriMatcher.addURI(MoEDataContract.getAuthority(context), "datapoints", DATAPOINTS);
    uriMatcher.addURI(MoEDataContract.getAuthority(context), "datapoints/#", DATAPOINT_ID);
    uriMatcher.addURI(MoEDataContract.getAuthority(context), "inapps", INAPPS);
    uriMatcher.addURI(MoEDataContract.getAuthority(context), "inapps/#", INAPP_ID);
    uriMatcher.addURI(MoEDataContract.getAuthority(context), "userattributes/", USER_ATTRIBUTES);
    uriMatcher.addURI(MoEDataContract.getAuthority(context), "userattributes/#",
        USER_ATTRIBUTES_ID);
    uriMatcher.addURI(MoEDataContract.getAuthority(context), "campaignlist/", CAMPAIGN_IDS);
    uriMatcher.addURI(MoEDataContract.getAuthority(context), "campaignlist/#", CAMPAIGN_ID);
    uriMatcher.addURI(MoEDataContract.getAuthority(context), "batchdata/", DATA_BATCHES);
    uriMatcher.addURI(MoEDataContract.getAuthority(context), "batchdata/#", DATA_BATCH);
    uriMatcher.addURI(MoEDataContract.getAuthority(context), "dtcampaign", DEVICE_TRIGGERS);
    uriMatcher.addURI(MoEDataContract.getAuthority(context), "dtcampaign/#", DEVICE_TRIGGER);
    uriMatcher.addURI(MoEDataContract.getAuthority(context), "inappv3", INAPPV3_CAMPAIGNS);
    uriMatcher.addURI(MoEDataContract.getAuthority(context), "inappv3/#", INAPPV3_CAMPAIGN);
    uriMatcher.addURI(MoEDataContract.getAuthority(context), "attributecache/", ATTRIBUTE_CACHE);
    uriMatcher.addURI(MoEDataContract.getAuthority(context), "attributecache/#",
        ATTRIBUTE_CACHE_ID);
    uriMatcher.addURI(MoEDataContract.getAuthority(context), "inappstats", INAPP_STATS);
    uriMatcher.addURI(MoEDataContract.getAuthority(context), "inappstats/#", INAPP_STAT);

    uriMatcher.addURI(MoEDataContract.getAuthority(context), "cards/", CARD_CAMPAIGNS);
    uriMatcher.addURI(MoEDataContract.getAuthority(context), "cards/#", CARD_CAMPAIGN);
  }

  /*
   * (non-Javadoc)
   *
   * @see android.content.ContentProvider#getType(android.net.Uri)
   */
  @Override public String getType(@NonNull Uri uri) {
    switch (uriMatcher.match(uri)) {
      case MESSAGES:
        return MessageEntity.CONTENT_TYPE;
      case MESSAGE_ID:
        return MessageEntity.CONTENT_ITEM_TYPE;
      case DATAPOINTS:
        return DataPointContract.DataPointEntity.CONTENT_TYPE;
      case DATAPOINT_ID:
        return DataPointContract.DataPointEntity.CONTENT_ITEM_TYPE;
      case INAPPS:
        return InAppMessageEntity.CONTENT_TYPE;
      case INAPP_ID:
        return InAppMessageEntity.CONTENT_ITEM_TYPE;
      case USER_ATTRIBUTES:
        return UserAttributeEntity.CONTENT_TYPE;
      case USER_ATTRIBUTES_ID:
        return UserAttributeEntity.CONTENT_ITEM_TYPE;
      case CAMPAIGN_IDS:
        return CampaignListEntity.CONTENT_TYPE;
      case CAMPAIGN_ID:
        return CampaignListEntity.CONTENT_ITEM_TYPE;
      case DATA_BATCHES:
        return BatchDataEntity.CONTENT_TYPE;
      case DATA_BATCH:
        return BatchDataEntity.CONTENT_ITEM_TYPE;
      case DEVICE_TRIGGERS:
        return DTEntity.CONTENT_TYPE;
      case DEVICE_TRIGGER:
        return DTEntity.CONTENT_ITEM_TYPE;
      case INAPPV3_CAMPAIGNS:
        return InAppV3Entity.CONTENT_TYPE;
      case INAPPV3_CAMPAIGN:
        return InAppV3Entity.CONTENT_ITEM_TYPE;
      case ATTRIBUTE_CACHE:
        return AttributeCacheEntity.CONTENT_TYPE;
      case ATTRIBUTE_CACHE_ID:
        return AttributeCacheEntity.CONTENT_ITEM_TYPE;
      case INAPP_STATS:
        return InAppStatsEntity.CONTENT_TYPE;
      case INAPP_STAT:
        return InAppStatsEntity.CONTENT_ITEM_TYPE;
      case CARD_CAMPAIGNS:
        return CardsEntity.CONTENT_TYPE;
      case CARD_CAMPAIGN:
        return CardsEntity.CONTENT_ITEM_TYPE;
      default:
        Logger.v("No Matching URI found");
        return null;
    }
  }

  public Cursor query(@NonNull Uri uri, String[] projection, String selection,
      String[] selectionArgs, String sortOrder) {
    // Get the database and run the query
    SQLiteDatabase db = databaseHelper.getReadableDatabase();
    updateIfRequired(db);
    SQLiteQueryBuilder qb = new SQLiteQueryBuilder();
    // If no sort order is specified use the default
    String orderBy = sortOrder;
    String limit = uri.getQueryParameter(MoEDataContract.QUERY_PARAMETER_LIMIT);
    switch (uriMatcher.match(uri)) {
      case MESSAGE_ID:
        qb.appendWhere(MessageEntity._ID + "=" + uri.getPathSegments().get(1));
      case MESSAGES:
        qb.setProjectionMap(messageProjectionMap);
        qb.setTables(MoEDatabaseHelper.TABLE_NAME_MSGS);
        if (TextUtils.isEmpty(sortOrder)) {
          orderBy = MessageEntity.DEFAULT_SORT_ORDER;
        }
        break;
      case DATAPOINT_ID:
        qb.appendWhere(DataPointContract.DataPointEntity._ID + "=" + uri.getPathSegments().get(1));
      case DATAPOINTS:
        qb.setProjectionMap(eventProjectionMap);
        qb.setTables(MoEDatabaseHelper.TABLE_NAME_DATAPOINTS);
        break;
      case INAPP_ID:
        qb.appendWhere(InAppMessageEntity._ID + "=" + uri.getPathSegments().get(1));
      case INAPPS:
        qb.setProjectionMap(inAppProjectionMap);
        qb.setTables(MoEDatabaseHelper.TABLE_NAME_INAPPS);
        if (TextUtils.isEmpty(sortOrder)) {
          orderBy = InAppMessageEntity.DEFAULT_SORT_ORDER;
        }
        break;
      case USER_ATTRIBUTES:
        qb.setProjectionMap(userAttributeProjectionMap);
        qb.setTables(MoEDatabaseHelper.TABLE_NAME_USER_ATTRIBUTES);
        break;
      case CAMPAIGN_IDS:
        qb.setProjectionMap(campaignListProjectionMap);
        qb.setTables(MoEDatabaseHelper.TABLE_NAME_CAMPAIGN_LIST);
        break;
      case DATA_BATCHES:
        qb.setProjectionMap(batchDataProjectionMap);
        qb.setTables(MoEDatabaseHelper.TABLE_NAME_BATCH_DATA);
        break;
      case DEVICE_TRIGGERS:
        qb.setProjectionMap(dTProjectionMap);
        qb.setTables(MoEDatabaseHelper.TABLE_NAME_DEVICE_TRIGGERS);
        break;
      case INAPPV3_CAMPAIGNS:
        qb.setProjectionMap(inAppV3ProjectionMap);
        qb.setTables(MoEDatabaseHelper.TABLE_NAME_INAPP_V3);
        break;
      case ATTRIBUTE_CACHE:
        qb.setProjectionMap(attributeCacheMap);
        qb.setTables(MoEDatabaseHelper.TABLE_NAME_ATTRIBUTE_CACHE);
        break;
      case INAPP_STATS:
        qb.setProjectionMap(inAppStatsMap);
        qb.setTables(MoEDatabaseHelper.TABLE_NAME_INAPP_STATS);
        break;
      case CARD_CAMPAIGNS:
        qb.setProjectionMap(cardCampaignMap);
        qb.setTables(MoEDatabaseHelper.TABLE_NAME_CARD_CAMPAIGNS);
        break;
      default:
        Logger.v("Unknown URI query() " + uri);
    }
    Cursor c = null;
    try {
      c = qb.query(db, projection, selection, selectionArgs, null, null, orderBy, limit);
    } catch (Exception e) {
      Logger.e(TAG + " query() : Exception: ", e);
    }
    // Tell the cursor what uri to watch, so it knows when its source data
    // changes
    return c;
  }

  public Uri insert(Uri uri, ContentValues values) {
    Uri newUri = null;
    try {
      if (null == values) return null;
      SQLiteDatabase db = databaseHelper.getWritableDatabase();
      updateIfRequired(db);
      long rowId = 0;
      newUri = null;
      switch (uriMatcher.match(uri)) {
        case MESSAGES:
          rowId = db.insert(MoEDatabaseHelper.TABLE_NAME_MSGS, null, values);
          if (rowId > 0) {
            newUri = ContentUris.withAppendedId(MessageEntity.getContentUri(getContext()), rowId);
          }
          break;
        case DATAPOINTS:
          rowId = db.insert(MoEDatabaseHelper.TABLE_NAME_DATAPOINTS, null, values);
          if (rowId > 0) {
            newUri = ContentUris.withAppendedId(DataPointContract.DataPointEntity.getContentUri(getContext()), rowId);
          }
          break;
        case INAPPS:
          rowId = db.insert(MoEDatabaseHelper.TABLE_NAME_INAPPS, null, values);
          if (rowId > 0) {
            newUri =
                ContentUris.withAppendedId(InAppMessageEntity.getContentUri(getContext()), rowId);
          }
          break;
        case USER_ATTRIBUTES:
          rowId = db.insert(MoEDatabaseHelper.TABLE_NAME_USER_ATTRIBUTES, null, values);
          if (rowId > 0) {
            newUri = ContentUris.withAppendedId(UserAttributeEntity.getContentUri(getContext()),
                rowId);
          }
          break;
        case CAMPAIGN_IDS:
          rowId = db.insert(MoEDatabaseHelper.TABLE_NAME_CAMPAIGN_LIST, null, values);
          if (rowId > 0) {
            newUri =
                ContentUris.withAppendedId(UserAttributeEntity.getContentUri(getContext()), rowId);
          }
          break;
        case DATA_BATCHES:
          rowId = db.insert(MoEDatabaseHelper.TABLE_NAME_BATCH_DATA, null, values);
          if (rowId > 0) {
            newUri = ContentUris.withAppendedId(BatchDataEntity.getContentUri(getContext()), rowId);
          }
          break;
        case DEVICE_TRIGGERS:
          rowId = db.insert(MoEDatabaseHelper.TABLE_NAME_DEVICE_TRIGGERS, null, values);
          if (rowId > 0) {
            newUri = ContentUris.withAppendedId(DTEntity.getContentUri(getContext()), rowId);
          }
          break;
        case ATTRIBUTE_CACHE:
          rowId = db.insert(MoEDatabaseHelper.TABLE_NAME_ATTRIBUTE_CACHE, null, values);
          if (rowId > 0) {
            newUri =
                ContentUris.withAppendedId(AttributeCacheEntity.getContentUri(getContext()), rowId);
          }
          break;
        case INAPPV3_CAMPAIGNS:
          rowId = db.insert(MoEDatabaseHelper.TABLE_NAME_INAPP_V3, null, values);
          if (rowId > 0) {
            newUri = ContentUris.withAppendedId(InAppV3Entity.getContentUri(getContext()), rowId);
          }
          break;
        case INAPP_STATS:
          rowId = db.insert(MoEDatabaseHelper.TABLE_NAME_INAPP_STATS, null, values);
          if (rowId > 0) {
            newUri =
                ContentUris.withAppendedId(InAppStatsEntity.getContentUri(getContext()), rowId);
          }
          break;
        case CARD_CAMPAIGNS:
          rowId = db.insert(MoEDatabaseHelper.TABLE_NAME_CARD_CAMPAIGNS, null, values);
          if (rowId > 0) {
            newUri = ContentUris.withAppendedId(CardsEntity.getContentUri(getContext()), rowId);
          }
          break;
        default:
          Logger.v("Unknown URI " + uri);
      }
      if (null != newUri) {
        Logger.v("MoEProvider: Added new record : " + newUri.toString());
        getContext().getContentResolver().notifyChange(uri, null);
      } else {
        Logger.v(TAG + ": Failed to add new record: " + uri);
      }
    } catch (Exception e) {
      Logger.e(TAG + " insert() : ", e);
    }
    return newUri;
  }

  public void updateIfRequired(SQLiteDatabase db) {
    if (MoEConstants.dbUpdateRequired) {
      databaseHelper.addMSGTagIfRequiredInbox(db);
      databaseHelper.addTableIfRequired(db, MoEDatabaseHelper.TABLE_NAME_USER_ATTRIBUTES);
      databaseHelper.addTableIfRequired(db, MoEDatabaseHelper.TABLE_NAME_CAMPAIGN_LIST);
      databaseHelper.addTableIfRequired(db, MoEDatabaseHelper.TABLE_NAME_BATCH_DATA);
      databaseHelper.addTableIfRequired(db, MoEDatabaseHelper.TABLE_NAME_DEVICE_TRIGGERS);
      databaseHelper.addTableIfRequired(db, MoEDatabaseHelper.TABLE_NAME_INAPP_V3);
      databaseHelper.addTableIfRequired(db, MoEDatabaseHelper.TABLE_NAME_ATTRIBUTE_CACHE);
      databaseHelper.addTableIfRequired(db, MoEDatabaseHelper.TABLE_NAME_INAPP_STATS);
      databaseHelper.addCampaignIdToMessageTable(db);
      databaseHelper.addTableIfRequired(db,MoEDatabaseHelper.TABLE_NAME_CARD_CAMPAIGNS);
      MoEConstants.dbUpdateRequired = false;
    }
  }


  @Override @NonNull public ContentProviderResult[] applyBatch(
      @NonNull ArrayList<ContentProviderOperation> operations)
      throws OperationApplicationException {
    ContentProviderResult[] result = new ContentProviderResult[operations.size()];
    int i = 0;
    // Opens the database object in "write" mode.
    SQLiteDatabase db = databaseHelper.getWritableDatabase();
    // Begin a transaction
    updateIfRequired(db);
    db.beginTransaction();
    try {
      for (ContentProviderOperation operation : operations) {
        // Chain the result for back references
        result[i++] = operation.apply(this, result, i);
      }
      db.setTransactionSuccessful();
    } catch (OperationApplicationException e) {
      Logger.e("MoEProvider : batch failed: ", e);
    } catch (Exception e) {
      Logger.e("MoEProvider : batch failed: ", e);
    } finally {
      db.endTransaction();
    }
    return result;
  }

  /*
   * (non-Javadoc)
   *
   * @see android.content.ContentProvider#delete(android.net.Uri,
   * java.lang.String, java.lang.String[])
   */
  public int delete(Uri uri, String selection, String[] selectionArgs) {
    int count = 0;
    try {
      if (null == uri) return 0;
      SQLiteDatabase db = databaseHelper.getWritableDatabase();
      updateIfRequired(db);
      switch (uriMatcher.match(uri)) {
        case MESSAGES:
          count = db.delete(MoEDatabaseHelper.TABLE_NAME_MSGS, selection, selectionArgs);
          break;
        case MESSAGE_ID:
          String noteId = uri.getPathSegments().get(1);
          count = db.delete(MoEDatabaseHelper.TABLE_NAME_MSGS,
              MessageEntity._ID + "=" + noteId + (!TextUtils.isEmpty(selection) ? " AND "
                  + selection
                  : ""), selectionArgs);
          break;
        case DATAPOINTS:
          count = db.delete(MoEDatabaseHelper.TABLE_NAME_DATAPOINTS, selection, selectionArgs);
          break;
        case DATAPOINT_ID:
          String eventId = uri.getPathSegments().get(1);
          count = db.delete(MoEDatabaseHelper.TABLE_NAME_DATAPOINTS,
              DataPointContract.DataPointEntity._ID + " = " + eventId + (!TextUtils.isEmpty(selection) ? " AND "
                  + selection : ""), selectionArgs);
          break;
        case INAPPS:
          count = db.delete(MoEDatabaseHelper.TABLE_NAME_INAPPS, selection, selectionArgs);
          break;
        case INAPP_ID:
          String inappId = uri.getPathSegments().get(1);
          count = db.delete(MoEDatabaseHelper.TABLE_NAME_INAPPS,
              DataPointContract.DataPointEntity._ID + "=" + inappId + (!TextUtils.isEmpty(selection) ? " AND "
                  + selection : ""), selectionArgs);
          break;
        case USER_ATTRIBUTES:
          count = db.delete(MoEDatabaseHelper.TABLE_NAME_USER_ATTRIBUTES, selection, selectionArgs);
          break;
        case USER_ATTRIBUTES_ID:
          String attributeId = uri.getPathSegments().get(1);
          count = db.delete(MoEDatabaseHelper.TABLE_NAME_USER_ATTRIBUTES,
              UserAttributeEntity._ID + "=" + attributeId + (!TextUtils.isEmpty(selection) ? " AND "
                  + selection : ""), selectionArgs);
          break;
        case CAMPAIGN_IDS:
          count = db.delete(MoEDatabaseHelper.TABLE_NAME_CAMPAIGN_LIST, selection, selectionArgs);
          break;
        case CAMPAIGN_ID:
          String campaignId = uri.getPathSegments().get(1);
          count = db.delete(MoEDatabaseHelper.TABLE_NAME_CAMPAIGN_LIST,
              CampaignListEntity._ID + "=" + campaignId + (!TextUtils.isEmpty(selection) ? " AND "
                  + selection : ""), selectionArgs);
          break;
        case DATA_BATCHES:
          count = db.delete(MoEDatabaseHelper.TABLE_NAME_BATCH_DATA, selection, selectionArgs);
          break;
        case DATA_BATCH:
          String batchId = uri.getPathSegments().get(1);
          count = db.delete(MoEDatabaseHelper.TABLE_NAME_BATCH_DATA,
              BatchDataEntity._ID + "=" + batchId + (!TextUtils.isEmpty(selection) ? " AND "
                  + selection : ""), selectionArgs);
          break;
        case DEVICE_TRIGGERS:
          count = db.delete(MoEDatabaseHelper.TABLE_NAME_DEVICE_TRIGGERS, selection, selectionArgs);
          break;
        case DEVICE_TRIGGER:
          String triggerId = uri.getPathSegments().get(1);
          count = db.delete(MoEDatabaseHelper.TABLE_NAME_DEVICE_TRIGGERS,
              DTEntity._ID + "=" + triggerId + (!TextUtils.isEmpty(selection) ? " AND "
                  + selection : ""), selectionArgs);
          break;
        case ATTRIBUTE_CACHE:
          count = db.delete(MoEDatabaseHelper.TABLE_NAME_ATTRIBUTE_CACHE, selection, selectionArgs);
          break;
        case ATTRIBUTE_CACHE_ID:
          String cachedAttributeId = uri.getPathSegments().get(1);
          count = db.delete(MoEDatabaseHelper.TABLE_NAME_ATTRIBUTE_CACHE,
              AttributeCacheEntity._ID + "=" + cachedAttributeId + (!TextUtils.isEmpty(selection) ?
                  " AND "
                      + selection : ""), selectionArgs);
          break;
        case INAPPV3_CAMPAIGNS:
          count = db.delete(MoEDatabaseHelper.TABLE_NAME_INAPP_V3, selection, selectionArgs);
          break;
        case INAPPV3_CAMPAIGN:
          String inAppCamapignId = uri.getPathSegments().get(1);
          count = db.delete(MoEDatabaseHelper.TABLE_NAME_INAPP_V3,
              InAppV3Entity._ID + "=" + inAppCamapignId + (!TextUtils.isEmpty(selection) ? " AND "
                  + selection
                  : ""), selectionArgs);
          break;
        case INAPP_STATS:
          count = db.delete(MoEDatabaseHelper.TABLE_NAME_INAPP_STATS, selection, selectionArgs);
          break;
        case INAPP_STAT:
          String statId = uri.getPathSegments().get(1);
          count = db.delete(MoEDatabaseHelper.TABLE_NAME_INAPP_STATS,
              InAppStatsEntity._ID + "=" + statId + (!TextUtils.isEmpty(selection) ?
                  " AND "
                      + selection : ""), selectionArgs);
          break;
        case CARD_CAMPAIGNS:
          count = db.delete(MoEDatabaseHelper.TABLE_NAME_CARD_CAMPAIGNS, selection, selectionArgs);
          break;
        case CARD_CAMPAIGN:
          String cardId = uri.getPathSegments().get(1);
          count = db.delete(MoEDatabaseHelper.TABLE_NAME_CARD_CAMPAIGNS,
              CardsEntity._ID + "=" + cardId + (!TextUtils.isEmpty(selection) ? " AND " + selection
                  : ""), selectionArgs);
          break;
        default:
          Logger.v("Unknown URI " + uri);
      }
      getContext().getContentResolver().notifyChange(uri, null);
      Logger.v("MoEProvider: Deleted " + count + " record(s) for URI: " + uri.toString());
    } catch (Exception e) {
      Logger.e(TAG + " delete() : ", e);
    }
    return count;
  }

  /*
   * (non-Javadoc)
   *
   * @see android.content.ContentProvider#update(android.net.Uri,
   * android.content.ContentValues, java.lang.String, java.lang.String[])
   */
  public int update(Uri uri, ContentValues values, String selection, String[] selectionArgs) {
    int count = 0;
    try {
      SQLiteDatabase db = databaseHelper.getWritableDatabase();
      updateIfRequired(db);
      switch (uriMatcher.match(uri)) {
        case MESSAGES:
          count = db.update(MoEDatabaseHelper.TABLE_NAME_MSGS, values, selection, selectionArgs);
          break;
        case MESSAGE_ID:
          String msgId = uri.getPathSegments().get(1);
          count = db.update(MoEDatabaseHelper.TABLE_NAME_MSGS, values,
              DataPointContract.DataPointEntity._ID + "=" + msgId + (!TextUtils.isEmpty(selection) ? " AND "
                  + selection
                  : ""), selectionArgs);
          break;
        case DATAPOINTS:
          count =
              db.update(MoEDatabaseHelper.TABLE_NAME_DATAPOINTS, values, selection, selectionArgs);
          break;
        case DATAPOINT_ID:
          String eventId = uri.getPathSegments().get(1);
          count = db.update(MoEDatabaseHelper.TABLE_NAME_DATAPOINTS, values,
              DataPointContract.DataPointEntity._ID + "=" + eventId + (!TextUtils.isEmpty(selection) ? " AND "
                  + selection : ""), selectionArgs);
          break;
        case INAPPS:
          count = db.update(MoEDatabaseHelper.TABLE_NAME_INAPPS, values, selection, selectionArgs);
          break;
        case INAPP_ID:
          String inAppId = uri.getPathSegments().get(1);
          count = db.update(MoEDatabaseHelper.TABLE_NAME_INAPPS, values,
              InAppMessageEntity._ID + "=" + inAppId + (!TextUtils.isEmpty(selection) ? " AND "
                  + selection : ""), selectionArgs);
          break;
        case USER_ATTRIBUTES:
          count = db.update(MoEDatabaseHelper.TABLE_NAME_USER_ATTRIBUTES, values, selection,
              selectionArgs);
          break;
        case USER_ATTRIBUTES_ID:
          String attributeId = uri.getPathSegments().get(1);
          count = db.update(MoEDatabaseHelper.TABLE_NAME_USER_ATTRIBUTES, values,
              UserAttributeEntity._ID + "=" + attributeId + (!TextUtils.isEmpty(selection) ? " AND "
                  + selection : ""), selectionArgs);
          break;
        case CAMPAIGN_IDS:
          count =
              db.update(MoEDatabaseHelper.TABLE_NAME_CAMPAIGN_LIST, values, selection,
                  selectionArgs);
          break;
        case CAMPAIGN_ID:
          String campaignId = uri.getPathSegments().get(1);
          count = db.update(MoEDatabaseHelper.TABLE_NAME_CAMPAIGN_LIST, values,
              CampaignListEntity._ID + "=" + campaignId + (!TextUtils.isEmpty(selection) ? " AND "
                  + selection : ""), selectionArgs);
          break;
        case DATA_BATCHES:
          count =
              db.update(MoEDatabaseHelper.TABLE_NAME_BATCH_DATA, values, selection, selectionArgs);
          break;
        case DATA_BATCH:
          String batchId = uri.getPathSegments().get(1);
          count = db.update(MoEDatabaseHelper.TABLE_NAME_BATCH_DATA, values,
              BatchDataEntity._ID + "=" + batchId + (!TextUtils.isEmpty(selection) ? " AND "
                  + selection : ""), selectionArgs);
          break;
        case DEVICE_TRIGGERS:
          count = db.update(MoEDatabaseHelper.TABLE_NAME_DEVICE_TRIGGERS, values, selection,
              selectionArgs);
          break;
        case DEVICE_TRIGGER:
          String triggerId = uri.getPathSegments().get(1);
          count = db.update(MoEDatabaseHelper.TABLE_NAME_DEVICE_TRIGGERS, values,
              DTEntity._ID + "=" + triggerId + (!TextUtils.isEmpty(selection) ? " AND "
                  + selection : ""), selectionArgs);
          break;
        case ATTRIBUTE_CACHE:
          count = db.update(MoEDatabaseHelper.TABLE_NAME_ATTRIBUTE_CACHE, values, selection,
              selectionArgs);
          break;
        case ATTRIBUTE_CACHE_ID:
          String cachedAttributeId = uri.getPathSegments().get(1);
          count = db.update(MoEDatabaseHelper.TABLE_NAME_ATTRIBUTE_CACHE, values,
              AttributeCacheEntity._ID + "=" + cachedAttributeId + (!TextUtils.isEmpty(selection) ?
                  " AND "
                      + selection : ""), selectionArgs);
          break;
        case INAPPV3_CAMPAIGNS:
          count =
              db.update(MoEDatabaseHelper.TABLE_NAME_INAPP_V3, values, selection, selectionArgs);
          break;
        case INAPPV3_CAMPAIGN:
          String inAppCampaignId = uri.getPathSegments().get(1);
          count = db.update(MoEDatabaseHelper.TABLE_NAME_INAPP_V3, values,
              InAppV3Entity._ID + "=" + inAppCampaignId + (!TextUtils.isEmpty(selection) ? " AND "
                  + selection : ""), selectionArgs);
          break;
        case INAPP_STATS:
          count =
              db.update(MoEDatabaseHelper.TABLE_NAME_INAPP_STATS, values, selection, selectionArgs);
          break;
        case INAPP_STAT:
          String statId = uri.getPathSegments().get(1);
          count = db.update(MoEDatabaseHelper.TABLE_NAME_INAPP_STATS, values,
              InAppStatsEntity._ID + "=" + statId + (!TextUtils.isEmpty(selection) ? " AND "
                  + selection : ""), selectionArgs);
          break;
        case CARD_CAMPAIGNS:
          count = db.update(MoEDatabaseHelper.TABLE_NAME_CARD_CAMPAIGNS, values, selection,
              selectionArgs);
          break;
        case CARD_CAMPAIGN:
          String cardId = uri.getPathSegments().get(1);
          count = db.update(MoEDatabaseHelper.TABLE_NAME_CARD_CAMPAIGNS, values,
              CardsEntity._ID + "=" + cardId + (!TextUtils.isEmpty(selection) ? " AND "
                  + selection : ""), selectionArgs);
          break;
        default:
          Logger.v("Unknown URI " + uri);
      }
      getContext().getContentResolver().notifyChange(uri, null);
      Logger.v("MoEProvider: Updated " + count + " record(s)");
    } catch (Exception e) {
      Logger.e(TAG + " update() : ", e);
    }
    return count;
  }
}
