/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.addon.trigger;

import android.content.Context;
import android.os.Build;
import androidx.annotation.Nullable;
import com.moe.pushlibrary.models.Event;
import com.moengage.core.APIManager;
import com.moengage.core.ConfigurationProvider;
import com.moengage.core.Logger;
import com.moengage.core.executor.SDKTask;
import com.moengage.core.executor.TaskResult;
import com.moengage.core.model.MoEJobParameters;
import com.moengage.core.rest.Response;
import java.util.LinkedList;
import org.json.JSONArray;
import org.json.JSONObject;

/**
 * @author Umang Chamaria
 */

public class DTNetworkTask extends SDKTask {

  private static final String TAG = DTConstants.MODULE_TAG + "DTNetworkTask";

  private DTController.NETWORK_CALL_TYPE mNetworkCallType;
  private Event mEvent;
  private TriggerMessage mCampaign;
  private MoEJobParameters jobParameters;

  DTNetworkTask(Context context, DTController.NETWORK_CALL_TYPE networkCallType) {
    super(context);
    this.mNetworkCallType = networkCallType;
  }

  DTNetworkTask(Context context, DTController.NETWORK_CALL_TYPE networkCallType, Event event,
      TriggerMessage campaign) {
    this(context, networkCallType);
    this.mEvent = event;
    this.mCampaign = campaign;
  }

  DTNetworkTask(Context context, DTController.NETWORK_CALL_TYPE networkCallType,
      @Nullable MoEJobParameters parameters) {
    this(context, networkCallType);
    this.jobParameters = parameters;
  }

  @Override public TaskResult execute() {
    try {
      Logger.v(TAG + " executing");
      Response response;
      switch (mNetworkCallType) {
        case SYNC_API:
          LinkedList<String> campaigns = DTDAO.getInstance(context).getActiveCampaignIds();
          JSONArray campaignArray = new JSONArray();
          if (campaigns != null) {
            for (String campaign : campaigns) {
              campaignArray.put(campaign);
            }
          }
          JSONObject request = new JSONObject();
          request.put(DTConstants.REQUEST_ATTR_CAMPAIGN_IDS, campaignArray);
          request.put(DTConstants.REQUEST_ATTR_LAST_SYNC_TIME,
              ConfigurationProvider.getInstance(context).getDTLastSyncTime());
          response = APIManager.deviceTriggerSyncRequest(context,
              DTConstants.ENDPOINT_SYNC_DT, null,
              request);
          DTParser.parseAndSaveSyncAPIResponse(response, context);
          break;
        case USER_IN_SEGMENT:
          JSONObject requestObject = new JSONObject(mEvent.details);
          requestObject.put(DTConstants.RESPONSE_ATTR_CAMPAIGN_ID, mCampaign.campaignId);
          response = APIManager.deviceTriggerSyncRequest(context,
              DTConstants.ENDPOINT_USER_IN_SEGMENT, null,
              requestObject);
          DTParser.parseUserInSegmentAPIResponse(response, mCampaign, context);
          break;
      }
      releaseJobLockIfRequired();
      Logger.v(TAG + "completed execution");
    } catch (Exception e) {
      Logger.e(TAG + "execute() : ", e);
    }
    return null;
  }

  private void releaseJobLockIfRequired() {
    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP
        && jobParameters != null) {
      Logger.v(TAG + " releaseJobLockIfRequired() : Trying to release job lock.");
      jobParameters.jobCompleteListener.jobComplete(jobParameters);
    }
  }

  @Override public String getTaskTag() {
    return TAG_DT_NETWORK_CALL_TASK;
  }

  @Override public boolean isSynchronous() {
    return false;
  }
}
