/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.addon.trigger;

import android.annotation.TargetApi;
import android.app.AlarmManager;
import android.app.PendingIntent;
import android.app.job.JobInfo;
import android.app.job.JobInfo.Builder;
import android.app.job.JobScheduler;
import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.os.Build;
import android.os.Build.VERSION_CODES;
import androidx.annotation.Keep;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import com.moengage.core.ConfigurationProvider;
import com.moengage.core.Logger;
import com.moengage.core.MoEDTManager.DTHandler;
import com.moengage.core.MoEDispatcher;
import com.moengage.core.MoEUtils;
import com.moengage.core.RemoteConfig;
import com.moengage.core.SdkConfig;
import com.moengage.core.model.MoEJobParameters;
import org.json.JSONObject;

/**
 * @author Umang Chamaria
 */
@Keep
public class DTHandlerImpl implements DTHandler {

  private static final String TAG = DTConstants.MODULE_TAG + "DTHandlerImpl";

  private static DTHandlerImpl _INSTANCE = null;

  public DTHandlerImpl(){
    _INSTANCE = this;
  }

  public static DTHandlerImpl getInstance(){
    if (_INSTANCE == null){
      _INSTANCE = new DTHandlerImpl();
    }
    return _INSTANCE;
  }

  @Override public void syncTriggersIfRequired(@NonNull Context context) {
    syncTriggersIfRequired(context, null);
  }

  @Override
  public void showTriggerCampaignIfPossible(@NonNull Context context, @NonNull String eventName,
      @NonNull JSONObject eventAttributes) {
    try {
      Logger.v(TAG + " showTriggerCampaignIfPossible() : trying to show trigger campaign");
      MoEDispatcher.getInstance(context).addTaskToQueueBeginning(new CheckAndShowDTCampaignTask
          (context, eventName, eventAttributes));
    } catch (Exception e) {
      Logger.e(TAG + " showTriggerCampaignIfPossible() : ", e);
    }
  }

  @Override public void forceSyncDeviceTriggers(@NonNull Context context) {
    forceSyncDeviceTriggers(context, null);
  }

  @Override public void scheduleBackgroundSync(Context context) {
    Logger.v(TAG + " scheduleBackgroundSync() : scheduling background fetch");
    try {
      if (!SdkConfig.getConfig().isRealTimeTriggerBackgroundSyncEnabled)return;
      long minimumDelayDuration =
          RemoteConfig.getConfig().realTimeTriggerBackgroundSyncInterval;
      Intent alarmIntent = new Intent(context, DTSyncAlarmReceiver.class);
      alarmIntent.setAction(DTConstants.ACTION_SYNC_MESSAGES);
      PendingIntent pendingIntent =
          PendingIntent.getBroadcast(context, DTConstants.REQ_CODE_FETCH_TRIGGERS,
              alarmIntent, PendingIntent.FLAG_UPDATE_CURRENT);
      AlarmManager alarmManager = (AlarmManager) context.getSystemService(Context.ALARM_SERVICE);
      if (Build.VERSION.SDK_INT >= VERSION_CODES.LOLLIPOP) {
        scheduleFetchJob(context, minimumDelayDuration);
      }else{
        alarmManager.set(AlarmManager.RTC_WAKEUP, System.currentTimeMillis() + minimumDelayDuration,
            pendingIntent);
      }
    } catch (Exception e) {
      Logger.e(TAG + "scheduleBackgroundSync() : ", e);
    }
  }

  @TargetApi(VERSION_CODES.LOLLIPOP)
  private void scheduleFetchJob(Context context, long minimumDelayDuration) {
    ComponentName serviceComponent = new ComponentName(context, DTSyncJob.class);
    Builder builder = new Builder(DTConstants.DT_SYNC_JOB_ID, serviceComponent);
    builder.setOverrideDeadline(System.currentTimeMillis() + minimumDelayDuration + (3600 *1000));
    builder.setMinimumLatency(minimumDelayDuration);
    builder.setRequiredNetworkType(JobInfo.NETWORK_TYPE_ANY);
    /*if (MoEHelperUtils.hasPermission(context, permission.RECEIVE_BOOT_COMPLETED)){
      builder.setPersisted(true);
    }*/
    JobScheduler jobScheduler =
        (JobScheduler) context.getSystemService(Context.JOB_SCHEDULER_SERVICE);
    if (jobScheduler != null) {
      jobScheduler.schedule(builder.build());
    }
  }

  void syncTriggersIfRequired(@NonNull Context context, @Nullable MoEJobParameters jobParameters){
    try {
      Logger.v(TAG + "syncTriggersIfRequired() : will sync if required");
      long lastSyncTime = ConfigurationProvider.getInstance(context).getDTLastSyncTime();
      long minimumSyncDelay =
          RemoteConfig.getConfig().realTimeTriggerBackgroundSyncInterval;
      if (lastSyncTime + minimumSyncDelay < MoEUtils.currentMillis()){
        forceSyncDeviceTriggers(context, jobParameters);
        scheduleBackgroundSync(context);
      }else {
        Logger.v(TAG + " syncTriggersIfRequired() : DT trigger sync not required");
      }
    } catch (Exception e) {
      Logger.e(TAG + " syncTriggersIfRequired() : ", e);
    }
  }

  void forceSyncDeviceTriggers(Context context, @Nullable MoEJobParameters jobComplete){
    try {
      Logger.v(TAG + " forceSyncDeviceTriggers() : force sync device trigger");
      MoEDispatcher.getInstance(context).startTask(new DTNetworkTask(context,
          DTController.NETWORK_CALL_TYPE.SYNC_API, jobComplete));
    } catch (Exception e) {
      Logger.e(TAG + " forceSyncDeviceTriggers() : ", e);
    }
  }
}
