/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.addon.trigger;

import android.content.ContentProviderOperation;
import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.net.Uri;
import android.text.TextUtils;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.WorkerThread;
import com.moe.pushlibrary.providers.MoEDataContract;
import com.moe.pushlibrary.providers.MoEDataContract.DTEntity;
import com.moengage.core.Logger;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import org.json.JSONObject;

/**
 * @author Umang Chamaria
 */

class DTDAO {

  private Context mContext;
  private Uri DT_CONTENT_URI;

  private DTDAO(Context context) {
    mContext = context;
    DT_CONTENT_URI = DTEntity.getContentUri(mContext);
  }

  private static DTDAO _INSTANCE = null;

  private static final String TAG = DTConstants.MODULE_TAG + "DTDAO";

  static DTDAO getInstance(Context context) {
    if (_INSTANCE == null) {
      _INSTANCE = new DTDAO(context);
    }
    return _INSTANCE;
  }

  /**
   * All active campaigns for a given event.
   *
   * @param eventName event name for which campaign is required
   * @return list of all campaigns with given event if any, else null
   */
  @Nullable @WorkerThread LinkedList<TriggerMessage> getCampaignsForEvent(String eventName) {
    Cursor cursor = null;
    LinkedList<TriggerMessage> triggerCampaign = new LinkedList<>();
    try {
      cursor = mContext.getContentResolver()
          .query(DT_CONTENT_URI, DTEntity.PROJECTION,
              DTEntity.TRIGGER_EVENT_NAME + " = ? AND " + DTEntity.STATUS + " = ?", new String[] {
                  eventName, DTConstants.CAMPAIGN_STATUS_ACTIVE
              }, DTEntity.DEFAULT_SORT_ORDER);
      if (cursor != null && cursor.moveToFirst()) {
        do {
          TriggerMessage triggerMessage = marshallTriggerMessage(cursor);
          if (triggerMessage != null) {
            triggerCampaign.add(triggerMessage);
          }
        } while (cursor.moveToNext());
      }
      return triggerCampaign;
    } catch (Exception e) {
      Logger.e(TAG + "getCampaignsForEvent() : ", e);
    } finally {
      closeCursor(cursor);
    }
    return null;
  }

  //get all campaigns
  @Nullable @WorkerThread LinkedList<TriggerMessage> getStoredCampaigns() {
    Cursor cursor = null;
    try {
      cursor = mContext.getContentResolver()
          .query(DT_CONTENT_URI, DTEntity.PROJECTION, null, null, null);
      if (cursor != null && cursor.moveToFirst()) {
        LinkedList<TriggerMessage> storedCampaignList = new LinkedList<>();
        do {
          TriggerMessage message = marshallTriggerMessage(cursor);
          if (message != null) storedCampaignList.add(message);
        } while (cursor.moveToNext());
        return storedCampaignList;
      }
    } catch (Exception e) {
      Logger.e(TAG + "getStoredCampaigns() : ", e);
    } finally {
      closeCursor(cursor);
    }
    return null;
  }

  // update all fields like last shown time, showCount etc.
  @WorkerThread boolean updateCampaignState(TriggerMessage triggerMessage) {
    Uri updateRec =
        DT_CONTENT_URI.buildUpon().appendPath(String.valueOf(triggerMessage._id)).build();
    ContentValues values = new ContentValues();
    values.put(DTEntity.LAST_SHOW_TIME, triggerMessage.state.lastShowTime);
    values.put(DTEntity.SHOW_COUNT, triggerMessage.state.showCount);
    int rowsUpdated = mContext.getContentResolver().update(updateRec, values, null, null);
    mContext.getContentResolver().notifyChange(updateRec, null);
    return rowsUpdated > 0;
  }

  /**
   * All unique trigger events
   *
   * @return set of trigger events if any, else null
   */
  @Nullable @WorkerThread HashSet<String> getTriggerEventsIfAny() {
    Cursor cursor = null;
    HashSet<String> triggerEvents = new HashSet<>();
    try {
      cursor = mContext.getContentResolver()
          .query(DT_CONTENT_URI, new String[] { DTEntity.TRIGGER_EVENT_NAME },
              DTEntity.STATUS + " = ?", new String[] { DTConstants.CAMPAIGN_STATUS_ACTIVE }, null);
      if (cursor != null && cursor.moveToFirst()) {
        do {
          triggerEvents.add(cursor.getString(0));
        } while (cursor.moveToNext());
      }
      return triggerEvents;
    } catch (Exception e) {
      Logger.e(TAG + "getTriggerEventsIfAny() : ", e);
    } finally {
      closeCursor(cursor);
    }
    return null;
  }

  /**
   * Add new campaigns and update existing campaigns.
   *
   * @param campaignList list of newly fetched campaigns
   */
  @Nullable @WorkerThread void addOrUpdateTriggerCampaigns(
      @NonNull LinkedList<TriggerMessage> campaignList) {
    try {
      LinkedList<TriggerMessage> storedCampaigns = getStoredCampaigns();

      ArrayList<ContentProviderOperation> addOperations = new ArrayList<>();
      ArrayList<ContentProviderOperation> updateOperations = new ArrayList<>();

      if (storedCampaigns == null) {
        for (TriggerMessage campaign : campaignList) {
          addOperations.add(ContentProviderOperation.newInsert(DT_CONTENT_URI)
              .withValues(getContentValues(campaign))
              .build());
        }
      } else {
        for (TriggerMessage campaign : campaignList) {
          boolean found = false;
          for (TriggerMessage storedCampaign : storedCampaigns) {
            if (storedCampaign.campaignId.equals(campaign.campaignId)) {
              campaign._id = storedCampaign._id;
              campaign.state.showCount = storedCampaign.state.showCount;
              campaign.state.lastShowTime = storedCampaign.state.lastShowTime;
              if (campaign.rules.expiryTime == -1){
                campaign.rules.expiryTime = storedCampaign.rules.expiryTime;
              }
              found = true;
              break;
            }
          }
          if (found) {
            updateOperations.add(ContentProviderOperation.newUpdate(
                DT_CONTENT_URI.buildUpon().appendPath(String.valueOf(campaign._id)).build())
                .withValues(getContentValues(campaign))
                .build());
          } else {
            addOperations.add(ContentProviderOperation.newInsert(DT_CONTENT_URI)
                .withValues(getContentValues(campaign))
                .build());
          }
        }
      }
      if (!updateOperations.isEmpty()) {
        mContext.getContentResolver()
            .applyBatch(MoEDataContract.getAuthority(mContext), updateOperations);
      }
      if (!addOperations.isEmpty()) {
        mContext.getContentResolver()
            .applyBatch(MoEDataContract.getAuthority(mContext), addOperations);
      }
    } catch (Exception e) {
      Logger.e(TAG + "addOrUpdateTriggerCampaigns() : ", e);
    }
  }

  @Nullable @WorkerThread LinkedHashMap<String, TriggerMessage> getAllActiveCampaigns() {
    Cursor cursor = null;
    try {
      cursor = mContext.getContentResolver()
          .query(DT_CONTENT_URI, DTEntity.PROJECTION, DTEntity.STATUS + " = ?", new String[] {
              DTConstants.CAMPAIGN_STATUS_ACTIVE
          }, DTEntity.DEFAULT_SORT_ORDER);
      if (cursor != null && cursor.moveToFirst()) {
        LinkedHashMap<String, TriggerMessage> activeCampaigns = new LinkedHashMap<>();
        do {
          TriggerMessage triggerMessage = marshallTriggerMessage(cursor);
          if (triggerMessage != null) {
            activeCampaigns.put(triggerMessage.campaignId, triggerMessage);
          }
        } while (cursor.moveToNext());
        return activeCampaigns;
      }
    } catch (Exception e) {
      Logger.e(TAG + "getAllActiveCampaigns() : ", e);
    } finally {
      closeCursor(cursor);
    }
    return null;
  }

  @Nullable @WorkerThread LinkedList<String> getActiveCampaignIds() {
    Cursor cursor = null;
    try {
      //get campaigns which are either active or paused
      cursor = mContext.getContentResolver()
          .query(DT_CONTENT_URI, new String[] { DTEntity.CAMPAIGN_ID }, DTEntity.STATUS + " = ?",
              new String[] { DTConstants.CAMPAIGN_STATUS_ACTIVE }, null);
      if (cursor == null || cursor.getCount() == 0) {
        if (cursor != null) closeCursor(cursor);
        return null;
      }
      LinkedList<String> campaignIdList = new LinkedList<>();
      while (cursor.moveToNext()) {
        String id = cursor.getString(0);
        if (!TextUtils.isEmpty(id)) {
          campaignIdList.add(id);
        }
      }
      return campaignIdList;
    } catch (Exception e) {
      Logger.e(TAG + "getActiveCampaignIds() : ", e);
    } finally {
      closeCursor(cursor);
    }
    return null;
  }

  void removeExpiredCampaigns() {
    try {
      int rows = mContext.getContentResolver()
          .delete(DT_CONTENT_URI, DTEntity.EXPIRY_TIME + " < ?",
              new String[] { Long.toString(System.currentTimeMillis()) });
      int rows1 = mContext.getContentResolver()
          .delete(DT_CONTENT_URI, DTEntity.STATUS + " = ?",
              new String[] { DTConstants.CAMPAIGN_STATUS_EXPIRED });
      Logger.v(TAG + "removeExpiredCampaigns(): Number of device triggers records deleted: " + (rows
          + rows1));
    } catch (Exception e) {
      Logger.e(TAG + "removeExpiredCampaigns() : ", e);
    }
  }

  @Nullable TriggerMessage getCampaignById(String campaignId) {
    Cursor cursor = null;
    try {
      cursor = mContext.getContentResolver()
          .query(DT_CONTENT_URI, DTEntity.PROJECTION, DTEntity.CAMPAIGN_ID + " = ?",
              new String[] { campaignId }, null);
      if (cursor != null && cursor.moveToFirst()) {
        return marshallTriggerMessage(cursor);
      }
    } catch (Exception e) {
      Logger.e(TAG + " getCampaignById() : ", e);
    } finally {
      closeCursor(cursor);
    }
    return null;
  }

  /**
   * Converts campaign object to ContentValues
   *
   * @param campaign campaign object
   * @return content value for the given campaign object.
   */
  @Nullable private ContentValues getContentValues(@NonNull TriggerMessage campaign) {
    if (campaign == null) return null;
    try {
      ContentValues values = new ContentValues();
      if (campaign._id != 0) {
        values.put(DTEntity._ID, campaign._id);
      }
      values.put(DTEntity.CAMPAIGN_ID, campaign.campaignId);
      values.put(DTEntity.CAMPAIGN_TYPE, campaign.campaignType);
      values.put(DTEntity.TRIGGER_EVENT_NAME, campaign.triggerEventName);
      if (campaign.payload != null) {
        values.put(DTEntity.PAYLOAD, campaign.payload.toString());
      }
      if (campaign.campaignPayload != null) {
        values.put(DTEntity.CAMPAIGN_PAYLOAD, campaign.campaignPayload.toString());
      }
      values.put(DTEntity.MAX_COUNT, campaign.rules.maxCount);
      values.put(DTEntity.MINIMUM_DELAY, campaign.rules.minimumDelay);
      values.put(DTEntity.SHOULD_SHOW_OFFLINE, campaign.rules.shouldShowOffline ? 1 : 0);
      values.put(DTEntity.MAX_SYNC_DELAY_TIME, campaign.rules.maxSyncDelay);
      values.put(DTEntity.EXPIRY_TIME, campaign.rules.expiryTime);
      values.put(DTEntity.PRIORITY, campaign.rules.priority);
      values.put(DTEntity.SHOULD_IGNORE_DND, campaign.rules.shouldIgnoreDND ? 1 : 0);
      values.put(DTEntity.DELAY_BEFORE_SHOWING_NOTIFICATION, campaign.rules.showDelay);

      values.put(DTEntity.STATUS, campaign.state.status);
      values.put(DTEntity.LAST_UPDATED_TIME, campaign.state.lastUpdatedTime);
      values.put(DTEntity.SHOW_COUNT, campaign.state.showCount);
      values.put(DTEntity.LAST_SHOW_TIME, campaign.state.lastShowTime);

      return values;
    } catch (Exception e) {
      Logger.e(TAG + "getContentValues() : ", e);
    }
    return null;
  }

  private void closeCursor(Cursor cursor) {
    if (cursor != null) {
      cursor.close();
    }
  }

  /**
   * Cursor object
   *
   * @param cursor cursor object
   * @return object of {@link TriggerMessage}
   */
  private TriggerMessage marshallTriggerMessage(Cursor cursor) {
    try {
      TriggerMessage triggerMessage = new TriggerMessage();
      triggerMessage._id = cursor.getLong(DTEntity.COLUMN_INDEX_ID);
      triggerMessage.campaignId = cursor.getString(DTEntity.COLUMN_INDEX_CAMPAIGN_ID);
      triggerMessage.triggerEventName = cursor.getString(DTEntity.COLUMN_INDEX_TRIGGER_EVENT_NAME);
      String pushPayload = cursor.getString(DTEntity.COLUMN_INDEX_PAYLOAD);
      if (!TextUtils.isEmpty(pushPayload)){
        triggerMessage.payload = new JSONObject(pushPayload);
      }
      String campaignPayloadString  = cursor.getString(DTEntity.COLUMN_INDEX_CAMPAIGN_PAYLOAD);
      if (!TextUtils.isEmpty(campaignPayloadString)){
        triggerMessage.campaignPayload = new JSONObject(campaignPayloadString);
      }
      triggerMessage.campaignType = cursor.getString(DTEntity.COLUMN_INDEX_CAMPAIGN_TYPE);
      triggerMessage.rules.maxCount = cursor.getLong(DTEntity.COLUMN_INDEX_MAX_COUNT);
      triggerMessage.rules.minimumDelay = cursor.getLong(DTEntity.COLUMN_INDEX_MINIMUM_DELAY);
      triggerMessage.rules.shouldShowOffline =
          cursor.getInt(DTEntity.COLUMN_INDEX_SHOULD_SHOW_OFFLINE) == 1;
      triggerMessage.rules.maxSyncDelay = cursor.getLong(DTEntity.COLUMN_INDEX_MAX_SYNC_DELAY_TIME);
      triggerMessage.rules.expiryTime = cursor.getLong(DTEntity.COLUMN_INDEX_EXPIRY_TIME);
      triggerMessage.rules.priority = cursor.getInt(DTEntity.COLUMN_INDEX_PRIORITY);
      triggerMessage.rules.shouldIgnoreDND =
          cursor.getInt(DTEntity.COLUMN_INDEX_SHOULD_IGNORE_DND) == 1;
      triggerMessage.rules.showDelay = cursor.getInt(DTEntity.COLUMN_INDEX_DELAY_BEFORE_SHOWING);
      triggerMessage.state.lastShowTime = cursor.getLong(DTEntity.COLUMN_INDEX_LAST_SHOW_TIME);
      triggerMessage.state.showCount = cursor.getLong(DTEntity.COLUMN_INDEX_SHOW_COUNT);
      triggerMessage.state.lastUpdatedTime =
          cursor.getLong(DTEntity.COLUMN_INDEX_LAST_UPDATED_TIME);
      triggerMessage.state.status = cursor.getString(DTEntity.COLUMN_INDEX_STATUS);
      triggerMessage.state.isActive =
          triggerMessage.state.status.equals(DTConstants.CAMPAIGN_STATUS_ACTIVE);
      return triggerMessage;
    } catch (Exception e) {
      Logger.e(TAG + "marshallTriggerMessage() : ", e);
    }
    return null;
  }
}
