/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.addon.trigger;

import android.app.AlarmManager;
import android.app.PendingIntent;
import android.content.Context;
import android.content.Intent;
import android.os.Bundle;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.WorkerThread;
import com.moe.pushlibrary.MoEHelper;
import com.moe.pushlibrary.utils.MoEHelperConstants;
import com.moe.pushlibrary.utils.MoEHelperUtils;
import com.moengage.core.ConfigurationProvider;
import com.moengage.core.Logger;
import com.moengage.core.MoEUtils;
import com.moengage.core.Properties;
import com.moengage.evaluator.ConditionEvaluator;
import com.moengage.pushbase.MoEPushHelper;
import com.moengage.pushbase.PushConstants;
import java.util.Calendar;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.TimeZone;
import org.json.JSONException;
import org.json.JSONObject;

/**
 * @author Umang Chamaria
 */

class DTController {

  public enum NETWORK_CALL_TYPE {
    SYNC_API, USER_IN_SEGMENT
  }

  private static final String TAG = DTConstants.MODULE_TAG + "DTController";

  private HashSet<String> mTriggerEvents;

  private Context mContext;

  private DTConditionEvaluator conditionEvaluator;

  private DTDAO dbHandler;

  private DTController(@NonNull Context context) {
    mContext = context;
    dbHandler = DTDAO.getInstance(mContext);
    updateDTCache();
    conditionEvaluator = new DTConditionEvaluator();
  }

  private static DTController _INSTANCE;

  public static DTController getInstance(Context context) {
    if (_INSTANCE == null) {
      _INSTANCE = new DTController(context);
    }
    return _INSTANCE;
  }

  @WorkerThread void updateDTCache() {
    try {
      mTriggerEvents = dbHandler.getTriggerEventsIfAny();
      if (mTriggerEvents != null){
        Logger.v(TAG + " updateDTCache() : device trigger event " + mTriggerEvents.toString());
      }else {
        Logger.v(TAG + " updateDTCache() : no device trigger events");
      }
      dbHandler.removeExpiredCampaigns();
    } catch (Exception e) {
      Logger.e(TAG + " updateDTCache() : ", e);
    }
  }

  @Nullable TriggerMessage getCampaignToShown(String eventName, JSONObject eventAttributes) {
    try {
      LinkedList<TriggerMessage> campaignsForEvent =
          dbHandler.getCampaignsForEvent(eventName);
      if (campaignsForEvent == null) return null;
      for (TriggerMessage campaign : campaignsForEvent) {
        try {
          Logger.v(TAG + " getCampaignToShown() : evaluating conditions for campaign id: " +
              campaign.campaignId);
          campaign.dump();
          if (!conditionEvaluator.canShowTriggerMessage(campaign, getLastSyncTime(),
              System.currentTimeMillis())) {
            continue;
          }
          //campaign payload sanity check
          if (campaign.campaignPayload == null) continue;
          if (!campaign.campaignPayload.has(DTConstants.RESPONSE_ATTR_CONDITION)) return campaign;
          JSONObject conditions = campaign.campaignPayload.getJSONObject(DTConstants
              .RESPONSE_ATTR_CONDITION);
          ConditionEvaluator evaluator = new ConditionEvaluator(conditions,
              MoEHelperUtils.transformEventAttributesForEvaluationPackage(eventAttributes));
          boolean evaluationResult = evaluator.evaluate();
          Logger.v(TAG + " getCampaignToShown() : Evaluation result: " + evaluationResult);
          if (evaluationResult) return campaign;
        } catch (Exception e) {
          Logger.e(TAG + "getCampaignToShown() : inside for loop ", e);
        }
      }
    } catch (Exception e) {
      Logger.e(TAG + "getCampaignToShown() : ", e);
    }
    return null;
  }


  /**
   * Show or schedule notification when user is online
   *
   * @param message Campaign object
   */
  void showOrScheduleNotificationOnline(TriggerMessage message) {
    if (conditionEvaluator.isPayloadEmpty(message)) return;
    if (message.rules.showDelay == 0) {
      showPushAndUpdateCounter(message);
    } else {
      schedulePushNotification(message, false);
    }
  }

  /**
   * Try to show notification offline is possible
   *
   * @param message Campaign Object
   * @throws JSONException pass on the exception to the caller
   */
  void showOrScheduleNotificationOffline(TriggerMessage message) throws JSONException {
    if (!message.rules.shouldShowOffline || conditionEvaluator.isPayloadEmpty(message)) return;
    // identifier to track notifications shown offline
    message.payload.put(PushConstants.REAL_TIME_TRIGGER_OFFLINE_IDENTIFIER, true);

    // show or schedule notification
    if (message.rules.showDelay == 0) {
      tryShowingNotificationOffline(message);
    } else {
      schedulePushNotification(message, true);
    }
  }

  /**
   * Update campaign counters.
   *
   * @param message Campaign object
   */
  private void updateCampaignState(TriggerMessage message) {
    message.state.showCount++;
    message.state.lastShowTime = System.currentTimeMillis();
    ConfigurationProvider.getInstance(mContext).saveLastDTShowTime(message.state.lastShowTime);
    dbHandler.updateCampaignState(message);
  }

  /**
   * Try to show offline notification if DND is not active.
   *
   * @param message Campaign object
   */
  void tryShowingNotificationOffline(TriggerMessage message){
    try {
      Calendar rightNow = Calendar.getInstance(TimeZone.getTimeZone("GMT"));
      int currentHour = rightNow.get(Calendar.HOUR_OF_DAY);
      int currentMinutes = rightNow.get(Calendar.MINUTE);

      if (conditionEvaluator.isDNDActive(getDNDStartTime(), getDNDEndTime(), currentHour,
          currentMinutes) && !message.rules.shouldIgnoreDND) {
        Logger.v(TAG + " tryShowingNotificationOffline() : dnd is active cannot show notification");
        return;
      }

      String campaignId = message.payload.getString(MoEHelperConstants.GCM_EXTRA_CAMPAIGN_ID);

      // event for backend stats job
      Properties properties = new Properties();
      properties.addAttribute(MoEHelperConstants.GCM_EXTRA_CAMPAIGN_ID, campaignId)
          .setNonInteractive();
      MoEHelper.getInstance(mContext)
          .trackEvent(MoEHelperConstants.NOTIFICATION_OFFLINE_MOE, properties);

      // adding additional timestamp to by pass duplicate check
      message.payload.put(MoEHelperConstants.GCM_EXTRA_CAMPAIGN_ID,
          campaignId + PushConstants.REAL_TIME_TRIGGER_IDENTIFIER + System.currentTimeMillis());
      showPushAndUpdateCounter(message);
    } catch (Exception e) {
      Logger.e( TAG + " tryShowingNotificationOffline() : ", e);
    }
  }

  void showScheduledNotification(String campaignId, boolean isOffline, String payload){
    try {
      TriggerMessage message = dbHandler.getCampaignById(campaignId);
      if (message != null){
        message.payload = new JSONObject(payload);
        if (isOffline) tryShowingNotificationOffline(message);
        else showPushAndUpdateCounter(message);
      }else {
        Logger.e( TAG + " showScheduledNotification() : did not find campaign with id: " +
            campaignId);
      }
    } catch (Exception e) {
      Logger.e(TAG + " showScheduledNotification() : ", e);
    }
  }

  /**
   * Pass push payload to push module
   *
   * @param message Campaign object
   */
  void showPushAndUpdateCounter(TriggerMessage message) {
    try {
      if (message.rules.expiryTime < System.currentTimeMillis()
          || conditionEvaluator.isPayloadEmpty(message)) {
        Logger.e(TAG + " showPushAndUpdateCounter() : cannot show trigger message for campaign "
            + "id: " + message.campaignId);
        return;
      }
      // convert payload from JSON to Bundle
      Bundle pushPayload = MoEUtils.jsonToBundle(message.payload);
      if (pushPayload == null){
        Logger.e(TAG + " showPushAndUpdateCounter() : could not convert json to bundle, cannot "
            + "show campaign for campaign id: " + message.campaignId);
        return;
      }
      //pass payload to push module
      MoEPushHelper.getInstance().handlePushPayload(mContext, pushPayload);
      // update count
      updateCampaignState(message);
    } catch (Exception e) {
      Logger.e(TAG + " showPushAndUpdateCounter() : ", e);
    }
  }

  /**
   * Schedule alarm to show notification later.
   *
   * @param message Campaign object
   * @param isOffline true if online notification was scheduled, else false
   */
  void schedulePushNotification(TriggerMessage message, boolean isOffline) {
    try {
      Logger.v(TAG + " schedulePushNotification() : will schedule notification for campaign id: "
          + message.campaignId);
      Intent laterIntent = new Intent(mContext, DTIntentService.class);
      laterIntent.putExtra(DTConstants.EXTRA_ATTR_OFFLINE, isOffline);
      laterIntent.putExtra(DTConstants.EXTRA_SHOW_NOTIFICATION, message.campaignId);
      laterIntent.putExtra(DTConstants.EXTRA_NOTIFICATION_PAYLOAD, message.payload.toString());
      laterIntent.setAction(DTConstants.ACTION_SHOW_NOTIFICATION);
      final int _id = (int)System.currentTimeMillis();
      PendingIntent alarmPendingIntent =
          PendingIntent.getService(mContext, _id, laterIntent, PendingIntent.FLAG_UPDATE_CURRENT);
      AlarmManager alarmManager = (AlarmManager) mContext.getSystemService(Context.ALARM_SERVICE);
      if (alarmManager != null) {
        alarmManager.set(AlarmManager.RTC_WAKEUP, System.currentTimeMillis() + message.rules.showDelay,
            alarmPendingIntent);
      }
      //track event to log campaign was scheduled
      Properties builder = new Properties();
      builder.addAttribute("campaign_id", message.campaignId);
      builder.setNonInteractive();
      MoEHelper.getInstance(mContext).trackEvent(MoEHelperConstants.DT_CAMPAIGN_SCHEDULED, builder);
    } catch (Exception e) {
      Logger.e(TAG + " schedulePushNotification() : ", e);
    }
  }

  @Nullable HashSet<String> getTriggerEvents() {
    return mTriggerEvents;
  }

  long getGlobalMinimumDelay(){
    return ConfigurationProvider.getInstance(mContext).getDTMinimumDelay();
  }

  long getLastShowTime(){
    return ConfigurationProvider.getInstance(mContext).getDTLastShowTime();
  }

  long getDNDStartTime(){
    return ConfigurationProvider.getInstance(mContext).getDTDNDStartTime();
  }

  long getDNDEndTime(){
    return ConfigurationProvider.getInstance(mContext).getDNDEndTime();
  }

  long getLastSyncTime(){
    return ConfigurationProvider.getInstance(mContext).getDTLastSyncTime();
  }
}
