/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.addon.trigger;

import androidx.annotation.Nullable;
import com.moengage.core.Logger;
import com.moengage.core.MoECoreEvaluator;
import java.util.HashSet;

/**
 * @author Umang Chamaria
 */
public class DTConditionEvaluator {

  private static final String TAG = DTConstants.MODULE_TAG + "DTConditionEvaluator";

  /**
   * Checks whether a given event is a device trigger event or not
   *
   * @param eventName triggered event name
   * @return true if the event is a device trigger event else false
   */
  boolean isDTEvent(@Nullable HashSet<String> triggerEvents, String eventName) {
    return triggerEvents != null && triggerEvents.contains(eventName);
  }

  boolean shouldShowTrigger(long lastShowTime, long globalMinimumDelay, long currentTime) {
    return lastShowTime + globalMinimumDelay < currentTime;
  }

  /**
   * Check whether DND is currently active or not.
   *
   * @return false if DND is not active, else true
   */
  boolean isDNDActive(long startTime, long endTime, int currentHour, int currentMinutes) {
    return new MoECoreEvaluator().canShowCampaignNow(startTime, endTime, currentHour,
        currentMinutes);
  }

  /**
   * Check whether a campaign is fit to be shown or not.
   *
   * @param triggerMessage campaign details.
   * @return true is the campaign is fit to be shown else false
   */
  boolean canShowTriggerMessage(TriggerMessage triggerMessage, long lastSyncTime, long
      currentTime) {
    //checking if active
    if (!triggerMessage.campaignType.equals(DTConstants.CAMPAIGN_TYPE_GENERAL)) {
      Logger.i(TAG
          + " canShowTriggerMessage() : "
          + triggerMessage.campaignId
          + " is not of type general. "
          + "Cannot show");
      return false;
    }
    if (!triggerMessage.state.isActive
        || triggerMessage.rules.expiryTime < currentTime) {
      Logger.i(TAG
          + " canShowTriggerMessage() : "
          + triggerMessage.campaignId
          + " is no longer active "
          + "cannot show");
      return false;
    }
    if (triggerMessage.rules.maxSyncDelay + lastSyncTime < currentTime && lastSyncTime != 0) {
      Logger.i(TAG
          + " canShowTriggerMessage() : "
          + triggerMessage.campaignId
          + " has not been updated"
          + " in a while. Cannot show without update");
      return false;
    }
    if (triggerMessage.rules.minimumDelay + triggerMessage.state.lastShowTime
        > currentTime) {
      Logger.i(TAG
          + " canShowTriggerMessage() : "
          + triggerMessage.campaignId
          + " was shown recently. "
          + "Cannot show now");
      return false;
    }
    if (triggerMessage.rules.maxCount != -9090909
        && triggerMessage.rules.maxCount <= triggerMessage.state.showCount) {
      Logger.i(TAG
          + "canShowTriggerMessage() : "
          + triggerMessage.campaignId
          + " has been shown "
          + "maximum number of times. Cannot be shown again");
      return false;
    }
    return true;
  }

  /**
   * Checks whether notification payload is empty or not
   *
   * @param message Campaign object
   * @return true if campaign has notification payload
   */
  boolean isPayloadEmpty(TriggerMessage message) {
    return message.payload == null || message.payload.length() == 0;
  }
}
