package com.moengage.addon.trigger;

import android.os.Parcel;
import android.os.Parcelable;
import com.moengage.core.Logger;
import org.json.JSONObject;

/**
 * @author Umang Chamaria
 */

class TriggerMessage implements Parcelable {

  /**
   * Database record id for trigger
   */
  long _id;
  /**
   * unique id for the campaign
   */
  String campaignId;
  /**
   * Event on which the campaign needs to be triggered.
   */
  String triggerEventName;
  /**
   * push payload for the campaign
   */
  JSONObject payload;
  /**
   * complete campaign payload
   */
  JSONObject campaignPayload;
  /**
   * Rules for the campaign
   */
  Rules rules;
  /**
   * Current state of the campaign
   */
  State state;

  /**
   * Campaign Type
   */
  String campaignType;

  TriggerMessage() {
    rules = new Rules();
    state = new State();
    payload = new JSONObject();
    campaignPayload = new JSONObject();
  }

  protected TriggerMessage(Parcel in) {
    if (rules == null) {
      rules = new Rules();
    }
    if (state == null) {
      state = new State();
    }
    try {
      _id = in.readLong();
      campaignId = in.readString();
      triggerEventName = in.readString();
      campaignType = in.readString();
      payload = new JSONObject(in.readString());
      campaignPayload = new JSONObject(in.readString());
      rules.maxCount = in.readLong();
      rules.showDelay = in.readLong();
      rules.minimumDelay = in.readLong();
      rules.shouldShowOffline = in.readInt() == 1;
      rules.maxSyncDelay = in.readLong();
      rules.expiryTime = in.readLong();
      rules.priority = in.readLong();
      rules.shouldIgnoreDND = in.readInt() == 1;
      state.lastShowTime = in.readLong();
      state.lastUpdatedTime = in.readLong();
      state.showCount = in.readLong();
      state.isActive = in.readInt() == 1;
      state.status = in.readString();
    } catch (Exception e) {
      Logger.f("TriggerMessage : TriggerMessage() : ", e);
    }
  }

  public static final Creator<TriggerMessage> CREATOR = new Creator<TriggerMessage>() {
    @Override public TriggerMessage createFromParcel(Parcel in) {
      return new TriggerMessage(in);
    }

    @Override public TriggerMessage[] newArray(int size) {
      return new TriggerMessage[size];
    }
  };

  @Override public int describeContents() {
    return 0;
  }

  @Override public void writeToParcel(Parcel dest, int flags) {
    dest.writeLong(_id);
    dest.writeString(campaignId);
    dest.writeString(triggerEventName);
    dest.writeString(campaignType);
    dest.writeString(payload.toString());
    dest.writeString(campaignPayload.toString());
    if (rules != null) {
      dest.writeLong(rules.maxCount);
      dest.writeLong(rules.showDelay);
      dest.writeLong(rules.minimumDelay);
      dest.writeInt(rules.shouldShowOffline ? 1 : 0);
      dest.writeLong(rules.maxSyncDelay);
      dest.writeLong(rules.expiryTime);
      dest.writeLong(rules.priority);
      dest.writeInt(rules.shouldIgnoreDND ? 1 : 0);
    }
    if (state != null) {
      dest.writeLong(state.lastShowTime);
      dest.writeLong(state.showCount);
      dest.writeLong(state.lastUpdatedTime);
      dest.writeInt(state.isActive ? 1 : 0);
      dest.writeString(state.status);
    }
  }

  static class Rules {
    /**
     * maximum times campaign can be shown
     */
    long maxCount;
    /**
     * Time after which campaign should be shown
     */
    long showDelay;
    /**
     * minimum time delay before the campaign cannot be shown again
     */
    long minimumDelay;
    /**
     * should the campaign be shown if device is offline
     */
    boolean shouldShowOffline;
    /**
     * maximum time which can elapse before next update.
     */
    long maxSyncDelay;
    /**
     * campaign expiry time.
     */
    long expiryTime;
    /**
     * priority of the campaign
     */
    long priority;
    /**
     * should the campaign honor DND setting
     */
    boolean shouldIgnoreDND;
  }

  static class State {
    /**
     * Time at which the campaign was last shown.
     */
    long lastShowTime;
    /**
     * number of times campaign has been shown.
     */
    long showCount;
    /**
     * last time campaign was updated.
     */
    long lastUpdatedTime;
    /**
     * is campaign active.
     */
    boolean isActive;
    /**
     * State of the campaign active/paused/stopped
     */
    String status;
  }

  public void dump() {
    try {
      StringBuilder builder = new StringBuilder("TriggerMessage: Details -> \n");
      builder.append("\n campaign-id: ")
          .append(campaignId)
          .append("\n trigger event: ")
          .append(triggerEventName);
      if (rules != null) {
        builder.append("\n expiry time: ")
            .append(rules.expiryTime)
            .append("\n max show count: ")
            .append(rules.maxCount)
            .append("\n minimum delay: ")
            .append(rules.minimumDelay)
            .append("\n priority: ")
            .append(rules.priority)
            .append("\n shouldIgnoreDND: ")
            .append(rules.shouldIgnoreDND)
            .append("\n shouldShowOffline: ")
            .append(rules.shouldShowOffline)
            .append("\n show delay: ")
            .append(rules.showDelay)
            .append("\n max sync delay: ")
            .append(rules.maxSyncDelay);
      }
      if (state != null) {
        builder.append("\n last show time: ")
            .append(state.lastShowTime)
            .append("\n last updated time: ")
            .append(state.lastUpdatedTime)
            .append("\n show count: ")
            .append(state.showCount)
            .append("\n status: ")
            .append(state.status);
      }
      if (payload != null) {
        builder.append("\n push payload: ").append(payload.toString());
      }
      if (campaignPayload != null && campaignPayload.has("condition")) {
        builder.append("\n conditions: ")
            .append(campaignPayload.getJSONObject("condition").toString());
      }
      Logger.v(builder.toString());
    } catch (Exception e) {
      Logger.e("TriggerMessage" + " dump() : ", e);
    }
  }
}
