package com.moengage.addon.messaging;

import android.Manifest.permission;
import android.annotation.TargetApi;
import android.app.AlarmManager;
import android.app.PendingIntent;
import android.app.job.JobInfo;
import android.app.job.JobInfo.Builder;
import android.app.job.JobParameters;
import android.app.job.JobScheduler;
import android.content.ComponentName;
import android.content.ContentValues;
import android.content.Context;
import android.content.Intent;
import android.os.Build;
import android.os.Build.VERSION_CODES;
import android.os.Bundle;
import android.support.annotation.Nullable;
import com.moe.pushlibrary.providers.MoEDataContract.CampaignListEntity;
import com.moe.pushlibrary.utils.MoEHelperUtils;
import com.moengage.core.ConfigurationProvider;
import com.moengage.core.FetchMessagesFromServerTask;
import com.moengage.core.Logger;
import com.moengage.core.MoEConstants;
import com.moengage.core.MoEDispatcher;
import com.moengage.core.MoEUtils;
import com.moengage.core.OnJobComplete;
import com.moengage.push.MoEMessagingManager.MessagingHandler;
import com.moengage.push.PushManager;
import com.moengage.push.PushManager.PushHandler;
import java.util.HashMap;
import java.util.Iterator;
import org.json.JSONArray;
import org.json.JSONObject;

/**
 * @author Umang Chamaria
 */

public class MessagingHandlerImpl implements MessagingHandler {

  private static MessagingHandlerImpl _INSTANCE;

  public MessagingHandlerImpl() {
    _INSTANCE = this;
  }

  public static MessagingHandlerImpl getInstance() {
    if (_INSTANCE == null) {
      new MessagingHandlerImpl();
    }
    return _INSTANCE;
  }

  @Override public void scheduleAndSyncMessages(Context context) {
    scheduleAndSyncMessages(context, null, null);
  }

  @Override public void parsePayloadAndShowPush(Context context, @Nullable JSONObject payload) {
    try {
      boolean isAppOpen = false;
      if (payload != null) {
        if (payload.has(MoEMessagingConstants.PARAM_IS_APP_OPEN)) {
          isAppOpen = payload.getBoolean(MoEMessagingConstants.PARAM_IS_APP_OPEN);
        }
        if (payload.has(MoEMessagingConstants.ATTR_RESPONSE_MESSAGE)) {
          JSONArray messageArray =
              payload.getJSONArray(MoEMessagingConstants.ATTR_RESPONSE_MESSAGE);
          for (int i = 0; i < messageArray.length(); i++) {
            JSONObject message = messageArray.getJSONObject(i);
            if (message.has(MoEMessagingConstants.ATTR_RESPONSE_DATA)) {
              JSONObject jsonObject =
                  message.getJSONObject(MoEMessagingConstants.ATTR_RESPONSE_DATA);
              Bundle bundle = jsonToBundle(jsonObject);
              if (bundle != null) {
                addSourceKeyToBundle(bundle);
                addForegroundKeyToBundle(bundle, isAppOpen);
                PushHandler handler = PushManager.getInstance().getPushHandler();
                if (handler != null) {
                  handler.handlePushPayload(context, bundle);
                }
              }
            }
          }
        }
      }
    } catch (Exception e) {
      Logger.f("MessagingHandleImpl#parsePayloadAndShowPush: Server Response Parsing exception ",
          e);
    }
  }

  @Override public void forceMessageSync(Context context, boolean isAppOpen) {
   forceMessageSync(context, isAppOpen, null, null);
  }

  @Override public void scheduleMessageSync(Context context) {
    try {
      if (!ConfigurationProvider.getInstance(context).isAppEnabled()
          || !ConfigurationProvider.getInstance(context).isPushAmpEnabled()) return;
      long minimumDelayDuration =
          ConfigurationProvider.getInstance(context).getPushAmpFetchDelayDuration();
      Intent alarmIntent = new Intent(context, MoEMessageSyncReceiver.class);
      alarmIntent.setAction(MoEMessagingConstants.ACTION_SYNC_MESSAGES);
      PendingIntent pendingIntent =
          PendingIntent.getBroadcast(context, MoEMessagingConstants.REQ_CODE_FETCH_MESSAGES,
              alarmIntent, PendingIntent.FLAG_UPDATE_CURRENT);
      AlarmManager alarmManager = (AlarmManager) context.getSystemService(Context.ALARM_SERVICE);
      if (Build.VERSION.SDK_INT >= VERSION_CODES.LOLLIPOP) {
        scheduleFetchJob(context, minimumDelayDuration);
      }else{
          alarmManager.set(AlarmManager.RTC_WAKEUP, System.currentTimeMillis() + minimumDelayDuration,
              pendingIntent);
      }
    } catch (Exception e) {
      Logger.f("MessagingHandleImpl#scheduleMessageSync: Exception ", e);
    }
  }

  @TargetApi(VERSION_CODES.LOLLIPOP)
  private void scheduleFetchJob(Context context, long minimumDelayDuration) {
    ComponentName serviceComponent = new ComponentName(context, MoEMessageSyncJob.class);
    Builder builder = new Builder(MoEMessagingConstants.MESSAGE_SYNC_JOB_ID, serviceComponent);
    builder.setOverrideDeadline(System.currentTimeMillis() + minimumDelayDuration + (3600 *1000));
    builder.setMinimumLatency(minimumDelayDuration);
    builder.setRequiredNetworkType(JobInfo.NETWORK_TYPE_ANY);
    if (MoEHelperUtils.hasPermission(context, permission.RECEIVE_BOOT_COMPLETED)){
      builder.setPersisted(true);
    }
    JobScheduler jobScheduler =
        (JobScheduler) context.getSystemService(Context.JOB_SCHEDULER_SERVICE);
      jobScheduler.schedule(builder.build());
  }

  @Override public void saveCampaignId(Context context, String campaignId) {
    try {
      ContentValues values = new ContentValues();
      values.put(CampaignListEntity.CAMPAIGN_ID, campaignId);
      values.put(CampaignListEntity.CAMPAIGN_ID_TTL,
          System.currentTimeMillis() + MoEMessagingConstants.DEFAULT_CAMPAIGN_TTL);
      context.getContentResolver().insert(CampaignListEntity.getContentUri(context), values);
    } catch (Exception e) {
      Logger.f("MessagingHandlerImpl#saveCampaignId: Exception : ", e);
    }
  }

  @Nullable private Bundle jsonToBundle(JSONObject json) {
    try {
      Bundle bundle = new Bundle();
      Iterator iter = json.keys();
      while (iter.hasNext()) {
        String key = (String) iter.next();
        String value = json.getString(key);
        bundle.putString(key, value);
      }
      return bundle;
    } catch (Exception e) {
      Logger.f("MessagingHandlerImpl#jsonToBundle : JSON Parsing Error", e);
    }
    return null;
  }

  private void addSourceKeyToBundle(Bundle bundle) {
    bundle.putString("received_from", "remote_inbox");
  }

  private void addForegroundKeyToBundle(Bundle bundle, boolean isAppOpen) {
    bundle.putBoolean("from_appOpen", isAppOpen);
  }

  void scheduleAndSyncMessages(Context context, @Nullable OnJobComplete onJobCompleteCallback,
      @Nullable JobParameters jobParameters) {
    try {
      Logger.v( "MessagingHandlerImpl scheduleAndSyncMessages() : will try to sync and ");
      if (!ConfigurationProvider.getInstance(context).isAppEnabled()
          || !ConfigurationProvider.getInstance(context).isPushAmpEnabled()) {
        return;
      }
      forceMessageSync(context, false, onJobCompleteCallback, jobParameters);
      scheduleMessageSync(context);
    } catch (Exception e) {
      Logger.f(" MessagingHandlerImpl: scheduleAndSyncMessages() : Exception ", e);
    }
  }

  private void forceMessageSync(Context context, boolean isAppOpen, @Nullable OnJobComplete
      onJobCompleteCallback, @Nullable JobParameters parameters){
    try {
      Logger.v("MessagingHandlerImpl forceMessageSync(): Will try to force sync messages ");
      if (!ConfigurationProvider.getInstance(context).isAppEnabled()
          || !ConfigurationProvider.getInstance(context).isPushAmpEnabled()) return;
      HashMap<String, String> paramsMap = new HashMap<>();
      long lastSyncTime = ConfigurationProvider.getInstance(context).getLastMessageFetchTime();
      paramsMap.put(MoEConstants.PARAM_LAST_UPDATED, Long.toString(lastSyncTime));
      if (isAppOpen) paramsMap.put(MoEMessagingConstants.PARAM_IS_APP_OPEN, "true");
      MoEDispatcher.getInstance(context)
          .addTaskToQueue(
              new FetchMessagesFromServerTask(context, MoEUtils.getAPIRoute(context) +
                  MoEMessagingConstants.MESSAGES_FETCH_ROUTE, paramsMap,
                  onJobCompleteCallback, parameters));
    } catch (Exception e) {
      Logger.f("MessagingHandleImpl: forceMessageSync() : Exception ", e);
    }
  }
}
