package com.miquido.securitylib.util

import android.Manifest
import android.annotation.SuppressLint
import android.annotation.TargetApi
import android.content.Context
import android.content.Intent
import android.content.pm.PackageManager
import android.content.pm.ResolveInfo
import android.os.Build
import android.support.v4.content.ContextCompat
import android.telephony.TelephonyManager
import android.text.TextUtils
import android.util.Log
import com.miquido.securitylib.util.EmulatorDetector.Companion.ANDY_FILES
import com.miquido.securitylib.util.EmulatorDetector.Companion.GENY_FILES
import com.miquido.securitylib.util.EmulatorDetector.Companion.NOX_FILES
import com.miquido.securitylib.util.EmulatorDetector.Companion.PIPES
import com.miquido.securitylib.util.EmulatorDetector.Companion.X86_FILES

import java.io.File
import java.io.FileInputStream
import java.io.InputStream
import java.lang.reflect.Method
import java.util.ArrayList

/**
 * Copyright 2016 Framgia, Inc.
 *
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 *
 * Created by Pham Quy Hai on 5/16/16.
 */

@TargetApi(Build.VERSION_CODES.ICE_CREAM_SANDWICH)
class EmulatorDetector private constructor(private val mContext: Context) {
    private var isDebug = false
    private var isCheckPackage = true
    private val mListPackageName = ArrayList<String>()

    val packageNameList: List<String>
        get() = this.mListPackageName

    private val isSupportTelePhony: Boolean
        get() {
            val packageManager = mContext.packageManager
            val isSupport = packageManager.hasSystemFeature(PackageManager.FEATURE_TELEPHONY)
            log("Supported TelePhony: " + isSupport)
            return isSupport
        }

    interface OnEmulatorDetectorListener {
        fun onResult(isEmulator: Boolean)
    }

    init {
        mListPackageName.add("com.google.android.launcher.layouts.genymotion")
        mListPackageName.add("com.bluestacks")
        mListPackageName.add("com.bignox.app")
    }

    fun setDebug(isDebug: Boolean): EmulatorDetector {
        this.isDebug = isDebug
        return this
    }

    fun isDebug(): Boolean {
        return isDebug
    }

    fun isCheckPackage(): Boolean {
        return isCheckPackage
    }

    fun setCheckPackage(chkPackage: Boolean): EmulatorDetector {
        this.isCheckPackage = chkPackage
        return this
    }

    fun addPackageName(pPackageName: String): EmulatorDetector {
        this.mListPackageName.add(pPackageName)
        return this
    }

    fun addPackageName(pListPackageName: List<String>): EmulatorDetector {
        this.mListPackageName.addAll(pListPackageName)
        return this
    }

    fun detect(pOnEmulatorDetectorListener: OnEmulatorDetectorListener?) {
        Thread(Runnable {
            val isEmulator = detect()
            log("This System is Emulator: " + isEmulator)
            pOnEmulatorDetectorListener?.onResult(isEmulator)
        }).start()
    }

    private fun detect(): Boolean {
        var result = false

        log(deviceInfo)

        // Check Basic
        if (!result) {
            result = checkBasic()
            log("Check basic " + result)
        }

        // Check Advanced
        if (!result) {
            result = checkAdvanced()
            log("Check Advanced " + result)
        }

        // Check Package Name
        if (!result) {
            result = checkPackageName()
            log("Check Package Name " + result)
        }

        return result
    }

    private fun checkBasic(): Boolean {
        var result = (Build.FINGERPRINT.startsWith("generic")
                || Build.MODEL.contains("google_sdk")
                || Build.MODEL.toLowerCase().contains("droid4x")
                || Build.MODEL.contains("Emulator")
                || Build.MODEL.contains("Android SDK built for x86")
                || Build.MANUFACTURER.contains("Genymotion")
                || Build.HARDWARE == "goldfish"
                || Build.HARDWARE == "vbox86"
                || Build.PRODUCT == "sdk"
                || Build.PRODUCT == "google_sdk"
                || Build.PRODUCT == "sdk_x86"
                || Build.PRODUCT == "vbox86p"
                || Build.BOARD.toLowerCase().contains("nox")
                || Build.BOOTLOADER.toLowerCase().contains("nox")
                || Build.HARDWARE.toLowerCase().contains("nox")
                || Build.PRODUCT.toLowerCase().contains("nox")
                || Build.SERIAL.toLowerCase().contains("nox"))

        if (result) return true
        result = result or (Build.BRAND.startsWith("generic") && Build.DEVICE.startsWith("generic"))
        if (result) return true
        result = result or ("google_sdk" == Build.PRODUCT)
        return result
    }

    private fun checkAdvanced(): Boolean {
        return (checkFiles(GENY_FILES, "Geny") ||
                checkFiles(ANDY_FILES, "Andy") ||
                checkFiles(NOX_FILES, "Nox") ||
                checkQEmuDrivers() ||
                checkFiles(PIPES, "Pipes") ||
                checkIp() ||
                checkQEmuProps() && checkFiles(X86_FILES, "X86"))
    }

    private fun checkPackageName(): Boolean {
        if (!isCheckPackage || mListPackageName.isEmpty()) {
            return false
        }
        val packageManager = mContext.packageManager
        for (pkgName in mListPackageName) {
            val tryIntent = packageManager.getLaunchIntentForPackage(pkgName)
            if (tryIntent != null) {
                val resolveInfos = packageManager.queryIntentActivities(tryIntent, PackageManager.MATCH_DEFAULT_ONLY)
                if (!resolveInfos.isEmpty()) {
                    return true
                }
            }
        }
        return false
    }


    private fun checkQEmuDrivers(): Boolean {
        for (drivers_file in arrayOf(File("/proc/tty/drivers"), File("/proc/cpuinfo"))) {
            if (drivers_file.exists() && drivers_file.canRead()) {
                val data = ByteArray(1024)
                try {
                    val `is` = FileInputStream(drivers_file)
                    `is`.read(data)
                    `is`.close()
                } catch (exception: Exception) {
                    exception.printStackTrace()
                }

                val driver_data = String(data)
                for (known_qemu_driver in QEMU_DRIVERS) {
                    if (driver_data.contains(known_qemu_driver)) {
                        log("Check QEmuDrivers is detected")
                        return true
                    }
                }
            }
        }

        return false
    }

    private fun checkFiles(targets: Array<String>, type: String): Boolean {
        for (pipe in targets) {
            val qemu_file = File(pipe)
            if (qemu_file.exists()) {
                log("Check $type is detected")
                return true
            }
        }
        return false
    }

    private fun checkQEmuProps(): Boolean {
        var found_props = 0

        for (property in PROPERTIES) {
            val property_value = getProp(mContext, property.name)
            if (property.seek_value == null && property_value != null) {
                found_props++
            }
            if (property.seek_value != null && property_value!!.contains(property.seek_value!!)) {
                found_props++
            }

        }

        if (found_props >= MIN_PROPERTIES_THRESHOLD) {
            log("Check QEmuProps is detected")
            return true
        }
        return false
    }

    private fun checkIp(): Boolean {
        var ipDetected = false
        if (ContextCompat.checkSelfPermission(mContext, Manifest.permission.INTERNET) == PackageManager.PERMISSION_GRANTED) {
            val args = arrayOf("/system/bin/netcfg")
            val stringBuilder = StringBuilder()
            try {
                val builder = ProcessBuilder(*args)
                builder.directory(File("/system/bin/"))
                builder.redirectErrorStream(true)
                val process = builder.start()
                val `in` = process.inputStream
                val re = ByteArray(1024)
                while (`in`.read(re) != -1) {
                    stringBuilder.append(String(re))
                }
                `in`.close()

            } catch (ex: Exception) {
                // empty catch
            }

            val netData = stringBuilder.toString()
            log("netcfg data -> " + netData)

            if (!TextUtils.isEmpty(netData)) {
                val array = netData.split("\n".toRegex()).dropLastWhile { it.isEmpty() }.toTypedArray()

                for (lan in array) {
                    if ((lan.contains("wlan0") || lan.contains("tunl0") || lan.contains("eth0")) && lan.contains(IP)) {
                        ipDetected = true
                        log("Check IP is detected")
                        break
                    }
                }

            }
        }
        return ipDetected
    }

    private fun getProp(context: Context, property: String): String? {
        try {
            val classLoader = context.classLoader
            val systemProperties = classLoader.loadClass("android.os.SystemProperties")

            val get = systemProperties.getMethod("get", String::class.java)

            val params = arrayOfNulls<Any>(1)
            params[0] = property

            return get.invoke(systemProperties, *params) as String
        } catch (exception: Exception) {
            // empty catch
        }

        return null
    }

    private fun log(str: String) {
        if (this.isDebug) {
            Log.d(javaClass.name, str)
        }
    }

    private class Property(var name: String, var seek_value: String?)

    companion object {

        private val GENY_FILES = arrayOf("/dev/socket/genyd", "/dev/socket/baseband_genyd")

        private val QEMU_DRIVERS = arrayOf("goldfish")

        private val PIPES = arrayOf("/dev/socket/qemud", "/dev/qemu_pipe")

        private val X86_FILES = arrayOf("ueventd.android_x86.rc", "x86.prop", "ueventd.ttVM_x86.rc", "init.ttVM_x86.rc", "fstab.ttVM_x86", "fstab.vbox86", "init.vbox86.rc", "ueventd.vbox86.rc")

        private val ANDY_FILES = arrayOf("fstab.andy", "ueventd.andy.rc")

        private val NOX_FILES = arrayOf("fstab.nox", "init.nox.rc", "ueventd.nox.rc")

        private val PROPERTIES = arrayOf(Property("init.svc.qemud", null), Property("init.svc.qemu-props", null), Property("qemu.hw.mainkeys", null), Property("qemu.sf.fake_camera", null), Property("qemu.sf.lcd_density", null), Property("ro.bootloader", "unknown"), Property("ro.bootmode", "unknown"), Property("ro.hardware", "goldfish"), Property("ro.kernel.android.qemud", null), Property("ro.kernel.qemu.gles", null), Property("ro.kernel.qemu", "1"), Property("ro.product.device", "generic"), Property("ro.product.model", "sdk"), Property("ro.product.name", "sdk"), Property("ro.serialno", null))

        private val IP = "10.0.2.15"

        private val MIN_PROPERTIES_THRESHOLD = 0x5

        @SuppressLint("StaticFieldLeak") //Since we use application context now this won't leak memory anymore. This is only to please Lint
        private var mEmulatorDetector: EmulatorDetector? = null

        fun with(pContext: Context?): EmulatorDetector {
            if (pContext == null) {
                throw IllegalArgumentException("Context must not be null.")
            }
            if (mEmulatorDetector == null)
                mEmulatorDetector = EmulatorDetector(pContext.applicationContext)
            return mEmulatorDetector!!
        }

        val deviceInfo: String
            get() = "Build.PRODUCT: " + Build.PRODUCT + "\n" +
                    "Build.MANUFACTURER: " + Build.MANUFACTURER + "\n" +
                    "Build.BRAND: " + Build.BRAND + "\n" +
                    "Build.DEVICE: " + Build.DEVICE + "\n" +
                    "Build.MODEL: " + Build.MODEL + "\n" +
                    "Build.HARDWARE: " + Build.HARDWARE + "\n" +
                    "Build.FINGERPRINT: " + Build.FINGERPRINT
    }
}