/*
 * Decompiled with CFR 0.152.
 */
package com.subgraph.orchid.directory;

import com.subgraph.orchid.Descriptor;
import com.subgraph.orchid.DirectoryStore;
import com.subgraph.orchid.Threading;
import com.subgraph.orchid.data.HexDigest;
import com.subgraph.orchid.directory.DescriptorCacheData;
import com.subgraph.orchid.directory.parsing.DocumentParser;
import com.subgraph.orchid.directory.parsing.DocumentParsingResult;
import com.subgraph.orchid.misc.GuardedBy;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.logging.Logger;

public abstract class DescriptorCache<T extends Descriptor> {
    private static final Logger logger = Logger.getLogger(DescriptorCache.class.getName());
    private final DescriptorCacheData<T> data;
    private final DirectoryStore store;
    private final ScheduledExecutorService rebuildExecutor = Threading.newScheduledPool("DescriptorCache rebuild worker");
    private final DirectoryStore.CacheFile cacheFile;
    private final DirectoryStore.CacheFile journalFile;
    @GuardedBy(value="this")
    private int droppedBytes;
    @GuardedBy(value="this")
    private int journalLength;
    @GuardedBy(value="this")
    private int cacheLength;
    @GuardedBy(value="this")
    private boolean initiallyLoaded;

    DescriptorCache(DirectoryStore store, DirectoryStore.CacheFile cacheFile, DirectoryStore.CacheFile journalFile) {
        this.data = new DescriptorCacheData();
        this.store = store;
        this.cacheFile = cacheFile;
        this.journalFile = journalFile;
        this.startRebuildTask();
    }

    public synchronized void initialLoad() {
        if (this.initiallyLoaded) {
            return;
        }
        this.reloadCache();
    }

    public void shutdown() {
        this.rebuildExecutor.shutdownNow();
    }

    public T getDescriptor(HexDigest digest) {
        return this.data.findByDigest(digest);
    }

    public synchronized void addDescriptors(List<T> descriptors) {
        ArrayList<Descriptor> journalDescriptors = new ArrayList<Descriptor>();
        int duplicateCount = 0;
        for (Descriptor d : descriptors) {
            if (this.data.addDescriptor(d)) {
                if (d.getCacheLocation() != Descriptor.CacheLocation.NOT_CACHED) continue;
                this.journalLength += d.getBodyLength();
                journalDescriptors.add(d);
                continue;
            }
            ++duplicateCount;
        }
        if (!journalDescriptors.isEmpty()) {
            this.store.appendDocumentList(this.journalFile, journalDescriptors);
        }
        if (duplicateCount > 0) {
            logger.info("Duplicate descriptors added to journal, count = " + duplicateCount);
        }
    }

    public void addDescriptor(T d) {
        ArrayList<T> descriptors = new ArrayList<T>();
        descriptors.add(d);
        this.addDescriptors(descriptors);
    }

    private synchronized void clearMemoryCache() {
        this.data.clear();
        this.journalLength = 0;
        this.cacheLength = 0;
        this.droppedBytes = 0;
    }

    private synchronized void reloadCache() {
        this.clearMemoryCache();
        ByteBuffer[] buffers = this.loadCacheBuffers();
        this.loadCacheFileBuffer(buffers[0]);
        this.loadJournalFileBuffer(buffers[1]);
        if (!this.initiallyLoaded) {
            this.initiallyLoaded = true;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private ByteBuffer[] loadCacheBuffers() {
        DirectoryStore directoryStore = this.store;
        synchronized (directoryStore) {
            ByteBuffer[] buffers = new ByteBuffer[]{this.store.loadCacheFile(this.cacheFile), this.store.loadCacheFile(this.journalFile)};
            return buffers;
        }
    }

    private void loadCacheFileBuffer(ByteBuffer buffer) {
        this.cacheLength = buffer.limit();
        if (this.cacheLength == 0) {
            return;
        }
        DocumentParser<T> parser = this.createDocumentParser(buffer);
        DocumentParsingResult<T> result = parser.parse();
        if (result.isOkay()) {
            for (Descriptor d : result.getParsedDocuments()) {
                d.setCacheLocation(Descriptor.CacheLocation.CACHED_CACHEFILE);
                this.data.addDescriptor(d);
            }
        }
    }

    private void loadJournalFileBuffer(ByteBuffer buffer) {
        this.journalLength = buffer.limit();
        if (this.journalLength == 0) {
            return;
        }
        DocumentParser<T> parser = this.createDocumentParser(buffer);
        DocumentParsingResult<T> result = parser.parse();
        if (result.isOkay()) {
            int duplicateCount = 0;
            logger.fine("Loaded " + result.getParsedDocuments().size() + " descriptors from journal");
            for (Descriptor d : result.getParsedDocuments()) {
                d.setCacheLocation(Descriptor.CacheLocation.CACHED_JOURNAL);
                if (this.data.addDescriptor(d)) continue;
                ++duplicateCount;
            }
            if (duplicateCount > 0) {
                logger.info("Found " + duplicateCount + " duplicate descriptors in journal file");
            }
        } else if (result.isInvalid()) {
            logger.warning("Invalid descriptor data parsing from journal file : " + result.getMessage());
        } else if (result.isError()) {
            logger.warning("Error parsing descriptors from journal file : " + result.getMessage());
        }
    }

    protected abstract DocumentParser<T> createDocumentParser(ByteBuffer var1);

    private ScheduledFuture<?> startRebuildTask() {
        return this.rebuildExecutor.scheduleAtFixedRate(new Runnable(){

            @Override
            public void run() {
                DescriptorCache.this.maybeRebuildCache();
            }
        }, 5L, 30L, TimeUnit.MINUTES);
    }

    private synchronized void maybeRebuildCache() {
        if (!this.initiallyLoaded) {
            return;
        }
        this.droppedBytes += this.data.cleanExpired();
        if (!this.shouldRebuildCache()) {
            return;
        }
        this.rebuildCache();
    }

    private boolean shouldRebuildCache() {
        if (this.journalLength < 16384) {
            return false;
        }
        if (this.droppedBytes > (this.journalLength + this.cacheLength) / 3) {
            return true;
        }
        return this.journalLength > this.cacheLength / 2;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void rebuildCache() {
        DirectoryStore directoryStore = this.store;
        synchronized (directoryStore) {
            this.store.writeDocumentList(this.cacheFile, this.data.getAllDescriptors());
            this.store.removeCacheFile(this.journalFile);
        }
        this.reloadCache();
    }
}

