// This file is generated by /utils/generate_types/index.js
/**
 * Copyright (c) Microsoft Corporation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import type { Browser, BrowserContext, BrowserContextOptions, Page, LaunchOptions, ViewportSize, Geolocation, HTTPCredentials } from './types';
import type { Expect } from './testExpect';

export type { Expect } from './testExpect';

export type ReporterDescription =
  ['dot'] |
  ['line'] |
  ['list'] |
  ['junit'] | ['junit', { outputFile?: string, stripANSIControlSequences?: boolean }] |
  ['json'] | ['json', { outputFile?: string }] |
  ['null'] |
  [string] | [string, any];

export type Shard = { total: number, current: number } | null;
export type ReportSlowTests = { max: number, threshold: number } | null;
export type PreserveOutput = 'always' | 'never' | 'failures-only';
export type UpdateSnapshots = 'all' | 'none' | 'missing';

type FixtureDefine<TestArgs extends KeyValue = {}, WorkerArgs extends KeyValue = {}> = { test: TestType<TestArgs, WorkerArgs>, fixtures: Fixtures<{}, {}, TestArgs, WorkerArgs> };

type ExpectSettings = {
  // Default timeout for async expect matchers in milliseconds, defaults to 5000ms.
  timeout?: number;
  toMatchSnapshot?: {
    // Pixel match threshold.
    threshold?: number
  }
};

/**
 * Playwright Test supports running multiple test projects at the same time. This is useful for running tests in multiple
 * configurations. For example, consider running tests against multiple browsers.
 *
 * `TestProject` encapsulates configuration specific to a single project. Projects are configured in
 * [testConfig.projects](https://playwright.dev/docs/api/class-testconfig#test-config-projects) specified in the
 * [configuration file](https://playwright.dev/docs/test-configuration). Note that all properties of [TestProject] are available in the top-level
 * [TestConfig], in which case they are shared between all projects.
 *
 * Here is an example configuration that runs every test in Chromium, Firefox and WebKit, both Desktop and Mobile versions.
 *
 * ```js js-flavor=js
 * // playwright.config.js
 * // @ts-check
 * const { devices } = require('@playwright/test');
 *
 * /** @type {import('@playwright/test').PlaywrightTestConfig} *\/
 * const config = {
 *   // Options shared for all projects.
 *   timeout: 30000,
 *   use: {
 *     ignoreHTTPSErrors: true,
 *   },
 *
 *   // Options specific to each project.
 *   projects: [
 *     {
 *       name: 'Desktop Chromium',
 *       use: {
 *         browserName: 'chromium',
 *         viewport: { width: 1280, height: 720 },
 *       },
 *     },
 *     {
 *       name: 'Desktop Safari',
 *       use: {
 *         browserName: 'webkit',
 *         viewport: { width: 1280, height: 720 },
 *       }
 *     },
 *     {
 *       name: 'Desktop Firefox',
 *       use: {
 *         browserName: 'firefox',
 *         viewport: { width: 1280, height: 720 },
 *       }
 *     },
 *     {
 *       name: 'Mobile Chrome',
 *       use: devices['Pixel 5'],
 *     },
 *     {
 *       name: 'Mobile Safari',
 *       use: devices['iPhone 12'],
 *     },
 *   ],
 * };
 *
 * module.exports = config;
 * ```
 *
 * ```js js-flavor=ts
 * // playwright.config.ts
 * import { PlaywrightTestConfig, devices } from '@playwright/test';
 *
 * const config: PlaywrightTestConfig = {
 *   // Options shared for all projects.
 *   timeout: 30000,
 *   use: {
 *     ignoreHTTPSErrors: true,
 *   },
 *
 *   // Options specific to each project.
 *   projects: [
 *     {
 *       name: 'Desktop Chromium',
 *       use: {
 *         browserName: 'chromium',
 *         viewport: { width: 1280, height: 720 },
 *       },
 *     },
 *     {
 *       name: 'Desktop Safari',
 *       use: {
 *         browserName: 'webkit',
 *         viewport: { width: 1280, height: 720 },
 *       }
 *     },
 *     {
 *       name: 'Desktop Firefox',
 *       use: {
 *         browserName: 'firefox',
 *         viewport: { width: 1280, height: 720 },
 *       }
 *     },
 *     {
 *       name: 'Mobile Chrome',
 *       use: devices['Pixel 5'],
 *     },
 *     {
 *       name: 'Mobile Safari',
 *       use: devices['iPhone 12'],
 *     },
 *   ],
 * };
 * export default config;
 * ```
 *
 */
interface TestProject {
  /**
   * Configuration for the `expect` assertion library.
   */
  expect?: ExpectSettings;
  /**
   * Any JSON-serializable metadata that will be put directly to the test report.
   */
  metadata?: any;
  /**
   * Project name is visible in the report and during test execution.
   */
  name?: string;
  /**
   * The output directory for files created during test execution. Defaults to `test-results`.
   *
   * This directory is cleaned at the start. When running a test, a unique subdirectory inside the
   * [testProject.outputDir](https://playwright.dev/docs/api/class-testproject#test-project-output-dir) is created,
   * guaranteeing that test running in parallel do not conflict. This directory can be accessed by
   * [testInfo.outputDir](https://playwright.dev/docs/api/class-testinfo#test-info-output-dir) and
   * [testInfo.outputPath(pathSegments)](https://playwright.dev/docs/api/class-testinfo#test-info-output-path).
   *
   * Here is an example that uses
   * [testInfo.outputPath(pathSegments)](https://playwright.dev/docs/api/class-testinfo#test-info-output-path) to create a
   * temporary file.
   *
   * ```js js-flavor=js
   * const { test, expect } = require('@playwright/test');
   * const fs = require('fs');
   *
   * test('example test', async ({}, testInfo) => {
   *   const file = testInfo.outputPath('temporary-file.txt');
   *   await fs.promises.writeFile(file, 'Put some data to the file', 'utf8');
   * });
   * ```
   *
   * ```js js-flavor=ts
   * import { test, expect } from '@playwright/test';
   * import fs from 'fs';
   *
   * test('example test', async ({}, testInfo) => {
   *   const file = testInfo.outputPath('temporary-file.txt');
   *   await fs.promises.writeFile(file, 'Put some data to the file', 'utf8');
   * });
   * ```
   *
   */
  outputDir?: string;
  /**
   * The number of times to repeat each test, useful for debugging flaky tests.
   */
  repeatEach?: number;
  /**
   * The maximum number of retry attempts given to failed tests. Learn more about [test retries](https://playwright.dev/docs/test-retries).
   */
  retries?: number;
  /**
   * Directory that will be recursively scanned for test files. Defaults to the directory of the configuration file.
   *
   * Each project can use a different directory. Here is an example that runs smoke tests in three browsers and all other
   * tests in stable Chrome browser.
   *
   * ```js js-flavor=js
   * // playwright.config.js
   * // @ts-check
   *
   * /** @type {import('@playwright/test').PlaywrightTestConfig} *\/
   * const config = {
   *   projects: [
   *     {
   *       name: 'Smoke Chromium',
   *       testDir: './smoke-tests',
   *       use: {
   *         browserName: 'chromium',
   *       }
   *     },
   *     {
   *       name: 'Smoke WebKit',
   *       testDir: './smoke-tests',
   *       use: {
   *         browserName: 'webkit',
   *       }
   *     },
   *     {
   *       name: 'Smoke Firefox',
   *       testDir: './smoke-tests',
   *       use: {
   *         browserName: 'firefox',
   *       }
   *     },
   *     {
   *       name: 'Chrome Stable',
   *       testDir: './',
   *       use: {
   *         browserName: 'chromium',
   *         channel: 'chrome',
   *       }
   *     },
   *   ],
   * };
   *
   * module.exports = config;
   * ```
   *
   * ```js js-flavor=ts
   * // playwright.config.ts
   * import { PlaywrightTestConfig } from '@playwright/test';
   *
   * const config: PlaywrightTestConfig = {
   *   projects: [
   *     {
   *       name: 'Smoke Chromium',
   *       testDir: './smoke-tests',
   *       use: {
   *         browserName: 'chromium',
   *       }
   *     },
   *     {
   *       name: 'Smoke WebKit',
   *       testDir: './smoke-tests',
   *       use: {
   *         browserName: 'webkit',
   *       }
   *     },
   *     {
   *       name: 'Smoke Firefox',
   *       testDir: './smoke-tests',
   *       use: {
   *         browserName: 'firefox',
   *       }
   *     },
   *     {
   *       name: 'Chrome Stable',
   *       testDir: './',
   *       use: {
   *         browserName: 'chromium',
   *         channel: 'chrome',
   *       }
   *     },
   *   ],
   * };
   * export default config;
   * ```
   *
   */
  testDir?: string;
  /**
   * Files matching one of these patterns are not executed as test files. Matching is performed against the absolute file
   * path. Strings are treated as glob patterns.
   *
   * For example, `'**\/test-assets/**'` will ignore any files in the `test-assets` directory.
   */
  testIgnore?: string | RegExp | (string | RegExp)[];
  /**
   * Only the files matching one of these patterns are executed as test files. Matching is performed against the absolute
   * file path. Strings are treated as glob patterns.
   *
   * By default, Playwright Test looks for files matching `.*(test|spec)\.(js|ts|mjs)`.
   */
  testMatch?: string | RegExp | (string | RegExp)[];
  /**
   * Timeout for each test in milliseconds. Defaults to 30 seconds.
   *
   * This is a base timeout for all tests. In addition, each test can configure its own timeout with
   * [test.setTimeout(timeout)](https://playwright.dev/docs/api/class-test#test-set-timeout).
   */
  timeout?: number;
}

/**
 * Playwright Test supports running multiple test projects at the same time. This is useful for running tests in multiple
 * configurations. For example, consider running tests against multiple browsers.
 *
 * `TestProject` encapsulates configuration specific to a single project. Projects are configured in
 * [testConfig.projects](https://playwright.dev/docs/api/class-testconfig#test-config-projects) specified in the
 * [configuration file](https://playwright.dev/docs/test-configuration). Note that all properties of [TestProject] are available in the top-level
 * [TestConfig], in which case they are shared between all projects.
 *
 * Here is an example configuration that runs every test in Chromium, Firefox and WebKit, both Desktop and Mobile versions.
 *
 * ```js js-flavor=js
 * // playwright.config.js
 * // @ts-check
 * const { devices } = require('@playwright/test');
 *
 * /** @type {import('@playwright/test').PlaywrightTestConfig} *\/
 * const config = {
 *   // Options shared for all projects.
 *   timeout: 30000,
 *   use: {
 *     ignoreHTTPSErrors: true,
 *   },
 *
 *   // Options specific to each project.
 *   projects: [
 *     {
 *       name: 'Desktop Chromium',
 *       use: {
 *         browserName: 'chromium',
 *         viewport: { width: 1280, height: 720 },
 *       },
 *     },
 *     {
 *       name: 'Desktop Safari',
 *       use: {
 *         browserName: 'webkit',
 *         viewport: { width: 1280, height: 720 },
 *       }
 *     },
 *     {
 *       name: 'Desktop Firefox',
 *       use: {
 *         browserName: 'firefox',
 *         viewport: { width: 1280, height: 720 },
 *       }
 *     },
 *     {
 *       name: 'Mobile Chrome',
 *       use: devices['Pixel 5'],
 *     },
 *     {
 *       name: 'Mobile Safari',
 *       use: devices['iPhone 12'],
 *     },
 *   ],
 * };
 *
 * module.exports = config;
 * ```
 *
 * ```js js-flavor=ts
 * // playwright.config.ts
 * import { PlaywrightTestConfig, devices } from '@playwright/test';
 *
 * const config: PlaywrightTestConfig = {
 *   // Options shared for all projects.
 *   timeout: 30000,
 *   use: {
 *     ignoreHTTPSErrors: true,
 *   },
 *
 *   // Options specific to each project.
 *   projects: [
 *     {
 *       name: 'Desktop Chromium',
 *       use: {
 *         browserName: 'chromium',
 *         viewport: { width: 1280, height: 720 },
 *       },
 *     },
 *     {
 *       name: 'Desktop Safari',
 *       use: {
 *         browserName: 'webkit',
 *         viewport: { width: 1280, height: 720 },
 *       }
 *     },
 *     {
 *       name: 'Desktop Firefox',
 *       use: {
 *         browserName: 'firefox',
 *         viewport: { width: 1280, height: 720 },
 *       }
 *     },
 *     {
 *       name: 'Mobile Chrome',
 *       use: devices['Pixel 5'],
 *     },
 *     {
 *       name: 'Mobile Safari',
 *       use: devices['iPhone 12'],
 *     },
 *   ],
 * };
 * export default config;
 * ```
 *
 */
export interface Project<TestArgs = {}, WorkerArgs = {}> extends TestProject {
  define?: FixtureDefine | FixtureDefine[];
  /**
   * Additional fixtures for this project. Most useful for specifying options, for example
   * [fixtures.browserName](https://playwright.dev/docs/api/class-fixtures#fixtures-browser-name). Learn more about
   * [Fixtures] and [configuration](https://playwright.dev/docs/test-configuration).
   *
   * ```js js-flavor=js
   * // playwright.config.js
   * // @ts-check
   *
   * /** @type {import('@playwright/test').PlaywrightTestConfig} *\/
   * const config = {
   *   projects: [
   *     {
   *       name: 'Chromium',
   *       use: {
   *         browserName: 'chromium',
   *       },
   *     },
   *   ],
   * };
   *
   * module.exports = config;
   * ```
   *
   * ```js js-flavor=ts
   * // playwright.config.ts
   * import { PlaywrightTestConfig } from '@playwright/test';
   *
   * const config: PlaywrightTestConfig = {
   *   projects: [
   *     {
   *       name: 'Chromium',
   *       use: {
   *         browserName: 'chromium',
   *       },
   *     },
   *   ],
   * };
   * export default config;
   * ```
   *
   */
  use?: Fixtures<{}, {}, TestArgs, WorkerArgs>;
}

export type FullProject<TestArgs = {}, WorkerArgs = {}> = Required<Project<TestArgs, WorkerArgs>>;

export type WebServerConfig = {
  /**
   * Shell command to start. For example `npm run start`.
   */
  command: string,
  /**
   * The port that your http server is expected to appear on. It does wait until it accepts connections.
   */
  port: number,
  /**
   * How long to wait for the process to start up and be available in milliseconds. Defaults to 60000.
   */
  timeout?: number,
  /**
   * If true, it will re-use an existing server on the port when available. If no server is running
   * on that port, it will run the command to start a new server.
   * If false, it will throw if an existing process is listening on the port.
   * This should commonly set to !process.env.CI to allow the local dev server when running tests locally.
   */
  reuseExistingServer?: boolean
  /**
   * Environment variables, process.env by default
   */
  env?: Record<string, string>,
  /**
   * Current working directory of the spawned process. Default is process.cwd().
   */
  cwd?: string,
};

type LiteralUnion<T extends U, U = string> = T | (U & { zz_IGNORE_ME?: never });

/**
 * Playwright Test provides many options to configure how your tests are collected and executed, for example `timeout` or
 * `testDir`. These options are described in the [TestConfig] object in the [configuration file](https://playwright.dev/docs/test-configuration).
 *
 * Playwright Test supports running multiple test projects at the same time. Project-specific options should be put to
 * [testConfig.projects](https://playwright.dev/docs/api/class-testconfig#test-config-projects), but top-level [TestConfig]
 * can also define base options shared between all projects.
 *
 * ```js js-flavor=js
 * // playwright.config.js
 * // @ts-check
 *
 * /** @type {import('@playwright/test').PlaywrightTestConfig} *\/
 * const config = {
 *   timeout: 30000,
 *   globalTimeout: 600000,
 *   reporter: 'list',
 *   testDir: './tests',
 * };
 *
 * module.exports = config;
 * ```
 *
 * ```js js-flavor=ts
 * // playwright.config.ts
 * import { PlaywrightTestConfig } from '@playwright/test';
 *
 * const config: PlaywrightTestConfig = {
 *   timeout: 30000,
 *   globalTimeout: 600000,
 *   reporter: 'list',
 *   testDir: './tests',
 * };
 * export default config;
 * ```
 *
 */
interface TestConfig {
  /**
   * Whether to exit with an error if any tests or groups are marked as
   * [test.only(title, testFunction)](https://playwright.dev/docs/api/class-test#test-only) or
   * [test.describe.only(title, callback)](https://playwright.dev/docs/api/class-test#test-describe-only). Useful on CI.
   */
  forbidOnly?: boolean;
  /**
   * Path to the global setup file. This file will be required and run before all the tests. It must export a single
   * function.
   *
   * Learn more about [global setup and teardown](https://playwright.dev/docs/test-advanced#global-setup-and-teardown).
   *
   * ```js js-flavor=js
   * // playwright.config.js
   * // @ts-check
   *
   * /** @type {import('@playwright/test').PlaywrightTestConfig} *\/
   * const config = {
   *   globalSetup: './global-setup',
   * };
   *
   * module.exports = config;
   * ```
   *
   * ```js js-flavor=ts
   * // playwright.config.ts
   * import { PlaywrightTestConfig, devices } from '@playwright/test';
   *
   * const config: PlaywrightTestConfig = {
   *   globalSetup: './global-setup',
   * };
   * export default config;
   * ```
   *
   */
  globalSetup?: string;
  /**
   * Path to the global teardown file. This file will be required and run after all the tests. It must export a single
   * function. See also [testConfig.globalSetup](https://playwright.dev/docs/api/class-testconfig#test-config-global-setup).
   *
   * Learn more about [global setup and teardown](https://playwright.dev/docs/test-advanced#global-setup-and-teardown).
   */
  globalTeardown?: string;
  /**
   * Maximum time in milliseconds the whole test suite can run. Zero timeout (default) disables this behavior. Useful on CI
   * to prevent broken setup from running too long and wasting resources.
   */
  globalTimeout?: number;
  /**
   * Filter to only run tests with a title matching one of the patterns. For example, passing `grep: /cart/` should only run
   * tests with "cart" in the title. Also available in the [command line](https://playwright.dev/docs/test-cli) with the `-g` option.
   *
   * `grep` option is also useful for [tagging tests](https://playwright.dev/docs/test-annotations#tag-tests).
   */
  grep?: RegExp | RegExp[];
  /**
   * Filter to only run tests with a title **not** matching one of the patterns. This is the opposite of
   * [testConfig.grep](https://playwright.dev/docs/api/class-testconfig#test-config-grep). Also available in the
   * [command line](https://playwright.dev/docs/test-cli) with the `--grep-invert` option.
   *
   * `grepInvert` option is also useful for [tagging tests](https://playwright.dev/docs/test-annotations#tag-tests).
   */
  grepInvert?: RegExp | RegExp[];
  /**
   * The maximum number of test failures for the whole test suite run. After reaching this number, testing will stop and exit
   * with an error. Setting to zero (default) disables this behavior.
   *
   * Also available in the [command line](https://playwright.dev/docs/test-cli) with the `--max-failures` and `-x` options.
   */
  maxFailures?: number;
  /**
   * Whether to preserve test output in the
   * [testConfig.outputDir](https://playwright.dev/docs/api/class-testconfig#test-config-output-dir). Defaults to `'always'`.
   * - `'always'` - preserve output for all tests;
   * - `'never'` - do not preserve output for any tests;
   * - `'failures-only'` - only preserve output for failed tests.
   */
  preserveOutput?: PreserveOutput;
  /**
   * Playwright Test supports running multiple test projects at the same time. See [TestProject] for more information.
   */
  projects?: Project[];
  /**
   * Whether to suppress stdio and stderr output from the tests.
   */
  quiet?: boolean;
  /**
   * The list of reporters to use. Each reporter can be:
   * - A builtin reporter name like `'list'` or `'json'`.
   * - A module name like `'my-awesome-reporter'`.
   * - A relative path to the reporter like `'./reporters/my-awesome-reporter.js'`.
   *
   * You can pass options to the reporter in a tuple like `['json', { outputFile: './report.json' }]`.
   *
   * Learn more in the [reporters guide](https://playwright.dev/docs/test-reporters).
   *
   * ```js js-flavor=js
   * // playwright.config.js
   * // @ts-check
   *
   * /** @type {import('@playwright/test').PlaywrightTestConfig} *\/
   * const config = {
   *   reporter: 'line',
   * };
   *
   * module.exports = config;
   * ```
   *
   * ```js js-flavor=ts
   * // playwright.config.ts
   * import { PlaywrightTestConfig } from '@playwright/test';
   *
   * const config: PlaywrightTestConfig = {
   *   reporter: 'line',
   * };
   * export default config;
   * ```
   *
   */
  reporter?: LiteralUnion<'list'|'dot'|'line'|'json'|'junit'|'null', string> | ReporterDescription[];
  /**
   * Whether to report slow tests. Pass `null` to disable this feature.
   *
   * Tests that took more than `threshold` milliseconds are considered slow, and the slowest ones are reported, no more than
   * `max` number of them. Passing zero as `max` reports all slow tests that exceed the threshold.
   */
  reportSlowTests?: ReportSlowTests;
  /**
   * Shard tests and execute only the selected shard. Specify in the one-based form like `{ total: 5, current: 2 }`.
   *
   * Learn more about [parallelism and sharding](https://playwright.dev/docs/test-parallel) with Playwright Test.
   */
  shard?: Shard;
  /**
   * Whether to update expected snapshots with the actual results produced by the test run. Defaults to `'missing'`.
   * - `'all'` - All tests that are executed will update snapshots.
   * - `'none'` - No snapshots are updated.
   * - `'missing'` - Missing snapshots are created, for example when authoring a new test and running it for the first
   *   time. This is the default.
   *
   * Learn more about [snapshots](https://playwright.dev/docs/test-snapshots).
   */
  updateSnapshots?: UpdateSnapshots;
  webServer?: WebServerConfig;
  /**
   * The maximum number of concurrent worker processes to use for parallelizing tests.
   *
   * Playwright Test uses worker processes to run tests. There is always at least one worker process, but more can be used to
   * speed up test execution.
   *
   * Defaults to one half of the number of CPU cores. Learn more about [parallelism and sharding](https://playwright.dev/docs/test-parallel) with
   * Playwright Test.
   */
  workers?: number;

  /**
   * Configuration for the `expect` assertion library.
   */
  expect?: ExpectSettings;
  /**
   * Any JSON-serializable metadata that will be put directly to the test report.
   */
  metadata?: any;
  name?: string;
  /**
   * The output directory for files created during test execution. Defaults to `test-results`.
   *
   * This directory is cleaned at the start. When running a test, a unique subdirectory inside the
   * [testConfig.outputDir](https://playwright.dev/docs/api/class-testconfig#test-config-output-dir) is created, guaranteeing
   * that test running in parallel do not conflict. This directory can be accessed by
   * [testInfo.outputDir](https://playwright.dev/docs/api/class-testinfo#test-info-output-dir) and
   * [testInfo.outputPath(pathSegments)](https://playwright.dev/docs/api/class-testinfo#test-info-output-path).
   *
   * Here is an example that uses
   * [testInfo.outputPath(pathSegments)](https://playwright.dev/docs/api/class-testinfo#test-info-output-path) to create a
   * temporary file.
   *
   * ```js js-flavor=js
   * const { test, expect } = require('@playwright/test');
   * const fs = require('fs');
   *
   * test('example test', async ({}, testInfo) => {
   *   const file = testInfo.outputPath('temporary-file.txt');
   *   await fs.promises.writeFile(file, 'Put some data to the file', 'utf8');
   * });
   * ```
   *
   * ```js js-flavor=ts
   * import { test, expect } from '@playwright/test';
   * import fs from 'fs';
   *
   * test('example test', async ({}, testInfo) => {
   *   const file = testInfo.outputPath('temporary-file.txt');
   *   await fs.promises.writeFile(file, 'Put some data to the file', 'utf8');
   * });
   * ```
   *
   */
  outputDir?: string;
  /**
   * The number of times to repeat each test, useful for debugging flaky tests.
   */
  repeatEach?: number;
  /**
   * The maximum number of retry attempts given to failed tests. Learn more about [test retries](https://playwright.dev/docs/test-retries).
   */
  retries?: number;
  /**
   * Directory that will be recursively scanned for test files. Defaults to the directory of the configuration file.
   */
  testDir?: string;
  /**
   * Files matching one of these patterns are not executed as test files. Matching is performed against the absolute file
   * path. Strings are treated as glob patterns.
   *
   * For example, `'**\/test-assets/**'` will ignore any files in the `test-assets` directory.
   */
  testIgnore?: string | RegExp | (string | RegExp)[];
  /**
   * Only the files matching one of these patterns are executed as test files. Matching is performed against the absolute
   * file path. Strings are treated as glob patterns.
   *
   * By default, Playwright Test looks for files matching `.*(test|spec)\.(js|ts|mjs)`.
   */
  testMatch?: string | RegExp | (string | RegExp)[];
  /**
   * Timeout for each test in milliseconds. Defaults to 30 seconds.
   *
   * This is a base timeout for all tests. In addition, each test can configure its own timeout with
   * [test.setTimeout(timeout)](https://playwright.dev/docs/api/class-test#test-set-timeout).
   */
  timeout?: number;
}

/**
 * Playwright Test provides many options to configure how your tests are collected and executed, for example `timeout` or
 * `testDir`. These options are described in the [TestConfig] object in the [configuration file](https://playwright.dev/docs/test-configuration).
 *
 * Playwright Test supports running multiple test projects at the same time. Project-specific options should be put to
 * [testConfig.projects](https://playwright.dev/docs/api/class-testconfig#test-config-projects), but top-level [TestConfig]
 * can also define base options shared between all projects.
 *
 * ```js js-flavor=js
 * // playwright.config.js
 * // @ts-check
 *
 * /** @type {import('@playwright/test').PlaywrightTestConfig} *\/
 * const config = {
 *   timeout: 30000,
 *   globalTimeout: 600000,
 *   reporter: 'list',
 *   testDir: './tests',
 * };
 *
 * module.exports = config;
 * ```
 *
 * ```js js-flavor=ts
 * // playwright.config.ts
 * import { PlaywrightTestConfig } from '@playwright/test';
 *
 * const config: PlaywrightTestConfig = {
 *   timeout: 30000,
 *   globalTimeout: 600000,
 *   reporter: 'list',
 *   testDir: './tests',
 * };
 * export default config;
 * ```
 *
 */
export interface Config<TestArgs = {}, WorkerArgs = {}> extends TestConfig {
  /**
   * Playwright Test supports running multiple test projects at the same time. See [TestProject] for more information.
   */
  projects?: Project<TestArgs, WorkerArgs>[];
  define?: FixtureDefine | FixtureDefine[];
  /**
   * Additional fixtures for this project. Most useful for specifying options, for example
   * [fixtures.browserName](https://playwright.dev/docs/api/class-fixtures#fixtures-browser-name). Learn more about
   * [Fixtures] and [configuration](https://playwright.dev/docs/test-configuration).
   *
   * ```js js-flavor=js
   * // playwright.config.js
   * // @ts-check
   *
   * /** @type {import('@playwright/test').PlaywrightTestConfig} *\/
   * const config = {
   *   use: {
   *     browserName: 'chromium',
   *   },
   * };
   *
   * module.exports = config;
   * ```
   *
   * ```js js-flavor=ts
   * // playwright.config.ts
   * import { PlaywrightTestConfig } from '@playwright/test';
   *
   * const config: PlaywrightTestConfig = {
   *   use: {
   *     browserName: 'chromium',
   *   },
   * };
   * export default config;
   * ```
   *
   */
  use?: Fixtures<{}, {}, TestArgs, WorkerArgs>;
}

/**
 * Playwright Test provides many options to configure how your tests are collected and executed, for example `timeout` or
 * `testDir`. These options are described in the [TestConfig] object in the [configuration file](https://playwright.dev/docs/test-configuration).
 *
 * Playwright Test supports running multiple test projects at the same time. Project-specific options should be put to
 * [testConfig.projects](https://playwright.dev/docs/api/class-testconfig#test-config-projects), but top-level [TestConfig]
 * can also define base options shared between all projects.
 *
 * ```js js-flavor=js
 * // playwright.config.js
 * // @ts-check
 *
 * /** @type {import('@playwright/test').PlaywrightTestConfig} *\/
 * const config = {
 *   timeout: 30000,
 *   globalTimeout: 600000,
 *   reporter: 'list',
 *   testDir: './tests',
 * };
 *
 * module.exports = config;
 * ```
 *
 * ```js js-flavor=ts
 * // playwright.config.ts
 * import { PlaywrightTestConfig } from '@playwright/test';
 *
 * const config: PlaywrightTestConfig = {
 *   timeout: 30000,
 *   globalTimeout: 600000,
 *   reporter: 'list',
 *   testDir: './tests',
 * };
 * export default config;
 * ```
 *
 */
export interface FullConfig {
  /**
   * Whether to exit with an error if any tests or groups are marked as
   * [test.only(title, testFunction)](https://playwright.dev/docs/api/class-test#test-only) or
   * [test.describe.only(title, callback)](https://playwright.dev/docs/api/class-test#test-describe-only). Useful on CI.
   */
  forbidOnly: boolean;
  /**
   * Path to the global setup file. This file will be required and run before all the tests. It must export a single
   * function.
   *
   * Learn more about [global setup and teardown](https://playwright.dev/docs/test-advanced#global-setup-and-teardown).
   *
   * ```js js-flavor=js
   * // playwright.config.js
   * // @ts-check
   *
   * /** @type {import('@playwright/test').PlaywrightTestConfig} *\/
   * const config = {
   *   globalSetup: './global-setup',
   * };
   *
   * module.exports = config;
   * ```
   *
   * ```js js-flavor=ts
   * // playwright.config.ts
   * import { PlaywrightTestConfig, devices } from '@playwright/test';
   *
   * const config: PlaywrightTestConfig = {
   *   globalSetup: './global-setup',
   * };
   * export default config;
   * ```
   *
   */
  globalSetup: string | null;
  /**
   * Path to the global teardown file. This file will be required and run after all the tests. It must export a single
   * function. See also [testConfig.globalSetup](https://playwright.dev/docs/api/class-testconfig#test-config-global-setup).
   *
   * Learn more about [global setup and teardown](https://playwright.dev/docs/test-advanced#global-setup-and-teardown).
   */
  globalTeardown: string | null;
  /**
   * Maximum time in milliseconds the whole test suite can run. Zero timeout (default) disables this behavior. Useful on CI
   * to prevent broken setup from running too long and wasting resources.
   */
  globalTimeout: number;
  /**
   * Filter to only run tests with a title matching one of the patterns. For example, passing `grep: /cart/` should only run
   * tests with "cart" in the title. Also available in the [command line](https://playwright.dev/docs/test-cli) with the `-g` option.
   *
   * `grep` option is also useful for [tagging tests](https://playwright.dev/docs/test-annotations#tag-tests).
   */
  grep: RegExp | RegExp[];
  /**
   * Filter to only run tests with a title **not** matching one of the patterns. This is the opposite of
   * [testConfig.grep](https://playwright.dev/docs/api/class-testconfig#test-config-grep). Also available in the
   * [command line](https://playwright.dev/docs/test-cli) with the `--grep-invert` option.
   *
   * `grepInvert` option is also useful for [tagging tests](https://playwright.dev/docs/test-annotations#tag-tests).
   */
  grepInvert: RegExp | RegExp[] | null;
  /**
   * The maximum number of test failures for the whole test suite run. After reaching this number, testing will stop and exit
   * with an error. Setting to zero (default) disables this behavior.
   *
   * Also available in the [command line](https://playwright.dev/docs/test-cli) with the `--max-failures` and `-x` options.
   */
  maxFailures: number;
  /**
   * Whether to preserve test output in the
   * [testConfig.outputDir](https://playwright.dev/docs/api/class-testconfig#test-config-output-dir). Defaults to `'always'`.
   * - `'always'` - preserve output for all tests;
   * - `'never'` - do not preserve output for any tests;
   * - `'failures-only'` - only preserve output for failed tests.
   */
  preserveOutput: PreserveOutput;
  /**
   * Playwright Test supports running multiple test projects at the same time. See [TestProject] for more information.
   */
  projects: FullProject[];
  /**
   * The list of reporters to use. Each reporter can be:
   * - A builtin reporter name like `'list'` or `'json'`.
   * - A module name like `'my-awesome-reporter'`.
   * - A relative path to the reporter like `'./reporters/my-awesome-reporter.js'`.
   *
   * You can pass options to the reporter in a tuple like `['json', { outputFile: './report.json' }]`.
   *
   * Learn more in the [reporters guide](https://playwright.dev/docs/test-reporters).
   *
   * ```js js-flavor=js
   * // playwright.config.js
   * // @ts-check
   *
   * /** @type {import('@playwright/test').PlaywrightTestConfig} *\/
   * const config = {
   *   reporter: 'line',
   * };
   *
   * module.exports = config;
   * ```
   *
   * ```js js-flavor=ts
   * // playwright.config.ts
   * import { PlaywrightTestConfig } from '@playwright/test';
   *
   * const config: PlaywrightTestConfig = {
   *   reporter: 'line',
   * };
   * export default config;
   * ```
   *
   */
  reporter: ReporterDescription[];
  /**
   * Whether to report slow tests. Pass `null` to disable this feature.
   *
   * Tests that took more than `threshold` milliseconds are considered slow, and the slowest ones are reported, no more than
   * `max` number of them. Passing zero as `max` reports all slow tests that exceed the threshold.
   */
  reportSlowTests: ReportSlowTests;
  rootDir: string;
  /**
   * Whether to suppress stdio and stderr output from the tests.
   */
  quiet: boolean;
  /**
   * Shard tests and execute only the selected shard. Specify in the one-based form like `{ total: 5, current: 2 }`.
   *
   * Learn more about [parallelism and sharding](https://playwright.dev/docs/test-parallel) with Playwright Test.
   */
  shard: Shard;
  /**
   * Whether to update expected snapshots with the actual results produced by the test run. Defaults to `'missing'`.
   * - `'all'` - All tests that are executed will update snapshots.
   * - `'none'` - No snapshots are updated.
   * - `'missing'` - Missing snapshots are created, for example when authoring a new test and running it for the first
   *   time. This is the default.
   *
   * Learn more about [snapshots](https://playwright.dev/docs/test-snapshots).
   */
  updateSnapshots: UpdateSnapshots;
  /**
   * The maximum number of concurrent worker processes to use for parallelizing tests.
   *
   * Playwright Test uses worker processes to run tests. There is always at least one worker process, but more can be used to
   * speed up test execution.
   *
   * Defaults to one half of the number of CPU cores. Learn more about [parallelism and sharding](https://playwright.dev/docs/test-parallel) with
   * Playwright Test.
   */
  workers: number;
  webServer: WebServerConfig | null;
}

export type TestStatus = 'passed' | 'failed' | 'timedOut' | 'skipped';

/**
 * Information about an error thrown during test execution.
 */
export interface TestError {
  /**
   * Error message. Set when [Error] (or its subclass) has been thrown.
   */
  message?: string;
  /**
   * Error stack. Set when [Error] (or its subclass) has been thrown.
   */
  stack?: string;
  /**
   * The value that was thrown. Set when anything except the [Error] (or its subclass) has been thrown.
   */
  value?: string;
}

/**
 * `WorkerInfo` contains information about the worker that is running tests. It is available to
 * [test.beforeAll(hookFunction)](https://playwright.dev/docs/api/class-test#test-before-all) and
 * [test.afterAll(hookFunction)](https://playwright.dev/docs/api/class-test#test-after-all) hooks and worker-scoped
 * fixtures.
 *
 * ```js js-flavor=js
 * const { test, expect } = require('@playwright/test');
 *
 * test.beforeAll(async ({ browserName }, workerInfo) => {
 *   console.log(`Running ${browserName} in worker #${workerInfo.workerIndex}`);
 * });
 * ```
 *
 * ```js js-flavor=ts
 * import { test, expect } from '@playwright/test';
 *
 * test.beforeAll(async ({ browserName }, workerInfo) => {
 *   console.log(`Running ${browserName} in worker #${workerInfo.workerIndex}`);
 * });
 * ```
 *
 */
export interface WorkerInfo {
  /**
   * Processed configuration from the [configuration file](https://playwright.dev/docs/test-configuration).
   */
  config: FullConfig;
  /**
   * Processed project configuration from the [configuration file](https://playwright.dev/docs/test-configuration).
   */
  project: FullProject;
  /**
   * The unique index of the worker process that is running the test. Also available as `process.env.TEST_WORKER_INDEX`.
   * Learn more about [parallelism and sharding](https://playwright.dev/docs/test-parallel) with Playwright Test.
   */
  workerIndex: number;
}

/**
 * `TestInfo` contains information about currently running test. It is available to any test function,
 * [test.beforeEach(hookFunction)](https://playwright.dev/docs/api/class-test#test-before-each) and
 * [test.afterEach(hookFunction)](https://playwright.dev/docs/api/class-test#test-after-each) hooks and test-scoped
 * fixtures. `TestInfo` provides utilities to control test execution: attach files, update test timeout, determine which
 * test is currently running and whether it was retried, etc.
 *
 * ```js js-flavor=js
 * const { test, expect } = require('@playwright/test');
 *
 * test('basic test', async ({ page }, testInfo) => {
 *   expect(testInfo.title).toBe('basic test');
 *   await page.screenshot(testInfo.outputPath('screenshot.png'));
 * });
 * ```
 *
 * ```js js-flavor=ts
 * import { test, expect } from '@playwright/test';
 *
 * test('basic test', async ({ page }, testInfo) => {
 *   expect(testInfo.title).toBe('basic test');
 *   await page.screenshot(testInfo.outputPath('screenshot.png'));
 * });
 * ```
 *
 */
export interface TestInfo {
  /**
   * Processed configuration from the [configuration file](https://playwright.dev/docs/test-configuration).
   */
  config: FullConfig;
  /**
   * Processed project configuration from the [configuration file](https://playwright.dev/docs/test-configuration).
   */
  project: FullProject;
  /**
   * The unique index of the worker process that is running the test. Also available as `process.env.TEST_WORKER_INDEX`.
   * Learn more about [parallelism and sharding](https://playwright.dev/docs/test-parallel) with Playwright Test.
   */
  workerIndex: number;

  /**
   * The title of the currently running test as passed to `test(title, testFunction)`.
   */
  title: string;
  /**
   * Absolute path to a file where the currently running test is declared.
   */
  file: string;
  /**
   * Line number where the currently running test is declared.
   */
  line: number;
  /**
   * Column number where the currently running test is declared.
   */
  column: number;
  /**
   * Test function as passed to `test(title, testFunction)`.
   */
  fn: Function;

  /**
   * Skips the currently running test. This is similar to
   * [test.skip(titleOrCondition, testFunctionOrDescription)](https://playwright.dev/docs/api/class-test#test-skip).
   * @param condition Optional condition - the test is skipped when the condition is `true`.
   * @param description Optional description that will be reflected in a test report.
   */
  skip(): void;
  skip(condition: boolean): void;
  skip(condition: boolean, description: string): void;

  /**
   * Marks the currently running test as "fixme". The test will be skipped, but the intention is to fix it. This is similar
   * to [test.fixme([condition, description])](https://playwright.dev/docs/api/class-test#test-fixme).
   * @param condition Optional condition - the test is marked as "fixme" when the condition is `true`.
   * @param description Optional description that will be reflected in a test report.
   */
  fixme(): void;
  fixme(condition: boolean): void;
  fixme(condition: boolean, description: string): void;

  /**
   * Marks the currently running test as "should fail". Playwright Test ensures that this test is actually failing. This is
   * similar to [test.fail([condition, description])](https://playwright.dev/docs/api/class-test#test-fail).
   * @param condition Optional condition - the test is marked as "should fail" when the condition is `true`.
   * @param description Optional description that will be reflected in a test report.
   */
  fail(): void;
  fail(condition: boolean): void;
  fail(condition: boolean, description: string): void;

  /**
   * Marks the currently running test as "slow", giving it triple the default timeout. This is similar to
   * [test.slow([condition, description])](https://playwright.dev/docs/api/class-test#test-slow).
   * @param condition Optional condition - the test is marked as "slow" when the condition is `true`.
   * @param description Optional description that will be reflected in a test report.
   */
  slow(): void;
  slow(condition: boolean): void;
  slow(condition: boolean, description: string): void;

  /**
   * Changes the timeout for the currently running test. Zero means no timeout.
   *
   * Timeout is usually specified in the [configuration file](https://playwright.dev/docs/test-configuration), but it could be useful to change the
   * timeout in certain scenarios:
   *
   * ```js js-flavor=js
   * const { test, expect } = require('@playwright/test');
   *
   * test.beforeEach(async ({ page }, testInfo) => {
   *   // Extend timeout for all tests running this hook by 30 seconds.
   *   testInfo.setTimeout(testInfo.timeout + 30000);
   * });
   * ```
   *
   * ```js js-flavor=ts
   * import { test, expect } from '@playwright/test';
   *
   * test.beforeEach(async ({ page }, testInfo) => {
   *   // Extend timeout for all tests running this hook by 30 seconds.
   *   testInfo.setTimeout(testInfo.timeout + 30000);
   * });
   * ```
   *
   * @param timeout Timeout in milliseconds.
   */
  setTimeout(timeout: number): void;
  /**
   * Expected status for the currently running test. This is usually `'passed'`, except for a few cases:
   * - `'skipped'` for skipped tests, e.g. with
   *   [test.skip(titleOrCondition, testFunctionOrDescription)](https://playwright.dev/docs/api/class-test#test-skip);
   * - `'failed'` for tests marked as failed with
   *   [test.fail([condition, description])](https://playwright.dev/docs/api/class-test#test-fail).
   *
   * Expected status is usually compared with the actual
   * [testInfo.status](https://playwright.dev/docs/api/class-testinfo#test-info-status):
   *
   * ```js js-flavor=js
   * const { test, expect } = require('@playwright/test');
   *
   * test.afterEach(async ({}, testInfo) => {
   *   if (testInfo.status !== testInfo.expectedStatus)
   *     console.log(`${testInfo.title} did not run as expected!`);
   * });
   * ```
   *
   * ```js js-flavor=ts
   * import { test, expect } from '@playwright/test';
   *
   * test.afterEach(async ({}, testInfo) => {
   *   if (testInfo.status !== testInfo.expectedStatus)
   *     console.log(`${testInfo.title} did not run as expected!`);
   * });
   * ```
   *
   */
  expectedStatus: TestStatus;
  /**
   * Timeout in milliseconds for the currently running test. Zero means no timeout. Timeout is usually specified in the
   * [configuration file](https://playwright.dev/docs/test-configuration)
   *
   * ```js js-flavor=js
   * const { test, expect } = require('@playwright/test');
   *
   * test.beforeEach(async ({ page }, testInfo) => {
   *   // Extend timeout for all tests running this hook by 30 seconds.
   *   testInfo.setTimeout(testInfo.timeout + 30000);
   * });
   * ```
   *
   * ```js js-flavor=ts
   * import { test, expect } from '@playwright/test';
   *
   * test.beforeEach(async ({ page }, testInfo) => {
   *   // Extend timeout for all tests running this hook by 30 seconds.
   *   testInfo.setTimeout(testInfo.timeout + 30000);
   * });
   * ```
   *
   */
  timeout: number;
  /**
   * The list of annotations applicable to the current test. Includes annotations from the test, annotations from all
   * [test.describe(title, callback)](https://playwright.dev/docs/api/class-test#test-describe) groups the test belongs to
   * and file-level annotations for the test file.
   *
   * Learn more about [test annotations](https://playwright.dev/docs/test-annotations).
   */
  annotations: { type: string, description?: string }[];
  /**
   * The list of files or buffers attached to the current test. Some reporters show test attachments. For example, you can
   * attach a screenshot to the test.
   *
   * ```js js-flavor=js
   * const { test, expect } = require('@playwright/test');
   *
   * test('basic test', async ({ page }, testInfo) => {
   *   await page.goto('https://playwright.dev');
   *
   *   // Capture a screenshot and attach it.
   *   const path = testInfo.outputPath('screenshot.png');
   *   await page.screenshot({ path });
   *   testInfo.attachments.push({ name: 'screenshot', path, contentType: 'image/png' });
   * });
   * ```
   *
   * ```js js-flavor=ts
   * import { test, expect } from '@playwright/test';
   *
   * test('basic test', async ({ page }, testInfo) => {
   *   await page.goto('https://playwright.dev');
   *
   *   // Capture a screenshot and attach it.
   *   const path = testInfo.outputPath('screenshot.png');
   *   await page.screenshot({ path });
   *   testInfo.attachments.push({ name: 'screenshot', path, contentType: 'image/png' });
   * });
   * ```
   *
   */
  attachments: { name: string, path?: string, body?: Buffer, contentType: string }[];
  /**
   * Specifies a unique repeat index when running in "repeat each" mode. This mode is enabled by passing `--repeat-each` to
   * the [command line](https://playwright.dev/docs/test-cli).
   */
  repeatEachIndex: number;
  /**
   * Specifies the retry number when the test is retried after a failure. The first test run has
   * [testInfo.retry](https://playwright.dev/docs/api/class-testinfo#test-info-retry) equal to zero, the first retry has it
   * equal to one, and so on. Learn more about [retries](https://playwright.dev/docs/test-retries).
   */
  retry: number;
  /**
   * The number of milliseconds the test took to finish. Always zero before the test finishes, either successfully or not.
   */
  duration: number;
  /**
   * Actual status for the currently running test. Available after the test has finished in
   * [test.afterEach(hookFunction)](https://playwright.dev/docs/api/class-test#test-after-each) hook and fixtures.
   *
   * Status is usually compared with the
   * [testInfo.expectedStatus](https://playwright.dev/docs/api/class-testinfo#test-info-expected-status):
   *
   * ```js js-flavor=js
   * const { test, expect } = require('@playwright/test');
   *
   * test.afterEach(async ({}, testInfo) => {
   *   if (testInfo.status !== testInfo.expectedStatus)
   *     console.log(`${testInfo.title} did not run as expected!`);
   * });
   * ```
   *
   * ```js js-flavor=ts
   * import { test, expect } from '@playwright/test';
   *
   * test.afterEach(async ({}, testInfo) => {
   *   if (testInfo.status !== testInfo.expectedStatus)
   *     console.log(`${testInfo.title} did not run as expected!`);
   * });
   * ```
   *
   */
  status?: TestStatus;
  /**
   * An error thrown during test execution, if any.
   */
  error?: TestError;
  /**
   * Output written to `process.stdout` or `console.log` during the test execution.
   */
  stdout: (string | Buffer)[];
  /**
   * Output written to `process.stderr` or `console.error` during the test execution.
   */
  stderr: (string | Buffer)[];
  /**
   * Suffix used to differentiate snapshots between multiple test configurations. For example, if snapshots depend on the
   * platform, you can set `testInfo.snapshotSuffix` equal to `process.platform`. In this case
   * `expect(value).toMatchSnapshot(snapshotName)` will use different snapshots depending on the platform. Learn more about
   * [snapshots](https://playwright.dev/docs/test-snapshots).
   */
  snapshotSuffix: string;
  /**
   * Absolute path to the output directory for this specific test run. Each test run gets its own directory so they cannot
   * conflict.
   */
  outputDir: string;
  /**
   * Returns a path to a snapshot file with the given `snapshotName`. Learn more about [snapshots](https://playwright.dev/docs/test-snapshots).
   * @param snapshotName
   */
  snapshotPath: (snapshotName: string) => string;
  /**
   * Returns a path inside the [testInfo.outputDir](https://playwright.dev/docs/api/class-testinfo#test-info-output-dir)
   * where the test can safely put a temporary file. Guarantees that tests running in parallel will not interfere with each
   * other.
   *
   * ```js js-flavor=js
   * const { test, expect } = require('@playwright/test');
   * const fs = require('fs');
   *
   * test('example test', async ({}, testInfo) => {
   *   const file = testInfo.outputPath('dir', 'temporary-file.txt');
   *   await fs.promises.writeFile(file, 'Put some data to the dir/temporary-file.txt', 'utf8');
   * });
   * ```
   *
   * ```js js-flavor=ts
   * import { test, expect } from '@playwright/test';
   * import fs from 'fs';
   *
   * test('example test', async ({}, testInfo) => {
   *   const file = testInfo.outputPath('dir', 'temporary-file.txt');
   *   await fs.promises.writeFile(file, 'Put some data to the dir/temporary-file.txt', 'utf8');
   * });
   * ```
   *
   * @param pathSegments Path segments to append at the end of the resulting path.
   */
  outputPath: (...pathSegments: string[]) => string;
}

interface SuiteFunction {
  (title: string, callback: () => void): void;
}

interface TestFunction<TestArgs> {
  (title: string, testFunction: (args: TestArgs, testInfo: TestInfo) => Promise<void> | void): void;
}

/**
 * Playwright Test provides a `test` function to declare tests and [`expect` function](https://jestjs.io/docs/expect) to
 * write assertions.
 *
 * ```js js-flavor=js
 * const { test, expect } = require('@playwright/test');
 *
 * test('basic test', async ({ page }) => {
 *   await page.goto('https://playwright.dev/');
 *   const name = await page.innerText('.navbar__title');
 *   expect(name).toBe('Playwright');
 * });
 * ```
 *
 * ```js js-flavor=ts
 * import { test, expect } from '@playwright/test';
 *
 * test('basic test', async ({ page }) => {
 *   await page.goto('https://playwright.dev/');
 *   const name = await page.innerText('.navbar__title');
 *   expect(name).toBe('Playwright');
 * });
 * ```
 *
 */
export interface TestType<TestArgs extends KeyValue, WorkerArgs extends KeyValue> extends TestFunction<TestArgs & WorkerArgs> {
  /**
   * Declares a focused test. If there are some focused tests or suites, all of them will be run but nothing else.
   *
   * ```js js-flavor=js
   * test.only('focus this test', async ({ page }) => {
   *   // Run only focused tests in the entire project.
   * });
   * ```
   *
   * ```js js-flavor=ts
   * test.only('focus this test', async ({ page }) => {
   *   // Run only focused tests in the entire project.
   * });
   * ```
   *
   * @param title Test title.
   * @param testFunction Test function that takes one or two arguments: an object with fixtures and optional [TestInfo].
   */
  only: TestFunction<TestArgs & WorkerArgs>;
  /**
   * Declares a group of tests.
   *
   * ```js js-flavor=js
   * test.describe('two tests', () => {
   *   test('one', async ({ page }) => {
   *     // ...
   *   });
   *
   *   test('two', async ({ page }) => {
   *     // ...
   *   });
   * });
   * ```
   *
   * ```js js-flavor=ts
   * test.describe('two tests', () => {
   *   test('one', async ({ page }) => {
   *     // ...
   *   });
   *
   *   test('two', async ({ page }) => {
   *     // ...
   *   });
   * });
   * ```
   *
   * @param title Group title.
   * @param callback A callback that is run immediately when calling [test.describe(title, callback)](https://playwright.dev/docs/api/class-test#test-describe). Any tests added in this
   * callback will belong to the group.
   */
  describe: SuiteFunction & {
    only: SuiteFunction;
    serial: SuiteFunction & {
      only: SuiteFunction;
    };
  };
  /**
   * Skips a test or a group of tests.
   *
   * Unconditionally skip a test, this is similar syntax to
   * [test.(call)(title, testFunction)](https://playwright.dev/docs/api/class-test#test-call):
   *
   * ```js js-flavor=js
   * const { test, expect } = require('@playwright/test');
   *
   * test.skip('broken test', async ({ page }) => {
   *   // ...
   * });
   * ```
   *
   * ```js js-flavor=ts
   * import { test, expect } from '@playwright/test';
   *
   * test.skip('broken test', async ({ page }) => {
   *   // ...
   * });
   * ```
   *
   * Conditionally skip a test with an optional description. In this case, call `test.skip()` inside the test function:
   *
   * ```js js-flavor=js
   * const { test, expect } = require('@playwright/test');
   *
   * test('skip in WebKit', async ({ page, browserName }) => {
   *   test.skip(browserName === 'webkit', 'This feature is not implemented for Mac');
   *   // ...
   * });
   * ```
   *
   * ```js js-flavor=ts
   * import { test, expect } from '@playwright/test';
   *
   * test('skip in WebKit', async ({ page, browserName }) => {
   *   test.skip(browserName === 'webkit', 'This feature is not implemented for Mac');
   *   // ...
   * });
   * ```
   *
   * Conditionally skip all tests in a file or
   * [test.describe(title, callback)](https://playwright.dev/docs/api/class-test#test-describe) group:
   *
   * ```js js-flavor=js
   * const { test, expect } = require('@playwright/test');
   *
   * test.skip(({ browserName }) => browserName === 'webkit');
   *
   * test('skip in WebKit 1', async ({ page }) => {
   *   // ...
   * });
   * test('skip in WebKit 2', async ({ page }) => {
   *   // ...
   * });
   * ```
   *
   * ```js js-flavor=ts
   * import { test, expect } from '@playwright/test';
   *
   * test.skip(({ browserName }) => browserName === 'webkit');
   *
   * test('skip in WebKit 1', async ({ page }) => {
   *   // ...
   * });
   * test('skip in WebKit 2', async ({ page }) => {
   *   // ...
   * });
   * ```
   *
   * Skip from a hook:
   *
   * ```js js-flavor=js
   * const { test, expect } = require('@playwright/test');
   *
   * test.beforeEach(async ({ page }) => {
   *   test.skip(process.env.APP_VERSION === 'v1', 'There are no settings in v1');
   *   await page.goto('/settings');
   * });
   * ```
   *
   * ```js js-flavor=ts
   * import { test, expect } from '@playwright/test';
   *
   * test.beforeEach(async ({ page }) => {
   *   test.skip(process.env.APP_VERSION === 'v1', 'There are no settings in v1');
   *   await page.goto('/settings');
   * });
   * ```
   *
   * @param titleOrCondition When used with `test.skip('test', () => {})` notation, first argument is a test title. Otherwise it is an optional skip condition - either a boolean value, or a function that takes a fixtures object and returns a boolean. Test or tests are
   * skipped when the condition is `true`.
   * @param testFunctionOrDescription When used with `test.skip('test', () => {})` notation, second argument is a test function. Otherwise it is an optional description that will be reflected in a test report.
   */
  skip(): void;
  skip(condition: boolean): void;
  skip(condition: boolean, description: string): void;
  skip(callback: (args: TestArgs & WorkerArgs) => boolean): void;
  skip(callback: (args: TestArgs & WorkerArgs) => boolean, description: string): void;
  /**
   * Marks a test or a group of tests as "fixme". These tests will not be run, but the intention is to fix them.
   *
   * Unconditional fixme:
   *
   * ```js js-flavor=js
   * const { test, expect } = require('@playwright/test');
   *
   * test('not yet ready', async ({ page }) => {
   *   test.fixme();
   *   // ...
   * });
   * ```
   *
   * ```js js-flavor=ts
   * import { test, expect } from '@playwright/test';
   *
   * test('not yet ready', async ({ page }) => {
   *   test.fixme();
   *   // ...
   * });
   * ```
   *
   * Conditional fixme a test with an optional description:
   *
   * ```js js-flavor=js
   * const { test, expect } = require('@playwright/test');
   *
   * test('fixme in WebKit', async ({ page, browserName }) => {
   *   test.fixme(browserName === 'webkit', 'This feature is not implemented for Mac yet');
   *   // ...
   * });
   * ```
   *
   * ```js js-flavor=ts
   * import { test, expect } from '@playwright/test';
   *
   * test('fixme in WebKit', async ({ page, browserName }) => {
   *   test.fixme(browserName === 'webkit', 'This feature is not implemented for Mac yet');
   *   // ...
   * });
   * ```
   *
   * Conditional fixme for all tests in a file or
   * [test.describe(title, callback)](https://playwright.dev/docs/api/class-test#test-describe) group:
   *
   * ```js js-flavor=js
   * const { test, expect } = require('@playwright/test');
   *
   * test.fixme(({ browserName }) => browserName === 'webkit');
   *
   * test('fixme in WebKit 1', async ({ page }) => {
   *   // ...
   * });
   * test('fixme in WebKit 2', async ({ page }) => {
   *   // ...
   * });
   * ```
   *
   * ```js js-flavor=ts
   * import { test, expect } from '@playwright/test';
   *
   * test.fixme(({ browserName }) => browserName === 'webkit');
   *
   * test('fixme in WebKit 1', async ({ page }) => {
   *   // ...
   * });
   * test('fixme in WebKit 2', async ({ page }) => {
   *   // ...
   * });
   * ```
   *
   * `fixme` from a hook:
   *
   * ```js js-flavor=js
   * const { test, expect } = require('@playwright/test');
   *
   * test.beforeEach(async ({ page }) => {
   *   test.fixme(process.env.APP_VERSION === 'v2', 'No settings in v2 yet');
   *   await page.goto('/settings');
   * });
   * ```
   *
   * ```js js-flavor=ts
   * import { test, expect } from '@playwright/test';
   *
   * test.beforeEach(async ({ page }) => {
   *   test.fixme(process.env.APP_VERSION === 'v2', 'No settings in v2 yet');
   *   await page.goto('/settings');
   * });
   * ```
   *
   * @param condition Optional condition - either a boolean value, or a function that takes a fixtures object and returns a boolean. Test or tests are marked as "fixme" when the condition is `true`.
   * @param description Optional description that will be reflected in a test report.
   */
  fixme(): void;
  fixme(condition: boolean): void;
  fixme(condition: boolean, description: string): void;
  fixme(callback: (args: TestArgs & WorkerArgs) => boolean): void;
  fixme(callback: (args: TestArgs & WorkerArgs) => boolean, description: string): void;
  /**
   * Marks a test or a group of tests as "should fail". Playwright Test runs these tests and ensures that they are actually
   * failing. This is useful for documentation purposes to acknowledge that some functionality is broken until it is fixed.
   *
   * Unconditional fail:
   *
   * ```js js-flavor=js
   * const { test, expect } = require('@playwright/test');
   *
   * test('not yet ready', async ({ page }) => {
   *   test.fail();
   *   // ...
   * });
   * ```
   *
   * ```js js-flavor=ts
   * import { test, expect } from '@playwright/test';
   *
   * test('not yet ready', async ({ page }) => {
   *   test.fail();
   *   // ...
   * });
   * ```
   *
   * Conditional fail a test with an optional description:
   *
   * ```js js-flavor=js
   * const { test, expect } = require('@playwright/test');
   *
   * test('fail in WebKit', async ({ page, browserName }) => {
   *   test.fail(browserName === 'webkit', 'This feature is not implemented for Mac yet');
   *   // ...
   * });
   * ```
   *
   * ```js js-flavor=ts
   * import { test, expect } from '@playwright/test';
   *
   * test('fail in WebKit', async ({ page, browserName }) => {
   *   test.fail(browserName === 'webkit', 'This feature is not implemented for Mac yet');
   *   // ...
   * });
   * ```
   *
   * Conditional fail for all tests in a file or
   * [test.describe(title, callback)](https://playwright.dev/docs/api/class-test#test-describe) group:
   *
   * ```js js-flavor=js
   * const { test, expect } = require('@playwright/test');
   *
   * test.fail(({ browserName }) => browserName === 'webkit');
   *
   * test('fail in WebKit 1', async ({ page }) => {
   *   // ...
   * });
   * test('fail in WebKit 2', async ({ page }) => {
   *   // ...
   * });
   * ```
   *
   * ```js js-flavor=ts
   * import { test, expect } from '@playwright/test';
   *
   * test.fail(({ browserName }) => browserName === 'webkit');
   *
   * test('fail in WebKit 1', async ({ page }) => {
   *   // ...
   * });
   * test('fail in WebKit 2', async ({ page }) => {
   *   // ...
   * });
   * ```
   *
   * @param condition Optional condition - either a boolean value, or a function that takes a fixtures object and returns a boolean. Test or tests are marked as "should fail" when the condition is `true`.
   * @param description Optional description that will be reflected in a test report.
   */
  fail(): void;
  fail(condition: boolean): void;
  fail(condition: boolean, description: string): void;
  fail(callback: (args: TestArgs & WorkerArgs) => boolean): void;
  fail(callback: (args: TestArgs & WorkerArgs) => boolean, description: string): void;
  /**
   * Marks a test or a group of tests as "slow". Slow tests will be given triple the default timeout.
   *
   * Unconditional slow:
   *
   * ```js js-flavor=js
   * const { test, expect } = require('@playwright/test');
   *
   * test('slow test', async ({ page }) => {
   *   test.slow();
   *   // ...
   * });
   * ```
   *
   * ```js js-flavor=ts
   * import { test, expect } from '@playwright/test';
   *
   * test('slow test', async ({ page }) => {
   *   test.slow();
   *   // ...
   * });
   * ```
   *
   * Conditional slow a test with an optional description:
   *
   * ```js js-flavor=js
   * const { test, expect } = require('@playwright/test');
   *
   * test('slow in WebKit', async ({ page, browserName }) => {
   *   test.slow(browserName === 'webkit', 'This feature is slow on Mac');
   *   // ...
   * });
   * ```
   *
   * ```js js-flavor=ts
   * import { test, expect } from '@playwright/test';
   *
   * test('slow in WebKit', async ({ page, browserName }) => {
   *   test.slow(browserName === 'webkit', 'This feature is slow on Mac');
   *   // ...
   * });
   * ```
   *
   * Conditional slow for all tests in a file or
   * [test.describe(title, callback)](https://playwright.dev/docs/api/class-test#test-describe) group:
   *
   * ```js js-flavor=js
   * const { test, expect } = require('@playwright/test');
   *
   * test.slow(({ browserName }) => browserName === 'webkit');
   *
   * test('slow in WebKit 1', async ({ page }) => {
   *   // ...
   * });
   * test('slow in WebKit 2', async ({ page }) => {
   *   // ...
   * });
   * ```
   *
   * ```js js-flavor=ts
   * import { test, expect } from '@playwright/test';
   *
   * test.slow(({ browserName }) => browserName === 'webkit');
   *
   * test('slow in WebKit 1', async ({ page }) => {
   *   // ...
   * });
   * test('fail in WebKit 2', async ({ page }) => {
   *   // ...
   * });
   * ```
   *
   * @param condition Optional condition - either a boolean value, or a function that takes a fixtures object and returns a boolean. Test or tests are marked as "slow" when the condition is `true`.
   * @param description Optional description that will be reflected in a test report.
   */
  slow(): void;
  slow(condition: boolean): void;
  slow(condition: boolean, description: string): void;
  slow(callback: (args: TestArgs & WorkerArgs) => boolean): void;
  slow(callback: (args: TestArgs & WorkerArgs) => boolean, description: string): void;
  /**
   * Changes the timeout for the test.
   *
   * ```js js-flavor=js
   * const { test, expect } = require('@playwright/test');
   *
   * test('very slow test', async ({ page }) => {
   *   test.setTimeout(120000);
   *   // ...
   * });
   * ```
   *
   * ```js js-flavor=ts
   * import { test, expect } from '@playwright/test';
   *
   * test('very slow test', async ({ page }) => {
   *   test.setTimeout(120000);
   *   // ...
   * });
   * ```
   *
   * Changing timeout from a slow hook:
   *
   * ```js js-flavor=js
   * const { test, expect } = require('@playwright/test');
   *
   * test.beforeEach(async ({ page }, testInfo) => {
   *   // Extend timeout for all tests running this hook by 30 seconds.
   *   test.setTimeout(testInfo.timeout + 30000);
   * });
   * ```
   *
   * ```js js-flavor=ts
   * import { test, expect } from '@playwright/test';
   *
   * test.beforeEach(async ({ page }, testInfo) => {
   *   // Extend timeout for all tests running this hook by 30 seconds.
   *   test.setTimeout(testInfo.timeout + 30000);
   * });
   * ```
   *
   * Timeout for the currently running test is available through
   * [testInfo.timeout](https://playwright.dev/docs/api/class-testinfo#test-info-timeout).
   * @param timeout Timeout in milliseconds.
   */
  setTimeout(timeout: number): void;
  /**
   * Declares a `beforeEach` hook that is executed before each test. When called in the scope of a test file, runs before
   * each test in the file. When called inside a
   * [test.describe(title, callback)](https://playwright.dev/docs/api/class-test#test-describe) group, runs before each test
   * in the group.
   *
   * ```js js-flavor=js
   * // example.spec.js
   * const { test, expect } = require('@playwright/test');
   *
   * test.beforeEach(async ({ page }) => {
   *   // Go to the starting url before each test.
   *   await page.goto('https://my.start.url/');
   * });
   *
   * test('my test', async ({ page }) => {
   *   expect(page.url()).toBe('https://my.start.url/');
   * });
   * ```
   *
   * ```js js-flavor=ts
   * // example.spec.ts
   * import { test, expect } from '@playwright/test';
   *
   * test.beforeEach(async ({ page }) => {
   *   // Go to the starting url before each test.
   *   await page.goto('https://my.start.url/');
   * });
   *
   * test('my test', async ({ page }) => {
   *   expect(page.url()).toBe('https://my.start.url/');
   * });
   * ```
   *
   * You can use [test.afterEach(hookFunction)](https://playwright.dev/docs/api/class-test#test-after-each) to teardown any
   * resources set up in `beforeEach`.
   * @param hookFunction Hook function that takes one or two arguments: an object with fixtures and optional [TestInfo].
   */
  beforeEach(inner: (args: TestArgs & WorkerArgs, testInfo: TestInfo) => Promise<any> | any): void;
  /**
   * Declares an `afterEach` hook that is executed after each test. When called in the scope of a test file, runs before each
   * test in the file. When called inside a
   * [test.describe(title, callback)](https://playwright.dev/docs/api/class-test#test-describe) group, runs before each test
   * in the group.
   * @param hookFunction Hook function that takes one or two arguments: an object with fixtures and optional [TestInfo].
   */
  afterEach(inner: (args: TestArgs & WorkerArgs, testInfo: TestInfo) => Promise<any> | any): void;
  /**
   * Declares a `beforeAll` hook that is executed once before all tests. When called in the scope of a test file, runs before
   * all tests in the file. When called inside a
   * [test.describe(title, callback)](https://playwright.dev/docs/api/class-test#test-describe) group, runs before all tests
   * in the group.
   *
   * ```js js-flavor=js
   * // example.spec.js
   * const { test, expect } = require('@playwright/test');
   *
   * test.beforeAll(async () => {
   *   console.log('Before tests');
   * });
   *
   * test.afterAll(async () => {
   *   console.log('After tests');
   * });
   *
   * test('my test', async ({ page }) => {
   *   // ...
   * });
   * ```
   *
   * ```js js-flavor=ts
   * // example.spec.ts
   * import { test, expect } from '@playwright/test';
   *
   * test.beforeAll(async () => {
   *   console.log('Before tests');
   * });
   *
   * test.afterAll(async () => {
   *   console.log('After tests');
   * });
   *
   * test('my test', async ({ page }) => {
   *   // ...
   * });
   * ```
   *
   * You can use [test.afterAll(hookFunction)](https://playwright.dev/docs/api/class-test#test-after-all) to teardown any
   * resources set up in `beforeAll`.
   * @param hookFunction Hook function that takes one or two arguments: an object with fixtures and optional [TestInfo].
   */
  beforeAll(inner: (args: TestArgs & WorkerArgs, testInfo: TestInfo) => Promise<any> | any): void;
  /**
   * Declares an `afterAll` hook that is executed once after all tests. When called in the scope of a test file, runs after
   * all tests in the file. When called inside a
   * [test.describe(title, callback)](https://playwright.dev/docs/api/class-test#test-describe) group, runs after all tests
   * in the group.
   * @param hookFunction Hook function that takes one or two arguments: an object with fixtures and optional [TestInfo].
   */
  afterAll(inner: (args: TestArgs & WorkerArgs, testInfo: TestInfo) => Promise<any> | any): void;
  /**
   * Specifies parameters or fixtures to use in a single test file or a
   * [test.describe(title, callback)](https://playwright.dev/docs/api/class-test#test-describe) group. Most useful to
   * configure a fixture, for example set `locale` to configure `context` fixture.
   *
   * ```js js-flavor=js
   * const { test, expect } = require('@playwright/test');
   *
   * test.use({ locale: 'en-US' });
   *
   * test('test with locale', async ({ page }) => {
   *   // Default context and page have locale as specified
   * });
   * ```
   *
   * ```js js-flavor=ts
   * import { test, expect } from '@playwright/test';
   *
   * test.use({ locale: 'en-US' });
   *
   * test('test with locale', async ({ page }) => {
   *   // Default context and page have locale as specified
   * });
   * ```
   *
   * It is possible not only to provide a fixture value, but also to override a fixture by providing a fixture function.
   *
   * ```js js-flavor=js
   * const { test, expect } = require('@playwright/test');
   *
   * test.use({
   *   locale: async ({}, use) => {
   *     // Read locale from some configuration file.
   *     const locale = await fs.promises.readFile('test-locale', 'utf-8');
   *     await use(locale);
   *   },
   * });
   *
   * test('test with locale', async ({ page }) => {
   *   // Default context and page have locale as specified
   * });
   * ```
   *
   * ```js js-flavor=ts
   * import { test, expect } from '@playwright/test';
   *
   * test.use({
   *   locale: async ({}, use) => {
   *     // Read locale from some configuration file.
   *     const locale = await fs.promises.readFile('test-locale', 'utf-8');
   *     await use(locale);
   *   },
   * });
   *
   * test('test with locale', async ({ page }) => {
   *   // Default context and page have locale as specified
   * });
   * ```
   *
   * @param fixtures An object with fixture definitions.
   */
  use(fixtures: Fixtures<{}, {}, TestArgs, WorkerArgs>): void;
  /**
   * Declares a test step.
   *
   * ```js js-flavor=js
   * const { test, expect } = require('@playwright/test');
   *
   * test('test', async ({ page }) => {
   *   await test.step('Log in', async () => {
   *     // ...
   *   });
   * });
   * ```
   *
   * ```js js-flavor=ts
   * import { test, expect } from '@playwright/test';
   *
   * test('test', async ({ page }) => {
   *   await test.step('Log in', async () => {
   *     // ...
   *   });
   * });
   * ```
   *
   * @param title Step name.
   * @param body Step body.
   */
  step(title: string, body: () => Promise<any>): Promise<any>;
  /**
   * `expect` function can be used to create test assertions. Read
   * [expect library documentation](https://jestjs.io/docs/expect) for more details.
   */
  expect: Expect;
  declare<T extends KeyValue = {}, W extends KeyValue = {}>(): TestType<TestArgs & T, WorkerArgs & W>;
  extend<T, W extends KeyValue = {}>(fixtures: Fixtures<T, W, TestArgs, WorkerArgs>): TestType<TestArgs & T, WorkerArgs & W>;
}

type KeyValue = { [key: string]: any };
export type TestFixture<R, Args extends KeyValue> = (args: Args, use: (r: R) => Promise<void>, testInfo: TestInfo) => any;
export type WorkerFixture<R, Args extends KeyValue> = (args: Args, use: (r: R) => Promise<void>, workerInfo: WorkerInfo) => any;
type TestFixtureValue<R, Args> = R | TestFixture<R, Args>;
type WorkerFixtureValue<R, Args> = R | WorkerFixture<R, Args>;
export type Fixtures<T extends KeyValue = {}, W extends KeyValue = {}, PT extends KeyValue = {}, PW extends KeyValue = {}> = {
  [K in keyof PW]?: WorkerFixtureValue<PW[K], W & PW> | [WorkerFixtureValue<PW[K], W & PW>, { scope: 'worker' }];
} & {
  [K in keyof PT]?: TestFixtureValue<PT[K], T & W & PT & PW> | [TestFixtureValue<PT[K], T & W & PT & PW>, { scope: 'test' }];
} & {
  [K in keyof W]?: [WorkerFixtureValue<W[K], W & PW>, { scope: 'worker', auto?: boolean }];
} & {
  [K in keyof T]?: TestFixtureValue<T[K], T & W & PT & PW> | [TestFixtureValue<T[K], T & W & PT & PW>, { scope?: 'test', auto?: boolean }];
};

type BrowserName = 'chromium' | 'firefox' | 'webkit';
type BrowserChannel = Exclude<LaunchOptions['channel'], undefined>;
type ColorScheme = Exclude<BrowserContextOptions['colorScheme'], undefined>;
type ExtraHTTPHeaders = Exclude<BrowserContextOptions['extraHTTPHeaders'], undefined>;
type Proxy = Exclude<BrowserContextOptions['proxy'], undefined>;
type StorageState = Exclude<BrowserContextOptions['storageState'], undefined>;

/**
 * Playwright Test is based on the concept of the [test fixtures](https://playwright.dev/docs/test-fixtures). Test fixtures are used to establish
 * environment for each test, giving the test everything it needs and nothing else.
 *
 * Playwright Test looks at each test declaration, analyses the set of fixtures the test needs and prepares those fixtures
 * specifically for the test. Values prepared by the fixtures are merged into a single object that is available to the
 * `test`, hooks, annotations and other fixtures as a first parameter.
 *
 * ```js js-flavor=js
 * const { test, expect } = require('@playwright/test');
 *
 * test('basic test', async ({ page }) => {
 *   // ...
 * });
 * ```
 *
 * ```js js-flavor=ts
 * import { test, expect } from '@playwright/test';
 *
 * test('basic test', async ({ page }) => {
 *   // ...
 * });
 * ```
 *
 * Given the test above, Playwright Test will set up the `page` fixture before running the test, and tear it down after the
 * test has finished. `page` fixture provides a [Page] object that is available to the test.
 *
 * Playwright Test comes with builtin fixtures listed below, and you can add your own fixtures as well. Many fixtures are
 * designed as "options" that you can set in your
 * [testConfig.use](https://playwright.dev/docs/api/class-testconfig#test-config-use) section.
 *
 * ```js js-flavor=js
 * // @ts-check
 *
 * /** @type {import('@playwright/test').PlaywrightTestConfig} *\/
 * const config = {
 *   use: {
 *     headless: false,
 *     viewport: { width: 1280, height: 720 },
 *     ignoreHTTPSErrors: true,
 *     video: 'on-first-retry',
 *   },
 * };
 *
 * module.exports = config;
 * ```
 *
 * ```js js-flavor=ts
 * import { PlaywrightTestConfig } from '@playwright/test';
 * const config: PlaywrightTestConfig = {
 *   use: {
 *     headless: false,
 *     viewport: { width: 1280, height: 720 },
 *     ignoreHTTPSErrors: true,
 *     video: 'on-first-retry',
 *   },
 * };
 * export default config;
 * ```
 *
 * Alternatively, with [test.use(fixtures)](https://playwright.dev/docs/api/class-test#test-use) you can override some
 * options for a file.
 *
 * ```js js-flavor=js
 * // example.spec.js
 * const { test, expect } = require('@playwright/test');
 *
 * // Run tests in this file with portrait-like viewport.
 * test.use({ viewport: { width: 600, height: 900 } });
 *
 * test('my portrait test', async ({ page }) => {
 *   // ...
 * });
 * ```
 *
 * ```js js-flavor=ts
 * // example.spec.ts
 * import { test, expect } from '@playwright/test';
 *
 * // Run tests in this file with portrait-like viewport.
 * test.use({ viewport: { width: 600, height: 900 } });
 *
 * test('my portrait test', async ({ page }) => {
 *   // ...
 * });
 * ```
 *
 */
export interface PlaywrightWorkerOptions {
  /**
   * Name of the browser that runs tests. Defaults to `'chromium'`. Most of the time you should set `browserName` in your
   * [TestConfig]:
   *
   * ```js js-flavor=js
   * // playwright.config.js
   * // @ts-check
   *
   * /** @type {import('@playwright/test').PlaywrightTestConfig} *\/
   * const config = {
   *   use: {
   *     browserName: 'firefox',
   *   },
   * };
   *
   * module.exports = config;
   * ```
   *
   * ```js js-flavor=ts
   * // playwright.config.ts
   * import { PlaywrightTestConfig, devices } from '@playwright/test';
   *
   * const config: PlaywrightTestConfig = {
   *   use: {
   *     browserName: 'firefox',
   *   },
   * };
   * export default config;
   * ```
   *
   */
  browserName: BrowserName;
  defaultBrowserType: BrowserName;
  /**
   * Whether to run browser in headless mode. More details for
   * [Chromium](https://developers.google.com/web/updates/2017/04/headless-chrome) and
   * [Firefox](https://developer.mozilla.org/en-US/docs/Mozilla/Firefox/Headless_mode). Defaults to `true` unless the
   * `devtools` option is `true`.
   */
  headless: boolean | undefined;
  /**
   * Browser distribution channel.  Supported values are "chrome", "chrome-beta", "chrome-dev", "chrome-canary", "msedge",
   * "msedge-beta", "msedge-dev", "msedge-canary". Read more about using
   * [Google Chrome and Microsoft Edge](https://playwright.dev/docs/browsers#google-chrome--microsoft-edge).
   */
  channel: BrowserChannel | undefined;
  /**
   * Options used to launch the browser, as passed to
   * [browserType.launch([options])](https://playwright.dev/docs/api/class-browsertype#browser-type-launch). Specific options
   * [fixtures.headless](https://playwright.dev/docs/api/class-fixtures#fixtures-headless) and
   * [fixtures.channel](https://playwright.dev/docs/api/class-fixtures#fixtures-channel) take priority over this.
   */
  launchOptions: LaunchOptions;
  /**
   * Whether to automatically capture a screenshot after each test. Defaults to `'off'`.
   * - `'off'`: Do not capture screenshots.
   * - `'on'`: Capture screenshot after each test.
   * - `'only-on-failure'`: Capture screenshot after each test failure.
   *
   * Learn more about [automatic screenshots](https://playwright.dev/docs/test-configuration#automatic-screenshots).
   */
  screenshot: 'off' | 'on' | 'only-on-failure';
  /**
   * Whether to record a trace for each test. Defaults to `'off'`.
   * - `'off'`: Do not record a trace.
   * - `'on'`: Record a trace for each test.
   * - `'retain-on-failure'`: Record a trace for each test, but remove it from successful test runs.
   * - `'on-first-retry'`: Record a trace only when retrying a test for the first time.
   *
   * Learn more about [recording trace](https://playwright.dev/docs/test-configuration#record-test-trace).
   */
  trace: 'off' | 'on' | 'retain-on-failure' | 'on-first-retry' | /** deprecated */ 'retry-with-trace';
  /**
   * Whether to record video for each test. Defaults to `'off'`.
   * - `'off'`: Do not record video.
   * - `'on'`: Record video for each test.
   * - `'retain-on-failure'`: Record video for each test, but remove all videos from successful test runs.
   * - `'on-first-retry'`: Record video only when retrying a test for the first time.
   *
   * Learn more about [recording video](https://playwright.dev/docs/test-configuration#record-video).
   */
  video: VideoMode | { mode: VideoMode, size: ViewportSize };
}

export type VideoMode = 'off' | 'on' | 'retain-on-failure' | 'on-first-retry' | /** deprecated */ 'retry-with-video';

/**
 * Playwright Test is based on the concept of the [test fixtures](https://playwright.dev/docs/test-fixtures). Test fixtures are used to establish
 * environment for each test, giving the test everything it needs and nothing else.
 *
 * Playwright Test looks at each test declaration, analyses the set of fixtures the test needs and prepares those fixtures
 * specifically for the test. Values prepared by the fixtures are merged into a single object that is available to the
 * `test`, hooks, annotations and other fixtures as a first parameter.
 *
 * ```js js-flavor=js
 * const { test, expect } = require('@playwright/test');
 *
 * test('basic test', async ({ page }) => {
 *   // ...
 * });
 * ```
 *
 * ```js js-flavor=ts
 * import { test, expect } from '@playwright/test';
 *
 * test('basic test', async ({ page }) => {
 *   // ...
 * });
 * ```
 *
 * Given the test above, Playwright Test will set up the `page` fixture before running the test, and tear it down after the
 * test has finished. `page` fixture provides a [Page] object that is available to the test.
 *
 * Playwright Test comes with builtin fixtures listed below, and you can add your own fixtures as well. Many fixtures are
 * designed as "options" that you can set in your
 * [testConfig.use](https://playwright.dev/docs/api/class-testconfig#test-config-use) section.
 *
 * ```js js-flavor=js
 * // @ts-check
 *
 * /** @type {import('@playwright/test').PlaywrightTestConfig} *\/
 * const config = {
 *   use: {
 *     headless: false,
 *     viewport: { width: 1280, height: 720 },
 *     ignoreHTTPSErrors: true,
 *     video: 'on-first-retry',
 *   },
 * };
 *
 * module.exports = config;
 * ```
 *
 * ```js js-flavor=ts
 * import { PlaywrightTestConfig } from '@playwright/test';
 * const config: PlaywrightTestConfig = {
 *   use: {
 *     headless: false,
 *     viewport: { width: 1280, height: 720 },
 *     ignoreHTTPSErrors: true,
 *     video: 'on-first-retry',
 *   },
 * };
 * export default config;
 * ```
 *
 * Alternatively, with [test.use(fixtures)](https://playwright.dev/docs/api/class-test#test-use) you can override some
 * options for a file.
 *
 * ```js js-flavor=js
 * // example.spec.js
 * const { test, expect } = require('@playwright/test');
 *
 * // Run tests in this file with portrait-like viewport.
 * test.use({ viewport: { width: 600, height: 900 } });
 *
 * test('my portrait test', async ({ page }) => {
 *   // ...
 * });
 * ```
 *
 * ```js js-flavor=ts
 * // example.spec.ts
 * import { test, expect } from '@playwright/test';
 *
 * // Run tests in this file with portrait-like viewport.
 * test.use({ viewport: { width: 600, height: 900 } });
 *
 * test('my portrait test', async ({ page }) => {
 *   // ...
 * });
 * ```
 *
 */
export interface PlaywrightTestOptions {
  /**
   * Whether to automatically download all the attachments. Defaults to `false` where all the downloads are canceled.
   */
  acceptDownloads: boolean | undefined;
  /**
   * Toggles bypassing page's Content-Security-Policy.
   */
  bypassCSP: boolean | undefined;
  /**
   * Emulates `'prefers-colors-scheme'` media feature, supported values are `'light'`, `'dark'`, `'no-preference'`. See
   * [page.emulateMedia([options])](https://playwright.dev/docs/api/class-page#page-emulate-media) for more details. Defaults
   * to `'light'`.
   */
  colorScheme: ColorScheme | undefined;
  /**
   * Specify device scale factor (can be thought of as dpr). Defaults to `1`.
   */
  deviceScaleFactor: number | undefined;
  /**
   * An object containing additional HTTP headers to be sent with every request. All header values must be strings.
   */
  extraHTTPHeaders: ExtraHTTPHeaders | undefined;
  geolocation: Geolocation | undefined;
  /**
   * Specifies if viewport supports touch events. Defaults to false.
   */
  hasTouch: boolean | undefined;
  /**
   * Credentials for [HTTP authentication](https://developer.mozilla.org/en-US/docs/Web/HTTP/Authentication).
   */
  httpCredentials: HTTPCredentials | undefined;
  /**
   * Whether to ignore HTTPS errors during navigation. Defaults to `false`.
   */
  ignoreHTTPSErrors: boolean | undefined;
  /**
   * Whether the `meta viewport` tag is taken into account and touch events are enabled. Defaults to `false`. Not supported
   * in Firefox.
   */
  isMobile: boolean | undefined;
  /**
   * Whether or not to enable JavaScript in the context. Defaults to `true`.
   */
  javaScriptEnabled: boolean | undefined;
  /**
   * Specify user locale, for example `en-GB`, `de-DE`, etc. Locale will affect `navigator.language` value, `Accept-Language`
   * request header value as well as number and date formatting rules.
   */
  locale: string | undefined;
  /**
   * Whether to emulate network being offline. Defaults to `false`.
   */
  offline: boolean | undefined;
  /**
   * A list of permissions to grant to all pages in this context. See
   * [browserContext.grantPermissions(permissions[, options])](https://playwright.dev/docs/api/class-browsercontext#browser-context-grant-permissions)
   * for more details.
   */
  permissions: string[] | undefined;
  /**
   * Network proxy settings.
   */
  proxy: Proxy | undefined;
  /**
   * Populates context with given storage state. This option can be used to initialize context with logged-in information
   * obtained via
   * [browserContext.storageState([options])](https://playwright.dev/docs/api/class-browsercontext#browser-context-storage-state).
   * Either a path to the file with saved storage, or an object with the following fields:
   */
  storageState: StorageState | undefined;
  /**
   * Changes the timezone of the context. See
   * [ICU's metaZones.txt](https://cs.chromium.org/chromium/src/third_party/icu/source/data/misc/metaZones.txt?rcl=faee8bc70570192d82d2978a71e2a615788597d1)
   * for a list of supported timezone IDs.
   */
  timezoneId: string | undefined;
  /**
   * Specific user agent to use in this context.
   */
  userAgent: string | undefined;
  /**
   * Emulates consistent viewport for each page. Defaults to an 1280x720 viewport. `null` disables the default viewport.
   */
  viewport: ViewportSize | null | undefined;
  /**
   * When using [page.goto(url[, options])](https://playwright.dev/docs/api/class-page#page-goto),
   * [page.route(url, handler)](https://playwright.dev/docs/api/class-page#page-route),
   * [page.waitForURL(url[, options])](https://playwright.dev/docs/api/class-page#page-wait-for-url),
   * [page.waitForRequest(urlOrPredicate[, options])](https://playwright.dev/docs/api/class-page#page-wait-for-request), or
   * [page.waitForResponse(urlOrPredicate[, options])](https://playwright.dev/docs/api/class-page#page-wait-for-response) it
   * takes the base URL in consideration by using the [`URL()`](https://developer.mozilla.org/en-US/docs/Web/API/URL/URL)
   * constructor for building the corresponding URL. Examples:
   * - baseURL: `http://localhost:3000` and navigating to `/bar.html` results in `http://localhost:3000/bar.html`
   * - baseURL: `http://localhost:3000/foo/` and navigating to `./bar.html` results in `http://localhost:3000/foo/bar.html`
   */
  baseURL: string | undefined;
  /**
   * Options used to create the context, as passed to
   * [browser.newContext([options])](https://playwright.dev/docs/api/class-browser#browser-new-context). Specific options
   * like [fixtures.viewport](https://playwright.dev/docs/api/class-fixtures#fixtures-viewport) take priority over this.
   */
  contextOptions: BrowserContextOptions;
  /**
   * Default timeout for each Playwright action in milliseconds, defaults to 0 (no timeout).
   *
   * This is a default timeout for all Playwright actions, same as configured via
   * [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout).
   */
  actionTimeout: number | undefined;
  /**
   * Timeout for each navigation action in milliseconds. Defaults to 0 (no timeout).
   *
   * This is a default navigation timeout, same as configured via
   * [page.setDefaultNavigationTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-navigation-timeout).
   */
  navigationTimeout: number | undefined;
}


/**
 * Playwright Test is based on the concept of the [test fixtures](https://playwright.dev/docs/test-fixtures). Test fixtures are used to establish
 * environment for each test, giving the test everything it needs and nothing else.
 *
 * Playwright Test looks at each test declaration, analyses the set of fixtures the test needs and prepares those fixtures
 * specifically for the test. Values prepared by the fixtures are merged into a single object that is available to the
 * `test`, hooks, annotations and other fixtures as a first parameter.
 *
 * ```js js-flavor=js
 * const { test, expect } = require('@playwright/test');
 *
 * test('basic test', async ({ page }) => {
 *   // ...
 * });
 * ```
 *
 * ```js js-flavor=ts
 * import { test, expect } from '@playwright/test';
 *
 * test('basic test', async ({ page }) => {
 *   // ...
 * });
 * ```
 *
 * Given the test above, Playwright Test will set up the `page` fixture before running the test, and tear it down after the
 * test has finished. `page` fixture provides a [Page] object that is available to the test.
 *
 * Playwright Test comes with builtin fixtures listed below, and you can add your own fixtures as well. Many fixtures are
 * designed as "options" that you can set in your
 * [testConfig.use](https://playwright.dev/docs/api/class-testconfig#test-config-use) section.
 *
 * ```js js-flavor=js
 * // @ts-check
 *
 * /** @type {import('@playwright/test').PlaywrightTestConfig} *\/
 * const config = {
 *   use: {
 *     headless: false,
 *     viewport: { width: 1280, height: 720 },
 *     ignoreHTTPSErrors: true,
 *     video: 'on-first-retry',
 *   },
 * };
 *
 * module.exports = config;
 * ```
 *
 * ```js js-flavor=ts
 * import { PlaywrightTestConfig } from '@playwright/test';
 * const config: PlaywrightTestConfig = {
 *   use: {
 *     headless: false,
 *     viewport: { width: 1280, height: 720 },
 *     ignoreHTTPSErrors: true,
 *     video: 'on-first-retry',
 *   },
 * };
 * export default config;
 * ```
 *
 * Alternatively, with [test.use(fixtures)](https://playwright.dev/docs/api/class-test#test-use) you can override some
 * options for a file.
 *
 * ```js js-flavor=js
 * // example.spec.js
 * const { test, expect } = require('@playwright/test');
 *
 * // Run tests in this file with portrait-like viewport.
 * test.use({ viewport: { width: 600, height: 900 } });
 *
 * test('my portrait test', async ({ page }) => {
 *   // ...
 * });
 * ```
 *
 * ```js js-flavor=ts
 * // example.spec.ts
 * import { test, expect } from '@playwright/test';
 *
 * // Run tests in this file with portrait-like viewport.
 * test.use({ viewport: { width: 600, height: 900 } });
 *
 * test('my portrait test', async ({ page }) => {
 *   // ...
 * });
 * ```
 *
 */
export interface PlaywrightWorkerArgs {
  playwright: typeof import('..');
  browser: Browser;
}

/**
 * Playwright Test is based on the concept of the [test fixtures](https://playwright.dev/docs/test-fixtures). Test fixtures are used to establish
 * environment for each test, giving the test everything it needs and nothing else.
 *
 * Playwright Test looks at each test declaration, analyses the set of fixtures the test needs and prepares those fixtures
 * specifically for the test. Values prepared by the fixtures are merged into a single object that is available to the
 * `test`, hooks, annotations and other fixtures as a first parameter.
 *
 * ```js js-flavor=js
 * const { test, expect } = require('@playwright/test');
 *
 * test('basic test', async ({ page }) => {
 *   // ...
 * });
 * ```
 *
 * ```js js-flavor=ts
 * import { test, expect } from '@playwright/test';
 *
 * test('basic test', async ({ page }) => {
 *   // ...
 * });
 * ```
 *
 * Given the test above, Playwright Test will set up the `page` fixture before running the test, and tear it down after the
 * test has finished. `page` fixture provides a [Page] object that is available to the test.
 *
 * Playwright Test comes with builtin fixtures listed below, and you can add your own fixtures as well. Many fixtures are
 * designed as "options" that you can set in your
 * [testConfig.use](https://playwright.dev/docs/api/class-testconfig#test-config-use) section.
 *
 * ```js js-flavor=js
 * // @ts-check
 *
 * /** @type {import('@playwright/test').PlaywrightTestConfig} *\/
 * const config = {
 *   use: {
 *     headless: false,
 *     viewport: { width: 1280, height: 720 },
 *     ignoreHTTPSErrors: true,
 *     video: 'on-first-retry',
 *   },
 * };
 *
 * module.exports = config;
 * ```
 *
 * ```js js-flavor=ts
 * import { PlaywrightTestConfig } from '@playwright/test';
 * const config: PlaywrightTestConfig = {
 *   use: {
 *     headless: false,
 *     viewport: { width: 1280, height: 720 },
 *     ignoreHTTPSErrors: true,
 *     video: 'on-first-retry',
 *   },
 * };
 * export default config;
 * ```
 *
 * Alternatively, with [test.use(fixtures)](https://playwright.dev/docs/api/class-test#test-use) you can override some
 * options for a file.
 *
 * ```js js-flavor=js
 * // example.spec.js
 * const { test, expect } = require('@playwright/test');
 *
 * // Run tests in this file with portrait-like viewport.
 * test.use({ viewport: { width: 600, height: 900 } });
 *
 * test('my portrait test', async ({ page }) => {
 *   // ...
 * });
 * ```
 *
 * ```js js-flavor=ts
 * // example.spec.ts
 * import { test, expect } from '@playwright/test';
 *
 * // Run tests in this file with portrait-like viewport.
 * test.use({ viewport: { width: 600, height: 900 } });
 *
 * test('my portrait test', async ({ page }) => {
 *   // ...
 * });
 * ```
 *
 */
export interface PlaywrightTestArgs {
  createContext: (options?: BrowserContextOptions) => Promise<BrowserContext>;
  /**
   * Isolated [BrowserContext] instance, created for each test. Since contexts are isolated between each other, every test
   * gets a fresh environment, even when multiple tests run in a single [Browser] for maximum efficiency.
   *
   * Learn how to [configure context](https://playwright.dev/docs/test-configuration) through other fixtures and options.
   *
   * The [fixtures.page](https://playwright.dev/docs/api/class-fixtures#fixtures-page) belongs to this context.
   */
  context: BrowserContext;
  /**
   * Isolated [Page] instance, created for each test. Pages are isolated between tests due to
   * [fixtures.context](https://playwright.dev/docs/api/class-fixtures#fixtures-context) isolation.
   *
   * This is the most common fixture used in a test.
   *
   * ```js js-flavor=js
   * const { test, expect } = require('@playwright/test');
   *
   * test('basic test', async ({ page }) => {
   *   await page.goto('/signin');
   *   await page.fill('#username', 'User');
   *   await page.fill('#password', 'pwd');
   *   await page.click('text=Sign in');
   *   // ...
   * });
   * ```
   *
   * ```js js-flavor=ts
   * import { test, expect } from '@playwright/test';
   *
   * test('basic test', async ({ page }) => {
   *   await page.goto('/signin');
   *   await page.fill('#username', 'User');
   *   await page.fill('#password', 'pwd');
   *   await page.click('text=Sign in');
   *   // ...
   * });
   * ```
   *
   */
  page: Page;
}

export type PlaywrightTestProject<TestArgs = {}, WorkerArgs = {}> = Project<PlaywrightTestOptions & TestArgs, PlaywrightWorkerOptions & WorkerArgs>;
export type PlaywrightTestConfig<TestArgs = {}, WorkerArgs = {}> = Config<PlaywrightTestOptions & TestArgs, PlaywrightWorkerOptions & WorkerArgs>;

/**
 * These tests are executed in Playwright environment that launches the browser
 * and provides a fresh page to each test.
 */
export const test: TestType<PlaywrightTestArgs & PlaywrightTestOptions, PlaywrightWorkerArgs & PlaywrightWorkerOptions>;
export default test;

export const _baseTest: TestType<{}, {}>;
export const expect: Expect;

// This is required to not export everything by default. See https://github.com/Microsoft/TypeScript/issues/19545#issuecomment-340490459
export {};




