// This file is generated by /utils/generate_types/index.js
/**
 * Copyright (c) Microsoft Corporation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
import { Protocol } from './protocol';
import { ChildProcess } from 'child_process';
import { EventEmitter } from 'events';
import { Readable } from 'stream';
import { Serializable, EvaluationArgument, PageFunction, PageFunctionOn, SmartHandle, ElementHandleForTag, BindingSource } from './structs';

type PageWaitForSelectorOptionsNotHidden = PageWaitForSelectorOptions & {
  state: 'visible'|'attached';
};
type ElementHandleWaitForSelectorOptionsNotHidden = ElementHandleWaitForSelectorOptions & {
  state: 'visible'|'attached';
};

/**
 * - extends: [EventEmitter]
 * 
 * Page provides methods to interact with a single tab in a [Browser], or an
 * [extension background page](https://developer.chrome.com/extensions/background_pages) in Chromium. One [Browser]
 * instance might have multiple [Page] instances.
 * 
 * This example creates a page, navigates it to a URL, and then saves a screenshot:
 * 
 * ```js
 * const { webkit } = require('playwright');  // Or 'chromium' or 'firefox'.
 * 
 * (async () => {
 *   const browser = await webkit.launch();
 *   const context = await browser.newContext();
 *   const page = await context.newPage();
 *   await page.goto('https://example.com');
 *   await page.screenshot({path: 'screenshot.png'});
 *   await browser.close();
 * })();
 * ```
 * 
 * The Page class emits various events (described below) which can be handled using any of Node's native
 * [`EventEmitter`](https://nodejs.org/api/events.html#events_class_eventemitter) methods, such as `on`, `once` or
 * `removeListener`.
 * 
 * This example logs a message for a single page `load` event:
 * 
 * ```js
 * page.once('load', () => console.log('Page loaded!'));
 * ```
 * 
 * To unsubscribe from events use the `removeListener` method:
 * 
 * ```js
 * function logRequest(interceptedRequest) {
 *   console.log('A request was made:', interceptedRequest.url());
 * }
 * page.on('request', logRequest);
 * // Sometime later...
 * page.removeListener('request', logRequest);
 * ```
 * 
 */
export interface Page {
  /**
   * Returns the value of the `pageFunction` invocation.
   * 
   * If the function passed to the
   * [page.evaluate(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageevaluate) returns a [Promise],
   * then [page.evaluate(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageevaluate) would wait for the
   * promise to resolve and return its value.
   * 
   * If the function passed to the
   * [page.evaluate(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageevaluate) returns a
   * non-[Serializable] value, then[ method: `Page.evaluate`] resolves to `undefined`. DevTools Protocol also supports
   * transferring some additional values that are not serializable by `JSON`: `-0`, `NaN`, `Infinity`, `-Infinity`, and
   * bigint literals.
   * 
   * Passing argument to `pageFunction`:
   * 
   * ```js
   * const result = await page.evaluate(([x, y]) => {
   *   return Promise.resolve(x * y);
   * }, [7, 8]);
   * console.log(result); // prints "56"
   * ```
   * 
   * A string can also be passed in instead of a function:
   * 
   * ```js
   * console.log(await page.evaluate('1 + 2')); // prints "3"
   * const x = 10;
   * console.log(await page.evaluate(`1 + ${x}`)); // prints "11"
   * ```
   * 
   * [ElementHandle] instances can be passed as an argument to the
   * [page.evaluate(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageevaluate):
   * 
   * ```js
   * const bodyHandle = await page.$('body');
   * const html = await page.evaluate(([body, suffix]) => body.innerHTML + suffix, [bodyHandle, 'hello']);
   * await bodyHandle.dispose();
   * ```
   * 
   * Shortcut for main frame's
   * [frame.evaluate(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#frameevaluate).
   * @param pageFunction Function to be evaluated in the page context
   * @param arg Optional argument to pass to `pageFunction`
   */
  evaluate<R, Arg>(pageFunction: PageFunction<Arg, R>, arg: Arg): Promise<R>;
  evaluate<R>(pageFunction: PageFunction<void, R>, arg?: any): Promise<R>;

  /**
   * Returns the value of the `pageFunction` invocation as in-page object (JSHandle).
   * 
   * The only difference between
   * [page.evaluate(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageevaluate) and
   * [page.evaluateHandle(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageevaluatehandle) is that
   * [page.evaluateHandle(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageevaluatehandle) returns
   * in-page object (JSHandle).
   * 
   * If the function passed to the
   * [page.evaluateHandle(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageevaluatehandle) returns a
   * [Promise], then
   * [page.evaluateHandle(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageevaluatehandle) would wait
   * for the promise to resolve and return its value.
   * 
   * ```js
   * const aWindowHandle = await page.evaluateHandle(() => Promise.resolve(window));
   * aWindowHandle; // Handle for the window object.
   * ```
   * 
   * A string can also be passed in instead of a function:
   * 
   * ```js
   * const aHandle = await page.evaluateHandle('document'); // Handle for the 'document'
   * ```
   * 
   * [JSHandle] instances can be passed as an argument to the
   * [page.evaluateHandle(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageevaluatehandle):
   * 
   * ```js
   * const aHandle = await page.evaluateHandle(() => document.body);
   * const resultHandle = await page.evaluateHandle(body => body.innerHTML, aHandle);
   * console.log(await resultHandle.jsonValue());
   * await resultHandle.dispose();
   * ```
   * 
   * @param pageFunction Function to be evaluated in the page context
   * @param arg Optional argument to pass to `pageFunction`
   */
  evaluateHandle<R, Arg>(pageFunction: PageFunction<Arg, R>, arg: Arg): Promise<SmartHandle<R>>;
  evaluateHandle<R>(pageFunction: PageFunction<void, R>, arg?: any): Promise<SmartHandle<R>>;

  /**
   * The method finds an element matching the specified selector within the page. If no elements match the selector, the
   * return value resolves to `null`.
   * 
   * Shortcut for main frame's [frame.$(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#frame).
   * @param selector A selector to query for. See [working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details.
   */
  $<K extends keyof HTMLElementTagNameMap>(selector: K): Promise<ElementHandleForTag<K> | null>;
  $(selector: string): Promise<ElementHandle<SVGElement | HTMLElement> | null>;

  /**
   * The method finds all elements matching the specified selector within the page. If no elements match the selector, the
   * return value resolves to `[]`.
   * 
   * Shortcut for main frame's [frame.$$(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#frame).
   * @param selector A selector to query for. See [working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details.
   */
  $$<K extends keyof HTMLElementTagNameMap>(selector: K): Promise<ElementHandleForTag<K>[]>;
  $$(selector: string): Promise<ElementHandle<SVGElement | HTMLElement>[]>;

  /**
   * The method finds an element matching the specified selector within the page and passes it as a first argument to
   * `pageFunction`. If no elements match the selector, the method throws an error. Returns the value of `pageFunction`.
   * 
   * If `pageFunction` returns a [Promise], then
   * [page.$eval(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageeval) would wait for the promise to
   * resolve and return its value.
   * 
   * Examples:
   * 
   * ```js
   * const searchValue = await page.$eval('#search', el => el.value);
   * const preloadHref = await page.$eval('link[rel=preload]', el => el.href);
   * const html = await page.$eval('.main-container', (e, suffix) => e.outerHTML + suffix, 'hello');
   * ```
   * 
   * Shortcut for main frame's [frame.$eval(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#frameeval).
   * @param selector A selector to query for. See [working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details.
   * @param pageFunction Function to be evaluated in browser context
   * @param arg Optional argument to pass to `pageFunction`
   */
  $eval<K extends keyof HTMLElementTagNameMap, R, Arg>(selector: K, pageFunction: PageFunctionOn<HTMLElementTagNameMap[K], Arg, R>, arg: Arg): Promise<R>;
  $eval<R, Arg, E extends SVGElement | HTMLElement = SVGElement | HTMLElement>(selector: string, pageFunction: PageFunctionOn<E, Arg, R>, arg: Arg): Promise<R>;
  $eval<K extends keyof HTMLElementTagNameMap, R>(selector: K, pageFunction: PageFunctionOn<HTMLElementTagNameMap[K], void, R>, arg?: any): Promise<R>;
  $eval<R, E extends SVGElement | HTMLElement = SVGElement | HTMLElement>(selector: string, pageFunction: PageFunctionOn<E, void, R>, arg?: any): Promise<R>;

  /**
   * The method finds all elements matching the specified selector within the page and passes an array of matched elements as
   * a first argument to `pageFunction`. Returns the result of `pageFunction` invocation.
   * 
   * If `pageFunction` returns a [Promise], then
   * [page.$$eval(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageeval) would wait for the promise to
   * resolve and return its value.
   * 
   * Examples:
   * 
   * ```js
   * const divCounts = await page.$$eval('div', (divs, min) => divs.length >= min, 10);
   * ```
   * 
   * @param selector A selector to query for. See [working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details.
   * @param pageFunction Function to be evaluated in browser context
   * @param arg Optional argument to pass to `pageFunction`
   */
  $$eval<K extends keyof HTMLElementTagNameMap, R, Arg>(selector: K, pageFunction: PageFunctionOn<HTMLElementTagNameMap[K][], Arg, R>, arg: Arg): Promise<R>;
  $$eval<R, Arg, E extends SVGElement | HTMLElement = SVGElement | HTMLElement>(selector: string, pageFunction: PageFunctionOn<E[], Arg, R>, arg: Arg): Promise<R>;
  $$eval<K extends keyof HTMLElementTagNameMap, R>(selector: K, pageFunction: PageFunctionOn<HTMLElementTagNameMap[K][], void, R>, arg?: any): Promise<R>;
  $$eval<R, E extends SVGElement | HTMLElement = SVGElement | HTMLElement>(selector: string, pageFunction: PageFunctionOn<E[], void, R>, arg?: any): Promise<R>;

  /**
   * Returns when the `pageFunction` returns a truthy value. It resolves to a JSHandle of the truthy value.
   * 
   * The `waitForFunction` can be used to observe viewport size change:
   * 
   * ```js
   * const { webkit } = require('playwright');  // Or 'chromium' or 'firefox'.
   * 
   * (async () => {
   *   const browser = await webkit.launch();
   *   const page = await browser.newPage();
   *   const watchDog = page.waitForFunction(() => window.innerWidth < 100);
   *   await page.setViewportSize({width: 50, height: 50});
   *   await watchDog;
   *   await browser.close();
   * })();
   * ```
   * 
   * To pass an argument to the predicate of
   * [page.waitForFunction(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagewaitforfunction) function:
   * 
   * ```js
   * const selector = '.foo';
   * await page.waitForFunction(selector => !!document.querySelector(selector), selector);
   * ```
   * 
   * Shortcut for main frame's
   * [frame.waitForFunction(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#framewaitforfunction).
   * @param pageFunction Function to be evaluated in browser context
   * @param arg Optional argument to pass to `pageFunction`
   * @param options 
   */
  waitForFunction<R, Arg>(pageFunction: PageFunction<Arg, R>, arg: Arg, options?: PageWaitForFunctionOptions): Promise<SmartHandle<R>>;
  waitForFunction<R>(pageFunction: PageFunction<void, R>, arg?: any, options?: PageWaitForFunctionOptions): Promise<SmartHandle<R>>;

  /**
   * Returns when element specified by selector satisfies `state` option. Returns `null` if waiting for `hidden` or
   * `detached`.
   * 
   * Wait for the `selector` to satisfy `state` option (either appear/disappear from dom, or become visible/hidden). If at
   * the moment of calling the method `selector` already satisfies the condition, the method will return immediately. If the
   * selector doesn't satisfy the condition for the `timeout` milliseconds, the function will throw.
   * 
   * This method works across navigations:
   * 
   * ```js
   * const { chromium } = require('playwright');  // Or 'firefox' or 'webkit'.
   * 
   * (async () => {
   *   const browser = await chromium.launch();
   *   const page = await browser.newPage();
   *   for (let currentURL of ['https://google.com', 'https://bbc.com']) {
   *     await page.goto(currentURL);
   *     const element = await page.waitForSelector('img');
   *     console.log('Loaded image: ' + await element.getAttribute('src'));
   *   }
   *   await browser.close();
   * })();
   * ```
   * 
   * @param selector A selector to query for. See [working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details.
   * @param options 
   */
  waitForSelector<K extends keyof HTMLElementTagNameMap>(selector: K, options?: PageWaitForSelectorOptionsNotHidden): Promise<ElementHandleForTag<K>>;
  waitForSelector(selector: string, options?: PageWaitForSelectorOptionsNotHidden): Promise<ElementHandle<SVGElement | HTMLElement>>;
  waitForSelector<K extends keyof HTMLElementTagNameMap>(selector: K, options: PageWaitForSelectorOptions): Promise<ElementHandleForTag<K> | null>;
  waitForSelector(selector: string, options: PageWaitForSelectorOptions): Promise<null|ElementHandle<SVGElement | HTMLElement>>;

  /**
   * The method adds a function called `name` on the `window` object of every frame in this page. When called, the function
   * executes `callback` and returns a [Promise] which resolves to the return value of `callback`. If the `callback` returns
   * a [Promise], it will be awaited.
   * 
   * The first argument of the `callback` function contains information about the caller: `{ browserContext: BrowserContext,
   * page: Page, frame: Frame }`.
   * 
   * See
   * [browserContext.exposeBinding(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextexposebinding)
   * for the context-wide version.
   * 
   * > NOTE: Functions installed via
   * [page.exposeBinding(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageexposebinding) survive
   * navigations.
   * 
   * An example of exposing page URL to all frames in a page:
   * 
   * ```js
   * const { webkit } = require('playwright');  // Or 'chromium' or 'firefox'.
   * 
   * (async () => {
   *   const browser = await webkit.launch({ headless: false });
   *   const context = await browser.newContext();
   *   const page = await context.newPage();
   *   await page.exposeBinding('pageURL', ({ page }) => page.url());
   *   await page.setContent(`
   *     <script>
   *       async function onClick() {
   *         document.querySelector('div').textContent = await window.pageURL();
   *       }
   *     </script>
   *     <button onclick="onClick()">Click me</button>
   *     <div></div>
   *   `);
   *   await page.click('button');
   * })();
   * ```
   * 
   * An example of passing an element handle:
   * 
   * ```js
   * await page.exposeBinding('clicked', async (source, element) => {
   *   console.log(await element.textContent());
   * }, { handle: true });
   * await page.setContent(`
   *   <script>
   *     document.addEventListener('click', event => window.clicked(event.target));
   *   </script>
   *   <div>Click me</div>
   *   <div>Or click me</div>
   * `);
   * ```
   * 
   * @param name Name of the function on the window object.
   * @param callback Callback function that will be called in the Playwright's context.
   * @param options 
   */
  exposeBinding(name: string, playwrightBinding: (source: BindingSource, arg: JSHandle) => any, options: { handle: true }): Promise<void>;
  exposeBinding(name: string, playwrightBinding: (source: BindingSource, ...args: any[]) => any, options?: { handle?: boolean }): Promise<void>;
  /**
   * Emitted when the page closes.
   */
  on(event: 'close', listener: () => void): this;

  /**
   * Emitted when JavaScript within the page calls one of console API methods, e.g. `console.log` or `console.dir`. Also
   * emitted if the page throws an error or a warning.
   * 
   * The arguments passed into `console.log` appear as arguments on the event handler.
   * 
   * An example of handling `console` event:
   * 
   * ```js
   * page.on('console', msg => {
   *   for (let i = 0; i < msg.args().length; ++i)
   *     console.log(`${i}: ${await msg.args()[i].jsonValue()}`);
   * });
   * page.evaluate(() => console.log('hello', 5, {foo: 'bar'}));
   * ```
   * 
   */
  on(event: 'console', listener: (consoleMessage: ConsoleMessage) => void): this;

  /**
   * Emitted when the page crashes. Browser pages might crash if they try to allocate too much memory. When the page crashes,
   * ongoing and subsequent operations will throw.
   * 
   * The most common way to deal with crashes is to catch an exception:
   * 
   * ```js
   * try {
   *   // Crash might happen during a click.
   *   await page.click('button');
   *   // Or while waiting for an event.
   *   await page.waitForEvent('popup');
   * } catch (e) {
   *   // When the page crashes, exception message contains 'crash'.
   * }
   * ```
   * 
   */
  on(event: 'crash', listener: () => void): this;

  /**
   * Emitted when a JavaScript dialog appears, such as `alert`, `prompt`, `confirm` or `beforeunload`. Playwright can respond
   * to the dialog via [dialog.accept(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#dialogaccept) or
   * [dialog.dismiss(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#dialogdismiss) methods.
   */
  on(event: 'dialog', listener: (dialog: Dialog) => void): this;

  /**
   * Emitted when the JavaScript [`DOMContentLoaded`](https://developer.mozilla.org/en-US/docs/Web/Events/DOMContentLoaded)
   * event is dispatched.
   */
  on(event: 'domcontentloaded', listener: () => void): this;

  /**
   * Emitted when attachment download started. User can access basic file operations on downloaded content via the passed
   * [Download] instance.
   * 
   * > NOTE: Browser context **must** be created with the `acceptDownloads` set to `true` when user needs access to the
   * downloaded content. If `acceptDownloads` is not set, download events are emitted, but the actual download is not
   * performed and user has no access to the downloaded files.
   */
  on(event: 'download', listener: (download: Download) => void): this;

  /**
   * Emitted when a file chooser is supposed to appear, such as after clicking the  `<input type=file>`. Playwright can
   * respond to it via setting the input files using
   * [fileChooser.setFiles(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#filechoosersetfiles) that can
   * be uploaded after that.
   * 
   * ```js
   * page.on('filechooser', async (fileChooser) => {
   *   await fileChooser.setFiles('/tmp/myfile.pdf');
   * });
   * ```
   * 
   */
  on(event: 'filechooser', listener: (fileChooser: FileChooser) => void): this;

  /**
   * Emitted when a frame is attached.
   */
  on(event: 'frameattached', listener: (frame: Frame) => void): this;

  /**
   * Emitted when a frame is detached.
   */
  on(event: 'framedetached', listener: (frame: Frame) => void): this;

  /**
   * Emitted when a frame is navigated to a new url.
   */
  on(event: 'framenavigated', listener: (frame: Frame) => void): this;

  /**
   * Emitted when the JavaScript [`load`](https://developer.mozilla.org/en-US/docs/Web/Events/load) event is dispatched.
   */
  on(event: 'load', listener: () => void): this;

  /**
   * Emitted when an uncaught exception happens within the page.
   */
  on(event: 'pageerror', listener: (error: Error) => void): this;

  /**
   * Emitted when the page opens a new tab or window. This event is emitted in addition to the
   * [browserContext.on('page')](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextonpage), but
   * only for popups relevant to this page.
   * 
   * The earliest moment that page is available is when it has navigated to the initial url. For example, when opening a
   * popup with `window.open('http://example.com')`, this event will fire when the network request to "http://example.com" is
   * done and its response has started loading in the popup.
   * 
   * ```js
   * const [popup] = await Promise.all([
   *   page.waitForEvent('popup'),
   *   page.evaluate(() => window.open('https://example.com')),
   * ]);
   * console.log(await popup.evaluate('location.href'));
   * ```
   * 
   * > NOTE: Use
   * [page.waitForLoadState(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagewaitforloadstate) to wait
   * until the page gets to a particular state (you should not need it in most cases).
   */
  on(event: 'popup', listener: (page: Page) => void): this;

  /**
   * Emitted when a page issues a request. The [request] object is read-only. In order to intercept and mutate requests, see
   * [page.route(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageroute) or
   * [browserContext.route(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextroute).
   */
  on(event: 'request', listener: (request: Request) => void): this;

  /**
   * Emitted when a request fails, for example by timing out.
   * 
   * > NOTE: HTTP Error responses, such as 404 or 503, are still successful responses from HTTP standpoint, so request will
   * complete with
   * [page.on('requestfinished')](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageonrequestfinished)
   * event and not with
   * [page.on('requestfailed')](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageonrequestfailed).
   */
  on(event: 'requestfailed', listener: (request: Request) => void): this;

  /**
   * Emitted when a request finishes successfully after downloading the response body. For a successful response, the
   * sequence of events is `request`, `response` and `requestfinished`.
   */
  on(event: 'requestfinished', listener: (request: Request) => void): this;

  /**
   * Emitted when [response] status and headers are received for a request. For a successful response, the sequence of events
   * is `request`, `response` and `requestfinished`.
   */
  on(event: 'response', listener: (response: Response) => void): this;

  /**
   * Emitted when <[WebSocket]> request is sent.
   */
  on(event: 'websocket', listener: (webSocket: WebSocket) => void): this;

  /**
   * Emitted when a dedicated [WebWorker](https://developer.mozilla.org/en-US/docs/Web/API/Web_Workers_API) is spawned by the
   * page.
   */
  on(event: 'worker', listener: (worker: Worker) => void): this;

  /**
   * Emitted when the page closes.
   */
  once(event: 'close', listener: () => void): this;

  /**
   * Emitted when JavaScript within the page calls one of console API methods, e.g. `console.log` or `console.dir`. Also
   * emitted if the page throws an error or a warning.
   * 
   * The arguments passed into `console.log` appear as arguments on the event handler.
   * 
   * An example of handling `console` event:
   * 
   * ```js
   * page.on('console', msg => {
   *   for (let i = 0; i < msg.args().length; ++i)
   *     console.log(`${i}: ${await msg.args()[i].jsonValue()}`);
   * });
   * page.evaluate(() => console.log('hello', 5, {foo: 'bar'}));
   * ```
   * 
   */
  once(event: 'console', listener: (consoleMessage: ConsoleMessage) => void): this;

  /**
   * Emitted when the page crashes. Browser pages might crash if they try to allocate too much memory. When the page crashes,
   * ongoing and subsequent operations will throw.
   * 
   * The most common way to deal with crashes is to catch an exception:
   * 
   * ```js
   * try {
   *   // Crash might happen during a click.
   *   await page.click('button');
   *   // Or while waiting for an event.
   *   await page.waitForEvent('popup');
   * } catch (e) {
   *   // When the page crashes, exception message contains 'crash'.
   * }
   * ```
   * 
   */
  once(event: 'crash', listener: () => void): this;

  /**
   * Emitted when a JavaScript dialog appears, such as `alert`, `prompt`, `confirm` or `beforeunload`. Playwright can respond
   * to the dialog via [dialog.accept(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#dialogaccept) or
   * [dialog.dismiss(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#dialogdismiss) methods.
   */
  once(event: 'dialog', listener: (dialog: Dialog) => void): this;

  /**
   * Emitted when the JavaScript [`DOMContentLoaded`](https://developer.mozilla.org/en-US/docs/Web/Events/DOMContentLoaded)
   * event is dispatched.
   */
  once(event: 'domcontentloaded', listener: () => void): this;

  /**
   * Emitted when attachment download started. User can access basic file operations on downloaded content via the passed
   * [Download] instance.
   * 
   * > NOTE: Browser context **must** be created with the `acceptDownloads` set to `true` when user needs access to the
   * downloaded content. If `acceptDownloads` is not set, download events are emitted, but the actual download is not
   * performed and user has no access to the downloaded files.
   */
  once(event: 'download', listener: (download: Download) => void): this;

  /**
   * Emitted when a file chooser is supposed to appear, such as after clicking the  `<input type=file>`. Playwright can
   * respond to it via setting the input files using
   * [fileChooser.setFiles(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#filechoosersetfiles) that can
   * be uploaded after that.
   * 
   * ```js
   * page.on('filechooser', async (fileChooser) => {
   *   await fileChooser.setFiles('/tmp/myfile.pdf');
   * });
   * ```
   * 
   */
  once(event: 'filechooser', listener: (fileChooser: FileChooser) => void): this;

  /**
   * Emitted when a frame is attached.
   */
  once(event: 'frameattached', listener: (frame: Frame) => void): this;

  /**
   * Emitted when a frame is detached.
   */
  once(event: 'framedetached', listener: (frame: Frame) => void): this;

  /**
   * Emitted when a frame is navigated to a new url.
   */
  once(event: 'framenavigated', listener: (frame: Frame) => void): this;

  /**
   * Emitted when the JavaScript [`load`](https://developer.mozilla.org/en-US/docs/Web/Events/load) event is dispatched.
   */
  once(event: 'load', listener: () => void): this;

  /**
   * Emitted when an uncaught exception happens within the page.
   */
  once(event: 'pageerror', listener: (error: Error) => void): this;

  /**
   * Emitted when the page opens a new tab or window. This event is emitted in addition to the
   * [browserContext.on('page')](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextonpage), but
   * only for popups relevant to this page.
   * 
   * The earliest moment that page is available is when it has navigated to the initial url. For example, when opening a
   * popup with `window.open('http://example.com')`, this event will fire when the network request to "http://example.com" is
   * done and its response has started loading in the popup.
   * 
   * ```js
   * const [popup] = await Promise.all([
   *   page.waitForEvent('popup'),
   *   page.evaluate(() => window.open('https://example.com')),
   * ]);
   * console.log(await popup.evaluate('location.href'));
   * ```
   * 
   * > NOTE: Use
   * [page.waitForLoadState(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagewaitforloadstate) to wait
   * until the page gets to a particular state (you should not need it in most cases).
   */
  once(event: 'popup', listener: (page: Page) => void): this;

  /**
   * Emitted when a page issues a request. The [request] object is read-only. In order to intercept and mutate requests, see
   * [page.route(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageroute) or
   * [browserContext.route(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextroute).
   */
  once(event: 'request', listener: (request: Request) => void): this;

  /**
   * Emitted when a request fails, for example by timing out.
   * 
   * > NOTE: HTTP Error responses, such as 404 or 503, are still successful responses from HTTP standpoint, so request will
   * complete with
   * [page.on('requestfinished')](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageonrequestfinished)
   * event and not with
   * [page.on('requestfailed')](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageonrequestfailed).
   */
  once(event: 'requestfailed', listener: (request: Request) => void): this;

  /**
   * Emitted when a request finishes successfully after downloading the response body. For a successful response, the
   * sequence of events is `request`, `response` and `requestfinished`.
   */
  once(event: 'requestfinished', listener: (request: Request) => void): this;

  /**
   * Emitted when [response] status and headers are received for a request. For a successful response, the sequence of events
   * is `request`, `response` and `requestfinished`.
   */
  once(event: 'response', listener: (response: Response) => void): this;

  /**
   * Emitted when <[WebSocket]> request is sent.
   */
  once(event: 'websocket', listener: (webSocket: WebSocket) => void): this;

  /**
   * Emitted when a dedicated [WebWorker](https://developer.mozilla.org/en-US/docs/Web/API/Web_Workers_API) is spawned by the
   * page.
   */
  once(event: 'worker', listener: (worker: Worker) => void): this;

  /**
   * Emitted when the page closes.
   */
  addListener(event: 'close', listener: () => void): this;

  /**
   * Emitted when JavaScript within the page calls one of console API methods, e.g. `console.log` or `console.dir`. Also
   * emitted if the page throws an error or a warning.
   * 
   * The arguments passed into `console.log` appear as arguments on the event handler.
   * 
   * An example of handling `console` event:
   * 
   * ```js
   * page.on('console', msg => {
   *   for (let i = 0; i < msg.args().length; ++i)
   *     console.log(`${i}: ${await msg.args()[i].jsonValue()}`);
   * });
   * page.evaluate(() => console.log('hello', 5, {foo: 'bar'}));
   * ```
   * 
   */
  addListener(event: 'console', listener: (consoleMessage: ConsoleMessage) => void): this;

  /**
   * Emitted when the page crashes. Browser pages might crash if they try to allocate too much memory. When the page crashes,
   * ongoing and subsequent operations will throw.
   * 
   * The most common way to deal with crashes is to catch an exception:
   * 
   * ```js
   * try {
   *   // Crash might happen during a click.
   *   await page.click('button');
   *   // Or while waiting for an event.
   *   await page.waitForEvent('popup');
   * } catch (e) {
   *   // When the page crashes, exception message contains 'crash'.
   * }
   * ```
   * 
   */
  addListener(event: 'crash', listener: () => void): this;

  /**
   * Emitted when a JavaScript dialog appears, such as `alert`, `prompt`, `confirm` or `beforeunload`. Playwright can respond
   * to the dialog via [dialog.accept(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#dialogaccept) or
   * [dialog.dismiss(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#dialogdismiss) methods.
   */
  addListener(event: 'dialog', listener: (dialog: Dialog) => void): this;

  /**
   * Emitted when the JavaScript [`DOMContentLoaded`](https://developer.mozilla.org/en-US/docs/Web/Events/DOMContentLoaded)
   * event is dispatched.
   */
  addListener(event: 'domcontentloaded', listener: () => void): this;

  /**
   * Emitted when attachment download started. User can access basic file operations on downloaded content via the passed
   * [Download] instance.
   * 
   * > NOTE: Browser context **must** be created with the `acceptDownloads` set to `true` when user needs access to the
   * downloaded content. If `acceptDownloads` is not set, download events are emitted, but the actual download is not
   * performed and user has no access to the downloaded files.
   */
  addListener(event: 'download', listener: (download: Download) => void): this;

  /**
   * Emitted when a file chooser is supposed to appear, such as after clicking the  `<input type=file>`. Playwright can
   * respond to it via setting the input files using
   * [fileChooser.setFiles(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#filechoosersetfiles) that can
   * be uploaded after that.
   * 
   * ```js
   * page.on('filechooser', async (fileChooser) => {
   *   await fileChooser.setFiles('/tmp/myfile.pdf');
   * });
   * ```
   * 
   */
  addListener(event: 'filechooser', listener: (fileChooser: FileChooser) => void): this;

  /**
   * Emitted when a frame is attached.
   */
  addListener(event: 'frameattached', listener: (frame: Frame) => void): this;

  /**
   * Emitted when a frame is detached.
   */
  addListener(event: 'framedetached', listener: (frame: Frame) => void): this;

  /**
   * Emitted when a frame is navigated to a new url.
   */
  addListener(event: 'framenavigated', listener: (frame: Frame) => void): this;

  /**
   * Emitted when the JavaScript [`load`](https://developer.mozilla.org/en-US/docs/Web/Events/load) event is dispatched.
   */
  addListener(event: 'load', listener: () => void): this;

  /**
   * Emitted when an uncaught exception happens within the page.
   */
  addListener(event: 'pageerror', listener: (error: Error) => void): this;

  /**
   * Emitted when the page opens a new tab or window. This event is emitted in addition to the
   * [browserContext.on('page')](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextonpage), but
   * only for popups relevant to this page.
   * 
   * The earliest moment that page is available is when it has navigated to the initial url. For example, when opening a
   * popup with `window.open('http://example.com')`, this event will fire when the network request to "http://example.com" is
   * done and its response has started loading in the popup.
   * 
   * ```js
   * const [popup] = await Promise.all([
   *   page.waitForEvent('popup'),
   *   page.evaluate(() => window.open('https://example.com')),
   * ]);
   * console.log(await popup.evaluate('location.href'));
   * ```
   * 
   * > NOTE: Use
   * [page.waitForLoadState(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagewaitforloadstate) to wait
   * until the page gets to a particular state (you should not need it in most cases).
   */
  addListener(event: 'popup', listener: (page: Page) => void): this;

  /**
   * Emitted when a page issues a request. The [request] object is read-only. In order to intercept and mutate requests, see
   * [page.route(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageroute) or
   * [browserContext.route(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextroute).
   */
  addListener(event: 'request', listener: (request: Request) => void): this;

  /**
   * Emitted when a request fails, for example by timing out.
   * 
   * > NOTE: HTTP Error responses, such as 404 or 503, are still successful responses from HTTP standpoint, so request will
   * complete with
   * [page.on('requestfinished')](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageonrequestfinished)
   * event and not with
   * [page.on('requestfailed')](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageonrequestfailed).
   */
  addListener(event: 'requestfailed', listener: (request: Request) => void): this;

  /**
   * Emitted when a request finishes successfully after downloading the response body. For a successful response, the
   * sequence of events is `request`, `response` and `requestfinished`.
   */
  addListener(event: 'requestfinished', listener: (request: Request) => void): this;

  /**
   * Emitted when [response] status and headers are received for a request. For a successful response, the sequence of events
   * is `request`, `response` and `requestfinished`.
   */
  addListener(event: 'response', listener: (response: Response) => void): this;

  /**
   * Emitted when <[WebSocket]> request is sent.
   */
  addListener(event: 'websocket', listener: (webSocket: WebSocket) => void): this;

  /**
   * Emitted when a dedicated [WebWorker](https://developer.mozilla.org/en-US/docs/Web/API/Web_Workers_API) is spawned by the
   * page.
   */
  addListener(event: 'worker', listener: (worker: Worker) => void): this;

  /**
   * Emitted when the page closes.
   */
  removeListener(event: 'close', listener: () => void): this;

  /**
   * Emitted when JavaScript within the page calls one of console API methods, e.g. `console.log` or `console.dir`. Also
   * emitted if the page throws an error or a warning.
   * 
   * The arguments passed into `console.log` appear as arguments on the event handler.
   * 
   * An example of handling `console` event:
   * 
   * ```js
   * page.on('console', msg => {
   *   for (let i = 0; i < msg.args().length; ++i)
   *     console.log(`${i}: ${await msg.args()[i].jsonValue()}`);
   * });
   * page.evaluate(() => console.log('hello', 5, {foo: 'bar'}));
   * ```
   * 
   */
  removeListener(event: 'console', listener: (consoleMessage: ConsoleMessage) => void): this;

  /**
   * Emitted when the page crashes. Browser pages might crash if they try to allocate too much memory. When the page crashes,
   * ongoing and subsequent operations will throw.
   * 
   * The most common way to deal with crashes is to catch an exception:
   * 
   * ```js
   * try {
   *   // Crash might happen during a click.
   *   await page.click('button');
   *   // Or while waiting for an event.
   *   await page.waitForEvent('popup');
   * } catch (e) {
   *   // When the page crashes, exception message contains 'crash'.
   * }
   * ```
   * 
   */
  removeListener(event: 'crash', listener: () => void): this;

  /**
   * Emitted when a JavaScript dialog appears, such as `alert`, `prompt`, `confirm` or `beforeunload`. Playwright can respond
   * to the dialog via [dialog.accept(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#dialogaccept) or
   * [dialog.dismiss(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#dialogdismiss) methods.
   */
  removeListener(event: 'dialog', listener: (dialog: Dialog) => void): this;

  /**
   * Emitted when the JavaScript [`DOMContentLoaded`](https://developer.mozilla.org/en-US/docs/Web/Events/DOMContentLoaded)
   * event is dispatched.
   */
  removeListener(event: 'domcontentloaded', listener: () => void): this;

  /**
   * Emitted when attachment download started. User can access basic file operations on downloaded content via the passed
   * [Download] instance.
   * 
   * > NOTE: Browser context **must** be created with the `acceptDownloads` set to `true` when user needs access to the
   * downloaded content. If `acceptDownloads` is not set, download events are emitted, but the actual download is not
   * performed and user has no access to the downloaded files.
   */
  removeListener(event: 'download', listener: (download: Download) => void): this;

  /**
   * Emitted when a file chooser is supposed to appear, such as after clicking the  `<input type=file>`. Playwright can
   * respond to it via setting the input files using
   * [fileChooser.setFiles(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#filechoosersetfiles) that can
   * be uploaded after that.
   * 
   * ```js
   * page.on('filechooser', async (fileChooser) => {
   *   await fileChooser.setFiles('/tmp/myfile.pdf');
   * });
   * ```
   * 
   */
  removeListener(event: 'filechooser', listener: (fileChooser: FileChooser) => void): this;

  /**
   * Emitted when a frame is attached.
   */
  removeListener(event: 'frameattached', listener: (frame: Frame) => void): this;

  /**
   * Emitted when a frame is detached.
   */
  removeListener(event: 'framedetached', listener: (frame: Frame) => void): this;

  /**
   * Emitted when a frame is navigated to a new url.
   */
  removeListener(event: 'framenavigated', listener: (frame: Frame) => void): this;

  /**
   * Emitted when the JavaScript [`load`](https://developer.mozilla.org/en-US/docs/Web/Events/load) event is dispatched.
   */
  removeListener(event: 'load', listener: () => void): this;

  /**
   * Emitted when an uncaught exception happens within the page.
   */
  removeListener(event: 'pageerror', listener: (error: Error) => void): this;

  /**
   * Emitted when the page opens a new tab or window. This event is emitted in addition to the
   * [browserContext.on('page')](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextonpage), but
   * only for popups relevant to this page.
   * 
   * The earliest moment that page is available is when it has navigated to the initial url. For example, when opening a
   * popup with `window.open('http://example.com')`, this event will fire when the network request to "http://example.com" is
   * done and its response has started loading in the popup.
   * 
   * ```js
   * const [popup] = await Promise.all([
   *   page.waitForEvent('popup'),
   *   page.evaluate(() => window.open('https://example.com')),
   * ]);
   * console.log(await popup.evaluate('location.href'));
   * ```
   * 
   * > NOTE: Use
   * [page.waitForLoadState(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagewaitforloadstate) to wait
   * until the page gets to a particular state (you should not need it in most cases).
   */
  removeListener(event: 'popup', listener: (page: Page) => void): this;

  /**
   * Emitted when a page issues a request. The [request] object is read-only. In order to intercept and mutate requests, see
   * [page.route(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageroute) or
   * [browserContext.route(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextroute).
   */
  removeListener(event: 'request', listener: (request: Request) => void): this;

  /**
   * Emitted when a request fails, for example by timing out.
   * 
   * > NOTE: HTTP Error responses, such as 404 or 503, are still successful responses from HTTP standpoint, so request will
   * complete with
   * [page.on('requestfinished')](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageonrequestfinished)
   * event and not with
   * [page.on('requestfailed')](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageonrequestfailed).
   */
  removeListener(event: 'requestfailed', listener: (request: Request) => void): this;

  /**
   * Emitted when a request finishes successfully after downloading the response body. For a successful response, the
   * sequence of events is `request`, `response` and `requestfinished`.
   */
  removeListener(event: 'requestfinished', listener: (request: Request) => void): this;

  /**
   * Emitted when [response] status and headers are received for a request. For a successful response, the sequence of events
   * is `request`, `response` and `requestfinished`.
   */
  removeListener(event: 'response', listener: (response: Response) => void): this;

  /**
   * Emitted when <[WebSocket]> request is sent.
   */
  removeListener(event: 'websocket', listener: (webSocket: WebSocket) => void): this;

  /**
   * Emitted when a dedicated [WebWorker](https://developer.mozilla.org/en-US/docs/Web/API/Web_Workers_API) is spawned by the
   * page.
   */
  removeListener(event: 'worker', listener: (worker: Worker) => void): this;

  /**
   * Emitted when the page closes.
   */
  off(event: 'close', listener: () => void): this;

  /**
   * Emitted when JavaScript within the page calls one of console API methods, e.g. `console.log` or `console.dir`. Also
   * emitted if the page throws an error or a warning.
   * 
   * The arguments passed into `console.log` appear as arguments on the event handler.
   * 
   * An example of handling `console` event:
   * 
   * ```js
   * page.on('console', msg => {
   *   for (let i = 0; i < msg.args().length; ++i)
   *     console.log(`${i}: ${await msg.args()[i].jsonValue()}`);
   * });
   * page.evaluate(() => console.log('hello', 5, {foo: 'bar'}));
   * ```
   * 
   */
  off(event: 'console', listener: (consoleMessage: ConsoleMessage) => void): this;

  /**
   * Emitted when the page crashes. Browser pages might crash if they try to allocate too much memory. When the page crashes,
   * ongoing and subsequent operations will throw.
   * 
   * The most common way to deal with crashes is to catch an exception:
   * 
   * ```js
   * try {
   *   // Crash might happen during a click.
   *   await page.click('button');
   *   // Or while waiting for an event.
   *   await page.waitForEvent('popup');
   * } catch (e) {
   *   // When the page crashes, exception message contains 'crash'.
   * }
   * ```
   * 
   */
  off(event: 'crash', listener: () => void): this;

  /**
   * Emitted when a JavaScript dialog appears, such as `alert`, `prompt`, `confirm` or `beforeunload`. Playwright can respond
   * to the dialog via [dialog.accept(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#dialogaccept) or
   * [dialog.dismiss(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#dialogdismiss) methods.
   */
  off(event: 'dialog', listener: (dialog: Dialog) => void): this;

  /**
   * Emitted when the JavaScript [`DOMContentLoaded`](https://developer.mozilla.org/en-US/docs/Web/Events/DOMContentLoaded)
   * event is dispatched.
   */
  off(event: 'domcontentloaded', listener: () => void): this;

  /**
   * Emitted when attachment download started. User can access basic file operations on downloaded content via the passed
   * [Download] instance.
   * 
   * > NOTE: Browser context **must** be created with the `acceptDownloads` set to `true` when user needs access to the
   * downloaded content. If `acceptDownloads` is not set, download events are emitted, but the actual download is not
   * performed and user has no access to the downloaded files.
   */
  off(event: 'download', listener: (download: Download) => void): this;

  /**
   * Emitted when a file chooser is supposed to appear, such as after clicking the  `<input type=file>`. Playwright can
   * respond to it via setting the input files using
   * [fileChooser.setFiles(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#filechoosersetfiles) that can
   * be uploaded after that.
   * 
   * ```js
   * page.on('filechooser', async (fileChooser) => {
   *   await fileChooser.setFiles('/tmp/myfile.pdf');
   * });
   * ```
   * 
   */
  off(event: 'filechooser', listener: (fileChooser: FileChooser) => void): this;

  /**
   * Emitted when a frame is attached.
   */
  off(event: 'frameattached', listener: (frame: Frame) => void): this;

  /**
   * Emitted when a frame is detached.
   */
  off(event: 'framedetached', listener: (frame: Frame) => void): this;

  /**
   * Emitted when a frame is navigated to a new url.
   */
  off(event: 'framenavigated', listener: (frame: Frame) => void): this;

  /**
   * Emitted when the JavaScript [`load`](https://developer.mozilla.org/en-US/docs/Web/Events/load) event is dispatched.
   */
  off(event: 'load', listener: () => void): this;

  /**
   * Emitted when an uncaught exception happens within the page.
   */
  off(event: 'pageerror', listener: (error: Error) => void): this;

  /**
   * Emitted when the page opens a new tab or window. This event is emitted in addition to the
   * [browserContext.on('page')](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextonpage), but
   * only for popups relevant to this page.
   * 
   * The earliest moment that page is available is when it has navigated to the initial url. For example, when opening a
   * popup with `window.open('http://example.com')`, this event will fire when the network request to "http://example.com" is
   * done and its response has started loading in the popup.
   * 
   * ```js
   * const [popup] = await Promise.all([
   *   page.waitForEvent('popup'),
   *   page.evaluate(() => window.open('https://example.com')),
   * ]);
   * console.log(await popup.evaluate('location.href'));
   * ```
   * 
   * > NOTE: Use
   * [page.waitForLoadState(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagewaitforloadstate) to wait
   * until the page gets to a particular state (you should not need it in most cases).
   */
  off(event: 'popup', listener: (page: Page) => void): this;

  /**
   * Emitted when a page issues a request. The [request] object is read-only. In order to intercept and mutate requests, see
   * [page.route(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageroute) or
   * [browserContext.route(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextroute).
   */
  off(event: 'request', listener: (request: Request) => void): this;

  /**
   * Emitted when a request fails, for example by timing out.
   * 
   * > NOTE: HTTP Error responses, such as 404 or 503, are still successful responses from HTTP standpoint, so request will
   * complete with
   * [page.on('requestfinished')](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageonrequestfinished)
   * event and not with
   * [page.on('requestfailed')](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageonrequestfailed).
   */
  off(event: 'requestfailed', listener: (request: Request) => void): this;

  /**
   * Emitted when a request finishes successfully after downloading the response body. For a successful response, the
   * sequence of events is `request`, `response` and `requestfinished`.
   */
  off(event: 'requestfinished', listener: (request: Request) => void): this;

  /**
   * Emitted when [response] status and headers are received for a request. For a successful response, the sequence of events
   * is `request`, `response` and `requestfinished`.
   */
  off(event: 'response', listener: (response: Response) => void): this;

  /**
   * Emitted when <[WebSocket]> request is sent.
   */
  off(event: 'websocket', listener: (webSocket: WebSocket) => void): this;

  /**
   * Emitted when a dedicated [WebWorker](https://developer.mozilla.org/en-US/docs/Web/API/Web_Workers_API) is spawned by the
   * page.
   */
  off(event: 'worker', listener: (worker: Worker) => void): this;

  accessibility: Accessibility;

  /**
   * Adds a script which would be evaluated in one of the following scenarios:
   * - Whenever the page is navigated.
   * - Whenever the child frame is attached or navigated. In this case, the script is evaluated in the context of the newly
   *   attached frame.
   * 
   * The script is evaluated after the document was created but before any of its scripts were run. This is useful to amend
   * the JavaScript environment, e.g. to seed `Math.random`.
   * 
   * An example of overriding `Math.random` before the page loads:
   * 
   * ```js browser
   * // preload.js
   * Math.random = () => 42;
   * ```
   * 
   * ```js
   * // In your playwright script, assuming the preload.js file is in same directory
   * await page.addInitScript({ path: './preload.js' });
   * ```
   * 
   * > NOTE: The order of evaluation of multiple scripts installed via
   * [browserContext.addInitScript(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextaddinitscript)
   * and [page.addInitScript(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageaddinitscript) is not
   * defined.
   * @param script Script to be evaluated in the page.
   * @param arg Optional argument to pass to `script` (only supported when passing a function).
   */
  addInitScript(script: Function|string|{
    /**
     * Path to the JavaScript file. If `path` is a relative path, then it is resolved relative to the current working
     * directory. Optional.
     */
    path?: string;

    /**
     * Raw script content. Optional.
     */
    content?: string;
  }, arg?: Serializable): Promise<void>;

  /**
   * Adds a `<script>` tag into the page with the desired url or content. Returns the added tag when the script's onload
   * fires or when the script content was injected into frame.
   * 
   * Shortcut for main frame's
   * [frame.addScriptTag(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#frameaddscripttag).
   * @param options 
   */
  addScriptTag(options?: {
    /**
     * Raw JavaScript content to be injected into frame.
     */
    content?: string;

    /**
     * Path to the JavaScript file to be injected into frame. If `path` is a relative path, then it is resolved relative to the
     * current working directory.
     */
    path?: string;

    /**
     * Script type. Use 'module' in order to load a Javascript ES6 module. See
     * [script](https://developer.mozilla.org/en-US/docs/Web/HTML/Element/script) for more details.
     */
    type?: string;

    /**
     * URL of a script to be added.
     */
    url?: string;
  }): Promise<ElementHandle>;

  /**
   * Adds a `<link rel="stylesheet">` tag into the page with the desired url or a `<style type="text/css">` tag with the
   * content. Returns the added tag when the stylesheet's onload fires or when the CSS content was injected into frame.
   * 
   * Shortcut for main frame's
   * [frame.addStyleTag(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#frameaddstyletag).
   * @param options 
   */
  addStyleTag(options?: {
    /**
     * Raw CSS content to be injected into frame.
     */
    content?: string;

    /**
     * Path to the CSS file to be injected into frame. If `path` is a relative path, then it is resolved relative to the
     * current working directory.
     */
    path?: string;

    /**
     * URL of the `<link>` tag.
     */
    url?: string;
  }): Promise<ElementHandle>;

  /**
   * Brings page to front (activates tab).
   */
  bringToFront(): Promise<void>;

  /**
   * This method checks an element matching `selector` by performing the following steps:
   * 1. Find an element match matching `selector`. If there is none, wait until a matching element is attached to the DOM.
   * 1. Ensure that matched element is a checkbox or a radio input. If not, this method rejects. If the element is already
   *    checked, this method returns immediately.
   * 1. Wait for [actionability](https://github.com/microsoft/playwright/blob/master/docs/actionability.md) checks on the matched element, unless `force` option is set. If the
   *    element is detached during the checks, the whole action is retried.
   * 1. Scroll the element into view if needed.
   * 1. Use [page.mouse](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagemouse) to click in the center
   *    of the element.
   * 1. Wait for initiated navigations to either succeed or fail, unless `noWaitAfter` option is set.
   * 1. Ensure that the element is now checked. If not, this method rejects.
   * 
   * When all steps combined have not finished during the specified `timeout`, this method rejects with a [TimeoutError].
   * Passing zero timeout disables this.
   * 
   * Shortcut for main frame's [frame.check(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#framecheck).
   * @param selector A selector to search for element. If there are multiple elements satisfying the selector, the first will be used. See [working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details.
   * @param options 
   */
  check(selector: string, options?: {
    /**
     * Whether to bypass the [actionability](https://github.com/microsoft/playwright/blob/master/docs/actionability.md) checks. Defaults to `false`.
     */
    force?: boolean;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You can
     * opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as navigating to
     * inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed by
     * using the
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * This method clicks an element matching `selector` by performing the following steps:
   * 1. Find an element match matching `selector`. If there is none, wait until a matching element is attached to the DOM.
   * 1. Wait for [actionability](https://github.com/microsoft/playwright/blob/master/docs/actionability.md) checks on the matched element, unless `force` option is set. If the
   *    element is detached during the checks, the whole action is retried.
   * 1. Scroll the element into view if needed.
   * 1. Use [page.mouse](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagemouse) to click in the center
   *    of the element, or the specified `position`.
   * 1. Wait for initiated navigations to either succeed or fail, unless `noWaitAfter` option is set.
   * 
   * When all steps combined have not finished during the specified `timeout`, this method rejects with a [TimeoutError].
   * Passing zero timeout disables this.
   * 
   * Shortcut for main frame's [frame.click(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#frameclick).
   * @param selector A selector to search for element. If there are multiple elements satisfying the selector, the first will be used. See [working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details.
   * @param options 
   */
  click(selector: string, options?: {
    /**
     * Defaults to `left`.
     */
    button?: "left"|"right"|"middle";

    /**
     * defaults to 1. See [UIEvent.detail].
     */
    clickCount?: number;

    /**
     * Time to wait between `mousedown` and `mouseup` in milliseconds. Defaults to 0.
     */
    delay?: number;

    /**
     * Whether to bypass the [actionability](https://github.com/microsoft/playwright/blob/master/docs/actionability.md) checks. Defaults to `false`.
     */
    force?: boolean;

    /**
     * Modifier keys to press. Ensures that only these modifiers are pressed during the operation, and then restores current
     * modifiers back. If not specified, currently pressed modifiers are used.
     */
    modifiers?: Array<"Alt"|"Control"|"Meta"|"Shift">;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You can
     * opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as navigating to
     * inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * A point to use relative to the top-left corner of element padding box. If not specified, uses some visible point of the
     * element.
     */
    position?: {
      x: number;

      y: number;
    };

    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed by
     * using the
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * If `runBeforeUnload` is `false`, does not run any unload handlers and waits for the page to be closed. If
   * `runBeforeUnload` is `true` the method will run unload handlers, but will **not** wait for the page to close.
   * 
   * By default, `page.close()` **does not** run `beforeunload` handlers.
   * 
   * > NOTE: if `runBeforeUnload` is passed as true, a `beforeunload` dialog might be summoned and should be handled manually
   * via [page.on('dialog')](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageondialog) event.
   * @param options 
   */
  close(options?: {
    /**
     * Defaults to `false`. Whether to run the
     * [before unload](https://developer.mozilla.org/en-US/docs/Web/Events/beforeunload) page handlers.
     */
    runBeforeUnload?: boolean;
  }): Promise<void>;

  /**
   * Gets the full HTML contents of the page, including the doctype.
   */
  content(): Promise<string>;

  /**
   * Get the browser context that the page belongs to.
   */
  context(): BrowserContext;

  /**
   * Browser-specific Coverage implementation, only available for Chromium atm. See
   * [ChromiumCoverage](https://github.com/microsoft/playwright/blob/master/docs/api.md#class-chromiumcoverage) for more details.
   */
  coverage: null|ChromiumCoverage;

  /**
   * This method double clicks an element matching `selector` by performing the following steps:
   * 1. Find an element match matching `selector`. If there is none, wait until a matching element is attached to the DOM.
   * 1. Wait for [actionability](https://github.com/microsoft/playwright/blob/master/docs/actionability.md) checks on the matched element, unless `force` option is set. If the
   *    element is detached during the checks, the whole action is retried.
   * 1. Scroll the element into view if needed.
   * 1. Use [page.mouse](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagemouse) to double click in the
   *    center of the element, or the specified `position`.
   * 1. Wait for initiated navigations to either succeed or fail, unless `noWaitAfter` option is set. Note that if the
   *    first click of the `dblclick()` triggers a navigation event, this method will reject.
   * 
   * When all steps combined have not finished during the specified `timeout`, this method rejects with a [TimeoutError].
   * Passing zero timeout disables this.
   * 
   * > NOTE: `page.dblclick()` dispatches two `click` events and a single `dblclick` event.
   * 
   * Shortcut for main frame's
   * [frame.dblclick(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#framedblclick).
   * @param selector A selector to search for element. If there are multiple elements satisfying the selector, the first will be used. See [working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details.
   * @param options 
   */
  dblclick(selector: string, options?: {
    /**
     * Defaults to `left`.
     */
    button?: "left"|"right"|"middle";

    /**
     * Time to wait between `mousedown` and `mouseup` in milliseconds. Defaults to 0.
     */
    delay?: number;

    /**
     * Whether to bypass the [actionability](https://github.com/microsoft/playwright/blob/master/docs/actionability.md) checks. Defaults to `false`.
     */
    force?: boolean;

    /**
     * Modifier keys to press. Ensures that only these modifiers are pressed during the operation, and then restores current
     * modifiers back. If not specified, currently pressed modifiers are used.
     */
    modifiers?: Array<"Alt"|"Control"|"Meta"|"Shift">;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You can
     * opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as navigating to
     * inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * A point to use relative to the top-left corner of element padding box. If not specified, uses some visible point of the
     * element.
     */
    position?: {
      x: number;

      y: number;
    };

    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed by
     * using the
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * The snippet below dispatches the `click` event on the element. Regardless of the visibility state of the elment, `click`
   * is dispatched. This is equivalend to calling
   * [element.click()](https://developer.mozilla.org/en-US/docs/Web/API/HTMLElement/click).
   * 
   * ```js
   * await page.dispatchEvent('button#submit', 'click');
   * ```
   * 
   * Under the hood, it creates an instance of an event based on the given `type`, initializes it with `eventInit` properties
   * and dispatches it on the element. Events are `composed`, `cancelable` and bubble by default.
   * 
   * Since `eventInit` is event-specific, please refer to the events documentation for the lists of initial properties:
   * - [DragEvent](https://developer.mozilla.org/en-US/docs/Web/API/DragEvent/DragEvent)
   * - [FocusEvent](https://developer.mozilla.org/en-US/docs/Web/API/FocusEvent/FocusEvent)
   * - [KeyboardEvent](https://developer.mozilla.org/en-US/docs/Web/API/KeyboardEvent/KeyboardEvent)
   * - [MouseEvent](https://developer.mozilla.org/en-US/docs/Web/API/MouseEvent/MouseEvent)
   * - [PointerEvent](https://developer.mozilla.org/en-US/docs/Web/API/PointerEvent/PointerEvent)
   * - [TouchEvent](https://developer.mozilla.org/en-US/docs/Web/API/TouchEvent/TouchEvent)
   * - [Event](https://developer.mozilla.org/en-US/docs/Web/API/Event/Event)
   * 
   * You can also specify `JSHandle` as the property value if you want live objects to be passed into the event:
   * 
   * ```js
   * // Note you can only create DataTransfer in Chromium and Firefox
   * const dataTransfer = await page.evaluateHandle(() => new DataTransfer());
   * await page.dispatchEvent('#source', 'dragstart', { dataTransfer });
   * ```
   * 
   * @param selector A selector to search for element. If there are multiple elements satisfying the selector, the first will be used. See [working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details.
   * @param type DOM event type: `"click"`, `"dragstart"`, etc.
   * @param eventInit Optional event-specific initialization properties.
   * @param options 
   */
  dispatchEvent(selector: string, type: string, eventInit?: EvaluationArgument, options?: {
    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed by
     * using the
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * ```js
   * await page.evaluate(() => matchMedia('screen').matches);
   * // → true
   * await page.evaluate(() => matchMedia('print').matches);
   * // → false
   * 
   * await page.emulateMedia({ media: 'print' });
   * await page.evaluate(() => matchMedia('screen').matches);
   * // → false
   * await page.evaluate(() => matchMedia('print').matches);
   * // → true
   * 
   * await page.emulateMedia({});
   * await page.evaluate(() => matchMedia('screen').matches);
   * // → true
   * await page.evaluate(() => matchMedia('print').matches);
   * // → false
   * ```
   * 
   * ```js
   * await page.emulateMedia({ colorScheme: 'dark' });
   * await page.evaluate(() => matchMedia('(prefers-color-scheme: dark)').matches);
   * // → true
   * await page.evaluate(() => matchMedia('(prefers-color-scheme: light)').matches);
   * // → false
   * await page.evaluate(() => matchMedia('(prefers-color-scheme: no-preference)').matches);
   * // → false
   * ```
   * 
   * @param params 
   */
  emulateMedia(params: {
    /**
     * Changes the CSS media type of the page. The only allowed values are `'screen'`, `'print'` and `null`. Passing `null`
     * disables CSS media emulation. Omitting `media` or passing `undefined` does not change the emulated value. Optional.
     */
    media?: null|"screen"|"print";

    /**
     * Emulates `'prefers-colors-scheme'` media feature, supported values are `'light'`, `'dark'`, `'no-preference'`. Passing
     * `null` disables color scheme emulation. Omitting `colorScheme` or passing `undefined` does not change the emulated
     * value. Optional.
     */
    colorScheme?: null|"light"|"dark"|"no-preference";
  }): Promise<void>;

  /**
   * The method adds a function called `name` on the `window` object of every frame in the page. When called, the function
   * executes `callback` and returns a [Promise] which resolves to the return value of `callback`.
   * 
   * If the `callback` returns a [Promise], it will be awaited.
   * 
   * See
   * [browserContext.exposeFunction(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextexposefunction)
   * for context-wide exposed function.
   * 
   * > NOTE: Functions installed via
   * [page.exposeFunction(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageexposefunction) survive
   * navigations.
   * 
   * An example of adding an `sha1` function to the page:
   * 
   * ```js
   * const { webkit } = require('playwright');  // Or 'chromium' or 'firefox'.
   * const crypto = require('crypto');
   * 
   * (async () => {
   *   const browser = await webkit.launch({ headless: false });
   *   const page = await browser.newPage();
   *   await page.exposeFunction('sha1', text => crypto.createHash('sha1').update(text).digest('hex'));
   *   await page.setContent(`
   *     <script>
   *       async function onClick() {
   *         document.querySelector('div').textContent = await window.sha1('PLAYWRIGHT');
   *       }
   *     </script>
   *     <button onclick="onClick()">Click me</button>
   *     <div></div>
   *   `);
   *   await page.click('button');
   * })();
   * ```
   * 
   * @param name Name of the function on the window object
   * @param callback Callback function which will be called in Playwright's context.
   */
  exposeFunction(name: string, callback: Function): Promise<void>;

  /**
   * This method waits for an element matching `selector`, waits for [actionability](https://github.com/microsoft/playwright/blob/master/docs/actionability.md) checks, focuses the
   * element, fills it and triggers an `input` event after filling. If the element matching `selector` is not an `<input>`,
   * `<textarea>` or `[contenteditable]` element, this method throws an error. Note that you can pass an empty string to
   * clear the input field.
   * 
   * To send fine-grained keyboard events, use
   * [page.type(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagetype).
   * 
   * Shortcut for main frame's [frame.fill(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#framefill)
   * @param selector A selector to search for element. If there are multiple elements satisfying the selector, the first will be used. See [working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details.
   * @param value Value to fill for the `<input>`, `<textarea>` or `[contenteditable]` element.
   * @param options 
   */
  fill(selector: string, value: string, options?: {
    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You can
     * opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as navigating to
     * inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed by
     * using the
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * This method fetches an element with `selector` and focuses it. If there's no element matching `selector`, the method
   * waits until a matching element appears in the DOM.
   * 
   * Shortcut for main frame's [frame.focus(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#framefocus).
   * @param selector A selector to search for element. If there are multiple elements satisfying the selector, the first will be used. See [working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details.
   * @param options 
   */
  focus(selector: string, options?: {
    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed by
     * using the
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * Returns frame matching the specified criteria. Either `name` or `url` must be specified.
   * 
   * ```js
   * const frame = page.frame('frame-name');
   * ```
   * 
   * ```js
   * const frame = page.frame({ url: /.*domain.*\/ });
   * ```
   * 
   * @param frameSelector Frame name or other frame lookup options.
   */
  frame(frameSelector: string|{
    /**
     * Frame name specified in the `iframe`'s `name` attribute. Optional.
     */
    name?: string;

    /**
     * A glob pattern, regex pattern or predicate receiving frame's `url` as a [URL] object. Optional.
     */
    url?: string|RegExp|((url: URL) => boolean);
  }): null|Frame;

  /**
   * An array of all frames attached to the page.
   */
  frames(): Array<Frame>;

  /**
   * Returns element attribute value.
   * @param selector A selector to search for element. If there are multiple elements satisfying the selector, the first will be used. See [working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details.
   * @param name Attribute name to get the value for.
   * @param options 
   */
  getAttribute(selector: string, name: string, options?: {
    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed by
     * using the
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;
  }): Promise<null|string>;

  /**
   * Returns the main resource response. In case of multiple redirects, the navigation will resolve with the response of the
   * last redirect. If can not go back, returns `null`.
   * 
   * Navigate to the previous page in history.
   * @param options 
   */
  goBack(options?: {
    /**
     * Maximum operation time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be
     * changed by using the
     * [browserContext.setDefaultNavigationTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaultnavigationtimeout),
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout),
     * [page.setDefaultNavigationTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaultnavigationtimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;

    /**
     * When to consider operation succeeded, defaults to `load`. Events can be either:
     * - `'domcontentloaded'` - consider operation to be finished when the `DOMContentLoaded` event is fired.
     * - `'load'` - consider operation to be finished when the `load` event is fired.
     * - `'networkidle'` - consider operation to be finished when there are no network connections for at least `500` ms.
     */
    waitUntil?: "load"|"domcontentloaded"|"networkidle";
  }): Promise<null|Response>;

  /**
   * Returns the main resource response. In case of multiple redirects, the navigation will resolve with the response of the
   * last redirect. If can not go forward, returns `null`.
   * 
   * Navigate to the next page in history.
   * @param options 
   */
  goForward(options?: {
    /**
     * Maximum operation time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be
     * changed by using the
     * [browserContext.setDefaultNavigationTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaultnavigationtimeout),
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout),
     * [page.setDefaultNavigationTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaultnavigationtimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;

    /**
     * When to consider operation succeeded, defaults to `load`. Events can be either:
     * - `'domcontentloaded'` - consider operation to be finished when the `DOMContentLoaded` event is fired.
     * - `'load'` - consider operation to be finished when the `load` event is fired.
     * - `'networkidle'` - consider operation to be finished when there are no network connections for at least `500` ms.
     */
    waitUntil?: "load"|"domcontentloaded"|"networkidle";
  }): Promise<null|Response>;

  /**
   * Returns the main resource response. In case of multiple redirects, the navigation will resolve with the response of the
   * last redirect.
   * 
   * `page.goto` will throw an error if:
   * - there's an SSL error (e.g. in case of self-signed certificates).
   * - target URL is invalid.
   * - the `timeout` is exceeded during navigation.
   * - the remote server does not respond or is unreachable.
   * - the main resource failed to load.
   * 
   * `page.goto` will not throw an error when any valid HTTP status code is returned by the remote server, including 404 "Not
   * Found" and 500 "Internal Server Error".  The status code for such responses can be retrieved by calling
   * [response.status(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#responsestatus).
   * 
   * > NOTE: `page.goto` either throws an error or returns a main resource response. The only exceptions are navigation to
   * `about:blank` or navigation to the same URL with a different hash, which would succeed and return `null`.
   * > NOTE: Headless mode doesn't support navigation to a PDF document. See the
   * [upstream issue](https://bugs.chromium.org/p/chromium/issues/detail?id=761295).
   * 
   * Shortcut for main frame's [frame.goto(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#framegoto)
   * @param url URL to navigate page to. The url should include scheme, e.g. `https://`.
   * @param options 
   */
  goto(url: string, options?: {
    /**
     * Referer header value. If provided it will take preference over the referer header value set by
     * [page.setExtraHTTPHeaders(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetextrahttpheaders).
     */
    referer?: string;

    /**
     * Maximum operation time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be
     * changed by using the
     * [browserContext.setDefaultNavigationTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaultnavigationtimeout),
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout),
     * [page.setDefaultNavigationTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaultnavigationtimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;

    /**
     * When to consider operation succeeded, defaults to `load`. Events can be either:
     * - `'domcontentloaded'` - consider operation to be finished when the `DOMContentLoaded` event is fired.
     * - `'load'` - consider operation to be finished when the `load` event is fired.
     * - `'networkidle'` - consider operation to be finished when there are no network connections for at least `500` ms.
     */
    waitUntil?: "load"|"domcontentloaded"|"networkidle";
  }): Promise<null|Response>;

  /**
   * This method hovers over an element matching `selector` by performing the following steps:
   * 1. Find an element match matching `selector`. If there is none, wait until a matching element is attached to the DOM.
   * 1. Wait for [actionability](https://github.com/microsoft/playwright/blob/master/docs/actionability.md) checks on the matched element, unless `force` option is set. If the
   *    element is detached during the checks, the whole action is retried.
   * 1. Scroll the element into view if needed.
   * 1. Use [page.mouse](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagemouse) to hover over the
   *    center of the element, or the specified `position`.
   * 1. Wait for initiated navigations to either succeed or fail, unless `noWaitAfter` option is set.
   * 
   * When all steps combined have not finished during the specified `timeout`, this method rejects with a [TimeoutError].
   * Passing zero timeout disables this.
   * 
   * Shortcut for main frame's [frame.hover(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#framehover).
   * @param selector A selector to search for element. If there are multiple elements satisfying the selector, the first will be used. See [working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details.
   * @param options 
   */
  hover(selector: string, options?: {
    /**
     * Whether to bypass the [actionability](https://github.com/microsoft/playwright/blob/master/docs/actionability.md) checks. Defaults to `false`.
     */
    force?: boolean;

    /**
     * Modifier keys to press. Ensures that only these modifiers are pressed during the operation, and then restores current
     * modifiers back. If not specified, currently pressed modifiers are used.
     */
    modifiers?: Array<"Alt"|"Control"|"Meta"|"Shift">;

    /**
     * A point to use relative to the top-left corner of element padding box. If not specified, uses some visible point of the
     * element.
     */
    position?: {
      x: number;

      y: number;
    };

    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed by
     * using the
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * Returns `element.innerHTML`.
   * @param selector A selector to search for element. If there are multiple elements satisfying the selector, the first will be used. See [working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details.
   * @param options 
   */
  innerHTML(selector: string, options?: {
    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed by
     * using the
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;
  }): Promise<string>;

  /**
   * Returns `element.innerText`.
   * @param selector A selector to search for element. If there are multiple elements satisfying the selector, the first will be used. See [working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details.
   * @param options 
   */
  innerText(selector: string, options?: {
    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed by
     * using the
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;
  }): Promise<string>;

  /**
   * Returns whether the element is checked. Throws if the element is not a checkbox or radio input.
   * @param selector A selector to search for element. If there are multiple elements satisfying the selector, the first will be used. See [working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details.
   * @param options 
   */
  isChecked(selector: string, options?: {
    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed by
     * using the
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;
  }): Promise<boolean>;

  /**
   * Indicates that the page has been closed.
   */
  isClosed(): boolean;

  /**
   * Returns whether the element is disabled, the opposite of [enabled](https://github.com/microsoft/playwright/blob/master/docs/actionability.md#enabled).
   * @param selector A selector to search for element. If there are multiple elements satisfying the selector, the first will be used. See [working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details.
   * @param options 
   */
  isDisabled(selector: string, options?: {
    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed by
     * using the
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;
  }): Promise<boolean>;

  /**
   * Returns whether the element is [editable](https://github.com/microsoft/playwright/blob/master/docs/actionability.md#editable).
   * @param selector A selector to search for element. If there are multiple elements satisfying the selector, the first will be used. See [working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details.
   * @param options 
   */
  isEditable(selector: string, options?: {
    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed by
     * using the
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;
  }): Promise<boolean>;

  /**
   * Returns whether the element is [enabled](https://github.com/microsoft/playwright/blob/master/docs/actionability.md#enabled).
   * @param selector A selector to search for element. If there are multiple elements satisfying the selector, the first will be used. See [working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details.
   * @param options 
   */
  isEnabled(selector: string, options?: {
    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed by
     * using the
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;
  }): Promise<boolean>;

  /**
   * Returns whether the element is hidden, the opposite of [visible](https://github.com/microsoft/playwright/blob/master/docs/actionability.md#visible).  `selector` that does not
   * match any elements is considered hidden.
   * @param selector A selector to search for element. If there are multiple elements satisfying the selector, the first will be used. See [working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details.
   * @param options 
   */
  isHidden(selector: string, options?: {
    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed by
     * using the
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;
  }): Promise<boolean>;

  /**
   * Returns whether the element is [visible](https://github.com/microsoft/playwright/blob/master/docs/actionability.md#visible). `selector` that does not match any elements is
   * considered not visible.
   * @param selector A selector to search for element. If there are multiple elements satisfying the selector, the first will be used. See [working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details.
   * @param options 
   */
  isVisible(selector: string, options?: {
    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed by
     * using the
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;
  }): Promise<boolean>;

  keyboard: Keyboard;

  /**
   * The page's main frame. Page is guaranteed to have a main frame which persists during navigations.
   */
  mainFrame(): Frame;

  mouse: Mouse;

  /**
   * Returns the opener for popup pages and `null` for others. If the opener has been closed already the returns `null`.
   */
  opener(): Promise<null|Page>;

  /**
   * Returns the PDF buffer.
   * 
   * > NOTE: Generating a pdf is currently only supported in Chromium headless.
   * 
   * `page.pdf()` generates a pdf of the page with `print` css media. To generate a pdf with `screen` media, call
   * [page.emulateMedia(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageemulatemedia) before calling
   * `page.pdf()`:
   * 
   * > NOTE: By default, `page.pdf()` generates a pdf with modified colors for printing. Use the
   * [`-webkit-print-color-adjust`](https://developer.mozilla.org/en-US/docs/Web/CSS/-webkit-print-color-adjust) property to
   * force rendering of exact colors.
   * 
   * ```js
   * // Generates a PDF with 'screen' media type.
   * await page.emulateMedia({media: 'screen'});
   * await page.pdf({path: 'page.pdf'});
   * ```
   * 
   * The `width`, `height`, and `margin` options accept values labeled with units. Unlabeled values are treated as pixels.
   * 
   * A few examples:
   * - `page.pdf({width: 100})` - prints with width set to 100 pixels
   * - `page.pdf({width: '100px'})` - prints with width set to 100 pixels
   * - `page.pdf({width: '10cm'})` - prints with width set to 10 centimeters.
   * 
   * All possible units are:
   * - `px` - pixel
   * - `in` - inch
   * - `cm` - centimeter
   * - `mm` - millimeter
   * 
   * The `format` options are:
   * - `Letter`: 8.5in x 11in
   * - `Legal`: 8.5in x 14in
   * - `Tabloid`: 11in x 17in
   * - `Ledger`: 17in x 11in
   * - `A0`: 33.1in x 46.8in
   * - `A1`: 23.4in x 33.1in
   * - `A2`: 16.54in x 23.4in
   * - `A3`: 11.7in x 16.54in
   * - `A4`: 8.27in x 11.7in
   * - `A5`: 5.83in x 8.27in
   * - `A6`: 4.13in x 5.83in
   * 
   * > NOTE: `headerTemplate` and `footerTemplate` markup have the following limitations: > 1. Script tags inside templates
   * are not evaluated. > 2. Page styles are not visible inside templates.
   * @param options 
   */
  pdf(options?: {
    /**
     * Display header and footer. Defaults to `false`.
     */
    displayHeaderFooter?: boolean;

    /**
     * HTML template for the print footer. Should use the same format as the `headerTemplate`.
     */
    footerTemplate?: string;

    /**
     * Paper format. If set, takes priority over `width` or `height` options. Defaults to 'Letter'.
     */
    format?: string;

    /**
     * HTML template for the print header. Should be valid HTML markup with following classes used to inject printing values
     * into them:
     * - `'date'` formatted print date
     * - `'title'` document title
     * - `'url'` document location
     * - `'pageNumber'` current page number
     * - `'totalPages'` total pages in the document
     */
    headerTemplate?: string;

    /**
     * Paper height, accepts values labeled with units.
     */
    height?: string|number;

    /**
     * Paper orientation. Defaults to `false`.
     */
    landscape?: boolean;

    /**
     * Paper margins, defaults to none.
     */
    margin?: {
      /**
       * Top margin, accepts values labeled with units. Defaults to `0`.
       */
      top?: string|number;

      /**
       * Right margin, accepts values labeled with units. Defaults to `0`.
       */
      right?: string|number;

      /**
       * Bottom margin, accepts values labeled with units. Defaults to `0`.
       */
      bottom?: string|number;

      /**
       * Left margin, accepts values labeled with units. Defaults to `0`.
       */
      left?: string|number;
    };

    /**
     * Paper ranges to print, e.g., '1-5, 8, 11-13'. Defaults to the empty string, which means print all pages.
     */
    pageRanges?: string;

    /**
     * The file path to save the PDF to. If `path` is a relative path, then it is resolved relative to the current working
     * directory. If no path is provided, the PDF won't be saved to the disk.
     */
    path?: string;

    /**
     * Give any CSS `@page` size declared in the page priority over what is declared in `width` and `height` or `format`
     * options. Defaults to `false`, which will scale the content to fit the paper size.
     */
    preferCSSPageSize?: boolean;

    /**
     * Print background graphics. Defaults to `false`.
     */
    printBackground?: boolean;

    /**
     * Scale of the webpage rendering. Defaults to `1`. Scale amount must be between 0.1 and 2.
     */
    scale?: number;

    /**
     * Paper width, accepts values labeled with units.
     */
    width?: string|number;
  }): Promise<Buffer>;

  /**
   * Focuses the element, and then uses
   * [keyboard.down(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#keyboarddown) and
   * [keyboard.up(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#keyboardup).
   * 
   * `key` can specify the intended [keyboardEvent.key](https://developer.mozilla.org/en-US/docs/Web/API/KeyboardEvent/key)
   * value or a single character to generate the text for. A superset of the `key` values can be found
   * [here](https://developer.mozilla.org/en-US/docs/Web/API/KeyboardEvent/key/Key_Values). Examples of the keys are:
   * 
   * `F1` - `F12`, `Digit0`- `Digit9`, `KeyA`- `KeyZ`, `Backquote`, `Minus`, `Equal`, `Backslash`, `Backspace`, `Tab`,
   * `Delete`, `Escape`, `ArrowDown`, `End`, `Enter`, `Home`, `Insert`, `PageDown`, `PageUp`, `ArrowRight`, `ArrowUp`, etc.
   * 
   * Following modification shortcuts are also supported: `Shift`, `Control`, `Alt`, `Meta`, `ShiftLeft`.
   * 
   * Holding down `Shift` will type the text that corresponds to the `key` in the upper case.
   * 
   * If `key` is a single character, it is case-sensitive, so the values `a` and `A` will generate different respective
   * texts.
   * 
   * Shortcuts such as `key: "Control+o"` or `key: "Control+Shift+T"` are supported as well. When speficied with the
   * modifier, modifier is pressed and being held while the subsequent key is being pressed.
   * 
   * ```js
   * const page = await browser.newPage();
   * await page.goto('https://keycode.info');
   * await page.press('body', 'A');
   * await page.screenshot({ path: 'A.png' });
   * await page.press('body', 'ArrowLeft');
   * await page.screenshot({ path: 'ArrowLeft.png' });
   * await page.press('body', 'Shift+O');
   * await page.screenshot({ path: 'O.png' });
   * await browser.close();
   * ```
   * 
   * @param selector A selector to search for element. If there are multiple elements satisfying the selector, the first will be used. See [working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details.
   * @param key Name of the key to press or a character to generate, such as `ArrowLeft` or `a`.
   * @param options 
   */
  press(selector: string, key: string, options?: {
    /**
     * Time to wait between `keydown` and `keyup` in milliseconds. Defaults to 0.
     */
    delay?: number;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You can
     * opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as navigating to
     * inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed by
     * using the
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * Returns the main resource response. In case of multiple redirects, the navigation will resolve with the response of the
   * last redirect.
   * @param options 
   */
  reload(options?: {
    /**
     * Maximum operation time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be
     * changed by using the
     * [browserContext.setDefaultNavigationTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaultnavigationtimeout),
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout),
     * [page.setDefaultNavigationTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaultnavigationtimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;

    /**
     * When to consider operation succeeded, defaults to `load`. Events can be either:
     * - `'domcontentloaded'` - consider operation to be finished when the `DOMContentLoaded` event is fired.
     * - `'load'` - consider operation to be finished when the `load` event is fired.
     * - `'networkidle'` - consider operation to be finished when there are no network connections for at least `500` ms.
     */
    waitUntil?: "load"|"domcontentloaded"|"networkidle";
  }): Promise<null|Response>;

  /**
   * Routing provides the capability to modify network requests that are made by a page.
   * 
   * Once routing is enabled, every request matching the url pattern will stall unless it's continued, fulfilled or aborted.
   * 
   * > NOTE: The handler will only be called for the first url if the response is a redirect.
   * 
   * An example of a naïve handler that aborts all image requests:
   * 
   * ```js
   * const page = await browser.newPage();
   * await page.route('**\/*.{png,jpg,jpeg}', route => route.abort());
   * await page.goto('https://example.com');
   * await browser.close();
   * ```
   * 
   * or the same snippet using a regex pattern instead:
   * 
   * ```js
   * const page = await browser.newPage();
   * await page.route(/(\.png$)|(\.jpg$)/, route => route.abort());
   * await page.goto('https://example.com');
   * await browser.close();
   * ```
   * 
   * Page routes take precedence over browser context routes (set up with
   * [browserContext.route(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextroute)) when
   * request matches both handlers.
   * 
   * > NOTE: Enabling routing disables http cache.
   * @param url A glob pattern, regex pattern or predicate receiving [URL] to match while routing.
   * @param handler handler function to route the request.
   */
  route(url: string|RegExp|((url: URL) => boolean), handler: ((route: Route, request: Request) => void)): Promise<void>;

  /**
   * Returns the buffer with the captured screenshot.
   * 
   * > NOTE: Screenshots take at least 1/6 second on Chromium OS X and Chromium Windows. See https://crbug.com/741689 for
   * discussion.
   * @param options 
   */
  screenshot(options?: {
    /**
     * An object which specifies clipping of the resulting image. Should have the following fields:
     */
    clip?: {
      /**
       * x-coordinate of top-left corner of clip area
       */
      x: number;

      /**
       * y-coordinate of top-left corner of clip area
       */
      y: number;

      /**
       * width of clipping area
       */
      width: number;

      /**
       * height of clipping area
       */
      height: number;
    };

    /**
     * When true, takes a screenshot of the full scrollable page, instead of the currently visible viewport. Defaults to
     * `false`.
     */
    fullPage?: boolean;

    /**
     * Hides default white background and allows capturing screenshots with transparency. Not applicable to `jpeg` images.
     * Defaults to `false`.
     */
    omitBackground?: boolean;

    /**
     * The file path to save the image to. The screenshot type will be inferred from file extension. If `path` is a relative
     * path, then it is resolved relative to the current working directory. If no path is provided, the image won't be saved to
     * the disk.
     */
    path?: string;

    /**
     * The quality of the image, between 0-100. Not applicable to `png` images.
     */
    quality?: number;

    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed by
     * using the
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;

    /**
     * Specify screenshot type, defaults to `png`.
     */
    type?: "png"|"jpeg";
  }): Promise<Buffer>;

  /**
   * Returns the array of option values that have been successfully selected.
   * 
   * Triggers a `change` and `input` event once all the provided options have been selected. If there's no `<select>` element
   * matching `selector`, the method throws an error.
   * 
   * Will wait until all specified options are present in the `<select>` element.
   * 
   * ```js
   * // single selection matching the value
   * page.selectOption('select#colors', 'blue');
   * 
   * // single selection matching the label
   * page.selectOption('select#colors', { label: 'Blue' });
   * 
   * // multiple selection
   * page.selectOption('select#colors', ['red', 'green', 'blue']);
   * 
   * ```
   * 
   * Shortcut for main frame's
   * [frame.selectOption(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#frameselectoption)
   * @param selector A selector to search for element. If there are multiple elements satisfying the selector, the first will be used. See [working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details.
   * @param values Options to select. If the `<select>` has the `multiple` attribute, all matching options are selected, otherwise only the first option matching one of the passed options is selected. String values are equivalent to `{value:'string'}`. Option
   * is considered matching if all specified properties match.
   * @param options 
   */
  selectOption(selector: string, values: null|string|ElementHandle|Array<string>|{
    /**
     * Matches by `option.value`. Optional.
     */
    value?: string;

    /**
     * Matches by `option.label`. Optional.
     */
    label?: string;

    /**
     * Matches by the index. Optional.
     */
    index?: number;
  }|Array<ElementHandle>|Array<{
    /**
     * Matches by `option.value`. Optional.
     */
    value?: string;

    /**
     * Matches by `option.label`. Optional.
     */
    label?: string;

    /**
     * Matches by the index. Optional.
     */
    index?: number;
  }>, options?: {
    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You can
     * opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as navigating to
     * inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed by
     * using the
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;
  }): Promise<Array<string>>;

  /**
   * @param html HTML markup to assign to the page.
   * @param options 
   */
  setContent(html: string, options?: {
    /**
     * Maximum operation time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be
     * changed by using the
     * [browserContext.setDefaultNavigationTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaultnavigationtimeout),
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout),
     * [page.setDefaultNavigationTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaultnavigationtimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;

    /**
     * When to consider operation succeeded, defaults to `load`. Events can be either:
     * - `'domcontentloaded'` - consider operation to be finished when the `DOMContentLoaded` event is fired.
     * - `'load'` - consider operation to be finished when the `load` event is fired.
     * - `'networkidle'` - consider operation to be finished when there are no network connections for at least `500` ms.
     */
    waitUntil?: "load"|"domcontentloaded"|"networkidle";
  }): Promise<void>;

  /**
   * This setting will change the default maximum navigation time for the following methods and related shortcuts:
   * - [page.goBack(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagegoback)
   * - [page.goForward(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagegoforward)
   * - [page.goto(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagegoto)
   * - [page.reload(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagereload)
   * - [page.setContent(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetcontent)
   * - [page.waitForNavigation(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagewaitfornavigation)
   * 
   * > NOTE:
   * [page.setDefaultNavigationTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaultnavigationtimeout)
   * takes priority over
   * [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout),
   * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
   * and
   * [browserContext.setDefaultNavigationTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaultnavigationtimeout).
   * @param timeout Maximum navigation time in milliseconds
   */
  setDefaultNavigationTimeout(timeout: number): void;

  /**
   * This setting will change the default maximum time for all the methods accepting `timeout` option.
   * 
   * > NOTE:
   * [page.setDefaultNavigationTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaultnavigationtimeout)
   * takes priority over
   * [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout).
   * @param timeout Maximum time in milliseconds
   */
  setDefaultTimeout(timeout: number): void;

  /**
   * The extra HTTP headers will be sent with every request the page initiates.
   * 
   * > NOTE:
   * [page.setExtraHTTPHeaders(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetextrahttpheaders)
   * does not guarantee the order of headers in the outgoing requests.
   * @param headers An object containing additional HTTP headers to be sent with every request. All header values must be strings.
   */
  setExtraHTTPHeaders(headers: { [key: string]: string; }): Promise<void>;

  /**
   * This method expects `selector` to point to an
   * [input element](https://developer.mozilla.org/en-US/docs/Web/HTML/Element/input).
   * 
   * Sets the value of the file input to these file paths or files. If some of the `filePaths` are relative paths, then they
   * are resolved relative to the the current working directory. For empty array, clears the selected files.
   * @param selector A selector to search for element. If there are multiple elements satisfying the selector, the first will be used. See [working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details.
   * @param files 
   * @param options 
   */
  setInputFiles(selector: string, files: string|Array<string>|{
    /**
     * [File] name
     */
    name: string;

    /**
     * [File] type
     */
    mimeType: string;

    /**
     * File content
     */
    buffer: Buffer;
  }|Array<{
    /**
     * [File] name
     */
    name: string;

    /**
     * [File] type
     */
    mimeType: string;

    /**
     * File content
     */
    buffer: Buffer;
  }>, options?: {
    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You can
     * opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as navigating to
     * inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed by
     * using the
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * In the case of multiple pages in a single browser, each page can have its own viewport size. However,
   * [browser.newContext(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsernewcontext) allows to set
   * viewport size (and more) for all pages in the context at once.
   * 
   * `page.setViewportSize` will resize the page. A lot of websites don't expect phones to change size, so you should set the
   * viewport size before navigating to the page.
   * 
   * ```js
   * const page = await browser.newPage();
   * await page.setViewportSize({
   *   width: 640,
   *   height: 480,
   * });
   * await page.goto('https://example.com');
   * ```
   * 
   * @param viewportSize 
   */
  setViewportSize(viewportSize: {
    /**
     * page width in pixels.
     */
    width: number;

    /**
     * page height in pixels.
     */
    height: number;
  }): Promise<void>;

  /**
   * This method taps an element matching `selector` by performing the following steps:
   * 1. Find an element match matching `selector`. If there is none, wait until a matching element is attached to the DOM.
   * 1. Wait for [actionability](https://github.com/microsoft/playwright/blob/master/docs/actionability.md) checks on the matched element, unless `force` option is set. If the
   *    element is detached during the checks, the whole action is retried.
   * 1. Scroll the element into view if needed.
   * 1. Use [page.touchscreen](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagetouchscreen) to tap the
   *    center of the element, or the specified `position`.
   * 1. Wait for initiated navigations to either succeed or fail, unless `noWaitAfter` option is set.
   * 
   * When all steps combined have not finished during the specified `timeout`, this method rejects with a [TimeoutError].
   * Passing zero timeout disables this.
   * 
   * > NOTE: [page.tap(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagetap) requires that the
   * `hasTouch` option of the browser context be set to true.
   * 
   * Shortcut for main frame's [frame.tap(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#frametap).
   * @param selector A selector to search for element. If there are multiple elements satisfying the selector, the first will be used. See [working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details.
   * @param options 
   */
  tap(selector: string, options?: {
    /**
     * Whether to bypass the [actionability](https://github.com/microsoft/playwright/blob/master/docs/actionability.md) checks. Defaults to `false`.
     */
    force?: boolean;

    /**
     * Modifier keys to press. Ensures that only these modifiers are pressed during the operation, and then restores current
     * modifiers back. If not specified, currently pressed modifiers are used.
     */
    modifiers?: Array<"Alt"|"Control"|"Meta"|"Shift">;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You can
     * opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as navigating to
     * inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * A point to use relative to the top-left corner of element padding box. If not specified, uses some visible point of the
     * element.
     */
    position?: {
      x: number;

      y: number;
    };

    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed by
     * using the
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * Returns `element.textContent`.
   * @param selector A selector to search for element. If there are multiple elements satisfying the selector, the first will be used. See [working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details.
   * @param options 
   */
  textContent(selector: string, options?: {
    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed by
     * using the
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;
  }): Promise<null|string>;

  /**
   * Returns the page's title. Shortcut for main frame's
   * [frame.title(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#frametitle).
   */
  title(): Promise<string>;

  touchscreen: Touchscreen;

  /**
   * Sends a `keydown`, `keypress`/`input`, and `keyup` event for each character in the text. `page.type` can be used to send
   * fine-grained keyboard events. To fill values in form fields, use
   * [page.fill(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagefill).
   * 
   * To press a special key, like `Control` or `ArrowDown`, use
   * [keyboard.press(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#keyboardpress).
   * 
   * ```js
   * await page.type('#mytextarea', 'Hello'); // Types instantly
   * await page.type('#mytextarea', 'World', {delay: 100}); // Types slower, like a user
   * ```
   * 
   * Shortcut for main frame's [frame.type(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#frametype).
   * @param selector A selector to search for element. If there are multiple elements satisfying the selector, the first will be used. See [working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details.
   * @param text A text to type into a focused element.
   * @param options 
   */
  type(selector: string, text: string, options?: {
    /**
     * Time to wait between key presses in milliseconds. Defaults to 0.
     */
    delay?: number;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You can
     * opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as navigating to
     * inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed by
     * using the
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * This method unchecks an element matching `selector` by performing the following steps:
   * 1. Find an element match matching `selector`. If there is none, wait until a matching element is attached to the DOM.
   * 1. Ensure that matched element is a checkbox or a radio input. If not, this method rejects. If the element is already
   *    unchecked, this method returns immediately.
   * 1. Wait for [actionability](https://github.com/microsoft/playwright/blob/master/docs/actionability.md) checks on the matched element, unless `force` option is set. If the
   *    element is detached during the checks, the whole action is retried.
   * 1. Scroll the element into view if needed.
   * 1. Use [page.mouse](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagemouse) to click in the center
   *    of the element.
   * 1. Wait for initiated navigations to either succeed or fail, unless `noWaitAfter` option is set.
   * 1. Ensure that the element is now unchecked. If not, this method rejects.
   * 
   * When all steps combined have not finished during the specified `timeout`, this method rejects with a [TimeoutError].
   * Passing zero timeout disables this.
   * 
   * Shortcut for main frame's
   * [frame.uncheck(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#frameuncheck).
   * @param selector A selector to search for element. If there are multiple elements satisfying the selector, the first will be used. See [working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details.
   * @param options 
   */
  uncheck(selector: string, options?: {
    /**
     * Whether to bypass the [actionability](https://github.com/microsoft/playwright/blob/master/docs/actionability.md) checks. Defaults to `false`.
     */
    force?: boolean;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You can
     * opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as navigating to
     * inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed by
     * using the
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * Removes a route created with [page.route(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageroute).
   * When `handler` is not specified, removes all routes for the `url`.
   * @param url A glob pattern, regex pattern or predicate receiving [URL] to match while routing.
   * @param handler Optional handler function to route the request.
   */
  unroute(url: string|RegExp|((url: URL) => boolean), handler?: ((route: Route, request: Request) => void)): Promise<void>;

  /**
   * Shortcut for main frame's [frame.url(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#frameurl).
   */
  url(): string;

  /**
   * Video object associated with this page.
   */
  video(): null|Video;

  viewportSize(): null|{
    /**
     * page width in pixels.
     */
    width: number;

    /**
     * page height in pixels.
     */
    height: number;
  };

  /**
   * Emitted when the page closes.
   */
  waitForEvent(event: 'close', optionsOrPredicate?: { predicate?: () => boolean, timeout?: number } | (() => boolean)): Promise<void>;

  /**
   * Emitted when JavaScript within the page calls one of console API methods, e.g. `console.log` or `console.dir`. Also
   * emitted if the page throws an error or a warning.
   * 
   * The arguments passed into `console.log` appear as arguments on the event handler.
   * 
   * An example of handling `console` event:
   * 
   * ```js
   * page.on('console', msg => {
   *   for (let i = 0; i < msg.args().length; ++i)
   *     console.log(`${i}: ${await msg.args()[i].jsonValue()}`);
   * });
   * page.evaluate(() => console.log('hello', 5, {foo: 'bar'}));
   * ```
   * 
   */
  waitForEvent(event: 'console', optionsOrPredicate?: { predicate?: (consoleMessage: ConsoleMessage) => boolean, timeout?: number } | ((consoleMessage: ConsoleMessage) => boolean)): Promise<ConsoleMessage>;

  /**
   * Emitted when the page crashes. Browser pages might crash if they try to allocate too much memory. When the page crashes,
   * ongoing and subsequent operations will throw.
   * 
   * The most common way to deal with crashes is to catch an exception:
   * 
   * ```js
   * try {
   *   // Crash might happen during a click.
   *   await page.click('button');
   *   // Or while waiting for an event.
   *   await page.waitForEvent('popup');
   * } catch (e) {
   *   // When the page crashes, exception message contains 'crash'.
   * }
   * ```
   * 
   */
  waitForEvent(event: 'crash', optionsOrPredicate?: { predicate?: () => boolean, timeout?: number } | (() => boolean)): Promise<void>;

  /**
   * Emitted when a JavaScript dialog appears, such as `alert`, `prompt`, `confirm` or `beforeunload`. Playwright can respond
   * to the dialog via [dialog.accept(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#dialogaccept) or
   * [dialog.dismiss(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#dialogdismiss) methods.
   */
  waitForEvent(event: 'dialog', optionsOrPredicate?: { predicate?: (dialog: Dialog) => boolean, timeout?: number } | ((dialog: Dialog) => boolean)): Promise<Dialog>;

  /**
   * Emitted when the JavaScript [`DOMContentLoaded`](https://developer.mozilla.org/en-US/docs/Web/Events/DOMContentLoaded)
   * event is dispatched.
   */
  waitForEvent(event: 'domcontentloaded', optionsOrPredicate?: { predicate?: () => boolean, timeout?: number } | (() => boolean)): Promise<void>;

  /**
   * Emitted when attachment download started. User can access basic file operations on downloaded content via the passed
   * [Download] instance.
   * 
   * > NOTE: Browser context **must** be created with the `acceptDownloads` set to `true` when user needs access to the
   * downloaded content. If `acceptDownloads` is not set, download events are emitted, but the actual download is not
   * performed and user has no access to the downloaded files.
   */
  waitForEvent(event: 'download', optionsOrPredicate?: { predicate?: (download: Download) => boolean, timeout?: number } | ((download: Download) => boolean)): Promise<Download>;

  /**
   * Emitted when a file chooser is supposed to appear, such as after clicking the  `<input type=file>`. Playwright can
   * respond to it via setting the input files using
   * [fileChooser.setFiles(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#filechoosersetfiles) that can
   * be uploaded after that.
   * 
   * ```js
   * page.on('filechooser', async (fileChooser) => {
   *   await fileChooser.setFiles('/tmp/myfile.pdf');
   * });
   * ```
   * 
   */
  waitForEvent(event: 'filechooser', optionsOrPredicate?: { predicate?: (fileChooser: FileChooser) => boolean, timeout?: number } | ((fileChooser: FileChooser) => boolean)): Promise<FileChooser>;

  /**
   * Emitted when a frame is attached.
   */
  waitForEvent(event: 'frameattached', optionsOrPredicate?: { predicate?: (frame: Frame) => boolean, timeout?: number } | ((frame: Frame) => boolean)): Promise<Frame>;

  /**
   * Emitted when a frame is detached.
   */
  waitForEvent(event: 'framedetached', optionsOrPredicate?: { predicate?: (frame: Frame) => boolean, timeout?: number } | ((frame: Frame) => boolean)): Promise<Frame>;

  /**
   * Emitted when a frame is navigated to a new url.
   */
  waitForEvent(event: 'framenavigated', optionsOrPredicate?: { predicate?: (frame: Frame) => boolean, timeout?: number } | ((frame: Frame) => boolean)): Promise<Frame>;

  /**
   * Emitted when the JavaScript [`load`](https://developer.mozilla.org/en-US/docs/Web/Events/load) event is dispatched.
   */
  waitForEvent(event: 'load', optionsOrPredicate?: { predicate?: () => boolean, timeout?: number } | (() => boolean)): Promise<void>;

  /**
   * Emitted when an uncaught exception happens within the page.
   */
  waitForEvent(event: 'pageerror', optionsOrPredicate?: { predicate?: (error: Error) => boolean, timeout?: number } | ((error: Error) => boolean)): Promise<Error>;

  /**
   * Emitted when the page opens a new tab or window. This event is emitted in addition to the
   * [browserContext.on('page')](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextonpage), but
   * only for popups relevant to this page.
   * 
   * The earliest moment that page is available is when it has navigated to the initial url. For example, when opening a
   * popup with `window.open('http://example.com')`, this event will fire when the network request to "http://example.com" is
   * done and its response has started loading in the popup.
   * 
   * ```js
   * const [popup] = await Promise.all([
   *   page.waitForEvent('popup'),
   *   page.evaluate(() => window.open('https://example.com')),
   * ]);
   * console.log(await popup.evaluate('location.href'));
   * ```
   * 
   * > NOTE: Use
   * [page.waitForLoadState(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagewaitforloadstate) to wait
   * until the page gets to a particular state (you should not need it in most cases).
   */
  waitForEvent(event: 'popup', optionsOrPredicate?: { predicate?: (page: Page) => boolean, timeout?: number } | ((page: Page) => boolean)): Promise<Page>;

  /**
   * Emitted when a page issues a request. The [request] object is read-only. In order to intercept and mutate requests, see
   * [page.route(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageroute) or
   * [browserContext.route(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextroute).
   */
  waitForEvent(event: 'request', optionsOrPredicate?: { predicate?: (request: Request) => boolean, timeout?: number } | ((request: Request) => boolean)): Promise<Request>;

  /**
   * Emitted when a request fails, for example by timing out.
   * 
   * > NOTE: HTTP Error responses, such as 404 or 503, are still successful responses from HTTP standpoint, so request will
   * complete with
   * [page.on('requestfinished')](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageonrequestfinished)
   * event and not with
   * [page.on('requestfailed')](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageonrequestfailed).
   */
  waitForEvent(event: 'requestfailed', optionsOrPredicate?: { predicate?: (request: Request) => boolean, timeout?: number } | ((request: Request) => boolean)): Promise<Request>;

  /**
   * Emitted when a request finishes successfully after downloading the response body. For a successful response, the
   * sequence of events is `request`, `response` and `requestfinished`.
   */
  waitForEvent(event: 'requestfinished', optionsOrPredicate?: { predicate?: (request: Request) => boolean, timeout?: number } | ((request: Request) => boolean)): Promise<Request>;

  /**
   * Emitted when [response] status and headers are received for a request. For a successful response, the sequence of events
   * is `request`, `response` and `requestfinished`.
   */
  waitForEvent(event: 'response', optionsOrPredicate?: { predicate?: (response: Response) => boolean, timeout?: number } | ((response: Response) => boolean)): Promise<Response>;

  /**
   * Emitted when <[WebSocket]> request is sent.
   */
  waitForEvent(event: 'websocket', optionsOrPredicate?: { predicate?: (webSocket: WebSocket) => boolean, timeout?: number } | ((webSocket: WebSocket) => boolean)): Promise<WebSocket>;

  /**
   * Emitted when a dedicated [WebWorker](https://developer.mozilla.org/en-US/docs/Web/API/Web_Workers_API) is spawned by the
   * page.
   */
  waitForEvent(event: 'worker', optionsOrPredicate?: { predicate?: (worker: Worker) => boolean, timeout?: number } | ((worker: Worker) => boolean)): Promise<Worker>;


  /**
   * Returns when the required load state has been reached.
   * 
   * This resolves when the page reaches a required load state, `load` by default. The navigation must have been committed
   * when this method is called. If current document has already reached the required state, resolves immediately.
   * 
   * ```js
   * await page.click('button'); // Click triggers navigation.
   * await page.waitForLoadState(); // The promise resolves after 'load' event.
   * ```
   * 
   * ```js
   * const [popup] = await Promise.all([
   *   page.waitForEvent('popup'),
   *   page.click('button'), // Click triggers a popup.
   * ])
   * await popup.waitForLoadState('domcontentloaded'); // The promise resolves after 'domcontentloaded' event.
   * console.log(await popup.title()); // Popup is ready to use.
   * ```
   * 
   * Shortcut for main frame's
   * [frame.waitForLoadState(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#framewaitforloadstate).
   * @param state Optional load state to wait for, defaults to `load`. If the state has been already reached while loading current document, the method resolves immediately. Can be one of:
   * - `'load'` - wait for the `load` event to be fired.
   * - `'domcontentloaded'` - wait for the `DOMContentLoaded` event to be fired.
   * - `'networkidle'` - wait until there are no network connections for at least `500` ms.
   * @param options 
   */
  waitForLoadState(state?: "load"|"domcontentloaded"|"networkidle", options?: {
    /**
     * Maximum operation time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be
     * changed by using the
     * [browserContext.setDefaultNavigationTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaultnavigationtimeout),
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout),
     * [page.setDefaultNavigationTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaultnavigationtimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * Waits for the main frame navigation and returns the main resource response. In case of multiple redirects, the
   * navigation will resolve with the response of the last redirect. In case of navigation to a different anchor or
   * navigation due to History API usage, the navigation will resolve with `null`.
   * 
   * This resolves when the page navigates to a new URL or reloads. It is useful for when you run code which will indirectly
   * cause the page to navigate. e.g. The click target has an `onclick` handler that triggers navigation from a `setTimeout`.
   * Consider this example:
   * 
   * ```js
   * const [response] = await Promise.all([
   *   page.waitForNavigation(), // The promise resolves after navigation has finished
   *   page.click('a.delayed-navigation'), // Clicking the link will indirectly cause a navigation
   * ]);
   * ```
   * 
   * > NOTE: Usage of the [History API](https://developer.mozilla.org/en-US/docs/Web/API/History_API) to change the URL is
   * considered a navigation.
   * 
   * Shortcut for main frame's
   * [frame.waitForNavigation(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#framewaitfornavigation).
   * @param options 
   */
  waitForNavigation(options?: {
    /**
     * Maximum operation time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be
     * changed by using the
     * [browserContext.setDefaultNavigationTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaultnavigationtimeout),
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout),
     * [page.setDefaultNavigationTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaultnavigationtimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;

    /**
     * A glob pattern, regex pattern or predicate receiving [URL] to match while waiting for the navigation.
     */
    url?: string|RegExp|((url: URL) => boolean);

    /**
     * When to consider operation succeeded, defaults to `load`. Events can be either:
     * - `'domcontentloaded'` - consider operation to be finished when the `DOMContentLoaded` event is fired.
     * - `'load'` - consider operation to be finished when the `load` event is fired.
     * - `'networkidle'` - consider operation to be finished when there are no network connections for at least `500` ms.
     */
    waitUntil?: "load"|"domcontentloaded"|"networkidle";
  }): Promise<null|Response>;

  /**
   * Waits for the matching request and returns it.
   * 
   * ```js
   * const firstRequest = await page.waitForRequest('http://example.com/resource');
   * const finalRequest = await page.waitForRequest(request => request.url() === 'http://example.com' && request.method() === 'GET');
   * return firstRequest.url();
   * ```
   * 
   * ```js
   * await page.waitForRequest(request => request.url().searchParams.get('foo') === 'bar' && request.url().searchParams.get('foo2') === 'bar2');
   * ```
   * 
   * @param urlOrPredicate Request URL string, regex or predicate receiving [Request] object.
   * @param options 
   */
  waitForRequest(urlOrPredicate: string|RegExp|((request: Request) => boolean), options?: {
    /**
     * Maximum wait time in milliseconds, defaults to 30 seconds, pass `0` to disable the timeout. The default value can be
     * changed by using the
     * [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * method.
     */
    timeout?: number;
  }): Promise<Request>;

  /**
   * Returns the matched response.
   * 
   * ```js
   * const firstResponse = await page.waitForResponse('https://example.com/resource');
   * const finalResponse = await page.waitForResponse(response => response.url() === 'https://example.com' && response.status() === 200);
   * return finalResponse.ok();
   * ```
   * 
   * @param urlOrPredicate Request URL string, regex or predicate receiving [Response] object.
   * @param options 
   */
  waitForResponse(urlOrPredicate: string|RegExp|((response: Response) => boolean), options?: {
    /**
     * Maximum wait time in milliseconds, defaults to 30 seconds, pass `0` to disable the timeout. The default value can be
     * changed by using the
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;
  }): Promise<Response>;

  /**
   * Waits for the given `timeout` in milliseconds.
   * 
   * Note that `page.waitForTimeout()` should only be used for debugging. Tests using the timer in production are going to be
   * flaky. Use signals such as network events, selectors becoming visible and others instead.
   * 
   * ```js
   * // wait for 1 second
   * await page.waitForTimeout(1000);
   * ```
   * 
   * Shortcut for main frame's
   * [frame.waitForTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#framewaitfortimeout).
   * @param timeout A timeout to wait for
   */
  waitForTimeout(timeout: number): Promise<void>;

  /**
   * This method returns all of the dedicated [WebWorkers](https://developer.mozilla.org/en-US/docs/Web/API/Web_Workers_API)
   * associated with the page.
   * 
   * > NOTE: This does not contain ServiceWorkers
   */
  workers(): Array<Worker>;}

/**
 * At every point of time, page exposes its current frame tree via the
 * [page.mainFrame(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagemainframe) and
 * [frame.childFrames(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#framechildframes) methods.
 * 
 * [Frame] object's lifecycle is controlled by three events, dispatched on the page object:
 * - [page.on('frameattached')](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageonframeattached) -
 *   fired when the frame gets attached to the page. A Frame can be attached to the page only once.
 * - [page.on('framenavigated')](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageonframenavigated) -
 *   fired when the frame commits navigation to a different URL.
 * - [page.on('framedetached')](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageonframedetached) -
 *   fired when the frame gets detached from the page.  A Frame can be detached from the page only once.
 * 
 * An example of dumping frame tree:
 * 
 * ```js
 * const { firefox } = require('playwright');  // Or 'chromium' or 'webkit'.
 * 
 * (async () => {
 *   const browser = await firefox.launch();
 *   const page = await browser.newPage();
 *   await page.goto('https://www.google.com/chrome/browser/canary.html');
 *   dumpFrameTree(page.mainFrame(), '');
 *   await browser.close();
 * 
 *   function dumpFrameTree(frame, indent) {
 *     console.log(indent + frame.url());
 *     for (const child of frame.childFrames()) {
 *       dumpFrameTree(child, indent + '  ');
 *     }
 *   }
 * })();
 * ```
 * 
 */
export interface Frame {
  /**
   * Returns the return value of `pageFunction`
   * 
   * If the function passed to the
   * [frame.evaluate(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#frameevaluate) returns a [Promise],
   * then [frame.evaluate(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#frameevaluate) would wait for
   * the promise to resolve and return its value.
   * 
   * If the function passed to the
   * [frame.evaluate(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#frameevaluate) returns a
   * non-[Serializable] value, then[ method: `Frame.evaluate`] returns `undefined`. DevTools Protocol also supports
   * transferring some additional values that are not serializable by `JSON`: `-0`, `NaN`, `Infinity`, `-Infinity`, and
   * bigint literals.
   * 
   * ```js
   * const result = await frame.evaluate(([x, y]) => {
   *   return Promise.resolve(x * y);
   * }, [7, 8]);
   * console.log(result); // prints "56"
   * ```
   * 
   * A string can also be passed in instead of a function.
   * 
   * ```js
   * console.log(await frame.evaluate('1 + 2')); // prints "3"
   * ```
   * 
   * [ElementHandle] instances can be passed as an argument to the
   * [frame.evaluate(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#frameevaluate):
   * 
   * ```js
   * const bodyHandle = await frame.$('body');
   * const html = await frame.evaluate(([body, suffix]) => body.innerHTML + suffix, [bodyHandle, 'hello']);
   * await bodyHandle.dispose();
   * ```
   * 
   * @param pageFunction Function to be evaluated in browser context
   * @param arg Optional argument to pass to `pageFunction`
   */
  evaluate<R, Arg>(pageFunction: PageFunction<Arg, R>, arg: Arg): Promise<R>;
  evaluate<R>(pageFunction: PageFunction<void, R>, arg?: any): Promise<R>;

  /**
   * Returns the return value of `pageFunction` as in-page object (JSHandle).
   * 
   * The only difference between
   * [frame.evaluate(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#frameevaluate) and
   * [frame.evaluateHandle(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#frameevaluatehandle) is
   * that[ method: Fframe.evaluateHandle`] returns in-page object (JSHandle).
   * 
   * If the function, passed to the
   * [frame.evaluateHandle(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#frameevaluatehandle), returns
   * a [Promise], then[ method: Fframe.evaluateHandle`] would wait for the promise to resolve and return its value.
   * 
   * ```js
   * const aWindowHandle = await frame.evaluateHandle(() => Promise.resolve(window));
   * aWindowHandle; // Handle for the window object.
   * ```
   * 
   * A string can also be passed in instead of a function.
   * 
   * ```js
   * const aHandle = await frame.evaluateHandle('document'); // Handle for the 'document'.
   * ```
   * 
   * [JSHandle] instances can be passed as an argument to the
   * [frame.evaluateHandle(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#frameevaluatehandle):
   * 
   * ```js
   * const aHandle = await frame.evaluateHandle(() => document.body);
   * const resultHandle = await frame.evaluateHandle(([body, suffix]) => body.innerHTML + suffix, [aHandle, 'hello']);
   * console.log(await resultHandle.jsonValue());
   * await resultHandle.dispose();
   * ```
   * 
   * @param pageFunction Function to be evaluated in the page context
   * @param arg Optional argument to pass to `pageFunction`
   */
  evaluateHandle<R, Arg>(pageFunction: PageFunction<Arg, R>, arg: Arg): Promise<SmartHandle<R>>;
  evaluateHandle<R>(pageFunction: PageFunction<void, R>, arg?: any): Promise<SmartHandle<R>>;

  /**
   * Returns the ElementHandle pointing to the frame element.
   * 
   * The method finds an element matching the specified selector within the frame. See
   * [Working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details. If no elements match the selector,
   * returns `null`.
   * @param selector A selector to query for. See [working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details.
   */
  $<K extends keyof HTMLElementTagNameMap>(selector: K): Promise<ElementHandleForTag<K> | null>;
  $(selector: string): Promise<ElementHandle<SVGElement | HTMLElement> | null>;

  /**
   * Returns the ElementHandles pointing to the frame elements.
   * 
   * The method finds all elements matching the specified selector within the frame. See
   * [Working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details. If no elements match the selector,
   * returns empty array.
   * @param selector A selector to query for. See [working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details.
   */
  $$<K extends keyof HTMLElementTagNameMap>(selector: K): Promise<ElementHandleForTag<K>[]>;
  $$(selector: string): Promise<ElementHandle<SVGElement | HTMLElement>[]>;

  /**
   * Returns the return value of `pageFunction`
   * 
   * The method finds an element matching the specified selector within the frame and passes it as a first argument to
   * `pageFunction`. See [Working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details. If no elements
   * match the selector, the method throws an error.
   * 
   * If `pageFunction` returns a [Promise], then `frame.$eval` would wait for the promise to resolve and return its value.
   * 
   * Examples:
   * 
   * ```js
   * const searchValue = await frame.$eval('#search', el => el.value);
   * const preloadHref = await frame.$eval('link[rel=preload]', el => el.href);
   * const html = await frame.$eval('.main-container', (e, suffix) => e.outerHTML + suffix, 'hello');
   * ```
   * 
   * @param selector A selector to query for. See [working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details.
   * @param pageFunction Function to be evaluated in browser context
   * @param arg Optional argument to pass to `pageFunction`
   */
  $eval<K extends keyof HTMLElementTagNameMap, R, Arg>(selector: K, pageFunction: PageFunctionOn<HTMLElementTagNameMap[K], Arg, R>, arg: Arg): Promise<R>;
  $eval<R, Arg, E extends SVGElement | HTMLElement = SVGElement | HTMLElement>(selector: string, pageFunction: PageFunctionOn<E, Arg, R>, arg: Arg): Promise<R>;
  $eval<K extends keyof HTMLElementTagNameMap, R>(selector: K, pageFunction: PageFunctionOn<HTMLElementTagNameMap[K], void, R>, arg?: any): Promise<R>;
  $eval<R, E extends SVGElement | HTMLElement = SVGElement | HTMLElement>(selector: string, pageFunction: PageFunctionOn<E, void, R>, arg?: any): Promise<R>;

  /**
   * Returns the return value of `pageFunction`
   * 
   * The method finds all elements matching the specified selector within the frame and passes an array of matched elements
   * as a first argument to `pageFunction`. See [Working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more
   * details.
   * 
   * If `pageFunction` returns a [Promise], then `frame.$$eval` would wait for the promise to resolve and return its value.
   * 
   * Examples:
   * 
   * ```js
   * const divsCounts = await frame.$$eval('div', (divs, min) => divs.length >= min, 10);
   * ```
   * 
   * @param selector A selector to query for. See [working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details.
   * @param pageFunction Function to be evaluated in browser context
   * @param arg Optional argument to pass to `pageFunction`
   */
  $$eval<K extends keyof HTMLElementTagNameMap, R, Arg>(selector: K, pageFunction: PageFunctionOn<HTMLElementTagNameMap[K][], Arg, R>, arg: Arg): Promise<R>;
  $$eval<R, Arg, E extends SVGElement | HTMLElement = SVGElement | HTMLElement>(selector: string, pageFunction: PageFunctionOn<E[], Arg, R>, arg: Arg): Promise<R>;
  $$eval<K extends keyof HTMLElementTagNameMap, R>(selector: K, pageFunction: PageFunctionOn<HTMLElementTagNameMap[K][], void, R>, arg?: any): Promise<R>;
  $$eval<R, E extends SVGElement | HTMLElement = SVGElement | HTMLElement>(selector: string, pageFunction: PageFunctionOn<E[], void, R>, arg?: any): Promise<R>;

  /**
   * Returns when the `pageFunction` returns a truthy value, returns that value.
   * 
   * The `waitForFunction` can be used to observe viewport size change:
   * 
   * ```js
   * const { firefox } = require('playwright');  // Or 'chromium' or 'webkit'.
   * 
   * (async () => {
   *   const browser = await firefox.launch();
   *   const page = await browser.newPage();
   *   const watchDog = page.mainFrame().waitForFunction('window.innerWidth < 100');
   *   page.setViewportSize({width: 50, height: 50});
   *   await watchDog;
   *   await browser.close();
   * })();
   * ```
   * 
   * To pass an argument to the predicate of `frame.waitForFunction` function:
   * 
   * ```js
   * const selector = '.foo';
   * await frame.waitForFunction(selector => !!document.querySelector(selector), selector);
   * ```
   * 
   * @param pageFunction Function to be evaluated in browser context
   * @param arg Optional argument to pass to `pageFunction`
   * @param options 
   */
  waitForFunction<R, Arg>(pageFunction: PageFunction<Arg, R>, arg: Arg, options?: PageWaitForFunctionOptions): Promise<SmartHandle<R>>;
  waitForFunction<R>(pageFunction: PageFunction<void, R>, arg?: any, options?: PageWaitForFunctionOptions): Promise<SmartHandle<R>>;

  /**
   * Returns when element specified by selector satisfies `state` option. Returns `null` if waiting for `hidden` or
   * `detached`.
   * 
   * Wait for the `selector` to satisfy `state` option (either appear/disappear from dom, or become visible/hidden). If at
   * the moment of calling the method `selector` already satisfies the condition, the method will return immediately. If the
   * selector doesn't satisfy the condition for the `timeout` milliseconds, the function will throw.
   * 
   * This method works across navigations:
   * 
   * ```js
   * const { chromium } = require('playwright');  // Or 'firefox' or 'webkit'.
   * 
   * (async () => {
   *   const browser = await chromium.launch();
   *   const page = await browser.newPage();
   *   for (let currentURL of ['https://google.com', 'https://bbc.com']) {
   *     await page.goto(currentURL);
   *     const element = await page.mainFrame().waitForSelector('img');
   *     console.log('Loaded image: ' + await element.getAttribute('src'));
   *   }
   *   await browser.close();
   * })();
   * ```
   * 
   * @param selector A selector to query for. See [working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details.
   * @param options 
   */
  waitForSelector<K extends keyof HTMLElementTagNameMap>(selector: K, options?: PageWaitForSelectorOptionsNotHidden): Promise<ElementHandleForTag<K>>;
  waitForSelector(selector: string, options?: PageWaitForSelectorOptionsNotHidden): Promise<ElementHandle<SVGElement | HTMLElement>>;
  waitForSelector<K extends keyof HTMLElementTagNameMap>(selector: K, options: PageWaitForSelectorOptions): Promise<ElementHandleForTag<K> | null>;
  waitForSelector(selector: string, options: PageWaitForSelectorOptions): Promise<null|ElementHandle<SVGElement | HTMLElement>>;
  /**
   * Returns the added tag when the script's onload fires or when the script content was injected into frame.
   * 
   * Adds a `<script>` tag into the page with the desired url or content.
   * @param options 
   */
  addScriptTag(options?: {
    /**
     * Raw JavaScript content to be injected into frame.
     */
    content?: string;

    /**
     * Path to the JavaScript file to be injected into frame. If `path` is a relative path, then it is resolved relative to the
     * current working directory.
     */
    path?: string;

    /**
     * Script type. Use 'module' in order to load a Javascript ES6 module. See
     * [script](https://developer.mozilla.org/en-US/docs/Web/HTML/Element/script) for more details.
     */
    type?: string;

    /**
     * URL of a script to be added.
     */
    url?: string;
  }): Promise<ElementHandle>;

  /**
   * Returns the added tag when the stylesheet's onload fires or when the CSS content was injected into frame.
   * 
   * Adds a `<link rel="stylesheet">` tag into the page with the desired url or a `<style type="text/css">` tag with the
   * content.
   * @param options 
   */
  addStyleTag(options?: {
    /**
     * Raw CSS content to be injected into frame.
     */
    content?: string;

    /**
     * Path to the CSS file to be injected into frame. If `path` is a relative path, then it is resolved relative to the
     * current working directory.
     */
    path?: string;

    /**
     * URL of the `<link>` tag.
     */
    url?: string;
  }): Promise<ElementHandle>;

  /**
   * This method checks an element matching `selector` by performing the following steps:
   * 1. Find an element match matching `selector`. If there is none, wait until a matching element is attached to the DOM.
   * 1. Ensure that matched element is a checkbox or a radio input. If not, this method rejects. If the element is already
   *    checked, this method returns immediately.
   * 1. Wait for [actionability](https://github.com/microsoft/playwright/blob/master/docs/actionability.md) checks on the matched element, unless `force` option is set. If the
   *    element is detached during the checks, the whole action is retried.
   * 1. Scroll the element into view if needed.
   * 1. Use [page.mouse](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagemouse) to click in the center
   *    of the element.
   * 1. Wait for initiated navigations to either succeed or fail, unless `noWaitAfter` option is set.
   * 1. Ensure that the element is now checked. If not, this method rejects.
   * 
   * When all steps combined have not finished during the specified `timeout`, this method rejects with a [TimeoutError].
   * Passing zero timeout disables this.
   * @param selector A selector to search for element. If there are multiple elements satisfying the selector, the first will be used. See [working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details.
   * @param options 
   */
  check(selector: string, options?: {
    /**
     * Whether to bypass the [actionability](https://github.com/microsoft/playwright/blob/master/docs/actionability.md) checks. Defaults to `false`.
     */
    force?: boolean;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You can
     * opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as navigating to
     * inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed by
     * using the
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;
  }): Promise<void>;

  childFrames(): Array<Frame>;

  /**
   * This method clicks an element matching `selector` by performing the following steps:
   * 1. Find an element match matching `selector`. If there is none, wait until a matching element is attached to the DOM.
   * 1. Wait for [actionability](https://github.com/microsoft/playwright/blob/master/docs/actionability.md) checks on the matched element, unless `force` option is set. If the
   *    element is detached during the checks, the whole action is retried.
   * 1. Scroll the element into view if needed.
   * 1. Use [page.mouse](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagemouse) to click in the center
   *    of the element, or the specified `position`.
   * 1. Wait for initiated navigations to either succeed or fail, unless `noWaitAfter` option is set.
   * 
   * When all steps combined have not finished during the specified `timeout`, this method rejects with a [TimeoutError].
   * Passing zero timeout disables this.
   * @param selector A selector to search for element. If there are multiple elements satisfying the selector, the first will be used. See [working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details.
   * @param options 
   */
  click(selector: string, options?: {
    /**
     * Defaults to `left`.
     */
    button?: "left"|"right"|"middle";

    /**
     * defaults to 1. See [UIEvent.detail].
     */
    clickCount?: number;

    /**
     * Time to wait between `mousedown` and `mouseup` in milliseconds. Defaults to 0.
     */
    delay?: number;

    /**
     * Whether to bypass the [actionability](https://github.com/microsoft/playwright/blob/master/docs/actionability.md) checks. Defaults to `false`.
     */
    force?: boolean;

    /**
     * Modifier keys to press. Ensures that only these modifiers are pressed during the operation, and then restores current
     * modifiers back. If not specified, currently pressed modifiers are used.
     */
    modifiers?: Array<"Alt"|"Control"|"Meta"|"Shift">;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You can
     * opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as navigating to
     * inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * A point to use relative to the top-left corner of element padding box. If not specified, uses some visible point of the
     * element.
     */
    position?: {
      x: number;

      y: number;
    };

    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed by
     * using the
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * Gets the full HTML contents of the frame, including the doctype.
   */
  content(): Promise<string>;

  /**
   * This method double clicks an element matching `selector` by performing the following steps:
   * 1. Find an element match matching `selector`. If there is none, wait until a matching element is attached to the DOM.
   * 1. Wait for [actionability](https://github.com/microsoft/playwright/blob/master/docs/actionability.md) checks on the matched element, unless `force` option is set. If the
   *    element is detached during the checks, the whole action is retried.
   * 1. Scroll the element into view if needed.
   * 1. Use [page.mouse](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagemouse) to double click in the
   *    center of the element, or the specified `position`.
   * 1. Wait for initiated navigations to either succeed or fail, unless `noWaitAfter` option is set. Note that if the
   *    first click of the `dblclick()` triggers a navigation event, this method will reject.
   * 
   * When all steps combined have not finished during the specified `timeout`, this method rejects with a [TimeoutError].
   * Passing zero timeout disables this.
   * 
   * > NOTE: `frame.dblclick()` dispatches two `click` events and a single `dblclick` event.
   * @param selector A selector to search for element. If there are multiple elements satisfying the selector, the first will be used. See [working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details.
   * @param options 
   */
  dblclick(selector: string, options?: {
    /**
     * Defaults to `left`.
     */
    button?: "left"|"right"|"middle";

    /**
     * Time to wait between `mousedown` and `mouseup` in milliseconds. Defaults to 0.
     */
    delay?: number;

    /**
     * Whether to bypass the [actionability](https://github.com/microsoft/playwright/blob/master/docs/actionability.md) checks. Defaults to `false`.
     */
    force?: boolean;

    /**
     * Modifier keys to press. Ensures that only these modifiers are pressed during the operation, and then restores current
     * modifiers back. If not specified, currently pressed modifiers are used.
     */
    modifiers?: Array<"Alt"|"Control"|"Meta"|"Shift">;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You can
     * opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as navigating to
     * inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * A point to use relative to the top-left corner of element padding box. If not specified, uses some visible point of the
     * element.
     */
    position?: {
      x: number;

      y: number;
    };

    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed by
     * using the
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * The snippet below dispatches the `click` event on the element. Regardless of the visibility state of the elment, `click`
   * is dispatched. This is equivalend to calling
   * [element.click()](https://developer.mozilla.org/en-US/docs/Web/API/HTMLElement/click).
   * 
   * ```js
   * await frame.dispatchEvent('button#submit', 'click');
   * ```
   * 
   * Under the hood, it creates an instance of an event based on the given `type`, initializes it with `eventInit` properties
   * and dispatches it on the element. Events are `composed`, `cancelable` and bubble by default.
   * 
   * Since `eventInit` is event-specific, please refer to the events documentation for the lists of initial properties:
   * - [DragEvent](https://developer.mozilla.org/en-US/docs/Web/API/DragEvent/DragEvent)
   * - [FocusEvent](https://developer.mozilla.org/en-US/docs/Web/API/FocusEvent/FocusEvent)
   * - [KeyboardEvent](https://developer.mozilla.org/en-US/docs/Web/API/KeyboardEvent/KeyboardEvent)
   * - [MouseEvent](https://developer.mozilla.org/en-US/docs/Web/API/MouseEvent/MouseEvent)
   * - [PointerEvent](https://developer.mozilla.org/en-US/docs/Web/API/PointerEvent/PointerEvent)
   * - [TouchEvent](https://developer.mozilla.org/en-US/docs/Web/API/TouchEvent/TouchEvent)
   * - [Event](https://developer.mozilla.org/en-US/docs/Web/API/Event/Event)
   * 
   * You can also specify `JSHandle` as the property value if you want live objects to be passed into the event:
   * 
   * ```js
   * // Note you can only create DataTransfer in Chromium and Firefox
   * const dataTransfer = await frame.evaluateHandle(() => new DataTransfer());
   * await frame.dispatchEvent('#source', 'dragstart', { dataTransfer });
   * ```
   * 
   * @param selector A selector to search for element. If there are multiple elements satisfying the selector, the first will be used. See [working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details.
   * @param type DOM event type: `"click"`, `"dragstart"`, etc.
   * @param eventInit Optional event-specific initialization properties.
   * @param options 
   */
  dispatchEvent(selector: string, type: string, eventInit?: EvaluationArgument, options?: {
    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed by
     * using the
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * This method waits for an element matching `selector`, waits for [actionability](https://github.com/microsoft/playwright/blob/master/docs/actionability.md) checks, focuses the
   * element, fills it and triggers an `input` event after filling. If the element matching `selector` is not an `<input>`,
   * `<textarea>` or `[contenteditable]` element, this method throws an error. Note that you can pass an empty string to
   * clear the input field.
   * 
   * To send fine-grained keyboard events, use
   * [frame.type(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#frametype).
   * @param selector A selector to search for element. If there are multiple elements satisfying the selector, the first will be used. See [working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details.
   * @param value Value to fill for the `<input>`, `<textarea>` or `[contenteditable]` element.
   * @param options 
   */
  fill(selector: string, value: string, options?: {
    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You can
     * opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as navigating to
     * inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed by
     * using the
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * This method fetches an element with `selector` and focuses it. If there's no element matching `selector`, the method
   * waits until a matching element appears in the DOM.
   * @param selector A selector to search for element. If there are multiple elements satisfying the selector, the first will be used. See [working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details.
   * @param options 
   */
  focus(selector: string, options?: {
    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed by
     * using the
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * Returns the `frame` or `iframe` element handle which corresponds to this frame.
   * 
   * This is an inverse of
   * [elementHandle.contentFrame(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#elementhandlecontentframe).
   * Note that returned handle actually belongs to the parent frame.
   * 
   * This method throws an error if the frame has been detached before `frameElement()` returns.
   * 
   * ```js
   * const frameElement = await frame.frameElement();
   * const contentFrame = await frameElement.contentFrame();
   * console.log(frame === contentFrame);  // -> true
   * ```
   * 
   */
  frameElement(): Promise<ElementHandle>;

  /**
   * Returns element attribute value.
   * @param selector A selector to search for element. If there are multiple elements satisfying the selector, the first will be used. See [working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details.
   * @param name Attribute name to get the value for.
   * @param options 
   */
  getAttribute(selector: string, name: string, options?: {
    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed by
     * using the
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;
  }): Promise<null|string>;

  /**
   * Returns the main resource response. In case of multiple redirects, the navigation will resolve with the response of the
   * last redirect.
   * 
   * `frame.goto` will throw an error if:
   * - there's an SSL error (e.g. in case of self-signed certificates).
   * - target URL is invalid.
   * - the `timeout` is exceeded during navigation.
   * - the remote server does not respond or is unreachable.
   * - the main resource failed to load.
   * 
   * `frame.goto` will not throw an error when any valid HTTP status code is returned by the remote server, including 404
   * "Not Found" and 500 "Internal Server Error".  The status code for such responses can be retrieved by calling
   * [response.status(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#responsestatus).
   * 
   * > NOTE: `frame.goto` either throws an error or returns a main resource response. The only exceptions are navigation to
   * `about:blank` or navigation to the same URL with a different hash, which would succeed and return `null`.
   * > NOTE: Headless mode doesn't support navigation to a PDF document. See the
   * [upstream issue](https://bugs.chromium.org/p/chromium/issues/detail?id=761295).
   * @param url URL to navigate frame to. The url should include scheme, e.g. `https://`.
   * @param options 
   */
  goto(url: string, options?: {
    /**
     * Referer header value. If provided it will take preference over the referer header value set by
     * [page.setExtraHTTPHeaders(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetextrahttpheaders).
     */
    referer?: string;

    /**
     * Maximum operation time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be
     * changed by using the
     * [browserContext.setDefaultNavigationTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaultnavigationtimeout),
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout),
     * [page.setDefaultNavigationTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaultnavigationtimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;

    /**
     * When to consider operation succeeded, defaults to `load`. Events can be either:
     * - `'domcontentloaded'` - consider operation to be finished when the `DOMContentLoaded` event is fired.
     * - `'load'` - consider operation to be finished when the `load` event is fired.
     * - `'networkidle'` - consider operation to be finished when there are no network connections for at least `500` ms.
     */
    waitUntil?: "load"|"domcontentloaded"|"networkidle";
  }): Promise<null|Response>;

  /**
   * This method hovers over an element matching `selector` by performing the following steps:
   * 1. Find an element match matching `selector`. If there is none, wait until a matching element is attached to the DOM.
   * 1. Wait for [actionability](https://github.com/microsoft/playwright/blob/master/docs/actionability.md) checks on the matched element, unless `force` option is set. If the
   *    element is detached during the checks, the whole action is retried.
   * 1. Scroll the element into view if needed.
   * 1. Use [page.mouse](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagemouse) to hover over the
   *    center of the element, or the specified `position`.
   * 1. Wait for initiated navigations to either succeed or fail, unless `noWaitAfter` option is set.
   * 
   * When all steps combined have not finished during the specified `timeout`, this method rejects with a [TimeoutError].
   * Passing zero timeout disables this.
   * @param selector A selector to search for element. If there are multiple elements satisfying the selector, the first will be used. See [working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details.
   * @param options 
   */
  hover(selector: string, options?: {
    /**
     * Whether to bypass the [actionability](https://github.com/microsoft/playwright/blob/master/docs/actionability.md) checks. Defaults to `false`.
     */
    force?: boolean;

    /**
     * Modifier keys to press. Ensures that only these modifiers are pressed during the operation, and then restores current
     * modifiers back. If not specified, currently pressed modifiers are used.
     */
    modifiers?: Array<"Alt"|"Control"|"Meta"|"Shift">;

    /**
     * A point to use relative to the top-left corner of element padding box. If not specified, uses some visible point of the
     * element.
     */
    position?: {
      x: number;

      y: number;
    };

    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed by
     * using the
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * Returns `element.innerHTML`.
   * @param selector A selector to search for element. If there are multiple elements satisfying the selector, the first will be used. See [working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details.
   * @param options 
   */
  innerHTML(selector: string, options?: {
    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed by
     * using the
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;
  }): Promise<string>;

  /**
   * Returns `element.innerText`.
   * @param selector A selector to search for element. If there are multiple elements satisfying the selector, the first will be used. See [working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details.
   * @param options 
   */
  innerText(selector: string, options?: {
    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed by
     * using the
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;
  }): Promise<string>;

  /**
   * Returns whether the element is checked. Throws if the element is not a checkbox or radio input.
   * @param selector A selector to search for element. If there are multiple elements satisfying the selector, the first will be used. See [working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details.
   * @param options 
   */
  isChecked(selector: string, options?: {
    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed by
     * using the
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;
  }): Promise<boolean>;

  /**
   * Returns `true` if the frame has been detached, or `false` otherwise.
   */
  isDetached(): boolean;

  /**
   * Returns whether the element is disabled, the opposite of [enabled](https://github.com/microsoft/playwright/blob/master/docs/actionability.md#enabled).
   * @param selector A selector to search for element. If there are multiple elements satisfying the selector, the first will be used. See [working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details.
   * @param options 
   */
  isDisabled(selector: string, options?: {
    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed by
     * using the
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;
  }): Promise<boolean>;

  /**
   * Returns whether the element is [editable](https://github.com/microsoft/playwright/blob/master/docs/actionability.md#editable).
   * @param selector A selector to search for element. If there are multiple elements satisfying the selector, the first will be used. See [working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details.
   * @param options 
   */
  isEditable(selector: string, options?: {
    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed by
     * using the
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;
  }): Promise<boolean>;

  /**
   * Returns whether the element is [enabled](https://github.com/microsoft/playwright/blob/master/docs/actionability.md#enabled).
   * @param selector A selector to search for element. If there are multiple elements satisfying the selector, the first will be used. See [working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details.
   * @param options 
   */
  isEnabled(selector: string, options?: {
    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed by
     * using the
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;
  }): Promise<boolean>;

  /**
   * Returns whether the element is hidden, the opposite of [visible](https://github.com/microsoft/playwright/blob/master/docs/actionability.md#visible).  `selector` that does not
   * match any elements is considered hidden.
   * @param selector A selector to search for element. If there are multiple elements satisfying the selector, the first will be used. See [working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details.
   * @param options 
   */
  isHidden(selector: string, options?: {
    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed by
     * using the
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;
  }): Promise<boolean>;

  /**
   * Returns whether the element is [visible](https://github.com/microsoft/playwright/blob/master/docs/actionability.md#visible). `selector` that does not match any elements is
   * considered not visible.
   * @param selector A selector to search for element. If there are multiple elements satisfying the selector, the first will be used. See [working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details.
   * @param options 
   */
  isVisible(selector: string, options?: {
    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed by
     * using the
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;
  }): Promise<boolean>;

  /**
   * Returns frame's name attribute as specified in the tag.
   * 
   * If the name is empty, returns the id attribute instead.
   * 
   * > NOTE: This value is calculated once when the frame is created, and will not update if the attribute is changed later.
   */
  name(): string;

  /**
   * Returns the page containing this frame.
   */
  page(): Page;

  /**
   * Parent frame, if any. Detached frames and main frames return `null`.
   */
  parentFrame(): null|Frame;

  /**
   * `key` can specify the intended [keyboardEvent.key](https://developer.mozilla.org/en-US/docs/Web/API/KeyboardEvent/key)
   * value or a single character to generate the text for. A superset of the `key` values can be found
   * [here](https://developer.mozilla.org/en-US/docs/Web/API/KeyboardEvent/key/Key_Values). Examples of the keys are:
   * 
   * `F1` - `F12`, `Digit0`- `Digit9`, `KeyA`- `KeyZ`, `Backquote`, `Minus`, `Equal`, `Backslash`, `Backspace`, `Tab`,
   * `Delete`, `Escape`, `ArrowDown`, `End`, `Enter`, `Home`, `Insert`, `PageDown`, `PageUp`, `ArrowRight`, `ArrowUp`, etc.
   * 
   * Following modification shortcuts are also supported: `Shift`, `Control`, `Alt`, `Meta`, `ShiftLeft`.
   * 
   * Holding down `Shift` will type the text that corresponds to the `key` in the upper case.
   * 
   * If `key` is a single character, it is case-sensitive, so the values `a` and `A` will generate different respective
   * texts.
   * 
   * Shortcuts such as `key: "Control+o"` or `key: "Control+Shift+T"` are supported as well. When speficied with the
   * modifier, modifier is pressed and being held while the subsequent key is being pressed.
   * @param selector A selector to search for element. If there are multiple elements satisfying the selector, the first will be used. See [working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details.
   * @param key Name of the key to press or a character to generate, such as `ArrowLeft` or `a`.
   * @param options 
   */
  press(selector: string, key: string, options?: {
    /**
     * Time to wait between `keydown` and `keyup` in milliseconds. Defaults to 0.
     */
    delay?: number;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You can
     * opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as navigating to
     * inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed by
     * using the
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * Returns the array of option values that have been successfully selected.
   * 
   * Triggers a `change` and `input` event once all the provided options have been selected. If there's no `<select>` element
   * matching `selector`, the method throws an error.
   * 
   * Will wait until all specified options are present in the `<select>` element.
   * 
   * ```js
   * // single selection matching the value
   * frame.selectOption('select#colors', 'blue');
   * 
   * // single selection matching both the value and the label
   * frame.selectOption('select#colors', { label: 'Blue' });
   * 
   * // multiple selection
   * frame.selectOption('select#colors', 'red', 'green', 'blue');
   * ```
   * 
   * @param selector A selector to query for. See [working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details.
   * @param values Options to select. If the `<select>` has the `multiple` attribute, all matching options are selected, otherwise only the first option matching one of the passed options is selected. String values are equivalent to `{value:'string'}`. Option
   * is considered matching if all specified properties match.
   * @param options 
   */
  selectOption(selector: string, values: null|string|ElementHandle|Array<string>|{
    /**
     * Matches by `option.value`. Optional.
     */
    value?: string;

    /**
     * Matches by `option.label`. Optional.
     */
    label?: string;

    /**
     * Matches by the index. Optional.
     */
    index?: number;
  }|Array<ElementHandle>|Array<{
    /**
     * Matches by `option.value`. Optional.
     */
    value?: string;

    /**
     * Matches by `option.label`. Optional.
     */
    label?: string;

    /**
     * Matches by the index. Optional.
     */
    index?: number;
  }>, options?: {
    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You can
     * opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as navigating to
     * inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed by
     * using the
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;
  }): Promise<Array<string>>;

  /**
   * @param html HTML markup to assign to the page.
   * @param options 
   */
  setContent(html: string, options?: {
    /**
     * Maximum operation time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be
     * changed by using the
     * [browserContext.setDefaultNavigationTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaultnavigationtimeout),
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout),
     * [page.setDefaultNavigationTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaultnavigationtimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;

    /**
     * When to consider operation succeeded, defaults to `load`. Events can be either:
     * - `'domcontentloaded'` - consider operation to be finished when the `DOMContentLoaded` event is fired.
     * - `'load'` - consider operation to be finished when the `load` event is fired.
     * - `'networkidle'` - consider operation to be finished when there are no network connections for at least `500` ms.
     */
    waitUntil?: "load"|"domcontentloaded"|"networkidle";
  }): Promise<void>;

  /**
   * This method expects `selector` to point to an
   * [input element](https://developer.mozilla.org/en-US/docs/Web/HTML/Element/input).
   * 
   * Sets the value of the file input to these file paths or files. If some of the `filePaths` are relative paths, then they
   * are resolved relative to the the current working directory. For empty array, clears the selected files.
   * @param selector A selector to search for element. If there are multiple elements satisfying the selector, the first will be used. See [working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details.
   * @param files 
   * @param options 
   */
  setInputFiles(selector: string, files: string|Array<string>|{
    /**
     * [File] name
     */
    name: string;

    /**
     * [File] type
     */
    mimeType: string;

    /**
     * File content
     */
    buffer: Buffer;
  }|Array<{
    /**
     * [File] name
     */
    name: string;

    /**
     * [File] type
     */
    mimeType: string;

    /**
     * File content
     */
    buffer: Buffer;
  }>, options?: {
    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You can
     * opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as navigating to
     * inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed by
     * using the
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * This method taps an element matching `selector` by performing the following steps:
   * 1. Find an element match matching `selector`. If there is none, wait until a matching element is attached to the DOM.
   * 1. Wait for [actionability](https://github.com/microsoft/playwright/blob/master/docs/actionability.md) checks on the matched element, unless `force` option is set. If the
   *    element is detached during the checks, the whole action is retried.
   * 1. Scroll the element into view if needed.
   * 1. Use [page.touchscreen](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagetouchscreen) to tap the
   *    center of the element, or the specified `position`.
   * 1. Wait for initiated navigations to either succeed or fail, unless `noWaitAfter` option is set.
   * 
   * When all steps combined have not finished during the specified `timeout`, this method rejects with a [TimeoutError].
   * Passing zero timeout disables this.
   * 
   * > NOTE: `frame.tap()` requires that the `hasTouch` option of the browser context be set to true.
   * @param selector A selector to search for element. If there are multiple elements satisfying the selector, the first will be used. See [working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details.
   * @param options 
   */
  tap(selector: string, options?: {
    /**
     * Whether to bypass the [actionability](https://github.com/microsoft/playwright/blob/master/docs/actionability.md) checks. Defaults to `false`.
     */
    force?: boolean;

    /**
     * Modifier keys to press. Ensures that only these modifiers are pressed during the operation, and then restores current
     * modifiers back. If not specified, currently pressed modifiers are used.
     */
    modifiers?: Array<"Alt"|"Control"|"Meta"|"Shift">;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You can
     * opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as navigating to
     * inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * A point to use relative to the top-left corner of element padding box. If not specified, uses some visible point of the
     * element.
     */
    position?: {
      x: number;

      y: number;
    };

    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed by
     * using the
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * Returns `element.textContent`.
   * @param selector A selector to search for element. If there are multiple elements satisfying the selector, the first will be used. See [working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details.
   * @param options 
   */
  textContent(selector: string, options?: {
    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed by
     * using the
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;
  }): Promise<null|string>;

  /**
   * Returns the page title.
   */
  title(): Promise<string>;

  /**
   * Sends a `keydown`, `keypress`/`input`, and `keyup` event for each character in the text. `frame.type` can be used to
   * send fine-grained keyboard events. To fill values in form fields, use
   * [frame.fill(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#framefill).
   * 
   * To press a special key, like `Control` or `ArrowDown`, use
   * [keyboard.press(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#keyboardpress).
   * 
   * ```js
   * await frame.type('#mytextarea', 'Hello'); // Types instantly
   * await frame.type('#mytextarea', 'World', {delay: 100}); // Types slower, like a user
   * ```
   * 
   * @param selector A selector to search for element. If there are multiple elements satisfying the selector, the first will be used. See [working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details.
   * @param text A text to type into a focused element.
   * @param options 
   */
  type(selector: string, text: string, options?: {
    /**
     * Time to wait between key presses in milliseconds. Defaults to 0.
     */
    delay?: number;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You can
     * opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as navigating to
     * inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed by
     * using the
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * This method checks an element matching `selector` by performing the following steps:
   * 1. Find an element match matching `selector`. If there is none, wait until a matching element is attached to the DOM.
   * 1. Ensure that matched element is a checkbox or a radio input. If not, this method rejects. If the element is already
   *    unchecked, this method returns immediately.
   * 1. Wait for [actionability](https://github.com/microsoft/playwright/blob/master/docs/actionability.md) checks on the matched element, unless `force` option is set. If the
   *    element is detached during the checks, the whole action is retried.
   * 1. Scroll the element into view if needed.
   * 1. Use [page.mouse](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagemouse) to click in the center
   *    of the element.
   * 1. Wait for initiated navigations to either succeed or fail, unless `noWaitAfter` option is set.
   * 1. Ensure that the element is now unchecked. If not, this method rejects.
   * 
   * When all steps combined have not finished during the specified `timeout`, this method rejects with a [TimeoutError].
   * Passing zero timeout disables this.
   * @param selector A selector to search for element. If there are multiple elements satisfying the selector, the first will be used. See [working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details.
   * @param options 
   */
  uncheck(selector: string, options?: {
    /**
     * Whether to bypass the [actionability](https://github.com/microsoft/playwright/blob/master/docs/actionability.md) checks. Defaults to `false`.
     */
    force?: boolean;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You can
     * opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as navigating to
     * inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed by
     * using the
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * Returns frame's url.
   */
  url(): string;

  /**
   * Waits for the required load state to be reached.
   * 
   * This returns when the frame reaches a required load state, `load` by default. The navigation must have been committed
   * when this method is called. If current document has already reached the required state, resolves immediately.
   * 
   * ```js
   * await frame.click('button'); // Click triggers navigation.
   * await frame.waitForLoadState(); // Waits for 'load' state by default.
   * ```
   * 
   * @param state Optional load state to wait for, defaults to `load`. If the state has been already reached while loading current document, the method resolves immediately. Can be one of:
   * - `'load'` - wait for the `load` event to be fired.
   * - `'domcontentloaded'` - wait for the `DOMContentLoaded` event to be fired.
   * - `'networkidle'` - wait until there are no network connections for at least `500` ms.
   * @param options 
   */
  waitForLoadState(state?: "load"|"domcontentloaded"|"networkidle", options?: {
    /**
     * Maximum operation time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be
     * changed by using the
     * [browserContext.setDefaultNavigationTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaultnavigationtimeout),
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout),
     * [page.setDefaultNavigationTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaultnavigationtimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * Waits for the frame navigation and returns the main resource response. In case of multiple redirects, the navigation
   * will resolve with the response of the last redirect. In case of navigation to a different anchor or navigation due to
   * History API usage, the navigation will resolve with `null`.
   * 
   * This method waits for the frame to navigate to a new URL. It is useful for when you run code which will indirectly cause
   * the frame to navigate. Consider this example:
   * 
   * ```js
   * const [response] = await Promise.all([
   *   frame.waitForNavigation(), // The promise resolves after navigation has finished
   *   frame.click('a.delayed-navigation'), // Clicking the link will indirectly cause a navigation
   * ]);
   * ```
   * 
   * > NOTE: Usage of the [History API](https://developer.mozilla.org/en-US/docs/Web/API/History_API) to change the URL is
   * considered a navigation.
   * @param options 
   */
  waitForNavigation(options?: {
    /**
     * Maximum operation time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be
     * changed by using the
     * [browserContext.setDefaultNavigationTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaultnavigationtimeout),
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout),
     * [page.setDefaultNavigationTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaultnavigationtimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;

    /**
     * URL string, URL regex pattern or predicate receiving [URL] to match while waiting for the navigation.
     */
    url?: string|RegExp|((url: URL) => boolean);

    /**
     * When to consider operation succeeded, defaults to `load`. Events can be either:
     * - `'domcontentloaded'` - consider operation to be finished when the `DOMContentLoaded` event is fired.
     * - `'load'` - consider operation to be finished when the `load` event is fired.
     * - `'networkidle'` - consider operation to be finished when there are no network connections for at least `500` ms.
     */
    waitUntil?: "load"|"domcontentloaded"|"networkidle";
  }): Promise<null|Response>;

  /**
   * Waits for the given `timeout` in milliseconds.
   * 
   * Note that `frame.waitForTimeout()` should only be used for debugging. Tests using the timer in production are going to
   * be flaky. Use signals such as network events, selectors becoming visible and others instead.
   * @param timeout A timeout to wait for
   */
  waitForTimeout(timeout: number): Promise<void>;}

/**
 * - extends: [EventEmitter]
 * 
 * BrowserContexts provide a way to operate multiple independent browser sessions.
 * 
 * If a page opens another page, e.g. with a `window.open` call, the popup will belong to the parent page's browser
 * context.
 * 
 * Playwright allows creation of "incognito" browser contexts with `browser.newContext()` method. "Incognito" browser
 * contexts don't write any browsing data to disk.
 * 
 * ```js
 * // Create a new incognito browser context
 * const context = await browser.newContext();
 * // Create a new page inside context.
 * const page = await context.newPage();
 * await page.goto('https://example.com');
 * // Dispose context once it's no longer needed.
 * await context.close();
 * ```
 * 
 */
export interface BrowserContext {
  /**
   * The method adds a function called `name` on the `window` object of every frame in every page in the context. When
   * called, the function executes `callback` and returns a [Promise] which resolves to the return value of `callback`. If
   * the `callback` returns a [Promise], it will be awaited.
   * 
   * The first argument of the `callback` function contains information about the caller: `{ browserContext: BrowserContext,
   * page: Page, frame: Frame }`.
   * 
   * See [page.exposeBinding(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageexposebinding) for
   * page-only version.
   * 
   * An example of exposing page URL to all frames in all pages in the context:
   * 
   * ```js
   * const { webkit } = require('playwright');  // Or 'chromium' or 'firefox'.
   * 
   * (async () => {
   *   const browser = await webkit.launch({ headless: false });
   *   const context = await browser.newContext();
   *   await context.exposeBinding('pageURL', ({ page }) => page.url());
   *   const page = await context.newPage();
   *   await page.setContent(`
   *     <script>
   *       async function onClick() {
   *         document.querySelector('div').textContent = await window.pageURL();
   *       }
   *     </script>
   *     <button onclick="onClick()">Click me</button>
   *     <div></div>
   *   `);
   *   await page.click('button');
   * })();
   * ```
   * 
   * An example of passing an element handle:
   * 
   * ```js
   * await context.exposeBinding('clicked', async (source, element) => {
   *   console.log(await element.textContent());
   * }, { handle: true });
   * await page.setContent(`
   *   <script>
   *     document.addEventListener('click', event => window.clicked(event.target));
   *   </script>
   *   <div>Click me</div>
   *   <div>Or click me</div>
   * `);
   * ```
   * 
   * @param name Name of the function on the window object.
   * @param callback Callback function that will be called in the Playwright's context.
   * @param options 
   */
  exposeBinding(name: string, playwrightBinding: (source: BindingSource, arg: JSHandle) => any, options: { handle: true }): Promise<void>;
  exposeBinding(name: string, playwrightBinding: (source: BindingSource, ...args: any[]) => any, options?: { handle?: boolean }): Promise<void>;
  /**
   * Emitted when Browser context gets closed. This might happen because of one of the following:
   * - Browser context is closed.
   * - Browser application is closed or crashed.
   * - The [browser.close(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browserclose) method was
   *   called.
   */
  on(event: 'close', listener: () => void): this;

  /**
   * The event is emitted when a new Page is created in the BrowserContext. The page may still be loading. The event will
   * also fire for popup pages. See also
   * [page.on('popup')](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageonpopup) to receive events about
   * popups relevant to a specific page.
   * 
   * The earliest moment that page is available is when it has navigated to the initial url. For example, when opening a
   * popup with `window.open('http://example.com')`, this event will fire when the network request to "http://example.com" is
   * done and its response has started loading in the popup.
   * 
   * ```js
   * const [page] = await Promise.all([
   *   context.waitForEvent('page'),
   *   page.click('a[target=_blank]'),
   * ]);
   * console.log(await page.evaluate('location.href'));
   * ```
   * 
   * > NOTE: Use
   * [page.waitForLoadState(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagewaitforloadstate) to wait
   * until the page gets to a particular state (you should not need it in most cases).
   */
  on(event: 'page', listener: (page: Page) => void): this;

  /**
   * Emitted when Browser context gets closed. This might happen because of one of the following:
   * - Browser context is closed.
   * - Browser application is closed or crashed.
   * - The [browser.close(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browserclose) method was
   *   called.
   */
  once(event: 'close', listener: () => void): this;

  /**
   * The event is emitted when a new Page is created in the BrowserContext. The page may still be loading. The event will
   * also fire for popup pages. See also
   * [page.on('popup')](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageonpopup) to receive events about
   * popups relevant to a specific page.
   * 
   * The earliest moment that page is available is when it has navigated to the initial url. For example, when opening a
   * popup with `window.open('http://example.com')`, this event will fire when the network request to "http://example.com" is
   * done and its response has started loading in the popup.
   * 
   * ```js
   * const [page] = await Promise.all([
   *   context.waitForEvent('page'),
   *   page.click('a[target=_blank]'),
   * ]);
   * console.log(await page.evaluate('location.href'));
   * ```
   * 
   * > NOTE: Use
   * [page.waitForLoadState(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagewaitforloadstate) to wait
   * until the page gets to a particular state (you should not need it in most cases).
   */
  once(event: 'page', listener: (page: Page) => void): this;

  /**
   * Emitted when Browser context gets closed. This might happen because of one of the following:
   * - Browser context is closed.
   * - Browser application is closed or crashed.
   * - The [browser.close(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browserclose) method was
   *   called.
   */
  addListener(event: 'close', listener: () => void): this;

  /**
   * The event is emitted when a new Page is created in the BrowserContext. The page may still be loading. The event will
   * also fire for popup pages. See also
   * [page.on('popup')](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageonpopup) to receive events about
   * popups relevant to a specific page.
   * 
   * The earliest moment that page is available is when it has navigated to the initial url. For example, when opening a
   * popup with `window.open('http://example.com')`, this event will fire when the network request to "http://example.com" is
   * done and its response has started loading in the popup.
   * 
   * ```js
   * const [page] = await Promise.all([
   *   context.waitForEvent('page'),
   *   page.click('a[target=_blank]'),
   * ]);
   * console.log(await page.evaluate('location.href'));
   * ```
   * 
   * > NOTE: Use
   * [page.waitForLoadState(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagewaitforloadstate) to wait
   * until the page gets to a particular state (you should not need it in most cases).
   */
  addListener(event: 'page', listener: (page: Page) => void): this;

  /**
   * Emitted when Browser context gets closed. This might happen because of one of the following:
   * - Browser context is closed.
   * - Browser application is closed or crashed.
   * - The [browser.close(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browserclose) method was
   *   called.
   */
  removeListener(event: 'close', listener: () => void): this;

  /**
   * The event is emitted when a new Page is created in the BrowserContext. The page may still be loading. The event will
   * also fire for popup pages. See also
   * [page.on('popup')](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageonpopup) to receive events about
   * popups relevant to a specific page.
   * 
   * The earliest moment that page is available is when it has navigated to the initial url. For example, when opening a
   * popup with `window.open('http://example.com')`, this event will fire when the network request to "http://example.com" is
   * done and its response has started loading in the popup.
   * 
   * ```js
   * const [page] = await Promise.all([
   *   context.waitForEvent('page'),
   *   page.click('a[target=_blank]'),
   * ]);
   * console.log(await page.evaluate('location.href'));
   * ```
   * 
   * > NOTE: Use
   * [page.waitForLoadState(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagewaitforloadstate) to wait
   * until the page gets to a particular state (you should not need it in most cases).
   */
  removeListener(event: 'page', listener: (page: Page) => void): this;

  /**
   * Emitted when Browser context gets closed. This might happen because of one of the following:
   * - Browser context is closed.
   * - Browser application is closed or crashed.
   * - The [browser.close(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browserclose) method was
   *   called.
   */
  off(event: 'close', listener: () => void): this;

  /**
   * The event is emitted when a new Page is created in the BrowserContext. The page may still be loading. The event will
   * also fire for popup pages. See also
   * [page.on('popup')](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageonpopup) to receive events about
   * popups relevant to a specific page.
   * 
   * The earliest moment that page is available is when it has navigated to the initial url. For example, when opening a
   * popup with `window.open('http://example.com')`, this event will fire when the network request to "http://example.com" is
   * done and its response has started loading in the popup.
   * 
   * ```js
   * const [page] = await Promise.all([
   *   context.waitForEvent('page'),
   *   page.click('a[target=_blank]'),
   * ]);
   * console.log(await page.evaluate('location.href'));
   * ```
   * 
   * > NOTE: Use
   * [page.waitForLoadState(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagewaitforloadstate) to wait
   * until the page gets to a particular state (you should not need it in most cases).
   */
  off(event: 'page', listener: (page: Page) => void): this;

  /**
   * Adds cookies into this browser context. All pages within this context will have these cookies installed. Cookies can be
   * obtained via
   * [browserContext.cookies(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextcookies).
   * 
   * ```js
   * await browserContext.addCookies([cookieObject1, cookieObject2]);
   * ```
   * 
   * @param cookies 
   */
  addCookies(cookies: Array<{
    name: string;

    value: string;

    /**
     * either url or domain / path are required. Optional.
     */
    url?: string;

    /**
     * either url or domain / path are required Optional.
     */
    domain?: string;

    /**
     * either url or domain / path are required Optional.
     */
    path?: string;

    /**
     * Unix time in seconds. Optional.
     */
    expires?: number;

    /**
     * Optional.
     */
    httpOnly?: boolean;

    /**
     * Optional.
     */
    secure?: boolean;

    /**
     * Optional.
     */
    sameSite?: "Strict"|"Lax"|"None";
  }>): Promise<void>;

  /**
   * Adds a script which would be evaluated in one of the following scenarios:
   * - Whenever a page is created in the browser context or is navigated.
   * - Whenever a child frame is attached or navigated in any page in the browser context. In this case, the script is
   *   evaluated in the context of the newly attached frame.
   * 
   * The script is evaluated after the document was created but before any of its scripts were run. This is useful to amend
   * the JavaScript environment, e.g. to seed `Math.random`.
   * 
   * An example of overriding `Math.random` before the page loads:
   * 
   * ```js browser
   * // preload.js
   * Math.random = () => 42;
   * ```
   * 
   * ```js
   * // In your playwright script, assuming the preload.js file is in same directory.
   * await browserContext.addInitScript({
   *   path: 'preload.js'
   * });
   * ```
   * 
   * > NOTE: The order of evaluation of multiple scripts installed via
   * [browserContext.addInitScript(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextaddinitscript)
   * and [page.addInitScript(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageaddinitscript) is not
   * defined.
   * @param script Script to be evaluated in all pages in the browser context.
   * @param arg Optional argument to pass to `script` (only supported when passing a function).
   */
  addInitScript(script: Function|string|{
    /**
     * Path to the JavaScript file. If `path` is a relative path, then it is resolved relative to the current working
     * directory. Optional.
     */
    path?: string;

    /**
     * Raw script content. Optional.
     */
    content?: string;
  }, arg?: Serializable): Promise<void>;

  /**
   * Returns the browser instance of the context. If it was launched as a persistent context null gets returned.
   */
  browser(): null|Browser;

  /**
   * Clears context cookies.
   */
  clearCookies(): Promise<void>;

  /**
   * Clears all permission overrides for the browser context.
   * 
   * ```js
   * const context = await browser.newContext();
   * await context.grantPermissions(['clipboard-read']);
   * // do stuff ..
   * context.clearPermissions();
   * ```
   * 
   */
  clearPermissions(): Promise<void>;

  /**
   * Closes the browser context. All the pages that belong to the browser context will be closed.
   * 
   * > NOTE: The default browser context cannot be closed.
   */
  close(): Promise<void>;

  /**
   * If no URLs are specified, this method returns all cookies. If URLs are specified, only cookies that affect those URLs
   * are returned.
   * @param urls Optional list of URLs.
   */
  cookies(urls?: string|Array<string>): Promise<Array<Cookie>>;

  /**
   * The method adds a function called `name` on the `window` object of every frame in every page in the context. When
   * called, the function executes `callback` and returns a [Promise] which resolves to the return value of `callback`.
   * 
   * If the `callback` returns a [Promise], it will be awaited.
   * 
   * See [page.exposeFunction(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageexposefunction) for
   * page-only version.
   * 
   * An example of adding an `md5` function to all pages in the context:
   * 
   * ```js
   * const { webkit } = require('playwright');  // Or 'chromium' or 'firefox'.
   * const crypto = require('crypto');
   * 
   * (async () => {
   *   const browser = await webkit.launch({ headless: false });
   *   const context = await browser.newContext();
   *   await context.exposeFunction('md5', text => crypto.createHash('md5').update(text).digest('hex'));
   *   const page = await context.newPage();
   *   await page.setContent(`
   *     <script>
   *       async function onClick() {
   *         document.querySelector('div').textContent = await window.md5('PLAYWRIGHT');
   *       }
   *     </script>
   *     <button onclick="onClick()">Click me</button>
   *     <div></div>
   *   `);
   *   await page.click('button');
   * })();
   * ```
   * 
   * @param name Name of the function on the window object.
   * @param callback Callback function that will be called in the Playwright's context.
   */
  exposeFunction(name: string, callback: Function): Promise<void>;

  /**
   * Grants specified permissions to the browser context. Only grants corresponding permissions to the given origin if
   * specified.
   * @param permissions A permission or an array of permissions to grant. Permissions can be one of the following values: - `'geolocation'`
   * - `'midi'`
   * - `'midi-sysex'` (system-exclusive midi)
   * - `'notifications'`
   * - `'push'`
   * - `'camera'`
   * - `'microphone'`
   * - `'background-sync'`
   * - `'ambient-light-sensor'`
   * - `'accelerometer'`
   * - `'gyroscope'`
   * - `'magnetometer'`
   * - `'accessibility-events'`
   * - `'clipboard-read'`
   * - `'clipboard-write'`
   * - `'payment-handler'`
   * @param options 
   */
  grantPermissions(permissions: Array<string>, options?: {
    /**
     * The [origin] to grant permissions to, e.g. "https://example.com".
     */
    origin?: string;
  }): Promise<void>;

  /**
   * Creates a new page in the browser context.
   */
  newPage(): Promise<Page>;

  /**
   * Returns all open pages in the context. Non visible pages, such as `"background_page"`, will not be listed here. You can
   * find them using
   * [chromiumBrowserContext.backgroundPages(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#chromiumbrowsercontextbackgroundpages).
   */
  pages(): Array<Page>;

  /**
   * Routing provides the capability to modify network requests that are made by any page in the browser context. Once route
   * is enabled, every request matching the url pattern will stall unless it's continued, fulfilled or aborted.
   * 
   * An example of a naïve handler that aborts all image requests:
   * 
   * ```js
   * const context = await browser.newContext();
   * await context.route('**\/*.{png,jpg,jpeg}', route => route.abort());
   * const page = await context.newPage();
   * await page.goto('https://example.com');
   * await browser.close();
   * ```
   * 
   * or the same snippet using a regex pattern instead:
   * 
   * ```js
   * const context = await browser.newContext();
   * await context.route(/(\.png$)|(\.jpg$)/, route => route.abort());
   * const page = await context.newPage();
   * await page.goto('https://example.com');
   * await browser.close();
   * ```
   * 
   * Page routes (set up with [page.route(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageroute))
   * take precedence over browser context routes when request matches both handlers.
   * 
   * > NOTE: Enabling routing disables http cache.
   * @param url A glob pattern, regex pattern or predicate receiving [URL] to match while routing.
   * @param handler handler function to route the request.
   */
  route(url: string|RegExp|((url: URL) => boolean), handler: ((route: Route, request: Request) => void)): Promise<void>;

  /**
   * This setting will change the default maximum navigation time for the following methods and related shortcuts:
   * - [page.goBack(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagegoback)
   * - [page.goForward(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagegoforward)
   * - [page.goto(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagegoto)
   * - [page.reload(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagereload)
   * - [page.setContent(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetcontent)
   * - [page.waitForNavigation(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagewaitfornavigation)
   * 
   * > NOTE:
   * [page.setDefaultNavigationTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaultnavigationtimeout)
   * and [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
   * take priority over
   * [browserContext.setDefaultNavigationTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaultnavigationtimeout).
   * @param timeout Maximum navigation time in milliseconds
   */
  setDefaultNavigationTimeout(timeout: number): void;

  /**
   * This setting will change the default maximum time for all the methods accepting `timeout` option.
   * 
   * > NOTE:
   * [page.setDefaultNavigationTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaultnavigationtimeout),
   * [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout) and
   * [browserContext.setDefaultNavigationTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaultnavigationtimeout)
   * take priority over
   * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout).
   * @param timeout Maximum time in milliseconds
   */
  setDefaultTimeout(timeout: number): void;

  /**
   * The extra HTTP headers will be sent with every request initiated by any page in the context. These headers are merged
   * with page-specific extra HTTP headers set with
   * [page.setExtraHTTPHeaders(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetextrahttpheaders).
   * If page overrides a particular header, page-specific header value will be used instead of the browser context header
   * value.
   * 
   * > NOTE:
   * [browserContext.setExtraHTTPHeaders(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetextrahttpheaders)
   * does not guarantee the order of headers in the outgoing requests.
   * @param headers An object containing additional HTTP headers to be sent with every request. All header values must be strings.
   */
  setExtraHTTPHeaders(headers: { [key: string]: string; }): Promise<void>;

  /**
   * Sets the context's geolocation. Passing `null` or `undefined` emulates position unavailable.
   * 
   * ```js
   * await browserContext.setGeolocation({latitude: 59.95, longitude: 30.31667});
   * ```
   * 
   * > NOTE: Consider using
   * [browserContext.grantPermissions(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextgrantpermissions)
   * to grant permissions for the browser context pages to read its geolocation.
   * @param geolocation 
   */
  setGeolocation(geolocation: null|{
    /**
     * Latitude between -90 and 90.
     */
    latitude: number;

    /**
     * Longitude between -180 and 180.
     */
    longitude: number;

    /**
     * Non-negative accuracy value. Defaults to `0`.
     */
    accuracy?: number;
  }): Promise<void>;

  /**
   * **DEPRECATED** Browsers may cache credentials after successful authentication. Create a new browser context instead.
   * @deprecated
   * @param httpCredentials 
   */
  setHTTPCredentials(httpCredentials: null|{
    username: string;

    password: string;
  }): Promise<void>;

  /**
   * @param offline Whether to emulate network being offline for the browser context.
   */
  setOffline(offline: boolean): Promise<void>;

  /**
   * Returns storage state for this browser context, contains current cookies and local storage snapshot.
   * @param options 
   */
  storageState(options?: {
    /**
     * The file path to save the storage state to. If `path` is a relative path, then it is resolved relative to current
     * working directory. If no path is provided, storage state is still returned, but won't be saved to the disk.
     */
    path?: string;
  }): Promise<{
    cookies: Array<{
      name: string;

      value: string;

      domain: string;

      path: string;

      /**
       * Unix time in seconds.
       */
      expires: number;

      httpOnly: boolean;

      secure: boolean;

      sameSite: "Strict"|"Lax"|"None";
    }>;

    origins: Array<{
      origin: string;

      localStorage: Array<{
        name: string;

        value: string;
      }>;
    }>;
  }>;

  /**
   * Removes a route created with
   * [browserContext.route(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextroute). When
   * `handler` is not specified, removes all routes for the `url`.
   * @param url A glob pattern, regex pattern or predicate receiving [URL] used to register a routing with [browserContext.route(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextroute).
   * @param handler Optional handler function used to register a routing with [browserContext.route(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextroute).
   */
  unroute(url: string|RegExp|((url: URL) => boolean), handler?: ((route: Route, request: Request) => void)): Promise<void>;

  /**
   * Emitted when Browser context gets closed. This might happen because of one of the following:
   * - Browser context is closed.
   * - Browser application is closed or crashed.
   * - The [browser.close(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browserclose) method was
   *   called.
   */
  waitForEvent(event: 'close', optionsOrPredicate?: { predicate?: () => boolean, timeout?: number } | (() => boolean)): Promise<void>;

  /**
   * The event is emitted when a new Page is created in the BrowserContext. The page may still be loading. The event will
   * also fire for popup pages. See also
   * [page.on('popup')](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageonpopup) to receive events about
   * popups relevant to a specific page.
   * 
   * The earliest moment that page is available is when it has navigated to the initial url. For example, when opening a
   * popup with `window.open('http://example.com')`, this event will fire when the network request to "http://example.com" is
   * done and its response has started loading in the popup.
   * 
   * ```js
   * const [page] = await Promise.all([
   *   context.waitForEvent('page'),
   *   page.click('a[target=_blank]'),
   * ]);
   * console.log(await page.evaluate('location.href'));
   * ```
   * 
   * > NOTE: Use
   * [page.waitForLoadState(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagewaitforloadstate) to wait
   * until the page gets to a particular state (you should not need it in most cases).
   */
  waitForEvent(event: 'page', optionsOrPredicate?: { predicate?: (page: Page) => boolean, timeout?: number } | ((page: Page) => boolean)): Promise<Page>;
}

/**
 * The Worker class represents a [WebWorker](https://developer.mozilla.org/en-US/docs/Web/API/Web_Workers_API). `worker`
 * event is emitted on the page object to signal a worker creation. `close` event is emitted on the worker object when the
 * worker is gone.
 * 
 * ```js
 * page.on('worker', worker => {
 *   console.log('Worker created: ' + worker.url());
 *   worker.on('close', worker => console.log('Worker destroyed: ' + worker.url()));
 * });
 * 
 * console.log('Current workers:');
 * for (const worker of page.workers())
 *   console.log('  ' + worker.url());
 * ```
 * 
 */
export interface Worker {
  /**
   * Returns the return value of `pageFunction`
   * 
   * If the function passed to the `worker.evaluate` returns a [Promise], then `worker.evaluate` would wait for the promise
   * to resolve and return its value.
   * 
   * If the function passed to the `worker.evaluate` returns a non-[Serializable] value, then `worker.evaluate` returns
   * `undefined`. DevTools Protocol also supports transferring some additional values that are not serializable by `JSON`:
   * `-0`, `NaN`, `Infinity`, `-Infinity`, and bigint literals.
   * @param pageFunction Function to be evaluated in the worker context
   * @param arg Optional argument to pass to `pageFunction`
   */
  evaluate<R, Arg>(pageFunction: PageFunction<Arg, R>, arg: Arg): Promise<R>;
  evaluate<R>(pageFunction: PageFunction<void, R>, arg?: any): Promise<R>;

  /**
   * Returns the return value of `pageFunction` as in-page object (JSHandle).
   * 
   * The only difference between `worker.evaluate` and `worker.evaluateHandle` is that `worker.evaluateHandle` returns
   * in-page object (JSHandle).
   * 
   * If the function passed to the `worker.evaluateHandle` returns a [Promise], then `worker.evaluateHandle` would wait for
   * the promise to resolve and return its value.
   * @param pageFunction Function to be evaluated in the page context
   * @param arg Optional argument to pass to `pageFunction`
   */
  evaluateHandle<R, Arg>(pageFunction: PageFunction<Arg, R>, arg: Arg): Promise<SmartHandle<R>>;
  evaluateHandle<R>(pageFunction: PageFunction<void, R>, arg?: any): Promise<SmartHandle<R>>;
  /**
   * Emitted when this dedicated [WebWorker](https://developer.mozilla.org/en-US/docs/Web/API/Web_Workers_API) is terminated.
   */
  on(event: 'close', listener: (worker: Worker) => void): this;

  /**
   * Emitted when this dedicated [WebWorker](https://developer.mozilla.org/en-US/docs/Web/API/Web_Workers_API) is terminated.
   */
  once(event: 'close', listener: (worker: Worker) => void): this;

  /**
   * Emitted when this dedicated [WebWorker](https://developer.mozilla.org/en-US/docs/Web/API/Web_Workers_API) is terminated.
   */
  addListener(event: 'close', listener: (worker: Worker) => void): this;

  /**
   * Emitted when this dedicated [WebWorker](https://developer.mozilla.org/en-US/docs/Web/API/Web_Workers_API) is terminated.
   */
  removeListener(event: 'close', listener: (worker: Worker) => void): this;

  /**
   * Emitted when this dedicated [WebWorker](https://developer.mozilla.org/en-US/docs/Web/API/Web_Workers_API) is terminated.
   */
  off(event: 'close', listener: (worker: Worker) => void): this;

  url(): string;}

/**
 * JSHandle represents an in-page JavaScript object. JSHandles can be created with the
 * [page.evaluateHandle(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageevaluatehandle) method.
 * 
 * ```js
 * const windowHandle = await page.evaluateHandle(() => window);
 * // ...
 * ```
 * 
 * JSHandle prevents the referenced JavaScript object being garbage collected unless the handle is exposed with
 * [jsHandle.dispose(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#jshandledispose). JSHandles are
 * auto-disposed when their origin frame gets navigated or the parent context gets destroyed.
 * 
 * JSHandle instances can be used as an argument in
 * [page.$eval(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageeval),
 * [page.evaluate(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageevaluate) and
 * [page.evaluateHandle(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageevaluatehandle) methods.
 */
export interface JSHandle<T = any> {
  /**
   * Returns the return value of `pageFunction`
   * 
   * This method passes this handle as the first argument to `pageFunction`.
   * 
   * If `pageFunction` returns a [Promise], then `handle.evaluate` would wait for the promise to resolve and return its
   * value.
   * 
   * Examples:
   * 
   * ```js
   * const tweetHandle = await page.$('.tweet .retweets');
   * expect(await tweetHandle.evaluate(node => node.innerText)).toBe('10 retweets');
   * ```
   * 
   * @param pageFunction Function to be evaluated in browser context
   * @param arg Optional argument to pass to `pageFunction`
   */
  evaluate<R, Arg, O extends T = T>(pageFunction: PageFunctionOn<O, Arg, R>, arg: Arg): Promise<R>;
  evaluate<R, O extends T = T>(pageFunction: PageFunctionOn<O, void, R>, arg?: any): Promise<R>;

  /**
   * Returns the return value of `pageFunction` as in-page object (JSHandle).
   * 
   * This method passes this handle as the first argument to `pageFunction`.
   * 
   * The only difference between `jsHandle.evaluate` and `jsHandle.evaluateHandle` is that `jsHandle.evaluateHandle` returns
   * in-page object (JSHandle).
   * 
   * If the function passed to the `jsHandle.evaluateHandle` returns a [Promise], then `jsHandle.evaluateHandle` would wait
   * for the promise to resolve and return its value.
   * 
   * See [page.evaluateHandle(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageevaluatehandle) for
   * more details.
   * @param pageFunction Function to be evaluated
   * @param arg Optional argument to pass to `pageFunction`
   */
  evaluateHandle<R, Arg, O extends T = T>(pageFunction: PageFunctionOn<O, Arg, R>, arg: Arg): Promise<SmartHandle<R>>;
  evaluateHandle<R, O extends T = T>(pageFunction: PageFunctionOn<O, void, R>, arg?: any): Promise<SmartHandle<R>>;

  /**
   * Returns a JSON representation of the object. If the object has a `toJSON` function, it **will not be called**.
   * 
   * > NOTE: The method will return an empty JSON object if the referenced object is not stringifiable. It will throw an
   * error if the object has circular references.
   */
  jsonValue(): Promise<T>;
  /**
   * Returns either `null` or the object handle itself, if the object handle is an instance of [ElementHandle].
   */
  asElement(): T extends Node ? ElementHandle<T> : null;
  /**
   * The `jsHandle.dispose` method stops referencing the element handle.
   */
  dispose(): Promise<void>;

  /**
   * The method returns a map with **own property names** as keys and JSHandle instances for the property values.
   * 
   * ```js
   * const handle = await page.evaluateHandle(() => ({window, document}));
   * const properties = await handle.getProperties();
   * const windowHandle = properties.get('window');
   * const documentHandle = properties.get('document');
   * await handle.dispose();
   * ```
   * 
   */
  getProperties(): Promise<Map<string, JSHandle>>;

  /**
   * Fetches a single property from the referenced object.
   * @param propertyName property to get
   */
  getProperty(propertyName: string): Promise<JSHandle>;}

/**
 * - extends: [JSHandle]
 * 
 * ElementHandle represents an in-page DOM element. ElementHandles can be created with the
 * [page.$(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#page) method.
 * 
 * ```js
 * const { chromium } = require('playwright');  // Or 'firefox' or 'webkit'.
 * 
 * (async () => {
 *   const browser = await chromium.launch();
 *   const page = await browser.newPage();
 *   await page.goto('https://example.com');
 *   const hrefElement = await page.$('a');
 *   await hrefElement.click();
 *   // ...
 * })();
 * ```
 * 
 * ElementHandle prevents DOM element from garbage collection unless the handle is disposed with
 * [jsHandle.dispose(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#jshandledispose). ElementHandles
 * are auto-disposed when their origin frame gets navigated.
 * 
 * ElementHandle instances can be used as an argument in
 * [page.$eval(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageeval) and
 * [page.evaluate(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageevaluate) methods.
 */
export interface ElementHandle<T=Node> extends JSHandle<T> {
  /**
   * The method finds an element matching the specified selector in the `ElementHandle`'s subtree. See
   * [Working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details. If no elements match the selector,
   * returns `null`.
   * @param selector A selector to query for. See [working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details.
   */
  $<K extends keyof HTMLElementTagNameMap>(selector: K): Promise<ElementHandleForTag<K> | null>;
  $(selector: string): Promise<ElementHandle<SVGElement | HTMLElement> | null>;

  /**
   * The method finds all elements matching the specified selector in the `ElementHandle`s subtree. See
   * [Working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details. If no elements match the selector,
   * returns empty array.
   * @param selector A selector to query for. See [working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details.
   */
  $$<K extends keyof HTMLElementTagNameMap>(selector: K): Promise<ElementHandleForTag<K>[]>;
  $$(selector: string): Promise<ElementHandle<SVGElement | HTMLElement>[]>;

  /**
   * Returns the return value of `pageFunction`
   * 
   * The method finds an element matching the specified selector in the `ElementHandle`s subtree and passes it as a first
   * argument to `pageFunction`. See [Working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details. If no
   * elements match the selector, the method throws an error.
   * 
   * If `pageFunction` returns a [Promise], then `frame.$eval` would wait for the promise to resolve and return its value.
   * 
   * Examples:
   * 
   * ```js
   * const tweetHandle = await page.$('.tweet');
   * expect(await tweetHandle.$eval('.like', node => node.innerText)).toBe('100');
   * expect(await tweetHandle.$eval('.retweets', node => node.innerText)).toBe('10');
   * ```
   * 
   * @param selector A selector to query for. See [working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details.
   * @param pageFunction Function to be evaluated in browser context
   * @param arg Optional argument to pass to `pageFunction`
   */
  $eval<K extends keyof HTMLElementTagNameMap, R, Arg>(selector: K, pageFunction: PageFunctionOn<HTMLElementTagNameMap[K], Arg, R>, arg: Arg): Promise<R>;
  $eval<R, Arg, E extends SVGElement | HTMLElement = SVGElement | HTMLElement>(selector: string, pageFunction: PageFunctionOn<E, Arg, R>, arg: Arg): Promise<R>;
  $eval<K extends keyof HTMLElementTagNameMap, R>(selector: K, pageFunction: PageFunctionOn<HTMLElementTagNameMap[K], void, R>, arg?: any): Promise<R>;
  $eval<R, E extends SVGElement | HTMLElement = SVGElement | HTMLElement>(selector: string, pageFunction: PageFunctionOn<E, void, R>, arg?: any): Promise<R>;

  /**
   * Returns the return value of `pageFunction`
   * 
   * The method finds all elements matching the specified selector in the `ElementHandle`'s subtree and passes an array of
   * matched elements as a first argument to `pageFunction`. See
   * [Working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details.
   * 
   * If `pageFunction` returns a [Promise], then `frame.$$eval` would wait for the promise to resolve and return its value.
   * 
   * Examples:
   * 
   * ```html
   * <div class="feed">
   *   <div class="tweet">Hello!</div>
   *   <div class="tweet">Hi!</div>
   * </div>
   * ```
   * 
   * ```js
   * const feedHandle = await page.$('.feed');
   * expect(await feedHandle.$$eval('.tweet', nodes => nodes.map(n => n.innerText))).toEqual(['Hello!', 'Hi!']);
   * ```
   * 
   * @param selector A selector to query for. See [working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details.
   * @param pageFunction Function to be evaluated in browser context
   * @param arg Optional argument to pass to `pageFunction`
   */
  $$eval<K extends keyof HTMLElementTagNameMap, R, Arg>(selector: K, pageFunction: PageFunctionOn<HTMLElementTagNameMap[K][], Arg, R>, arg: Arg): Promise<R>;
  $$eval<R, Arg, E extends SVGElement | HTMLElement = SVGElement | HTMLElement>(selector: string, pageFunction: PageFunctionOn<E[], Arg, R>, arg: Arg): Promise<R>;
  $$eval<K extends keyof HTMLElementTagNameMap, R>(selector: K, pageFunction: PageFunctionOn<HTMLElementTagNameMap[K][], void, R>, arg?: any): Promise<R>;
  $$eval<R, E extends SVGElement | HTMLElement = SVGElement | HTMLElement>(selector: string, pageFunction: PageFunctionOn<E[], void, R>, arg?: any): Promise<R>;

  /**
   * Returns element specified by selector when it satisfies `state` option. Returns `null` if waiting for `hidden` or
   * `detached`.
   * 
   * Wait for the `selector` relative to the element handle to satisfy `state` option (either appear/disappear from dom, or
   * become visible/hidden). If at the moment of calling the method `selector` already satisfies the condition, the method
   * will return immediately. If the selector doesn't satisfy the condition for the `timeout` milliseconds, the function will
   * throw.
   * 
   * ```js
   * await page.setContent(`<div><span></span></div>`);
   * const div = await page.$('div');
   * // Waiting for the 'span' selector relative to the div.
   * const span = await div.waitForSelector('span', { state: 'attached' });
   * ```
   * 
   * > NOTE: This method does not work across navigations, use
   * [page.waitForSelector(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagewaitforselector) instead.
   * @param selector A selector to query for. See [working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more details.
   * @param options 
   */
  waitForSelector<K extends keyof HTMLElementTagNameMap>(selector: K, options?: ElementHandleWaitForSelectorOptionsNotHidden): Promise<ElementHandleForTag<K>>;
  waitForSelector(selector: string, options?: ElementHandleWaitForSelectorOptionsNotHidden): Promise<ElementHandle<SVGElement | HTMLElement>>;
  waitForSelector<K extends keyof HTMLElementTagNameMap>(selector: K, options: ElementHandleWaitForSelectorOptions): Promise<ElementHandleForTag<K> | null>;
  waitForSelector(selector: string, options: ElementHandleWaitForSelectorOptions): Promise<null|ElementHandle<SVGElement | HTMLElement>>;
  /**
   * This method returns the bounding box of the element, or `null` if the element is not visible. The bounding box is
   * calculated relative to the main frame viewport - which is usually the same as the browser window.
   * 
   * Scrolling affects the returned bonding box, similarly to
   * [Element.getBoundingClientRect](https://developer.mozilla.org/en-US/docs/Web/API/Element/getBoundingClientRect). That
   * means `x` and/or `y` may be negative.
   * 
   * Elements from child frames return the bounding box relative to the main frame, unlike the
   * [Element.getBoundingClientRect](https://developer.mozilla.org/en-US/docs/Web/API/Element/getBoundingClientRect).
   * 
   * Assuming the page is static, it is safe to use bounding box coordinates to perform input. For example, the following
   * snippet should click the center of the element.
   * 
   * ```js
   * const box = await elementHandle.boundingBox();
   * await page.mouse.click(box.x + box.width / 2, box.y + box.height / 2);
   * ```
   * 
   */
  boundingBox(): Promise<null|{
    /**
     * the x coordinate of the element in pixels.
     */
    x: number;

    /**
     * the y coordinate of the element in pixels.
     */
    y: number;

    /**
     * the width of the element in pixels.
     */
    width: number;

    /**
     * the height of the element in pixels.
     */
    height: number;
  }>;

  /**
   * This method checks the element by performing the following steps:
   * 1. Ensure that element is a checkbox or a radio input. If not, this method rejects. If the element is already
   *    checked, this method returns immediately.
   * 1. Wait for [actionability](https://github.com/microsoft/playwright/blob/master/docs/actionability.md) checks on the element, unless `force` option is set.
   * 1. Scroll the element into view if needed.
   * 1. Use [page.mouse](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagemouse) to click in the center
   *    of the element.
   * 1. Wait for initiated navigations to either succeed or fail, unless `noWaitAfter` option is set.
   * 1. Ensure that the element is now checked. If not, this method rejects.
   * 
   * If the element is detached from the DOM at any moment during the action, this method rejects.
   * 
   * When all steps combined have not finished during the specified `timeout`, this method rejects with a [TimeoutError].
   * Passing zero timeout disables this.
   * @param options 
   */
  check(options?: {
    /**
     * Whether to bypass the [actionability](https://github.com/microsoft/playwright/blob/master/docs/actionability.md) checks. Defaults to `false`.
     */
    force?: boolean;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You can
     * opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as navigating to
     * inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed by
     * using the
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * This method clicks the element by performing the following steps:
   * 1. Wait for [actionability](https://github.com/microsoft/playwright/blob/master/docs/actionability.md) checks on the element, unless `force` option is set.
   * 1. Scroll the element into view if needed.
   * 1. Use [page.mouse](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagemouse) to click in the center
   *    of the element, or the specified `position`.
   * 1. Wait for initiated navigations to either succeed or fail, unless `noWaitAfter` option is set.
   * 
   * If the element is detached from the DOM at any moment during the action, this method rejects.
   * 
   * When all steps combined have not finished during the specified `timeout`, this method rejects with a [TimeoutError].
   * Passing zero timeout disables this.
   * @param options 
   */
  click(options?: {
    /**
     * Defaults to `left`.
     */
    button?: "left"|"right"|"middle";

    /**
     * defaults to 1. See [UIEvent.detail].
     */
    clickCount?: number;

    /**
     * Time to wait between `mousedown` and `mouseup` in milliseconds. Defaults to 0.
     */
    delay?: number;

    /**
     * Whether to bypass the [actionability](https://github.com/microsoft/playwright/blob/master/docs/actionability.md) checks. Defaults to `false`.
     */
    force?: boolean;

    /**
     * Modifier keys to press. Ensures that only these modifiers are pressed during the operation, and then restores current
     * modifiers back. If not specified, currently pressed modifiers are used.
     */
    modifiers?: Array<"Alt"|"Control"|"Meta"|"Shift">;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You can
     * opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as navigating to
     * inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * A point to use relative to the top-left corner of element padding box. If not specified, uses some visible point of the
     * element.
     */
    position?: {
      x: number;

      y: number;
    };

    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed by
     * using the
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * Returns the content frame for element handles referencing iframe nodes, or `null` otherwise
   */
  contentFrame(): Promise<null|Frame>;

  /**
   * This method double clicks the element by performing the following steps:
   * 1. Wait for [actionability](https://github.com/microsoft/playwright/blob/master/docs/actionability.md) checks on the element, unless `force` option is set.
   * 1. Scroll the element into view if needed.
   * 1. Use [page.mouse](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagemouse) to double click in the
   *    center of the element, or the specified `position`.
   * 1. Wait for initiated navigations to either succeed or fail, unless `noWaitAfter` option is set. Note that if the
   *    first click of the `dblclick()` triggers a navigation event, this method will reject.
   * 
   * If the element is detached from the DOM at any moment during the action, this method rejects.
   * 
   * When all steps combined have not finished during the specified `timeout`, this method rejects with a [TimeoutError].
   * Passing zero timeout disables this.
   * 
   * > NOTE: `elementHandle.dblclick()` dispatches two `click` events and a single `dblclick` event.
   * @param options 
   */
  dblclick(options?: {
    /**
     * Defaults to `left`.
     */
    button?: "left"|"right"|"middle";

    /**
     * Time to wait between `mousedown` and `mouseup` in milliseconds. Defaults to 0.
     */
    delay?: number;

    /**
     * Whether to bypass the [actionability](https://github.com/microsoft/playwright/blob/master/docs/actionability.md) checks. Defaults to `false`.
     */
    force?: boolean;

    /**
     * Modifier keys to press. Ensures that only these modifiers are pressed during the operation, and then restores current
     * modifiers back. If not specified, currently pressed modifiers are used.
     */
    modifiers?: Array<"Alt"|"Control"|"Meta"|"Shift">;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You can
     * opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as navigating to
     * inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * A point to use relative to the top-left corner of element padding box. If not specified, uses some visible point of the
     * element.
     */
    position?: {
      x: number;

      y: number;
    };

    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed by
     * using the
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * The snippet below dispatches the `click` event on the element. Regardless of the visibility state of the elment, `click`
   * is dispatched. This is equivalend to calling
   * [element.click()](https://developer.mozilla.org/en-US/docs/Web/API/HTMLElement/click).
   * 
   * ```js
   * await elementHandle.dispatchEvent('click');
   * ```
   * 
   * Under the hood, it creates an instance of an event based on the given `type`, initializes it with `eventInit` properties
   * and dispatches it on the element. Events are `composed`, `cancelable` and bubble by default.
   * 
   * Since `eventInit` is event-specific, please refer to the events documentation for the lists of initial properties:
   * - [DragEvent](https://developer.mozilla.org/en-US/docs/Web/API/DragEvent/DragEvent)
   * - [FocusEvent](https://developer.mozilla.org/en-US/docs/Web/API/FocusEvent/FocusEvent)
   * - [KeyboardEvent](https://developer.mozilla.org/en-US/docs/Web/API/KeyboardEvent/KeyboardEvent)
   * - [MouseEvent](https://developer.mozilla.org/en-US/docs/Web/API/MouseEvent/MouseEvent)
   * - [PointerEvent](https://developer.mozilla.org/en-US/docs/Web/API/PointerEvent/PointerEvent)
   * - [TouchEvent](https://developer.mozilla.org/en-US/docs/Web/API/TouchEvent/TouchEvent)
   * - [Event](https://developer.mozilla.org/en-US/docs/Web/API/Event/Event)
   * 
   * You can also specify `JSHandle` as the property value if you want live objects to be passed into the event:
   * 
   * ```js
   * // Note you can only create DataTransfer in Chromium and Firefox
   * const dataTransfer = await page.evaluateHandle(() => new DataTransfer());
   * await elementHandle.dispatchEvent('dragstart', { dataTransfer });
   * ```
   * 
   * @param type DOM event type: `"click"`, `"dragstart"`, etc.
   * @param eventInit Optional event-specific initialization properties.
   */
  dispatchEvent(type: string, eventInit?: EvaluationArgument): Promise<void>;

  /**
   * This method waits for [actionability](https://github.com/microsoft/playwright/blob/master/docs/actionability.md) checks, focuses the element, fills it and triggers an `input`
   * event after filling. If the element is not an `<input>`, `<textarea>` or `[contenteditable]` element, this method throws
   * an error. Note that you can pass an empty string to clear the input field.
   * @param value Value to set for the `<input>`, `<textarea>` or `[contenteditable]` element.
   * @param options 
   */
  fill(value: string, options?: {
    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You can
     * opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as navigating to
     * inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed by
     * using the
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * Calls [focus](https://developer.mozilla.org/en-US/docs/Web/API/HTMLElement/focus) on the element.
   */
  focus(): Promise<void>;

  /**
   * Returns element attribute value.
   * @param name Attribute name to get the value for.
   */
  getAttribute(name: string): Promise<null|string>;

  /**
   * This method hovers over the element by performing the following steps:
   * 1. Wait for [actionability](https://github.com/microsoft/playwright/blob/master/docs/actionability.md) checks on the element, unless `force` option is set.
   * 1. Scroll the element into view if needed.
   * 1. Use [page.mouse](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagemouse) to hover over the
   *    center of the element, or the specified `position`.
   * 1. Wait for initiated navigations to either succeed or fail, unless `noWaitAfter` option is set.
   * 
   * If the element is detached from the DOM at any moment during the action, this method rejects.
   * 
   * When all steps combined have not finished during the specified `timeout`, this method rejects with a [TimeoutError].
   * Passing zero timeout disables this.
   * @param options 
   */
  hover(options?: {
    /**
     * Whether to bypass the [actionability](https://github.com/microsoft/playwright/blob/master/docs/actionability.md) checks. Defaults to `false`.
     */
    force?: boolean;

    /**
     * Modifier keys to press. Ensures that only these modifiers are pressed during the operation, and then restores current
     * modifiers back. If not specified, currently pressed modifiers are used.
     */
    modifiers?: Array<"Alt"|"Control"|"Meta"|"Shift">;

    /**
     * A point to use relative to the top-left corner of element padding box. If not specified, uses some visible point of the
     * element.
     */
    position?: {
      x: number;

      y: number;
    };

    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed by
     * using the
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * Returns the `element.innerHTML`.
   */
  innerHTML(): Promise<string>;

  /**
   * Returns the `element.innerText`.
   */
  innerText(): Promise<string>;

  /**
   * Returns whether the element is checked. Throws if the element is not a checkbox or radio input.
   */
  isChecked(): Promise<boolean>;

  /**
   * Returns whether the element is disabled, the opposite of [enabled](https://github.com/microsoft/playwright/blob/master/docs/actionability.md#enabled).
   */
  isDisabled(): Promise<boolean>;

  /**
   * Returns whether the element is [editable](https://github.com/microsoft/playwright/blob/master/docs/actionability.md#editable).
   */
  isEditable(): Promise<boolean>;

  /**
   * Returns whether the element is [enabled](https://github.com/microsoft/playwright/blob/master/docs/actionability.md#enabled).
   */
  isEnabled(): Promise<boolean>;

  /**
   * Returns whether the element is hidden, the opposite of [visible](https://github.com/microsoft/playwright/blob/master/docs/actionability.md#visible).
   */
  isHidden(): Promise<boolean>;

  /**
   * Returns whether the element is [visible](https://github.com/microsoft/playwright/blob/master/docs/actionability.md#visible).
   */
  isVisible(): Promise<boolean>;

  /**
   * Returns the frame containing the given element.
   */
  ownerFrame(): Promise<null|Frame>;

  /**
   * Focuses the element, and then uses
   * [keyboard.down(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#keyboarddown) and
   * [keyboard.up(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#keyboardup).
   * 
   * `key` can specify the intended [keyboardEvent.key](https://developer.mozilla.org/en-US/docs/Web/API/KeyboardEvent/key)
   * value or a single character to generate the text for. A superset of the `key` values can be found
   * [here](https://developer.mozilla.org/en-US/docs/Web/API/KeyboardEvent/key/Key_Values). Examples of the keys are:
   * 
   * `F1` - `F12`, `Digit0`- `Digit9`, `KeyA`- `KeyZ`, `Backquote`, `Minus`, `Equal`, `Backslash`, `Backspace`, `Tab`,
   * `Delete`, `Escape`, `ArrowDown`, `End`, `Enter`, `Home`, `Insert`, `PageDown`, `PageUp`, `ArrowRight`, `ArrowUp`, etc.
   * 
   * Following modification shortcuts are also supported: `Shift`, `Control`, `Alt`, `Meta`, `ShiftLeft`.
   * 
   * Holding down `Shift` will type the text that corresponds to the `key` in the upper case.
   * 
   * If `key` is a single character, it is case-sensitive, so the values `a` and `A` will generate different respective
   * texts.
   * 
   * Shortcuts such as `key: "Control+o"` or `key: "Control+Shift+T"` are supported as well. When speficied with the
   * modifier, modifier is pressed and being held while the subsequent key is being pressed.
   * @param key Name of the key to press or a character to generate, such as `ArrowLeft` or `a`.
   * @param options 
   */
  press(key: string, options?: {
    /**
     * Time to wait between `keydown` and `keyup` in milliseconds. Defaults to 0.
     */
    delay?: number;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You can
     * opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as navigating to
     * inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed by
     * using the
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * Returns the buffer with the captured screenshot.
   * 
   * This method waits for the [actionability](https://github.com/microsoft/playwright/blob/master/docs/actionability.md) checks, then scrolls element into view before taking a
   * screenshot. If the element is detached from DOM, the method throws an error.
   * @param options 
   */
  screenshot(options?: {
    /**
     * Hides default white background and allows capturing screenshots with transparency. Not applicable to `jpeg` images.
     * Defaults to `false`.
     */
    omitBackground?: boolean;

    /**
     * The file path to save the image to. The screenshot type will be inferred from file extension. If `path` is a relative
     * path, then it is resolved relative to the current working directory. If no path is provided, the image won't be saved to
     * the disk.
     */
    path?: string;

    /**
     * The quality of the image, between 0-100. Not applicable to `png` images.
     */
    quality?: number;

    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed by
     * using the
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;

    /**
     * Specify screenshot type, defaults to `png`.
     */
    type?: "png"|"jpeg";
  }): Promise<Buffer>;

  /**
   * This method waits for [actionability](https://github.com/microsoft/playwright/blob/master/docs/actionability.md) checks, then tries to scroll element into view, unless it is
   * completely visible as defined by
   * [IntersectionObserver](https://developer.mozilla.org/en-US/docs/Web/API/Intersection_Observer_API)'s `ratio`.
   * 
   * Throws when `elementHandle` does not point to an element
   * [connected](https://developer.mozilla.org/en-US/docs/Web/API/Node/isConnected) to a Document or a ShadowRoot.
   * @param options 
   */
  scrollIntoViewIfNeeded(options?: {
    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed by
     * using the
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * Returns the array of option values that have been successfully selected.
   * 
   * Triggers a `change` and `input` event once all the provided options have been selected. If element is not a `<select>`
   * element, the method throws an error.
   * 
   * Will wait until all specified options are present in the `<select>` element.
   * 
   * ```js
   * // single selection matching the value
   * handle.selectOption('blue');
   * 
   * // single selection matching the label
   * handle.selectOption({ label: 'Blue' });
   * 
   * // multiple selection
   * handle.selectOption(['red', 'green', 'blue']);
   * ```
   * 
   * @param values Options to select. If the `<select>` has the `multiple` attribute, all matching options are selected, otherwise only the first option matching one of the passed options is selected. String values are equivalent to `{value:'string'}`. Option
   * is considered matching if all specified properties match.
   * @param options 
   */
  selectOption(values: null|string|ElementHandle|Array<string>|{
    /**
     * Matches by `option.value`. Optional.
     */
    value?: string;

    /**
     * Matches by `option.label`. Optional.
     */
    label?: string;

    /**
     * Matches by the index. Optional.
     */
    index?: number;
  }|Array<ElementHandle>|Array<{
    /**
     * Matches by `option.value`. Optional.
     */
    value?: string;

    /**
     * Matches by `option.label`. Optional.
     */
    label?: string;

    /**
     * Matches by the index. Optional.
     */
    index?: number;
  }>, options?: {
    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You can
     * opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as navigating to
     * inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed by
     * using the
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;
  }): Promise<Array<string>>;

  /**
   * This method waits for [actionability](https://github.com/microsoft/playwright/blob/master/docs/actionability.md) checks, then focuses the element and selects all its text
   * content.
   * @param options 
   */
  selectText(options?: {
    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed by
     * using the
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * This method expects `elementHandle` to point to an
   * [input element](https://developer.mozilla.org/en-US/docs/Web/HTML/Element/input).
   * 
   * Sets the value of the file input to these file paths or files. If some of the `filePaths` are relative paths, then they
   * are resolved relative to the the current working directory. For empty array, clears the selected files.
   * @param files 
   * @param options 
   */
  setInputFiles(files: string|Array<string>|{
    /**
     * [File] name
     */
    name: string;

    /**
     * [File] type
     */
    mimeType: string;

    /**
     * File content
     */
    buffer: Buffer;
  }|Array<{
    /**
     * [File] name
     */
    name: string;

    /**
     * [File] type
     */
    mimeType: string;

    /**
     * File content
     */
    buffer: Buffer;
  }>, options?: {
    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You can
     * opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as navigating to
     * inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed by
     * using the
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * This method taps the element by performing the following steps:
   * 1. Wait for [actionability](https://github.com/microsoft/playwright/blob/master/docs/actionability.md) checks on the element, unless `force` option is set.
   * 1. Scroll the element into view if needed.
   * 1. Use [page.touchscreen](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagetouchscreen) to tap the
   *    center of the element, or the specified `position`.
   * 1. Wait for initiated navigations to either succeed or fail, unless `noWaitAfter` option is set.
   * 
   * If the element is detached from the DOM at any moment during the action, this method rejects.
   * 
   * When all steps combined have not finished during the specified `timeout`, this method rejects with a [TimeoutError].
   * Passing zero timeout disables this.
   * 
   * > NOTE: `elementHandle.tap()` requires that the `hasTouch` option of the browser context be set to true.
   * @param options 
   */
  tap(options?: {
    /**
     * Whether to bypass the [actionability](https://github.com/microsoft/playwright/blob/master/docs/actionability.md) checks. Defaults to `false`.
     */
    force?: boolean;

    /**
     * Modifier keys to press. Ensures that only these modifiers are pressed during the operation, and then restores current
     * modifiers back. If not specified, currently pressed modifiers are used.
     */
    modifiers?: Array<"Alt"|"Control"|"Meta"|"Shift">;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You can
     * opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as navigating to
     * inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * A point to use relative to the top-left corner of element padding box. If not specified, uses some visible point of the
     * element.
     */
    position?: {
      x: number;

      y: number;
    };

    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed by
     * using the
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * Returns the `node.textContent`.
   */
  textContent(): Promise<null|string>;

  /**
   * Focuses the element, and then sends a `keydown`, `keypress`/`input`, and `keyup` event for each character in the text.
   * 
   * To press a special key, like `Control` or `ArrowDown`, use
   * [elementHandle.press(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#elementhandlepress).
   * 
   * ```js
   * await elementHandle.type('Hello'); // Types instantly
   * await elementHandle.type('World', {delay: 100}); // Types slower, like a user
   * ```
   * 
   * An example of typing into a text field and then submitting the form:
   * 
   * ```js
   * const elementHandle = await page.$('input');
   * await elementHandle.type('some text');
   * await elementHandle.press('Enter');
   * ```
   * 
   * @param text A text to type into a focused element.
   * @param options 
   */
  type(text: string, options?: {
    /**
     * Time to wait between key presses in milliseconds. Defaults to 0.
     */
    delay?: number;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You can
     * opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as navigating to
     * inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed by
     * using the
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * This method checks the element by performing the following steps:
   * 1. Ensure that element is a checkbox or a radio input. If not, this method rejects. If the element is already
   *    unchecked, this method returns immediately.
   * 1. Wait for [actionability](https://github.com/microsoft/playwright/blob/master/docs/actionability.md) checks on the element, unless `force` option is set.
   * 1. Scroll the element into view if needed.
   * 1. Use [page.mouse](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagemouse) to click in the center
   *    of the element.
   * 1. Wait for initiated navigations to either succeed or fail, unless `noWaitAfter` option is set.
   * 1. Ensure that the element is now unchecked. If not, this method rejects.
   * 
   * If the element is detached from the DOM at any moment during the action, this method rejects.
   * 
   * When all steps combined have not finished during the specified `timeout`, this method rejects with a [TimeoutError].
   * Passing zero timeout disables this.
   * @param options 
   */
  uncheck(options?: {
    /**
     * Whether to bypass the [actionability](https://github.com/microsoft/playwright/blob/master/docs/actionability.md) checks. Defaults to `false`.
     */
    force?: boolean;

    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You can
     * opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as navigating to
     * inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed by
     * using the
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * Returns when the element satisfies the `state`.
   * 
   * Depending on the `state` parameter, this method waits for one of the [actionability](https://github.com/microsoft/playwright/blob/master/docs/actionability.md) checks to pass.
   * This method throws when the element is detached while waiting, unless waiting for the `"hidden"` state.
   * - `"visible"` Wait until the element is [visible](https://github.com/microsoft/playwright/blob/master/docs/actionability.md#visible).
   * - `"hidden"` Wait until the element is [not visible](https://github.com/microsoft/playwright/blob/master/docs/actionability.md#visible) or
   *   [not attached](https://github.com/microsoft/playwright/blob/master/docs/actionability.md#attached). Note that waiting for hidden does not throw when the element detaches.
   * - `"stable"` Wait until the element is both [visible](https://github.com/microsoft/playwright/blob/master/docs/actionability.md#visible) and
   *   [stable](https://github.com/microsoft/playwright/blob/master/docs/actionability.md#stable).
   * - `"enabled"` Wait until the element is [enabled](https://github.com/microsoft/playwright/blob/master/docs/actionability.md#enabled).
   * - `"disabled"` Wait until the element is [not enabled](https://github.com/microsoft/playwright/blob/master/docs/actionability.md#enabled).
   * - `"editable"` Wait until the element is [editable](https://github.com/microsoft/playwright/blob/master/docs/actionability.md#editable).
   * 
   * If the element does not satisfy the condition for the `timeout` milliseconds, this method will throw.
   * @param state A state to wait for, see below for more details.
   * @param options 
   */
  waitForElementState(state: "visible"|"hidden"|"stable"|"enabled"|"disabled"|"editable", options?: {
    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed by
     * using the
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;
  }): Promise<void>;}

/**
 * BrowserType provides methods to launch a specific browser instance or connect to an existing one. The following is a
 * typical example of using Playwright to drive automation:
 * 
 * ```js
 * const { chromium } = require('playwright');  // Or 'firefox' or 'webkit'.
 * 
 * (async () => {
 *   const browser = await chromium.launch();
 *   const page = await browser.newPage();
 *   await page.goto('https://example.com');
 *   // other actions...
 *   await browser.close();
 * })();
 * ```
 * 
 */
export interface BrowserType<Browser> {

  /**
   * This methods attaches Playwright to an existing browser instance.
   * @param params 
   */
  connect(params: ConnectOptions): Promise<Browser>;

  /**
   * A path where Playwright expects to find a bundled browser executable.
   */
  executablePath(): string;

  /**
   * Returns the browser instance.
   * 
   * You can use `ignoreDefaultArgs` to filter out `--mute-audio` from default arguments:
   * 
   * ```js
   * const browser = await chromium.launch({  // Or 'firefox' or 'webkit'.
   *   ignoreDefaultArgs: ['--mute-audio']
   * });
   * ```
   * 
   * > **Chromium-only** Playwright can also be used to control the Chrome browser, but it works best with the version of
   * Chromium it is bundled with. There is no guarantee it will work with any other version. Use `executablePath` option with
   * extreme caution.
   * >
   * > If Google Chrome (rather than Chromium) is preferred, a
   * [Chrome Canary](https://www.google.com/chrome/browser/canary.html) or
   * [Dev Channel](https://www.chromium.org/getting-involved/dev-channel) build is suggested.
   * >
   * > In [browserType.launch(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsertypelaunch) above,
   * any mention of Chromium also applies to Chrome.
   * >
   * > See [`this article`](https://www.howtogeek.com/202825/what%E2%80%99s-the-difference-between-chromium-and-chrome/) for
   * a description of the differences between Chromium and Chrome.
   * [`This article`](https://chromium.googlesource.com/chromium/src/+/lkgr/docs/chromium_browser_vs_google_chrome.md)
   * describes some differences for Linux users.
   * @param options 
   */
  launch(options?: LaunchOptions): Promise<Browser>;

  /**
   * Returns the persistent browser context instance.
   * 
   * Launches browser that uses persistent storage located at `userDataDir` and returns the only context. Closing this
   * context will automatically close the browser.
   * @param userDataDir Path to a User Data Directory, which stores browser session data like cookies and local storage. More details for [Chromium](https://chromium.googlesource.com/chromium/src/+/master/docs/user_data_dir.md) and
   * [Firefox](https://developer.mozilla.org/en-US/docs/Mozilla/Command_Line_Options#User_Profile).
   * @param options 
   */
  launchPersistentContext(userDataDir: string, options?: {
    /**
     * Whether to automatically download all the attachments. Defaults to `false` where all the downloads are canceled.
     */
    acceptDownloads?: boolean;

    /**
     * Additional arguments to pass to the browser instance. The list of Chromium flags can be found
     * [here](http://peter.sh/experiments/chromium-command-line-switches/).
     */
    args?: Array<string>;

    /**
     * Toggles bypassing page's Content-Security-Policy.
     */
    bypassCSP?: boolean;

    /**
     * Enable Chromium sandboxing. Defaults to `true`.
     */
    chromiumSandbox?: boolean;

    /**
     * Emulates `'prefers-colors-scheme'` media feature, supported values are `'light'`, `'dark'`, `'no-preference'`. See
     * [page.emulateMedia(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageemulatemedia) for more
     * details. Defaults to '`light`'.
     */
    colorScheme?: "light"|"dark"|"no-preference";

    /**
     * Specify device scale factor (can be thought of as dpr). Defaults to `1`.
     */
    deviceScaleFactor?: number;

    /**
     * **Chromium-only** Whether to auto-open a Developer Tools panel for each tab. If this option is `true`, the `headless`
     * option will be set `false`.
     */
    devtools?: boolean;

    /**
     * If specified, accepted downloads are downloaded into this directory. Otherwise, temporary directory is created and is
     * deleted when browser is closed.
     */
    downloadsPath?: string;

    /**
     * Specify environment variables that will be visible to the browser. Defaults to `process.env`.
     */
    env?: { [key: string]: string|number|boolean; };

    /**
     * Path to a browser executable to run instead of the bundled one. If `executablePath` is a relative path, then it is
     * resolved relative to the current working directory. **BEWARE**: Playwright is only guaranteed to work with the bundled
     * Chromium, Firefox or WebKit, use at your own risk.
     */
    executablePath?: string;

    /**
     * An object containing additional HTTP headers to be sent with every request. All header values must be strings.
     */
    extraHTTPHeaders?: { [key: string]: string; };

    geolocation?: {
      /**
       * Latitude between -90 and 90.
       */
      latitude: number;

      /**
       * Longitude between -180 and 180.
       */
      longitude: number;

      /**
       * Non-negative accuracy value. Defaults to `0`.
       */
      accuracy?: number;
    };

    /**
     * Close the browser process on SIGHUP. Defaults to `true`.
     */
    handleSIGHUP?: boolean;

    /**
     * Close the browser process on Ctrl-C. Defaults to `true`.
     */
    handleSIGINT?: boolean;

    /**
     * Close the browser process on SIGTERM. Defaults to `true`.
     */
    handleSIGTERM?: boolean;

    /**
     * Specifies if viewport supports touch events. Defaults to false.
     */
    hasTouch?: boolean;

    /**
     * Whether to run browser in headless mode. More details for
     * [Chromium](https://developers.google.com/web/updates/2017/04/headless-chrome) and
     * [Firefox](https://developer.mozilla.org/en-US/docs/Mozilla/Firefox/Headless_mode). Defaults to `true` unless the
     * `devtools` option is `true`.
     */
    headless?: boolean;

    /**
     * Credentials for [HTTP authentication](https://developer.mozilla.org/en-US/docs/Web/HTTP/Authentication).
     */
    httpCredentials?: {
      username: string;

      password: string;
    };

    /**
     * If `true`, then do not use any of the default arguments. If an array is given, then filter out the given default
     * arguments. Dangerous option; use with care. Defaults to `false`.
     */
    ignoreDefaultArgs?: boolean|Array<string>;

    /**
     * Whether to ignore HTTPS errors during navigation. Defaults to `false`.
     */
    ignoreHTTPSErrors?: boolean;

    /**
     * Whether the `meta viewport` tag is taken into account and touch events are enabled. Defaults to `false`. Not supported
     * in Firefox.
     */
    isMobile?: boolean;

    /**
     * Whether or not to enable JavaScript in the context. Defaults to `true`.
     */
    javaScriptEnabled?: boolean;

    /**
     * Specify user locale, for example `en-GB`, `de-DE`, etc. Locale will affect `navigator.language` value, `Accept-Language`
     * request header value as well as number and date formatting rules.
     */
    locale?: string;

    /**
     * Logger sink for Playwright logging.
     */
    logger?: Logger;

    /**
     * Whether to emulate network being offline. Defaults to `false`.
     */
    offline?: boolean;

    /**
     * A list of permissions to grant to all pages in this context. See
     * [browserContext.grantPermissions(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextgrantpermissions)
     * for more details.
     */
    permissions?: Array<string>;

    /**
     * Network proxy settings.
     */
    proxy?: {
      /**
       * Proxy to be used for all requests. HTTP and SOCKS proxies are supported, for example `http://myproxy.com:3128` or
       * `socks5://myproxy.com:3128`. Short form `myproxy.com:3128` is considered an HTTP proxy.
       */
      server: string;

      /**
       * Optional coma-separated domains to bypass proxy, for example `".com, chromium.org, .domain.com"`.
       */
      bypass?: string;

      /**
       * Optional username to use if HTTP proxy requires authentication.
       */
      username?: string;

      /**
       * Optional password to use if HTTP proxy requires authentication.
       */
      password?: string;
    };

    /**
     * Enables [HAR](http://www.softwareishard.com/blog/har-12-spec) recording for all pages into `recordHar.path` file. If not
     * specified, the HAR is not recorded. Make sure to await
     * [browserContext.close(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextclose) for the
     * HAR to be saved.
     */
    recordHar?: {
      /**
       * Optional setting to control whether to omit request content from the HAR. Defaults to `false`.
       */
      omitContent?: boolean;

      /**
       * Path on the filesystem to write the HAR file to.
       */
      path: string;
    };

    /**
     * Enables video recording for all pages into `recordVideo.dir` directory. If not specified videos are not recorded. Make
     * sure to await
     * [browserContext.close(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextclose) for
     * videos to be saved.
     */
    recordVideo?: {
      /**
       * Path to the directory to put videos into.
       */
      dir: string;

      /**
       * Optional dimensions of the recorded videos. If not specified the size will be equal to `viewport`. If `viewport` is not
       * configured explicitly the video size defaults to 1280x720. Actual picture of each page will be scaled down if necessary
       * to fit the specified size.
       */
      size?: {
        /**
         * Video frame width.
         */
        width: number;

        /**
         * Video frame height.
         */
        height: number;
      };
    };

    /**
     * Slows down Playwright operations by the specified amount of milliseconds. Useful so that you can see what is going on.
     * Defaults to 0.
     */
    slowMo?: number;

    /**
     * Maximum time in milliseconds to wait for the browser instance to start. Defaults to `30000` (30 seconds). Pass `0` to
     * disable timeout.
     */
    timeout?: number;

    /**
     * Changes the timezone of the context. See
     * [ICU's metaZones.txt](https://cs.chromium.org/chromium/src/third_party/icu/source/data/misc/metaZones.txt?rcl=faee8bc70570192d82d2978a71e2a615788597d1)
     * for a list of supported timezone IDs.
     */
    timezoneId?: string;

    /**
     * Specific user agent to use in this context.
     */
    userAgent?: string;

    /**
     * **DEPRECATED** Use `recordVideo` instead.
     * @deprecated
     */
    videoSize?: {
      /**
       * Video frame width.
       */
      width: number;

      /**
       * Video frame height.
       */
      height: number;
    };

    /**
     * **DEPRECATED** Use `recordVideo` instead.
     * @deprecated
     */
    videosPath?: string;

    /**
     * Sets a consistent viewport for each page. Defaults to an 1280x720 viewport. `null` disables the default viewport.
     */
    viewport?: null|{
      /**
       * page width in pixels.
       */
      width: number;

      /**
       * page height in pixels.
       */
      height: number;
    };
  }): Promise<BrowserContext>;

  /**
   * Returns the browser app instance.
   * 
   * Launches browser server that client can connect to. An example of launching a browser executable and connecting to it
   * later:
   * 
   * ```js
   * const { chromium } = require('playwright');  // Or 'webkit' or 'firefox'.
   * 
   * (async () => {
   *   const browserServer = await chromium.launchServer();
   *   const wsEndpoint = browserServer.wsEndpoint();
   *   // Use web socket endpoint later to establish a connection.
   *   const browser = await chromium.connect({ wsEndpoint });
   *   // Close browser instance.
   *   await browserServer.close();
   * })();
   * ```
   * 
   * @param options 
   */
  launchServer(options?: {
    /**
     * Additional arguments to pass to the browser instance. The list of Chromium flags can be found
     * [here](http://peter.sh/experiments/chromium-command-line-switches/).
     */
    args?: Array<string>;

    /**
     * Enable Chromium sandboxing. Defaults to `true`.
     */
    chromiumSandbox?: boolean;

    /**
     * **Chromium-only** Whether to auto-open a Developer Tools panel for each tab. If this option is `true`, the `headless`
     * option will be set `false`.
     */
    devtools?: boolean;

    /**
     * If specified, accepted downloads are downloaded into this directory. Otherwise, temporary directory is created and is
     * deleted when browser is closed.
     */
    downloadsPath?: string;

    /**
     * Specify environment variables that will be visible to the browser. Defaults to `process.env`.
     */
    env?: { [key: string]: string|number|boolean; };

    /**
     * Path to a browser executable to run instead of the bundled one. If `executablePath` is a relative path, then it is
     * resolved relative to the current working directory. **BEWARE**: Playwright is only guaranteed to work with the bundled
     * Chromium, Firefox or WebKit, use at your own risk.
     */
    executablePath?: string;

    /**
     * Firefox user preferences. Learn more about the Firefox user preferences at
     * [`about:config`](https://support.mozilla.org/en-US/kb/about-config-editor-firefox).
     */
    firefoxUserPrefs?: { [key: string]: string|number|boolean; };

    /**
     * Close the browser process on SIGHUP. Defaults to `true`.
     */
    handleSIGHUP?: boolean;

    /**
     * Close the browser process on Ctrl-C. Defaults to `true`.
     */
    handleSIGINT?: boolean;

    /**
     * Close the browser process on SIGTERM. Defaults to `true`.
     */
    handleSIGTERM?: boolean;

    /**
     * Whether to run browser in headless mode. More details for
     * [Chromium](https://developers.google.com/web/updates/2017/04/headless-chrome) and
     * [Firefox](https://developer.mozilla.org/en-US/docs/Mozilla/Firefox/Headless_mode). Defaults to `true` unless the
     * `devtools` option is `true`.
     */
    headless?: boolean;

    /**
     * If `true`, then do not use any of the default arguments. If an array is given, then filter out the given default
     * arguments. Dangerous option; use with care. Defaults to `false`.
     */
    ignoreDefaultArgs?: boolean|Array<string>;

    /**
     * Logger sink for Playwright logging.
     */
    logger?: Logger;

    /**
     * Port to use for the web socket. Defaults to 0 that picks any available port.
     */
    port?: number;

    /**
     * Network proxy settings.
     */
    proxy?: {
      /**
       * Proxy to be used for all requests. HTTP and SOCKS proxies are supported, for example `http://myproxy.com:3128` or
       * `socks5://myproxy.com:3128`. Short form `myproxy.com:3128` is considered an HTTP proxy.
       */
      server: string;

      /**
       * Optional coma-separated domains to bypass proxy, for example `".com, chromium.org, .domain.com"`.
       */
      bypass?: string;

      /**
       * Optional username to use if HTTP proxy requires authentication.
       */
      username?: string;

      /**
       * Optional password to use if HTTP proxy requires authentication.
       */
      password?: string;
    };

    /**
     * Maximum time in milliseconds to wait for the browser instance to start. Defaults to `30000` (30 seconds). Pass `0` to
     * disable timeout.
     */
    timeout?: number;
  }): Promise<BrowserServer>;

  /**
   * Returns browser name. For example: `'chromium'`, `'webkit'` or `'firefox'`.
   */
  name(): string;}

/**
 * - extends: [Browser]
 * 
 * Chromium-specific features including Tracing, service worker support, etc. You can use
 * [chromiumBrowser.startTracing(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#chromiumbrowserstarttracing)
 * and
 * [chromiumBrowser.stopTracing(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#chromiumbrowserstoptracing)
 * to create a trace file which can be opened in Chrome DevTools or
 * [timeline viewer](https://chromedevtools.github.io/timeline-viewer/).
 * 
 * ```js
 * await browser.startTracing(page, {path: 'trace.json'});
 * await page.goto('https://www.google.com');
 * await browser.stopTracing();
 * ```
 * 
 * [ChromiumBrowser] can also be used for testing Chrome Extensions.
 * 
 * > NOTE: Extensions in Chrome / Chromium currently only work in non-headless mode.
 * 
 * The following is code for getting a handle to the
 * [background page](https://developer.chrome.com/extensions/background_pages) of an extension whose source is located in
 * `./my-extension`:
 * 
 * ```js
 * const { chromium } = require('playwright');
 * 
 * (async () => {
 *   const pathToExtension = require('path').join(__dirname, 'my-extension');
 *   const userDataDir = '/tmp/test-user-data-dir';
 *   const browserContext = await chromium.launchPersistentContext(userDataDir,{
 *     headless: false,
 *     args: [
 *       `--disable-extensions-except=${pathToExtension}`,
 *       `--load-extension=${pathToExtension}`
 *     ]
 *   });
 *   const backgroundPage = browserContext.backgroundPages()[0];
 *   // Test the background page as you would any other page.
 *   await browserContext.close();
 * })();
 * ```
 * 
 */
export interface ChromiumBrowser extends Browser {
  /**
   * Returns an array of all open browser contexts. In a newly created browser, this will return zero browser contexts.
   * 
   * ```js
   * const browser = await pw.webkit.launch();
   * console.log(browser.contexts().length); // prints `0`
   * 
   * const context = await browser.newContext();
   * console.log(browser.contexts().length); // prints `1`
   * ```
   * 
   */
  contexts(): Array<ChromiumBrowserContext>;
  /**
   * Creates a new browser context. It won't share cookies/cache with other browser contexts.
   * 
   * ```js
   * (async () => {
   *   const browser = await playwright.firefox.launch();  // Or 'chromium' or 'webkit'.
   *   // Create a new incognito browser context.
   *   const context = await browser.newContext();
   *   // Create a new page in a pristine context.
   *   const page = await context.newPage();
   *   await page.goto('https://example.com');
   * })();
   * ```
   * 
   * @param options 
   */
  newContext(options?: BrowserContextOptions): Promise<ChromiumBrowserContext>;
  /**
   * Returns the newly created browser session.
   */
  newBrowserCDPSession(): Promise<CDPSession>;

  /**
   * Only one trace can be active at a time per browser.
   * @param page Optional, if specified, tracing includes screenshots of the given page.
   * @param options 
   */
  startTracing(page?: Page, options?: {
    /**
     * specify custom categories to use instead of default.
     */
    categories?: Array<string>;

    /**
     * A path to write the trace file to.
     */
    path?: string;

    /**
     * captures screenshots in the trace.
     */
    screenshots?: boolean;
  }): Promise<void>;

  /**
   * Returns the buffer with trace data.
   */
  stopTracing(): Promise<Buffer>;}

/**
 * - extends: [EventEmitter]
 * 
 * The `CDPSession` instances are used to talk raw Chrome Devtools Protocol:
 * - protocol methods can be called with `session.send` method.
 * - protocol events can be subscribed to with `session.on` method.
 * 
 * Useful links:
 * - Documentation on DevTools Protocol can be found here:
 *   [DevTools Protocol Viewer](https://chromedevtools.github.io/devtools-protocol/).
 * - Getting Started with DevTools Protocol:
 *   https://github.com/aslushnikov/getting-started-with-cdp/blob/master/README.md
 * 
 * ```js
 * const client = await page.context().newCDPSession(page);
 * await client.send('Animation.enable');
 * client.on('Animation.animationCreated', () => console.log('Animation created!'));
 * const response = await client.send('Animation.getPlaybackRate');
 * console.log('playback rate is ' + response.playbackRate);
 * await client.send('Animation.setPlaybackRate', {
 *   playbackRate: response.playbackRate / 2
 * });
 * ```
 * 
 */
export interface CDPSession {
  on: <T extends keyof Protocol.Events | symbol>(event: T, listener: (payload: T extends symbol ? any : Protocol.Events[T extends keyof Protocol.Events ? T : never]) => void) => this;
  addListener: <T extends keyof Protocol.Events | symbol>(event: T, listener: (payload: T extends symbol ? any : Protocol.Events[T extends keyof Protocol.Events ? T : never]) => void) => this;
  off: <T extends keyof Protocol.Events | symbol>(event: T, listener: (payload: T extends symbol ? any : Protocol.Events[T extends keyof Protocol.Events ? T : never]) => void) => this;
  removeListener: <T extends keyof Protocol.Events | symbol>(event: T, listener: (payload: T extends symbol ? any : Protocol.Events[T extends keyof Protocol.Events ? T : never]) => void) => this;
  once: <T extends keyof Protocol.Events | symbol>(event: T, listener: (payload: T extends symbol ? any : Protocol.Events[T extends keyof Protocol.Events ? T : never]) => void) => this;
  /**
   * @param method protocol method name
   * @param params Optional method parameters
   */
  send<T extends keyof Protocol.CommandParameters>(
    method: T,
    params?: Protocol.CommandParameters[T]
  ): Promise<Protocol.CommandReturnValues[T]>;
  /**
   * Detaches the CDPSession from the target. Once detached, the CDPSession object won't emit any events and can't be used to
   * send messages.
   */
  detach(): Promise<void>;}

type DeviceDescriptor = {
  viewport: ViewportSize;
  userAgent: string;
  deviceScaleFactor: number;
  isMobile: boolean;
  hasTouch: boolean;
  defaultBrowserType: 'chromium' | 'firefox' | 'webkit';
};

export namespace errors {

/**
 * - extends: [Error]
 * 
 * TimeoutError is emitted whenever certain operations are terminated due to timeout, e.g.
 * [page.waitForSelector(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagewaitforselector) or
 * [browserType.launch(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsertypelaunch).
 */
class TimeoutError extends Error {}

}

/**
 * The Accessibility class provides methods for inspecting Chromium's accessibility tree. The accessibility tree is used by
 * assistive technology such as [screen readers](https://en.wikipedia.org/wiki/Screen_reader) or
 * [switches](https://en.wikipedia.org/wiki/Switch_access).
 * 
 * Accessibility is a very platform-specific thing. On different platforms, there are different screen readers that might
 * have wildly different output.
 * 
 * Rendering engines of Chromium, Firefox and Webkit have a concept of "accessibility tree", which is then translated into
 * different platform-specific APIs. Accessibility namespace gives access to this Accessibility Tree.
 * 
 * Most of the accessibility tree gets filtered out when converting from internal browser AX Tree to Platform-specific
 * AX-Tree or by assistive technologies themselves. By default, Playwright tries to approximate this filtering, exposing
 * only the "interesting" nodes of the tree.
 */
export interface Accessibility {
  /**
   * Captures the current state of the accessibility tree. The returned object represents the root accessible node of the
   * page.
   * 
   * > NOTE: The Chromium accessibility tree contains nodes that go unused on most platforms and by most screen readers.
   * Playwright will discard them as well for an easier to process tree, unless `interestingOnly` is set to `false`.
   * 
   * An example of dumping the entire accessibility tree:
   * 
   * ```js
   * const snapshot = await page.accessibility.snapshot();
   * console.log(snapshot);
   * ```
   * 
   * An example of logging the focused node's name:
   * 
   * ```js
   * const snapshot = await page.accessibility.snapshot();
   * const node = findFocusedNode(snapshot);
   * console.log(node && node.name);
   * 
   * function findFocusedNode(node) {
   *   if (node.focused)
   *     return node;
   *   for (const child of node.children || []) {
   *     const foundNode = findFocusedNode(child);
   *     return foundNode;
   *   }
   *   return null;
   * }
   * ```
   * 
   * @param options 
   */
  snapshot(options?: {
    /**
     * Prune uninteresting nodes from the tree. Defaults to `true`.
     */
    interestingOnly?: boolean;

    /**
     * The root DOM element for the snapshot. Defaults to the whole page.
     */
    root?: ElementHandle;
  }): Promise<null|AccessibilityNode>;
}

type AccessibilityNode = {
  role: string;
  name: string;
  value?: string|number;
  description?: string;
  keyshortcuts?: string;
  roledescription?: string;
  valuetext?: string;
  disabled?: boolean;
  expanded?: boolean;
  focused?: boolean;
  modal?: boolean;
  multiline?: boolean;
  multiselectable?: boolean;
  readonly?: boolean;
  required?: boolean;
  selected?: boolean;
  checked?: boolean|"mixed";
  pressed?: boolean|"mixed";
  level?: number;
  valuemin?: number;
  valuemax?: number;
  autocomplete?: string;
  haspopup?: string;
  invalid?: string;
  orientation?: string;
  children?: AccessibilityNode[];
}

export const selectors: Selectors;
export const devices: Devices & DeviceDescriptor[];

// This is required to not export everything by default. See https://github.com/Microsoft/TypeScript/issues/19545#issuecomment-340490459
export {};


/**
 * - extends: [EventEmitter]
 * 
 * A Browser is created via
 * [browserType.launch(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsertypelaunch). An example
 * of using a [Browser] to create a [Page]:
 * 
 * ```js
 * const { firefox } = require('playwright');  // Or 'chromium' or 'webkit'.
 * 
 * (async () => {
 *   const browser = await firefox.launch();
 *   const page = await browser.newPage();
 *   await page.goto('https://example.com');
 *   await browser.close();
 * })();
 * ```
 * 
 */
export interface Browser extends EventEmitter {
  /**
   * Emitted when Browser gets disconnected from the browser application. This might happen because of one of the following:
   * - Browser application is closed or crashed.
   * - The [browser.close(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browserclose) method was
   *   called.
   */
  on(event: 'disconnected', listener: () => void): this;

  /**
   * Emitted when Browser gets disconnected from the browser application. This might happen because of one of the following:
   * - Browser application is closed or crashed.
   * - The [browser.close(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browserclose) method was
   *   called.
   */
  once(event: 'disconnected', listener: () => void): this;

  /**
   * Emitted when Browser gets disconnected from the browser application. This might happen because of one of the following:
   * - Browser application is closed or crashed.
   * - The [browser.close(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browserclose) method was
   *   called.
   */
  addListener(event: 'disconnected', listener: () => void): this;

  /**
   * Emitted when Browser gets disconnected from the browser application. This might happen because of one of the following:
   * - Browser application is closed or crashed.
   * - The [browser.close(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browserclose) method was
   *   called.
   */
  removeListener(event: 'disconnected', listener: () => void): this;

  /**
   * Emitted when Browser gets disconnected from the browser application. This might happen because of one of the following:
   * - Browser application is closed or crashed.
   * - The [browser.close(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browserclose) method was
   *   called.
   */
  off(event: 'disconnected', listener: () => void): this;

  /**
   * In case this browser is obtained using
   * [browserType.launch(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsertypelaunch), closes the
   * browser and all of its pages (if any were opened).
   * 
   * In case this browser is connected to, clears all created contexts belonging to this browser and disconnects from the
   * browser server.
   * 
   * The [Browser] object itself is considered to be disposed and cannot be used anymore.
   */
  close(): Promise<void>;

  /**
   * Returns an array of all open browser contexts. In a newly created browser, this will return zero browser contexts.
   * 
   * ```js
   * const browser = await pw.webkit.launch();
   * console.log(browser.contexts().length); // prints `0`
   * 
   * const context = await browser.newContext();
   * console.log(browser.contexts().length); // prints `1`
   * ```
   * 
   */
  contexts(): Array<BrowserContext>;

  /**
   * Indicates that the browser is connected.
   */
  isConnected(): boolean;

  /**
   * Creates a new browser context. It won't share cookies/cache with other browser contexts.
   * 
   * ```js
   * (async () => {
   *   const browser = await playwright.firefox.launch();  // Or 'chromium' or 'webkit'.
   *   // Create a new incognito browser context.
   *   const context = await browser.newContext();
   *   // Create a new page in a pristine context.
   *   const page = await context.newPage();
   *   await page.goto('https://example.com');
   * })();
   * ```
   * 
   * @param options 
   */
  newContext(options?: BrowserContextOptions): Promise<BrowserContext>;

  /**
   * Creates a new page in a new browser context. Closing this page will close the context as well.
   * 
   * This is a convenience API that should only be used for the single-page scenarios and short snippets. Production code and
   * testing frameworks should explicitly create
   * [browser.newContext(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsernewcontext) followed by
   * the [browserContext.newPage(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextnewpage)
   * to control their exact life times.
   * @param options 
   */
  newPage(options?: {
    /**
     * Whether to automatically download all the attachments. Defaults to `false` where all the downloads are canceled.
     */
    acceptDownloads?: boolean;

    /**
     * Toggles bypassing page's Content-Security-Policy.
     */
    bypassCSP?: boolean;

    /**
     * Emulates `'prefers-colors-scheme'` media feature, supported values are `'light'`, `'dark'`, `'no-preference'`. See
     * [page.emulateMedia(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageemulatemedia) for more
     * details. Defaults to '`light`'.
     */
    colorScheme?: "light"|"dark"|"no-preference";

    /**
     * Specify device scale factor (can be thought of as dpr). Defaults to `1`.
     */
    deviceScaleFactor?: number;

    /**
     * An object containing additional HTTP headers to be sent with every request. All header values must be strings.
     */
    extraHTTPHeaders?: { [key: string]: string; };

    geolocation?: {
      /**
       * Latitude between -90 and 90.
       */
      latitude: number;

      /**
       * Longitude between -180 and 180.
       */
      longitude: number;

      /**
       * Non-negative accuracy value. Defaults to `0`.
       */
      accuracy?: number;
    };

    /**
     * Specifies if viewport supports touch events. Defaults to false.
     */
    hasTouch?: boolean;

    /**
     * Credentials for [HTTP authentication](https://developer.mozilla.org/en-US/docs/Web/HTTP/Authentication).
     */
    httpCredentials?: {
      username: string;

      password: string;
    };

    /**
     * Whether to ignore HTTPS errors during navigation. Defaults to `false`.
     */
    ignoreHTTPSErrors?: boolean;

    /**
     * Whether the `meta viewport` tag is taken into account and touch events are enabled. Defaults to `false`. Not supported
     * in Firefox.
     */
    isMobile?: boolean;

    /**
     * Whether or not to enable JavaScript in the context. Defaults to `true`.
     */
    javaScriptEnabled?: boolean;

    /**
     * Specify user locale, for example `en-GB`, `de-DE`, etc. Locale will affect `navigator.language` value, `Accept-Language`
     * request header value as well as number and date formatting rules.
     */
    locale?: string;

    /**
     * Logger sink for Playwright logging.
     */
    logger?: Logger;

    /**
     * Whether to emulate network being offline. Defaults to `false`.
     */
    offline?: boolean;

    /**
     * A list of permissions to grant to all pages in this context. See
     * [browserContext.grantPermissions(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextgrantpermissions)
     * for more details.
     */
    permissions?: Array<string>;

    /**
     * Network proxy settings to use with this context. Note that browser needs to be launched with the global proxy for this
     * option to work. If all contexts override the proxy, global proxy will be never used and can be any string, for example
     * `launch({ proxy: { server: 'per-context' } })`.
     */
    proxy?: {
      /**
       * Proxy to be used for all requests. HTTP and SOCKS proxies are supported, for example `http://myproxy.com:3128` or
       * `socks5://myproxy.com:3128`. Short form `myproxy.com:3128` is considered an HTTP proxy.
       */
      server: string;

      /**
       * Optional coma-separated domains to bypass proxy, for example `".com, chromium.org, .domain.com"`.
       */
      bypass?: string;

      /**
       * Optional username to use if HTTP proxy requires authentication.
       */
      username?: string;

      /**
       * Optional password to use if HTTP proxy requires authentication.
       */
      password?: string;
    };

    /**
     * Enables [HAR](http://www.softwareishard.com/blog/har-12-spec) recording for all pages into `recordHar.path` file. If not
     * specified, the HAR is not recorded. Make sure to await
     * [browserContext.close(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextclose) for the
     * HAR to be saved.
     */
    recordHar?: {
      /**
       * Optional setting to control whether to omit request content from the HAR. Defaults to `false`.
       */
      omitContent?: boolean;

      /**
       * Path on the filesystem to write the HAR file to.
       */
      path: string;
    };

    /**
     * Enables video recording for all pages into `recordVideo.dir` directory. If not specified videos are not recorded. Make
     * sure to await
     * [browserContext.close(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextclose) for
     * videos to be saved.
     */
    recordVideo?: {
      /**
       * Path to the directory to put videos into.
       */
      dir: string;

      /**
       * Optional dimensions of the recorded videos. If not specified the size will be equal to `viewport`. If `viewport` is not
       * configured explicitly the video size defaults to 1280x720. Actual picture of each page will be scaled down if necessary
       * to fit the specified size.
       */
      size?: {
        /**
         * Video frame width.
         */
        width: number;

        /**
         * Video frame height.
         */
        height: number;
      };
    };

    /**
     * Populates context with given storage state. This method can be used to initialize context with logged-in information
     * obtained via
     * [browserContext.storageState(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextstoragestate).
     * Either a path to the file with saved storage, or an object with the following fields:
     */
    storageState?: string|{
      /**
       * Optional cookies to set for context
       */
      cookies?: Array<{
        name: string;

        value: string;

        /**
         * Optional either url or domain / path are required
         */
        url?: string;

        /**
         * Optional either url or domain / path are required
         */
        domain?: string;

        /**
         * Optional either url or domain / path are required
         */
        path?: string;

        /**
         * Optional Unix time in seconds.
         */
        expires?: number;

        /**
         * Optional httpOnly flag
         */
        httpOnly?: boolean;

        /**
         * Optional secure flag
         */
        secure?: boolean;

        /**
         * Optional sameSite flag
         */
        sameSite?: "Strict"|"Lax"|"None";
      }>;

      /**
       * Optional localStorage to set for context
       */
      origins?: Array<{
        origin: string;

        localStorage: Array<{
          name: string;

          value: string;
        }>;
      }>;
    };

    /**
     * Changes the timezone of the context. See
     * [ICU's metaZones.txt](https://cs.chromium.org/chromium/src/third_party/icu/source/data/misc/metaZones.txt?rcl=faee8bc70570192d82d2978a71e2a615788597d1)
     * for a list of supported timezone IDs.
     */
    timezoneId?: string;

    /**
     * Specific user agent to use in this context.
     */
    userAgent?: string;

    /**
     * **DEPRECATED** Use `recordVideo` instead.
     * @deprecated
     */
    videoSize?: {
      /**
       * Video frame width.
       */
      width: number;

      /**
       * Video frame height.
       */
      height: number;
    };

    /**
     * **DEPRECATED** Use `recordVideo` instead.
     * @deprecated
     */
    videosPath?: string;

    /**
     * Sets a consistent viewport for each page. Defaults to an 1280x720 viewport. `null` disables the default viewport.
     */
    viewport?: null|{
      /**
       * page width in pixels.
       */
      width: number;

      /**
       * page height in pixels.
       */
      height: number;
    };
  }): Promise<Page>;

  /**
   * Returns the browser version.
   */
  version(): string;
}

export interface BrowserServer {
  /**
   * Emitted when the browser server closes.
   */
  on(event: 'close', listener: () => void): this;

  /**
   * Emitted when the browser server closes.
   */
  once(event: 'close', listener: () => void): this;

  /**
   * Emitted when the browser server closes.
   */
  addListener(event: 'close', listener: () => void): this;

  /**
   * Emitted when the browser server closes.
   */
  removeListener(event: 'close', listener: () => void): this;

  /**
   * Emitted when the browser server closes.
   */
  off(event: 'close', listener: () => void): this;

  /**
   * Closes the browser gracefully and makes sure the process is terminated.
   */
  close(): Promise<void>;

  /**
   * Kills the browser process and waits for the process to exit.
   */
  kill(): Promise<void>;

  /**
   * Spawned browser application process.
   */
  process(): ChildProcess;

  /**
   * Browser websocket url.
   * 
   * Browser websocket endpoint which can be used as an argument to
   * [browserType.connect(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsertypeconnect) to
   * establish connection to the browser.
   */
  wsEndpoint(): string;
}

/**
 * - extends: [BrowserContext]
 * 
 * Chromium-specific features including background pages, service worker support, etc.
 * 
 * ```js
 * const backgroundPage = await context.waitForEvent('backgroundpage');
 * ```
 * 
 */
export interface ChromiumBrowserContext extends BrowserContext {
  /**
   * Emitted when new background page is created in the context.
   * 
   * > NOTE: Only works with persistent context.
   */
  on(event: 'backgroundpage', listener: (page: Page) => void): this;

  /**
   * Emitted when new service worker is created in the context.
   */
  on(event: 'serviceworker', listener: (worker: Worker) => void): this;

  /**
   * Emitted when Browser context gets closed. This might happen because of one of the following:
   * - Browser context is closed.
   * - Browser application is closed or crashed.
   * - The [browser.close(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browserclose) method was
   *   called.
   */
  on(event: 'close', listener: () => void): this;

  /**
   * The event is emitted when a new Page is created in the BrowserContext. The page may still be loading. The event will
   * also fire for popup pages. See also
   * [page.on('popup')](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageonpopup) to receive events about
   * popups relevant to a specific page.
   * 
   * The earliest moment that page is available is when it has navigated to the initial url. For example, when opening a
   * popup with `window.open('http://example.com')`, this event will fire when the network request to "http://example.com" is
   * done and its response has started loading in the popup.
   * 
   * ```js
   * const [page] = await Promise.all([
   *   context.waitForEvent('page'),
   *   page.click('a[target=_blank]'),
   * ]);
   * console.log(await page.evaluate('location.href'));
   * ```
   * 
   * > NOTE: Use
   * [page.waitForLoadState(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagewaitforloadstate) to wait
   * until the page gets to a particular state (you should not need it in most cases).
   */
  on(event: 'page', listener: (page: Page) => void): this;

  /**
   * Emitted when new background page is created in the context.
   * 
   * > NOTE: Only works with persistent context.
   */
  once(event: 'backgroundpage', listener: (page: Page) => void): this;

  /**
   * Emitted when new service worker is created in the context.
   */
  once(event: 'serviceworker', listener: (worker: Worker) => void): this;

  /**
   * Emitted when Browser context gets closed. This might happen because of one of the following:
   * - Browser context is closed.
   * - Browser application is closed or crashed.
   * - The [browser.close(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browserclose) method was
   *   called.
   */
  once(event: 'close', listener: () => void): this;

  /**
   * The event is emitted when a new Page is created in the BrowserContext. The page may still be loading. The event will
   * also fire for popup pages. See also
   * [page.on('popup')](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageonpopup) to receive events about
   * popups relevant to a specific page.
   * 
   * The earliest moment that page is available is when it has navigated to the initial url. For example, when opening a
   * popup with `window.open('http://example.com')`, this event will fire when the network request to "http://example.com" is
   * done and its response has started loading in the popup.
   * 
   * ```js
   * const [page] = await Promise.all([
   *   context.waitForEvent('page'),
   *   page.click('a[target=_blank]'),
   * ]);
   * console.log(await page.evaluate('location.href'));
   * ```
   * 
   * > NOTE: Use
   * [page.waitForLoadState(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagewaitforloadstate) to wait
   * until the page gets to a particular state (you should not need it in most cases).
   */
  once(event: 'page', listener: (page: Page) => void): this;

  /**
   * Emitted when new background page is created in the context.
   * 
   * > NOTE: Only works with persistent context.
   */
  addListener(event: 'backgroundpage', listener: (page: Page) => void): this;

  /**
   * Emitted when new service worker is created in the context.
   */
  addListener(event: 'serviceworker', listener: (worker: Worker) => void): this;

  /**
   * Emitted when Browser context gets closed. This might happen because of one of the following:
   * - Browser context is closed.
   * - Browser application is closed or crashed.
   * - The [browser.close(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browserclose) method was
   *   called.
   */
  addListener(event: 'close', listener: () => void): this;

  /**
   * The event is emitted when a new Page is created in the BrowserContext. The page may still be loading. The event will
   * also fire for popup pages. See also
   * [page.on('popup')](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageonpopup) to receive events about
   * popups relevant to a specific page.
   * 
   * The earliest moment that page is available is when it has navigated to the initial url. For example, when opening a
   * popup with `window.open('http://example.com')`, this event will fire when the network request to "http://example.com" is
   * done and its response has started loading in the popup.
   * 
   * ```js
   * const [page] = await Promise.all([
   *   context.waitForEvent('page'),
   *   page.click('a[target=_blank]'),
   * ]);
   * console.log(await page.evaluate('location.href'));
   * ```
   * 
   * > NOTE: Use
   * [page.waitForLoadState(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagewaitforloadstate) to wait
   * until the page gets to a particular state (you should not need it in most cases).
   */
  addListener(event: 'page', listener: (page: Page) => void): this;

  /**
   * Emitted when new background page is created in the context.
   * 
   * > NOTE: Only works with persistent context.
   */
  removeListener(event: 'backgroundpage', listener: (page: Page) => void): this;

  /**
   * Emitted when new service worker is created in the context.
   */
  removeListener(event: 'serviceworker', listener: (worker: Worker) => void): this;

  /**
   * Emitted when Browser context gets closed. This might happen because of one of the following:
   * - Browser context is closed.
   * - Browser application is closed or crashed.
   * - The [browser.close(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browserclose) method was
   *   called.
   */
  removeListener(event: 'close', listener: () => void): this;

  /**
   * The event is emitted when a new Page is created in the BrowserContext. The page may still be loading. The event will
   * also fire for popup pages. See also
   * [page.on('popup')](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageonpopup) to receive events about
   * popups relevant to a specific page.
   * 
   * The earliest moment that page is available is when it has navigated to the initial url. For example, when opening a
   * popup with `window.open('http://example.com')`, this event will fire when the network request to "http://example.com" is
   * done and its response has started loading in the popup.
   * 
   * ```js
   * const [page] = await Promise.all([
   *   context.waitForEvent('page'),
   *   page.click('a[target=_blank]'),
   * ]);
   * console.log(await page.evaluate('location.href'));
   * ```
   * 
   * > NOTE: Use
   * [page.waitForLoadState(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagewaitforloadstate) to wait
   * until the page gets to a particular state (you should not need it in most cases).
   */
  removeListener(event: 'page', listener: (page: Page) => void): this;

  /**
   * Emitted when new background page is created in the context.
   * 
   * > NOTE: Only works with persistent context.
   */
  off(event: 'backgroundpage', listener: (page: Page) => void): this;

  /**
   * Emitted when new service worker is created in the context.
   */
  off(event: 'serviceworker', listener: (worker: Worker) => void): this;

  /**
   * Emitted when Browser context gets closed. This might happen because of one of the following:
   * - Browser context is closed.
   * - Browser application is closed or crashed.
   * - The [browser.close(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browserclose) method was
   *   called.
   */
  off(event: 'close', listener: () => void): this;

  /**
   * The event is emitted when a new Page is created in the BrowserContext. The page may still be loading. The event will
   * also fire for popup pages. See also
   * [page.on('popup')](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageonpopup) to receive events about
   * popups relevant to a specific page.
   * 
   * The earliest moment that page is available is when it has navigated to the initial url. For example, when opening a
   * popup with `window.open('http://example.com')`, this event will fire when the network request to "http://example.com" is
   * done and its response has started loading in the popup.
   * 
   * ```js
   * const [page] = await Promise.all([
   *   context.waitForEvent('page'),
   *   page.click('a[target=_blank]'),
   * ]);
   * console.log(await page.evaluate('location.href'));
   * ```
   * 
   * > NOTE: Use
   * [page.waitForLoadState(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagewaitforloadstate) to wait
   * until the page gets to a particular state (you should not need it in most cases).
   */
  off(event: 'page', listener: (page: Page) => void): this;

  /**
   * All existing background pages in the context.
   */
  backgroundPages(): Array<Page>;

  /**
   * Returns the newly created session.
   * @param page Page to create new session for.
   */
  newCDPSession(page: Page): Promise<CDPSession>;

  /**
   * All existing service workers in the context.
   */
  serviceWorkers(): Array<Worker>;

  /**
   * Emitted when new background page is created in the context.
   * 
   * > NOTE: Only works with persistent context.
   */
  waitForEvent(event: 'backgroundpage', optionsOrPredicate?: { predicate?: (page: Page) => boolean, timeout?: number } | ((page: Page) => boolean)): Promise<Page>;

  /**
   * Emitted when new service worker is created in the context.
   */
  waitForEvent(event: 'serviceworker', optionsOrPredicate?: { predicate?: (worker: Worker) => boolean, timeout?: number } | ((worker: Worker) => boolean)): Promise<Worker>;

  /**
   * Emitted when Browser context gets closed. This might happen because of one of the following:
   * - Browser context is closed.
   * - Browser application is closed or crashed.
   * - The [browser.close(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browserclose) method was
   *   called.
   */
  waitForEvent(event: 'close', optionsOrPredicate?: { predicate?: () => boolean, timeout?: number } | (() => boolean)): Promise<void>;

  /**
   * The event is emitted when a new Page is created in the BrowserContext. The page may still be loading. The event will
   * also fire for popup pages. See also
   * [page.on('popup')](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageonpopup) to receive events about
   * popups relevant to a specific page.
   * 
   * The earliest moment that page is available is when it has navigated to the initial url. For example, when opening a
   * popup with `window.open('http://example.com')`, this event will fire when the network request to "http://example.com" is
   * done and its response has started loading in the popup.
   * 
   * ```js
   * const [page] = await Promise.all([
   *   context.waitForEvent('page'),
   *   page.click('a[target=_blank]'),
   * ]);
   * console.log(await page.evaluate('location.href'));
   * ```
   * 
   * > NOTE: Use
   * [page.waitForLoadState(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagewaitforloadstate) to wait
   * until the page gets to a particular state (you should not need it in most cases).
   */
  waitForEvent(event: 'page', optionsOrPredicate?: { predicate?: (page: Page) => boolean, timeout?: number } | ((page: Page) => boolean)): Promise<Page>;

}

/**
 * Coverage gathers information about parts of JavaScript and CSS that were used by the page.
 * 
 * An example of using JavaScript coverage to produce Istambul report for page load:
 * 
 * ```js
 * const { chromium } = require('playwright');
 * const v8toIstanbul = require('v8-to-istanbul');
 * 
 * (async() => {
 *   const browser = await chromium.launch();
 *   const page = await browser.newPage();
 *   await page.coverage.startJSCoverage();
 *   await page.goto('https://chromium.org');
 *   const coverage = await page.coverage.stopJSCoverage();
 *   for (const entry of coverage) {
 *     const converter = new v8toIstanbul('', 0, { source: entry.source });
 *     await converter.load();
 *     converter.applyCoverage(entry.functions);
 *     console.log(JSON.stringify(converter.toIstanbul()));
 *   }
 *   await browser.close();
 * })();
 * ```
 * 
 */
export interface ChromiumCoverage {
  /**
   * Returns coverage is started
   * @param options 
   */
  startCSSCoverage(options?: {
    /**
     * Whether to reset coverage on every navigation. Defaults to `true`.
     */
    resetOnNavigation?: boolean;
  }): Promise<void>;

  /**
   * Returns coverage is started
   * 
   * > NOTE: Anonymous scripts are ones that don't have an associated url. These are scripts that are dynamically created on
   * the page using `eval` or `new Function`. If `reportAnonymousScripts` is set to `true`, anonymous scripts will have
   * `__playwright_evaluation_script__` as their URL.
   * @param options 
   */
  startJSCoverage(options?: {
    /**
     * Whether anonymous scripts generated by the page should be reported. Defaults to `false`.
     */
    reportAnonymousScripts?: boolean;

    /**
     * Whether to reset coverage on every navigation. Defaults to `true`.
     */
    resetOnNavigation?: boolean;
  }): Promise<void>;

  /**
   * Returns the array of coverage reports for all stylesheets
   * 
   * > NOTE: CSS Coverage doesn't include dynamically injected style tags without sourceURLs.
   */
  stopCSSCoverage(): Promise<Array<{
    /**
     * StyleSheet URL
     */
    url: string;

    /**
     * StyleSheet content, if available.
     */
    text?: string;

    /**
     * StyleSheet ranges that were used. Ranges are sorted and non-overlapping.
     */
    ranges: Array<{
      /**
       * A start offset in text, inclusive
       */
      start: number;

      /**
       * An end offset in text, exclusive
       */
      end: number;
    }>;
  }>>;

  /**
   * Returns the array of coverage reports for all scripts
   * 
   * > NOTE: JavaScript Coverage doesn't include anonymous scripts by default. However, scripts with sourceURLs are reported.
   */
  stopJSCoverage(): Promise<Array<{
    /**
     * Script URL
     */
    url: string;

    /**
     * Script ID
     */
    scriptId: string;

    /**
     * Script content, if applicable.
     */
    source?: string;

    /**
     * V8-specific coverage format.
     */
    functions: Array<{
      functionName: string;

      isBlockCoverage: boolean;

      ranges: Array<{
        count: number;

        startOffset: number;

        endOffset: number;
      }>;
    }>;
  }>>;
}

/**
 * [ConsoleMessage] objects are dispatched by page via the
 * [page.on('console')](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageonconsole) event.
 */
export interface ConsoleMessage {
  args(): Array<JSHandle>;

  location(): {
    /**
     * URL of the resource.
     */
    url: string;

    /**
     * 0-based line number in the resource.
     */
    lineNumber: number;

    /**
     * 0-based column number in the resource.
     */
    columnNumber: number;
  };

  text(): string;

  /**
   * One of the following values: `'log'`, `'debug'`, `'info'`, `'error'`, `'warning'`, `'dir'`, `'dirxml'`, `'table'`,
   * `'trace'`, `'clear'`, `'startGroup'`, `'startGroupCollapsed'`, `'endGroup'`, `'assert'`, `'profile'`, `'profileEnd'`,
   * `'count'`, `'timeEnd'`.
   */
  type(): string;
}

/**
 * [Dialog] objects are dispatched by page via the
 * [page.on('dialog')](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageondialog) event.
 * 
 * An example of using `Dialog` class:
 * 
 * ```js
 * const { chromium } = require('playwright');  // Or 'firefox' or 'webkit'.
 * 
 * (async () => {
 *   const browser = await chromium.launch();
 *   const page = await browser.newPage();
 *   page.on('dialog', async dialog => {
 *     console.log(dialog.message());
 *     await dialog.dismiss();
 *     await browser.close();
 *   });
 *   page.evaluate(() => alert('1'));
 * })();
 * ```
 * 
 */
export interface Dialog {
  /**
   * Returns when the dialog has been accepted.
   * @param promptText A text to enter in prompt. Does not cause any effects if the dialog's `type` is not prompt. Optional.
   */
  accept(promptText?: string): Promise<void>;

  /**
   * If dialog is prompt, returns default prompt value. Otherwise, returns empty string.
   */
  defaultValue(): string;

  /**
   * Returns when the dialog has been dismissed.
   */
  dismiss(): Promise<void>;

  /**
   * A message displayed in the dialog.
   */
  message(): string;

  /**
   * Returns dialog's type, can be one of `alert`, `beforeunload`, `confirm` or `prompt`.
   */
  type(): string;
}

/**
 * [Download] objects are dispatched by page via the
 * [page.on('download')](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageondownload) event.
 * 
 * All the downloaded files belonging to the browser context are deleted when the browser context is closed. All downloaded
 * files are deleted when the browser closes.
 * 
 * Download event is emitted once the download starts. Download path becomes available once download completes:
 * 
 * ```js
 * const [ download ] = await Promise.all([
 *   page.waitForEvent('download'), // wait for download to start
 *   page.click('a')
 * ]);
 * // wait for download to complete
 * const path = await download.path();
 * ```
 * 
 * > NOTE: Browser context **must** be created with the `acceptDownloads` set to `true` when user needs access to the
 * downloaded content. If `acceptDownloads` is not set, download events are emitted, but the actual download is not
 * performed and user has no access to the downloaded files.
 */
export interface Download {
  /**
   * Returns readable stream for current download or `null` if download failed.
   */
  createReadStream(): Promise<null|Readable>;

  /**
   * Deletes the downloaded file.
   */
  delete(): Promise<void>;

  /**
   * Returns download error if any.
   */
  failure(): Promise<null|string>;

  /**
   * Returns path to the downloaded file in case of successful download.
   */
  path(): Promise<null|string>;

  /**
   * Saves the download to a user-specified path.
   * @param path Path where the download should be saved.
   */
  saveAs(path: string): Promise<void>;

  /**
   * Returns suggested filename for this download. It is typically computed by the browser from the
   * [`Content-Disposition`](https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Content-Disposition) response header
   * or the `download` attribute. See the spec on [whatwg](https://html.spec.whatwg.org/#downloading-resources). Different
   * browsers can use different logic for computing it.
   */
  suggestedFilename(): string;

  /**
   * Returns downloaded url.
   */
  url(): string;
}

/**
 * [FileChooser] objects are dispatched by the page in the
 * [page.on('filechooser')](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageonfilechooser) event.
 * 
 * ```js
 * const [fileChooser] = await Promise.all([
 *   page.waitForEvent('filechooser'),
 *   page.click('upload')
 * ]);
 * await fileChooser.setFiles('myfile.pdf');
 * ```
 * 
 */
export interface FileChooser {
  /**
   * Returns input element associated with this file chooser.
   */
  element(): ElementHandle;

  /**
   * Returns whether this file chooser accepts multiple files.
   */
  isMultiple(): boolean;

  /**
   * Returns page this file chooser belongs to.
   */
  page(): Page;

  /**
   * Sets the value of the file input this chooser is associated with. If some of the `filePaths` are relative paths, then
   * they are resolved relative to the the current working directory. For empty array, clears the selected files.
   * @param files 
   * @param options 
   */
  setFiles(files: string|Array<string>|{
    /**
     * [File] name
     */
    name: string;

    /**
     * [File] type
     */
    mimeType: string;

    /**
     * File content
     */
    buffer: Buffer;
  }|Array<{
    /**
     * [File] name
     */
    name: string;

    /**
     * [File] type
     */
    mimeType: string;

    /**
     * File content
     */
    buffer: Buffer;
  }>, options?: {
    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You can
     * opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as navigating to
     * inaccessible pages. Defaults to `false`.
     */
    noWaitAfter?: boolean;

    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed by
     * using the
     * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
     * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
     * methods.
     */
    timeout?: number;
  }): Promise<void>;
}

/**
 * - extends: [Browser]
 * 
 * Firefox browser instance does not expose Firefox-specific features.
 */
export interface FirefoxBrowser extends Browser {

}

/**
 * Keyboard provides an api for managing a virtual keyboard. The high level api is
 * [keyboard.type(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#keyboardtype), which takes raw
 * characters and generates proper keydown, keypress/input, and keyup events on your page.
 * 
 * For finer control, you can use
 * [keyboard.down(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#keyboarddown),
 * [keyboard.up(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#keyboardup), and
 * [keyboard.insertText(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#keyboardinserttext) to manually
 * fire events as if they were generated from a real keyboard.
 * 
 * An example of holding down `Shift` in order to select and delete some text:
 * 
 * ```js
 * await page.keyboard.type('Hello World!');
 * await page.keyboard.press('ArrowLeft');
 * 
 * await page.keyboard.down('Shift');
 * for (let i = 0; i < ' World'.length; i++)
 *   await page.keyboard.press('ArrowLeft');
 * await page.keyboard.up('Shift');
 * 
 * await page.keyboard.press('Backspace');
 * // Result text will end up saying 'Hello!'
 * ```
 * 
 * An example of pressing uppercase `A`
 * 
 * ```js
 * await page.keyboard.press('Shift+KeyA');
 * // or
 * await page.keyboard.press('Shift+A');
 * ```
 * 
 * An example to trigger select-all with the keyboard
 * 
 * ```js
 * // on Windows and Linux
 * await page.keyboard.press('Control+A');
 * // on macOS
 * await page.keyboard.press('Meta+A');
 * ```
 * 
 */
export interface Keyboard {
  /**
   * Dispatches a `keydown` event.
   * 
   * `key` can specify the intended [keyboardEvent.key](https://developer.mozilla.org/en-US/docs/Web/API/KeyboardEvent/key)
   * value or a single character to generate the text for. A superset of the `key` values can be found
   * [here](https://developer.mozilla.org/en-US/docs/Web/API/KeyboardEvent/key/Key_Values). Examples of the keys are:
   * 
   * `F1` - `F12`, `Digit0`- `Digit9`, `KeyA`- `KeyZ`, `Backquote`, `Minus`, `Equal`, `Backslash`, `Backspace`, `Tab`,
   * `Delete`, `Escape`, `ArrowDown`, `End`, `Enter`, `Home`, `Insert`, `PageDown`, `PageUp`, `ArrowRight`, `ArrowUp`, etc.
   * 
   * Following modification shortcuts are also supported: `Shift`, `Control`, `Alt`, `Meta`, `ShiftLeft`.
   * 
   * Holding down `Shift` will type the text that corresponds to the `key` in the upper case.
   * 
   * If `key` is a single character, it is case-sensitive, so the values `a` and `A` will generate different respective
   * texts.
   * 
   * If `key` is a modifier key, `Shift`, `Meta`, `Control`, or `Alt`, subsequent key presses will be sent with that modifier
   * active. To release the modifier key, use
   * [keyboard.up(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#keyboardup).
   * 
   * After the key is pressed once, subsequent calls to
   * [keyboard.down(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#keyboarddown) will have
   * [repeat](https://developer.mozilla.org/en-US/docs/Web/API/KeyboardEvent/repeat) set to true. To release the key, use
   * [keyboard.up(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#keyboardup).
   * 
   * > NOTE: Modifier keys DO influence `keyboard.down`. Holding down `Shift` will type the text in upper case.
   * @param key Name of the key to press or a character to generate, such as `ArrowLeft` or `a`.
   */
  down(key: string): Promise<void>;

  /**
   * Dispatches only `input` event, does not emit the `keydown`, `keyup` or `keypress` events.
   * 
   * ```js
   * page.keyboard.insertText('嗨');
   * ```
   * 
   * > NOTE: Modifier keys DO NOT effect `keyboard.insertText`. Holding down `Shift` will not type the text in upper case.
   * @param text Sets input to the specified text value.
   */
  insertText(text: string): Promise<void>;

  /**
   * `key` can specify the intended [keyboardEvent.key](https://developer.mozilla.org/en-US/docs/Web/API/KeyboardEvent/key)
   * value or a single character to generate the text for. A superset of the `key` values can be found
   * [here](https://developer.mozilla.org/en-US/docs/Web/API/KeyboardEvent/key/Key_Values). Examples of the keys are:
   * 
   * `F1` - `F12`, `Digit0`- `Digit9`, `KeyA`- `KeyZ`, `Backquote`, `Minus`, `Equal`, `Backslash`, `Backspace`, `Tab`,
   * `Delete`, `Escape`, `ArrowDown`, `End`, `Enter`, `Home`, `Insert`, `PageDown`, `PageUp`, `ArrowRight`, `ArrowUp`, etc.
   * 
   * Following modification shortcuts are also supported: `Shift`, `Control`, `Alt`, `Meta`, `ShiftLeft`.
   * 
   * Holding down `Shift` will type the text that corresponds to the `key` in the upper case.
   * 
   * If `key` is a single character, it is case-sensitive, so the values `a` and `A` will generate different respective
   * texts.
   * 
   * Shortcuts such as `key: "Control+o"` or `key: "Control+Shift+T"` are supported as well. When speficied with the
   * modifier, modifier is pressed and being held while the subsequent key is being pressed.
   * 
   * ```js
   * const page = await browser.newPage();
   * await page.goto('https://keycode.info');
   * await page.keyboard.press('A');
   * await page.screenshot({ path: 'A.png' });
   * await page.keyboard.press('ArrowLeft');
   * await page.screenshot({ path: 'ArrowLeft.png' });
   * await page.keyboard.press('Shift+O');
   * await page.screenshot({ path: 'O.png' });
   * await browser.close();
   * ```
   * 
   * Shortcut for [keyboard.down(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#keyboarddown) and
   * [keyboard.up(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#keyboardup).
   * @param key Name of the key to press or a character to generate, such as `ArrowLeft` or `a`.
   * @param options 
   */
  press(key: string, options?: {
    /**
     * Time to wait between `keydown` and `keyup` in milliseconds. Defaults to 0.
     */
    delay?: number;
  }): Promise<void>;

  /**
   * Sends a `keydown`, `keypress`/`input`, and `keyup` event for each character in the text.
   * 
   * To press a special key, like `Control` or `ArrowDown`, use
   * [keyboard.press(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#keyboardpress).
   * 
   * ```js
   * await page.keyboard.type('Hello'); // Types instantly
   * await page.keyboard.type('World', {delay: 100}); // Types slower, like a user
   * ```
   * 
   * > NOTE: Modifier keys DO NOT effect `keyboard.type`. Holding down `Shift` will not type the text in upper case.
   * @param text A text to type into a focused element.
   * @param options 
   */
  type(text: string, options?: {
    /**
     * Time to wait between key presses in milliseconds. Defaults to 0.
     */
    delay?: number;
  }): Promise<void>;

  /**
   * Dispatches a `keyup` event.
   * @param key Name of the key to press or a character to generate, such as `ArrowLeft` or `a`.
   */
  up(key: string): Promise<void>;
}

/**
 * Playwright generates a lot of logs and they are accessible via the pluggable logger sink.
 * 
 * ```js
 * const { chromium } = require('playwright');  // Or 'firefox' or 'webkit'.
 * 
 * (async () => {
 *   const browser = await chromium.launch({
 *     logger: {
 *       isEnabled: (name, severity) => name === 'browser',
 *       log: (name, severity, message, args) => console.log(`${name} ${message}`)
 *     }
 *   });
 *   ...
 * })();
 * ```
 * 
 */
export interface Logger {
  /**
   * Determines whether sink is interested in the logger with the given name and severity.
   * @param name logger name
   * @param severity 
   */
  isEnabled(name: string, severity: "verbose"|"info"|"warning"|"error"): boolean;

  /**
   * @param name logger name
   * @param severity 
   * @param message log message format
   * @param args message arguments
   * @param hints optional formatting hints
   */
  log(name: string, severity: "verbose"|"info"|"warning"|"error", message: string|Error, args: Array<Object>, hints: {
    /**
     * Optional preferred logger color.
     */
    color?: string;
  }): void;
}

/**
 * The Mouse class operates in main-frame CSS pixels relative to the top-left corner of the viewport.
 * 
 * Every `page` object has its own Mouse, accessible with
 * [page.mouse](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagemouse).
 * 
 * ```js
 * // Using ‘page.mouse’ to trace a 100x100 square.
 * await page.mouse.move(0, 0);
 * await page.mouse.down();
 * await page.mouse.move(0, 100);
 * await page.mouse.move(100, 100);
 * await page.mouse.move(100, 0);
 * await page.mouse.move(0, 0);
 * await page.mouse.up();
 * ```
 * 
 */
export interface Mouse {
  /**
   * Shortcut for [mouse.move(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#mousemove),
   * [mouse.down(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#mousedown),
   * [mouse.up(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#mouseup).
   * @param x 
   * @param y 
   * @param options 
   */
  click(x: number, y: number, options?: {
    /**
     * Defaults to `left`.
     */
    button?: "left"|"right"|"middle";

    /**
     * defaults to 1. See [UIEvent.detail].
     */
    clickCount?: number;

    /**
     * Time to wait between `mousedown` and `mouseup` in milliseconds. Defaults to 0.
     */
    delay?: number;
  }): Promise<void>;

  /**
   * Shortcut for [mouse.move(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#mousemove),
   * [mouse.down(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#mousedown),
   * [mouse.up(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#mouseup),
   * [mouse.down(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#mousedown) and
   * [mouse.up(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#mouseup).
   * @param x 
   * @param y 
   * @param options 
   */
  dblclick(x: number, y: number, options?: {
    /**
     * Defaults to `left`.
     */
    button?: "left"|"right"|"middle";

    /**
     * Time to wait between `mousedown` and `mouseup` in milliseconds. Defaults to 0.
     */
    delay?: number;
  }): Promise<void>;

  /**
   * Dispatches a `mousedown` event.
   * @param options 
   */
  down(options?: {
    /**
     * Defaults to `left`.
     */
    button?: "left"|"right"|"middle";

    /**
     * defaults to 1. See [UIEvent.detail].
     */
    clickCount?: number;
  }): Promise<void>;

  /**
   * Dispatches a `mousemove` event.
   * @param x 
   * @param y 
   * @param options 
   */
  move(x: number, y: number, options?: {
    /**
     * defaults to 1. Sends intermediate `mousemove` events.
     */
    steps?: number;
  }): Promise<void>;

  /**
   * Dispatches a `mouseup` event.
   * @param options 
   */
  up(options?: {
    /**
     * Defaults to `left`.
     */
    button?: "left"|"right"|"middle";

    /**
     * defaults to 1. See [UIEvent.detail].
     */
    clickCount?: number;
  }): Promise<void>;
}

/**
 * Whenever the page sends a request for a network resource the following sequence of events are emitted by [Page]:
 * - [page.on('request')](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageonrequest) emitted when the
 *   request is issued by the page.
 * - [page.on('response')](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageonresponse) emitted
 *   when/if the response status and headers are received for the request.
 * - [page.on('requestfinished')](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageonrequestfinished)
 *   emitted when the response body is downloaded and the request is complete.
 * 
 * If request fails at some point, then instead of `'requestfinished'` event (and possibly instead of 'response' event),
 * the  [page.on('requestfailed')](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageonrequestfailed)
 * event is emitted.
 * 
 * > NOTE: HTTP Error responses, such as 404 or 503, are still successful responses from HTTP standpoint, so request will
 * complete with `'requestfinished'` event.
 * 
 * If request gets a 'redirect' response, the request is successfully finished with the 'requestfinished' event, and a new
 * request is  issued to a redirected url.
 */
export interface Request {
  /**
   * The method returns `null` unless this request has failed, as reported by `requestfailed` event.
   * 
   * Example of logging of all the failed requests:
   * 
   * ```js
   * page.on('requestfailed', request => {
   *   console.log(request.url() + ' ' + request.failure().errorText);
   * });
   * ```
   * 
   */
  failure(): null|{
    /**
     * Human-readable error message, e.g. `'net::ERR_FAILED'`.
     */
    errorText: string;
  };

  /**
   * Returns the [Frame] that initiated this request.
   */
  frame(): Frame;

  /**
   * An object with HTTP headers associated with the request. All header names are lower-case.
   */
  headers(): { [key: string]: string; };

  /**
   * Whether this request is driving frame's navigation.
   */
  isNavigationRequest(): boolean;

  /**
   * Request's method (GET, POST, etc.)
   */
  method(): string;

  /**
   * Request's post body, if any.
   */
  postData(): null|string;

  /**
   * Request's post body in a binary form, if any.
   */
  postDataBuffer(): null|Buffer;

  /**
   * Returns parsed request's body for `form-urlencoded` and JSON as a fallback if any.
   * 
   * When the response is `application/x-www-form-urlencoded` then a key/value object of the values will be returned.
   * Otherwise it will be parsed as JSON.
   */
  postDataJSON(): null|any;

  /**
   * Request that was redirected by the server to this one, if any.
   * 
   * When the server responds with a redirect, Playwright creates a new [Request] object. The two requests are connected by
   * `redirectedFrom()` and `redirectedTo()` methods. When multiple server redirects has happened, it is possible to
   * construct the whole redirect chain by repeatedly calling `redirectedFrom()`.
   * 
   * For example, if the website `http://example.com` redirects to `https://example.com`:
   * 
   * ```js
   * const response = await page.goto('http://example.com');
   * console.log(response.request().redirectedFrom().url()); // 'http://example.com'
   * ```
   * 
   * If the website `https://google.com` has no redirects:
   * 
   * ```js
   * const response = await page.goto('https://google.com');
   * console.log(response.request().redirectedFrom()); // null
   * ```
   * 
   */
  redirectedFrom(): null|Request;

  /**
   * New request issued by the browser if the server responded with redirect.
   * 
   * This method is the opposite of
   * [request.redirectedFrom(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#requestredirectedfrom):
   * 
   * ```js
   * console.log(request.redirectedFrom().redirectedTo() === request); // true
   * ```
   * 
   */
  redirectedTo(): null|Request;

  /**
   * Contains the request's resource type as it was perceived by the rendering engine. ResourceType will be one of the
   * following: `document`, `stylesheet`, `image`, `media`, `font`, `script`, `texttrack`, `xhr`, `fetch`, `eventsource`,
   * `websocket`, `manifest`, `other`.
   */
  resourceType(): string;

  /**
   * Returns the matching [Response] object, or `null` if the response was not received due to error.
   */
  response(): Promise<null|Response>;

  /**
   * Returns resource timing information for given request. Most of the timing values become available upon the response,
   * `responseEnd` becomes available when request finishes. Find more information at
   * [Resource Timing API](https://developer.mozilla.org/en-US/docs/Web/API/PerformanceResourceTiming).
   * 
   * ```js
   * const [request] = await Promise.all([
   *   page.waitForEvent('requestfinished'),
   *   page.goto('http://example.com')
   * ]);
   * console.log(request.timing());
   * ```
   * 
   */
  timing(): {
    /**
     * Request start time in milliseconds elapsed since January 1, 1970 00:00:00 UTC
     */
    startTime: number;

    /**
     * Time immediately before the browser starts the domain name lookup for the resource. The value is given in milliseconds
     * relative to `startTime`, -1 if not available.
     */
    domainLookupStart: number;

    /**
     * Time immediately after the browser starts the domain name lookup for the resource. The value is given in milliseconds
     * relative to `startTime`, -1 if not available.
     */
    domainLookupEnd: number;

    /**
     * Time immediately before the user agent starts establishing the connection to the server to retrieve the resource. The
     * value is given in milliseconds relative to `startTime`, -1 if not available.
     */
    connectStart: number;

    /**
     * Time immediately before the browser starts the handshake process to secure the current connection. The value is given in
     * milliseconds relative to `startTime`, -1 if not available.
     */
    secureConnectionStart: number;

    /**
     * Time immediately before the user agent starts establishing the connection to the server to retrieve the resource. The
     * value is given in milliseconds relative to `startTime`, -1 if not available.
     */
    connectEnd: number;

    /**
     * Time immediately before the browser starts requesting the resource from the server, cache, or local resource. The value
     * is given in milliseconds relative to `startTime`, -1 if not available.
     */
    requestStart: number;

    /**
     * Time immediately after the browser starts requesting the resource from the server, cache, or local resource. The value
     * is given in milliseconds relative to `startTime`, -1 if not available.
     */
    responseStart: number;

    /**
     * Time immediately after the browser receives the last byte of the resource or immediately before the transport connection
     * is closed, whichever comes first. The value is given in milliseconds relative to `startTime`, -1 if not available.
     */
    responseEnd: number;
  };

  /**
   * URL of the request.
   */
  url(): string;
}

/**
 * [Response] class represents responses which are received by page.
 */
export interface Response {
  /**
   * Returns the buffer with response body.
   */
  body(): Promise<Buffer>;

  /**
   * Waits for this response to finish, returns failure error if request failed.
   */
  finished(): Promise<null|Error>;

  /**
   * Returns the [Frame] that initiated this response.
   */
  frame(): Frame;

  /**
   * Returns the object with HTTP headers associated with the response. All header names are lower-case.
   */
  headers(): { [key: string]: string; };

  /**
   * Returns the JSON representation of response body.
   * 
   * This method will throw if the response body is not parsable via `JSON.parse`.
   */
  json(): Promise<Serializable>;

  /**
   * Contains a boolean stating whether the response was successful (status in the range 200-299) or not.
   */
  ok(): boolean;

  /**
   * Returns the matching [Request] object.
   */
  request(): Request;

  /**
   * Contains the status code of the response (e.g., 200 for a success).
   */
  status(): number;

  /**
   * Contains the status text of the response (e.g. usually an "OK" for a success).
   */
  statusText(): string;

  /**
   * Returns the text representation of response body.
   */
  text(): Promise<string>;

  /**
   * Contains the URL of the response.
   */
  url(): string;
}

/**
 * Whenever a network route is set up with
 * [page.route(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageroute) or
 * [browserContext.route(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextroute), the
 * `Route` object allows to handle the route.
 */
export interface Route {
  /**
   * Aborts the route's request.
   * @param errorCode Optional error code. Defaults to `failed`, could be one of the following: - `'aborted'` - An operation was aborted (due to user action)
   * - `'accessdenied'` - Permission to access a resource, other than the network, was denied
   * - `'addressunreachable'` - The IP address is unreachable. This usually means that there is no route to the specified
   *   host or network.
   * - `'blockedbyclient'` - The client chose to block the request.
   * - `'blockedbyresponse'` - The request failed because the response was delivered along with requirements which are not
   *   met ('X-Frame-Options' and 'Content-Security-Policy' ancestor checks, for instance).
   * - `'connectionaborted'` - A connection timed out as a result of not receiving an ACK for data sent.
   * - `'connectionclosed'` - A connection was closed (corresponding to a TCP FIN).
   * - `'connectionfailed'` - A connection attempt failed.
   * - `'connectionrefused'` - A connection attempt was refused.
   * - `'connectionreset'` - A connection was reset (corresponding to a TCP RST).
   * - `'internetdisconnected'` - The Internet connection has been lost.
   * - `'namenotresolved'` - The host name could not be resolved.
   * - `'timedout'` - An operation timed out.
   * - `'failed'` - A generic failure occurred.
   */
  abort(errorCode?: string): Promise<void>;

  /**
   * Continues route's request with optional overrides.
   * 
   * ```js
   * await page.route('**\/*', (route, request) => {
   *   // Override headers
   *   const headers = {
   *     ...request.headers(),
   *     foo: 'bar', // set "foo" header
   *     origin: undefined, // remove "origin" header
   *   };
   *   route.continue({headers});
   * });
   * ```
   * 
   * @param options 
   */
  continue(options?: {
    /**
     * If set changes the request HTTP headers. Header values will be converted to a string.
     */
    headers?: { [key: string]: string; };

    /**
     * If set changes the request method (e.g. GET or POST)
     */
    method?: string;

    /**
     * If set changes the post data of request
     */
    postData?: string|Buffer;

    /**
     * If set changes the request URL. New URL must have same protocol as original one.
     */
    url?: string;
  }): Promise<void>;

  /**
   * Fulfills route's request with given response.
   * 
   * An example of fulfilling all requests with 404 responses:
   * 
   * ```js
   * await page.route('**\/*', route => {
   *   route.fulfill({
   *     status: 404,
   *     contentType: 'text/plain',
   *     body: 'Not Found!'
   *   });
   * });
   * ```
   * 
   * An example of serving static file:
   * 
   * ```js
   * await page.route('**\/xhr_endpoint', route => route.fulfill({ path: 'mock_data.json' }));
   * ```
   * 
   * @param options 
   */
  fulfill(options?: {
    /**
     * Response body.
     */
    body?: string|Buffer;

    /**
     * If set, equals to setting `Content-Type` response header.
     */
    contentType?: string;

    /**
     * Response headers. Header values will be converted to a string.
     */
    headers?: { [key: string]: string; };

    /**
     * File path to respond with. The content type will be inferred from file extension. If `path` is a relative path, then it
     * is resolved relative to the current working directory.
     */
    path?: string;

    /**
     * Response status code, defaults to `200`.
     */
    status?: number;
  }): Promise<void>;

  /**
   * A request to be routed.
   */
  request(): Request;
}

/**
 * Selectors can be used to install custom selector engines. See
 * [Working with selectors](https://github.com/microsoft/playwright/blob/master/docs/selectors.md#working-with-selectors) for more information.
 */
export interface Selectors {
  /**
   * An example of registering selector engine that queries elements based on a tag name:
   * 
   * ```js
   * const { selectors, firefox } = require('playwright');  // Or 'chromium' or 'webkit'.
   * 
   * (async () => {
   *   // Must be a function that evaluates to a selector engine instance.
   *   const createTagNameEngine = () => ({
   *     // Returns the first element matching given selector in the root's subtree.
   *     query(root, selector) {
   *       return root.querySelector(selector);
   *     },
   * 
   *     // Returns all elements matching given selector in the root's subtree.
   *     queryAll(root, selector) {
   *       return Array.from(root.querySelectorAll(selector));
   *     }
   *   });
   * 
   *   // Register the engine. Selectors will be prefixed with "tag=".
   *   await selectors.register('tag', createTagNameEngine);
   * 
   *   const browser = await firefox.launch();
   *   const page = await browser.newPage();
   *   await page.setContent(`<div><button>Click me</button></div>`);
   * 
   *   // Use the selector prefixed with its name.
   *   const button = await page.$('tag=button');
   *   // Combine it with other selector engines.
   *   await page.click('tag=div >> text="Click me"');
   *   // Can use it in any methods supporting selectors.
   *   const buttonCount = await page.$$eval('tag=button', buttons => buttons.length);
   * 
   *   await browser.close();
   * })();
   * ```
   * 
   * @param name Name that is used in selectors as a prefix, e.g. `{name: 'foo'}` enables `foo=myselectorbody` selectors. May only contain `[a-zA-Z0-9_]` characters.
   * @param script Script that evaluates to a selector engine instance.
   * @param options 
   */
  register(name: string, script: Function|string|{
    /**
     * Path to the JavaScript file. If `path` is a relative path, then it is resolved relative to the current working
     * directory. Optional.
     */
    path?: string;

    /**
     * Raw script content. Optional.
     */
    content?: string;
  }, options?: {
    /**
     * Whether to run this selector engine in isolated JavaScript environment. This environment has access to the same DOM, but
     * not any JavaScript objects from the frame's scripts. Defaults to `false`. Note that running as a content script is not
     * guaranteed when this engine is used together with other registered engines.
     */
    contentScript?: boolean;
  }): Promise<void>;
}

/**
 * The Touchscreen class operates in main-frame CSS pixels relative to the top-left corner of the viewport. Methods on the
 * touchscreen can only be used in browser contexts that have been intialized with `hasTouch` set to true.
 */
export interface Touchscreen {
  /**
   * Dispatches a `touchstart` and `touchend` event with a single touch at the position (`x`,`y`).
   * @param x 
   * @param y 
   */
  tap(x: number, y: number): Promise<void>;
}

/**
 * When browser context is created with the `videosPath` option, each page has a video object associated with it.
 * 
 * ```js
 * console.log(await page.video().path());
 * ```
 * 
 */
export interface Video {
  /**
   * Returns the file system path this video will be recorded to. The video is guaranteed to be written to the filesystem
   * upon closing the browser context.
   */
  path(): Promise<string>;
}

/**
 * - extends: [Browser]
 * 
 * WebKit browser instance does not expose WebKit-specific features.
 */
export interface WebKitBrowser extends Browser {

}

/**
 * The [WebSocket] class represents websocket connections in the page.
 */
export interface WebSocket {
  /**
   * Fired when the websocket closes.
   */
  on(event: 'close', listener: () => void): this;

  /**
   * Fired when the websocket recieves a frame.
   */
  on(event: 'framereceived', listener: (data: {
  /**
   * frame payload
   */
  payload: string|Buffer;
}) => void): this;

  /**
   * Fired when the websocket sends a frame.
   */
  on(event: 'framesent', listener: (data: {
  /**
   * frame payload
   */
  payload: string|Buffer;
}) => void): this;

  /**
   * Fired when the websocket has an error.
   */
  on(event: 'socketerror', listener: (string: String) => void): this;

  /**
   * Fired when the websocket closes.
   */
  once(event: 'close', listener: () => void): this;

  /**
   * Fired when the websocket recieves a frame.
   */
  once(event: 'framereceived', listener: (data: {
  /**
   * frame payload
   */
  payload: string|Buffer;
}) => void): this;

  /**
   * Fired when the websocket sends a frame.
   */
  once(event: 'framesent', listener: (data: {
  /**
   * frame payload
   */
  payload: string|Buffer;
}) => void): this;

  /**
   * Fired when the websocket has an error.
   */
  once(event: 'socketerror', listener: (string: String) => void): this;

  /**
   * Fired when the websocket closes.
   */
  addListener(event: 'close', listener: () => void): this;

  /**
   * Fired when the websocket recieves a frame.
   */
  addListener(event: 'framereceived', listener: (data: {
  /**
   * frame payload
   */
  payload: string|Buffer;
}) => void): this;

  /**
   * Fired when the websocket sends a frame.
   */
  addListener(event: 'framesent', listener: (data: {
  /**
   * frame payload
   */
  payload: string|Buffer;
}) => void): this;

  /**
   * Fired when the websocket has an error.
   */
  addListener(event: 'socketerror', listener: (string: String) => void): this;

  /**
   * Fired when the websocket closes.
   */
  removeListener(event: 'close', listener: () => void): this;

  /**
   * Fired when the websocket recieves a frame.
   */
  removeListener(event: 'framereceived', listener: (data: {
  /**
   * frame payload
   */
  payload: string|Buffer;
}) => void): this;

  /**
   * Fired when the websocket sends a frame.
   */
  removeListener(event: 'framesent', listener: (data: {
  /**
   * frame payload
   */
  payload: string|Buffer;
}) => void): this;

  /**
   * Fired when the websocket has an error.
   */
  removeListener(event: 'socketerror', listener: (string: String) => void): this;

  /**
   * Fired when the websocket closes.
   */
  off(event: 'close', listener: () => void): this;

  /**
   * Fired when the websocket recieves a frame.
   */
  off(event: 'framereceived', listener: (data: {
  /**
   * frame payload
   */
  payload: string|Buffer;
}) => void): this;

  /**
   * Fired when the websocket sends a frame.
   */
  off(event: 'framesent', listener: (data: {
  /**
   * frame payload
   */
  payload: string|Buffer;
}) => void): this;

  /**
   * Fired when the websocket has an error.
   */
  off(event: 'socketerror', listener: (string: String) => void): this;

  /**
   * Indicates that the web socket has been closed.
   */
  isClosed(): boolean;

  /**
   * Contains the URL of the WebSocket.
   */
  url(): string;

  /**
   * Fired when the websocket closes.
   */
  waitForEvent(event: 'close', optionsOrPredicate?: { predicate?: () => boolean, timeout?: number } | (() => boolean)): Promise<void>;

  /**
   * Fired when the websocket recieves a frame.
   */
  waitForEvent(event: 'framereceived', optionsOrPredicate?: { predicate?: (data: {
  /**
   * frame payload
   */
  payload: string|Buffer;
}) => boolean, timeout?: number } | ((data: {
  /**
   * frame payload
   */
  payload: string|Buffer;
}) => boolean)): Promise<{
  /**
   * frame payload
   */
  payload: string|Buffer;
}>;

  /**
   * Fired when the websocket sends a frame.
   */
  waitForEvent(event: 'framesent', optionsOrPredicate?: { predicate?: (data: {
  /**
   * frame payload
   */
  payload: string|Buffer;
}) => boolean, timeout?: number } | ((data: {
  /**
   * frame payload
   */
  payload: string|Buffer;
}) => boolean)): Promise<{
  /**
   * frame payload
   */
  payload: string|Buffer;
}>;

  /**
   * Fired when the websocket has an error.
   */
  waitForEvent(event: 'socketerror', optionsOrPredicate?: { predicate?: (string: String) => boolean, timeout?: number } | ((string: String) => boolean)): Promise<String>;

}

export interface BrowserContextOptions {
  /**
   * Whether to automatically download all the attachments. Defaults to `false` where all the downloads are canceled.
   */
  acceptDownloads?: boolean;

  /**
   * Toggles bypassing page's Content-Security-Policy.
   */
  bypassCSP?: boolean;

  /**
   * Emulates `'prefers-colors-scheme'` media feature, supported values are `'light'`, `'dark'`, `'no-preference'`. See
   * [page.emulateMedia(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pageemulatemedia) for more
   * details. Defaults to '`light`'.
   */
  colorScheme?: "light"|"dark"|"no-preference";

  /**
   * Specify device scale factor (can be thought of as dpr). Defaults to `1`.
   */
  deviceScaleFactor?: number;

  /**
   * An object containing additional HTTP headers to be sent with every request. All header values must be strings.
   */
  extraHTTPHeaders?: { [key: string]: string; };

  geolocation?: Geolocation;

  /**
   * Specifies if viewport supports touch events. Defaults to false.
   */
  hasTouch?: boolean;

  /**
   * Credentials for [HTTP authentication](https://developer.mozilla.org/en-US/docs/Web/HTTP/Authentication).
   */
  httpCredentials?: HTTPCredentials;

  /**
   * Whether to ignore HTTPS errors during navigation. Defaults to `false`.
   */
  ignoreHTTPSErrors?: boolean;

  /**
   * Whether the `meta viewport` tag is taken into account and touch events are enabled. Defaults to `false`. Not supported
   * in Firefox.
   */
  isMobile?: boolean;

  /**
   * Whether or not to enable JavaScript in the context. Defaults to `true`.
   */
  javaScriptEnabled?: boolean;

  /**
   * Specify user locale, for example `en-GB`, `de-DE`, etc. Locale will affect `navigator.language` value, `Accept-Language`
   * request header value as well as number and date formatting rules.
   */
  locale?: string;

  /**
   * Logger sink for Playwright logging.
   */
  logger?: Logger;

  /**
   * Whether to emulate network being offline. Defaults to `false`.
   */
  offline?: boolean;

  /**
   * A list of permissions to grant to all pages in this context. See
   * [browserContext.grantPermissions(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextgrantpermissions)
   * for more details.
   */
  permissions?: Array<string>;

  /**
   * Network proxy settings to use with this context. Note that browser needs to be launched with the global proxy for this
   * option to work. If all contexts override the proxy, global proxy will be never used and can be any string, for example
   * `launch({ proxy: { server: 'per-context' } })`.
   */
  proxy?: {
    /**
     * Proxy to be used for all requests. HTTP and SOCKS proxies are supported, for example `http://myproxy.com:3128` or
     * `socks5://myproxy.com:3128`. Short form `myproxy.com:3128` is considered an HTTP proxy.
     */
    server: string;

    /**
     * Optional coma-separated domains to bypass proxy, for example `".com, chromium.org, .domain.com"`.
     */
    bypass?: string;

    /**
     * Optional username to use if HTTP proxy requires authentication.
     */
    username?: string;

    /**
     * Optional password to use if HTTP proxy requires authentication.
     */
    password?: string;
  };

  /**
   * Enables [HAR](http://www.softwareishard.com/blog/har-12-spec) recording for all pages into `recordHar.path` file. If not
   * specified, the HAR is not recorded. Make sure to await
   * [browserContext.close(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextclose) for the
   * HAR to be saved.
   */
  recordHar?: {
    /**
     * Optional setting to control whether to omit request content from the HAR. Defaults to `false`.
     */
    omitContent?: boolean;

    /**
     * Path on the filesystem to write the HAR file to.
     */
    path: string;
  };

  /**
   * Enables video recording for all pages into `recordVideo.dir` directory. If not specified videos are not recorded. Make
   * sure to await
   * [browserContext.close(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextclose) for
   * videos to be saved.
   */
  recordVideo?: {
    /**
     * Path to the directory to put videos into.
     */
    dir: string;

    /**
     * Optional dimensions of the recorded videos. If not specified the size will be equal to `viewport`. If `viewport` is not
     * configured explicitly the video size defaults to 1280x720. Actual picture of each page will be scaled down if necessary
     * to fit the specified size.
     */
    size?: {
      /**
       * Video frame width.
       */
      width: number;

      /**
       * Video frame height.
       */
      height: number;
    };
  };

  /**
   * Populates context with given storage state. This method can be used to initialize context with logged-in information
   * obtained via
   * [browserContext.storageState(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextstoragestate).
   * Either a path to the file with saved storage, or an object with the following fields:
   */
  storageState?: string|{
    /**
     * Optional cookies to set for context
     */
    cookies?: Array<{
      name: string;

      value: string;

      /**
       * Optional either url or domain / path are required
       */
      url?: string;

      /**
       * Optional either url or domain / path are required
       */
      domain?: string;

      /**
       * Optional either url or domain / path are required
       */
      path?: string;

      /**
       * Optional Unix time in seconds.
       */
      expires?: number;

      /**
       * Optional httpOnly flag
       */
      httpOnly?: boolean;

      /**
       * Optional secure flag
       */
      secure?: boolean;

      /**
       * Optional sameSite flag
       */
      sameSite?: "Strict"|"Lax"|"None";
    }>;

    /**
     * Optional localStorage to set for context
     */
    origins?: Array<{
      origin: string;

      localStorage: Array<{
        name: string;

        value: string;
      }>;
    }>;
  };

  /**
   * Changes the timezone of the context. See
   * [ICU's metaZones.txt](https://cs.chromium.org/chromium/src/third_party/icu/source/data/misc/metaZones.txt?rcl=faee8bc70570192d82d2978a71e2a615788597d1)
   * for a list of supported timezone IDs.
   */
  timezoneId?: string;

  /**
   * Specific user agent to use in this context.
   */
  userAgent?: string;

  /**
   * **DEPRECATED** Use `recordVideo` instead.
   * @deprecated
   */
  videoSize?: {
    /**
     * Video frame width.
     */
    width: number;

    /**
     * Video frame height.
     */
    height: number;
  };

  /**
   * **DEPRECATED** Use `recordVideo` instead.
   * @deprecated
   */
  videosPath?: string;

  /**
   * Sets a consistent viewport for each page. Defaults to an 1280x720 viewport. `null` disables the default viewport.
   */
  viewport?: null|ViewportSize;
}

export interface ViewportSize {
  /**
   * page width in pixels.
   */
  width: number;

  /**
   * page height in pixels.
   */
  height: number;
}

export interface HTTPCredentials {
  username: string;

  password: string;
}

export interface Geolocation {
  /**
   * Latitude between -90 and 90.
   */
  latitude: number;

  /**
   * Longitude between -180 and 180.
   */
  longitude: number;

  /**
   * Non-negative accuracy value. Defaults to `0`.
   */
  accuracy?: number;
}

export interface LaunchOptions {
  /**
   * Additional arguments to pass to the browser instance. The list of Chromium flags can be found
   * [here](http://peter.sh/experiments/chromium-command-line-switches/).
   */
  args?: Array<string>;

  /**
   * Enable Chromium sandboxing. Defaults to `false`.
   */
  chromiumSandbox?: boolean;

  /**
   * **Chromium-only** Whether to auto-open a Developer Tools panel for each tab. If this option is `true`, the `headless`
   * option will be set `false`.
   */
  devtools?: boolean;

  /**
   * If specified, accepted downloads are downloaded into this directory. Otherwise, temporary directory is created and is
   * deleted when browser is closed.
   */
  downloadsPath?: string;

  /**
   * Specify environment variables that will be visible to the browser. Defaults to `process.env`.
   */
  env?: { [key: string]: string|number|boolean; };

  /**
   * Path to a browser executable to run instead of the bundled one. If `executablePath` is a relative path, then it is
   * resolved relative to the current working directory. Note that Playwright only works with the bundled Chromium, Firefox
   * or WebKit, use at your own risk.
   */
  executablePath?: string;

  /**
   * Firefox user preferences. Learn more about the Firefox user preferences at
   * [`about:config`](https://support.mozilla.org/en-US/kb/about-config-editor-firefox).
   */
  firefoxUserPrefs?: { [key: string]: string|number|boolean; };

  /**
   * Close the browser process on SIGHUP. Defaults to `true`.
   */
  handleSIGHUP?: boolean;

  /**
   * Close the browser process on Ctrl-C. Defaults to `true`.
   */
  handleSIGINT?: boolean;

  /**
   * Close the browser process on SIGTERM. Defaults to `true`.
   */
  handleSIGTERM?: boolean;

  /**
   * Whether to run browser in headless mode. More details for
   * [Chromium](https://developers.google.com/web/updates/2017/04/headless-chrome) and
   * [Firefox](https://developer.mozilla.org/en-US/docs/Mozilla/Firefox/Headless_mode). Defaults to `true` unless the
   * `devtools` option is `true`.
   */
  headless?: boolean;

  /**
   * If `true`, Playwright does not pass its own configurations args and only uses the ones from `args`. If an array is
   * given, then filters out the given default arguments. Dangerous option; use with care. Defaults to `false`.
   */
  ignoreDefaultArgs?: boolean|Array<string>;

  /**
   * Logger sink for Playwright logging.
   */
  logger?: Logger;

  /**
   * Network proxy settings.
   */
  proxy?: {
    /**
     * Proxy to be used for all requests. HTTP and SOCKS proxies are supported, for example `http://myproxy.com:3128` or
     * `socks5://myproxy.com:3128`. Short form `myproxy.com:3128` is considered an HTTP proxy.
     */
    server: string;

    /**
     * Optional coma-separated domains to bypass proxy, for example `".com, chromium.org, .domain.com"`.
     */
    bypass?: string;

    /**
     * Optional username to use if HTTP proxy requires authentication.
     */
    username?: string;

    /**
     * Optional password to use if HTTP proxy requires authentication.
     */
    password?: string;
  };

  /**
   * Slows down Playwright operations by the specified amount of milliseconds. Useful so that you can see what is going on.
   */
  slowMo?: number;

  /**
   * Maximum time in milliseconds to wait for the browser instance to start. Defaults to `30000` (30 seconds). Pass `0` to
   * disable timeout.
   */
  timeout?: number;
}

export interface ConnectOptions {
  /**
   * A browser websocket endpoint to connect to.
   */
  wsEndpoint: string;

  /**
   * Slows down Playwright operations by the specified amount of milliseconds. Useful so that you can see what is going on.
   * Defaults to 0.
   */
  slowMo?: number;

  /**
   * Logger sink for Playwright logging. Optional.
   */
  logger?: Logger;

  /**
   * Maximum time in milliseconds to wait for the connection to be established. Defaults to `30000` (30 seconds). Pass `0` to
   * disable timeout.
   */
  timeout?: number;
}

interface ElementHandleWaitForSelectorOptions {
  /**
   * Defaults to `'visible'`. Can be either:
   * - `'attached'` - wait for element to be present in DOM.
   * - `'detached'` - wait for element to not be present in DOM.
   * - `'visible'` - wait for element to have non-empty bounding box and no `visibility:hidden`. Note that element without
   *   any content or with `display:none` has an empty bounding box and is not considered visible.
   * - `'hidden'` - wait for element to be either detached from DOM, or have an empty bounding box or `visibility:hidden`.
   *   This is opposite to the `'visible'` option.
   */
  state?: "attached"|"detached"|"visible"|"hidden";

  /**
   * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed by
   * using the
   * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
   * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
   * methods.
   */
  timeout?: number;
}

export interface Cookie {
  name: string;

  value: string;

  domain: string;

  path: string;

  /**
   * Unix time in seconds.
   */
  expires: number;

  httpOnly: boolean;

  secure: boolean;

  sameSite: "Strict"|"Lax"|"None";
}

interface PageWaitForSelectorOptions {
  /**
   * Defaults to `'visible'`. Can be either:
   * - `'attached'` - wait for element to be present in DOM.
   * - `'detached'` - wait for element to not be present in DOM.
   * - `'visible'` - wait for element to have non-empty bounding box and no `visibility:hidden`. Note that element without
   *   any content or with `display:none` has an empty bounding box and is not considered visible.
   * - `'hidden'` - wait for element to be either detached from DOM, or have an empty bounding box or `visibility:hidden`.
   *   This is opposite to the `'visible'` option.
   */
  state?: "attached"|"detached"|"visible"|"hidden";

  /**
   * Maximum time in milliseconds, defaults to 30 seconds, pass `0` to disable timeout. The default value can be changed by
   * using the
   * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout)
   * or [page.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#pagesetdefaulttimeout)
   * methods.
   */
  timeout?: number;
}

interface PageWaitForFunctionOptions {
  /**
   * If `polling` is `'raf'`, then `pageFunction` is constantly executed in `requestAnimationFrame` callback. If `polling` is
   * a number, then it is treated as an interval in milliseconds at which the function would be executed. Defaults to `raf`.
   */
  polling?: number|"raf";

  /**
   * maximum time to wait for in milliseconds. Defaults to `30000` (30 seconds). Pass `0` to disable timeout. The default
   * value can be changed by using the
   * [browserContext.setDefaultTimeout(…)](https://github.com/microsoft/playwright/blob/master/docs/api.md#browsercontextsetdefaulttimeout).
   */
  timeout?: number;
}

type Devices = {
  "Blackberry PlayBook": DeviceDescriptor;
  "Blackberry PlayBook landscape": DeviceDescriptor;
  "BlackBerry Z30": DeviceDescriptor;
  "BlackBerry Z30 landscape": DeviceDescriptor;
  "Galaxy Note 3": DeviceDescriptor;
  "Galaxy Note 3 landscape": DeviceDescriptor;
  "Galaxy Note II": DeviceDescriptor;
  "Galaxy Note II landscape": DeviceDescriptor;
  "Galaxy S III": DeviceDescriptor;
  "Galaxy S III landscape": DeviceDescriptor;
  "Galaxy S5": DeviceDescriptor;
  "Galaxy S5 landscape": DeviceDescriptor;
  "iPad (gen 6)": DeviceDescriptor;
  "iPad (gen 6) landscape": DeviceDescriptor;
  "iPad (gen 7)": DeviceDescriptor;
  "iPad (gen 7) landscape": DeviceDescriptor;
  "iPad Mini": DeviceDescriptor;
  "iPad Mini landscape": DeviceDescriptor;
  "iPad Pro 11": DeviceDescriptor;
  "iPad Pro 11 landscape": DeviceDescriptor;
  "iPhone 6": DeviceDescriptor;
  "iPhone 6 landscape": DeviceDescriptor;
  "iPhone 6 Plus": DeviceDescriptor;
  "iPhone 6 Plus landscape": DeviceDescriptor;
  "iPhone 7": DeviceDescriptor;
  "iPhone 7 landscape": DeviceDescriptor;
  "iPhone 7 Plus": DeviceDescriptor;
  "iPhone 7 Plus landscape": DeviceDescriptor;
  "iPhone 8": DeviceDescriptor;
  "iPhone 8 landscape": DeviceDescriptor;
  "iPhone 8 Plus": DeviceDescriptor;
  "iPhone 8 Plus landscape": DeviceDescriptor;
  "iPhone SE": DeviceDescriptor;
  "iPhone SE landscape": DeviceDescriptor;
  "iPhone X": DeviceDescriptor;
  "iPhone X landscape": DeviceDescriptor;
  "iPhone XR": DeviceDescriptor;
  "iPhone XR landscape": DeviceDescriptor;
  "iPhone 11": DeviceDescriptor;
  "iPhone 11 landscape": DeviceDescriptor;
  "iPhone 11 Pro": DeviceDescriptor;
  "iPhone 11 Pro landscape": DeviceDescriptor;
  "iPhone 11 Pro Max": DeviceDescriptor;
  "iPhone 11 Pro Max landscape": DeviceDescriptor;
  "JioPhone 2": DeviceDescriptor;
  "JioPhone 2 landscape": DeviceDescriptor;
  "Kindle Fire HDX": DeviceDescriptor;
  "Kindle Fire HDX landscape": DeviceDescriptor;
  "LG Optimus L70": DeviceDescriptor;
  "LG Optimus L70 landscape": DeviceDescriptor;
  "Microsoft Lumia 550": DeviceDescriptor;
  "Microsoft Lumia 550 landscape": DeviceDescriptor;
  "Microsoft Lumia 950": DeviceDescriptor;
  "Microsoft Lumia 950 landscape": DeviceDescriptor;
  "Nexus 10": DeviceDescriptor;
  "Nexus 10 landscape": DeviceDescriptor;
  "Nexus 4": DeviceDescriptor;
  "Nexus 4 landscape": DeviceDescriptor;
  "Nexus 5": DeviceDescriptor;
  "Nexus 5 landscape": DeviceDescriptor;
  "Nexus 5X": DeviceDescriptor;
  "Nexus 5X landscape": DeviceDescriptor;
  "Nexus 6": DeviceDescriptor;
  "Nexus 6 landscape": DeviceDescriptor;
  "Nexus 6P": DeviceDescriptor;
  "Nexus 6P landscape": DeviceDescriptor;
  "Nexus 7": DeviceDescriptor;
  "Nexus 7 landscape": DeviceDescriptor;
  "Nokia Lumia 520": DeviceDescriptor;
  "Nokia Lumia 520 landscape": DeviceDescriptor;
  "Nokia N9": DeviceDescriptor;
  "Nokia N9 landscape": DeviceDescriptor;
  "Pixel 2": DeviceDescriptor;
  "Pixel 2 landscape": DeviceDescriptor;
  "Pixel 2 XL": DeviceDescriptor;
  "Pixel 2 XL landscape": DeviceDescriptor;
  [key: string]: DeviceDescriptor;
}
